# Import necessary PyQt and QGIS classes
from qgis.PyQt.QtWidgets import (QDialog, QPushButton, QLineEdit, 
                                 QVBoxLayout, QHBoxLayout, QMessageBox, QLabel)
from qgis.PyQt.QtGui import QColor
from qgis.core import (QgsFeature, QgsGeometry, QgsPointXY, QgsProject,
                       QgsCoordinateReferenceSystem, QgsCoordinateTransform,
                       QgsWkbTypes)
from qgis.gui import QgsRubberBand

class LatLongDigitizerDialog(QDialog):
    """
    This is the custom QDialog window for the plugin.
    """

    def __init__(self, layer, iface, parent=None):
        """
        Constructor. Pass in the active layer and the QGIS interface (iface).
        """
        super(LatLongDigitizerDialog, self).__init__(parent)
        
        # Store references to the layer and iface
        self.layer = layer
        self.iface = iface
        
        # This list will store the vertices as QgsPointXY objects
        self.vertices = []  
        self.first_point = None
        
        # --- Setup the Rubber Band for live preview ---
        # Apply int() wrapper for older QGIS versions
        self.rubberBand = QgsRubberBand(self.iface.mapCanvas(), int(QgsWkbTypes.LineString))
        self.rubberBand.setColor(QColor(255, 0, 0, 150)) # Set color to semi-transparent red
        self.rubberBand.setWidth(2)

        # --- Define Coordinate Reference Systems (CRS) ---
        self.crs_input = QgsCoordinateReferenceSystem("EPSG:4326")
        self.crs_layer = self.layer.crs()
        self.transform = QgsCoordinateTransform(self.crs_input, 
                                                self.crs_layer, 
                                                QgsProject.instance())

        # --- Setup the GUI ---
        self.setWindowTitle("Digitize with Lat/Long")
        layout = QVBoxLayout()
        
        # Add a label for instructions
        info_label = QLabel("Enter coordinates as 'Latitude, Longitude'")
        layout.addWidget(info_label)

        # Add the text input box
        self.coord_input = QLineEdit()
        self.coord_input.setPlaceholderText("e.g., 40.7128, -74.0060")
        layout.addWidget(self.coord_input)
        
        # Horizontal layout for Add/Undo buttons
        button_layout = QHBoxLayout()
        self.btn_add_point = QPushButton("Add Point")
        self.btn_undo = QPushButton("Undo Last Point")
        button_layout.addWidget(self.btn_add_point)
        button_layout.addWidget(self.btn_undo)
        layout.addLayout(button_layout)
        
        # Add the "Close Polygon" button
        self.btn_close_polygon = QPushButton("Close Polygon and Save")
        layout.addWidget(self.btn_close_polygon)
        
        # Add the "Cancel" button
        self.btn_cancel = QPushButton("Cancel")
        layout.addWidget(self.btn_cancel)
        
        self.setLayout(layout)
        
        # --- Connect button clicks to methods ---
        self.btn_add_point.clicked.connect(self.add_point)
        self.btn_undo.clicked.connect(self.undo_point) 
        self.btn_close_polygon.clicked.connect(self.close_polygon)
        self.btn_cancel.clicked.connect(self.reject) # 'reject' closes the dialog

    def update_rubber_band(self):
        """Updates the temporary polygon preview on the map."""
        
        # Clear any existing rubber band
        # Apply int() wrapper for older QGIS versions
        self.rubberBand.reset(int(QgsWkbTypes.LineString))
        
        if len(self.vertices) < 1:
            return # Nothing to draw
        
        # Create a geometry from the points we have
        geom = QgsGeometry.fromPolylineXY(self.vertices)
        self.rubberBand.setToGeometry(geom, self.layer)
        

    def add_point(self):
        """
        Reads text from the QLineEdit, parses it, transforms it,
        and adds it to the vertices list.
        """
        text = self.coord_input.text()
        if not text:
            QMessageBox.warning(self, "Empty Input", "Please enter coordinates.")
            return

        try:
            parts = text.split(',')
            if len(parts) != 2:
                raise ValueError("Input must have two parts (lat, lon).")

            lat = float(parts[0].strip())
            lon = float(parts[1].strip())
            
            input_point = QgsPointXY(lon, lat)
            transformed_point = self.transform.transform(input_point)
            
            if not self.first_point:
                self.first_point = transformed_point
                
            self.vertices.append(transformed_point)
            
            # Update the live preview on the map
            self.update_rubber_band()
            
            # Give user feedback
            QMessageBox.information(self, "Point Added", 
                                    f"Added point {len(self.vertices)}.\n"
                                    f"Original: ({lat}, {lon})\n"
                                    f"Transformed: ({transformed_point.x():.5f}, {transformed_point.y():.5f})")
            
            self.coord_input.clear()
            self.coord_input.setFocus() 
            
        except Exception as e:
            QMessageBox.critical(self, "Input Error", 
                                 f"Could not parse coordinates. \n"
                                 f"Use 'lat, lon' format. \n\nError: {e}")

    def undo_point(self):
        """
        Removes the last added point from the list and updates the preview.
        """
        if self.vertices:
            # Remove the last point from the list
            self.vertices.pop()
            
            # If we just removed the *first* point, reset first_point
            if not self.vertices:
                self.first_point = None
            
            # Update the map preview
            self.update_rubber_band() 
            
            QMessageBox.information(self, "Point Removed", 
                                    f"Removed point {len(self.vertices) + 1}.")
        else:
            QMessageBox.warning(self, "No Points", "There are no points to undo.")

    def close_polygon(self):
        """
        Closes the polygon ring, creates the feature, adds it to the layer,
        and closes the dialog.
        """
        if len(self.vertices) < 3:
            QMessageBox.warning(self, "Error", 
                                "You need at least 3 points to make a polygon.")
            return
            
        try:
            polygon_ring = list(self.vertices)
            polygon_ring.append(self.first_point) # Close the ring
            
            geometry = QgsGeometry.fromPolygonXY([polygon_ring])
            
            feature = QgsFeature(self.layer.fields())
            feature.setGeometry(geometry)
            self.layer.addFeature(feature)
            
            self.iface.mapCanvas().refresh()
            
            # --- FINAL FIX ---
            # Hide the rubber band *before* showing the success message
            self.rubberBand.reset(int(QgsWkbTypes.LineString))
            # -----------------
            
            QMessageBox.information(self, "Success", "Polygon feature created!")
            
            # 'accept' will trigger the closeEvent
            self.accept()
            
        except Exception as e:
            QMessageBox.critical(self, "Error", 
                                 f"Could not create polygon. \n\nError: {e}")

    def closeEvent(self, event):
        """
        Called when the dialog is closed (with 'X', 'Cancel', or 'Save').
        Ensures the rubber band is always cleaned up.
        """
        # Clean up the rubber band from the map
        # Apply int() wrapper for older QGIS versions
        self.rubberBand.reset(int(QgsWkbTypes.LineString))
        
        # Clean up local variables
        self.vertices = []
        self.first_point = None
        event.accept()
