import os
from qgis.PyQt.QtWidgets import QAction, QMessageBox
from qgis.PyQt.QtGui import QIcon
from qgis.core import QgsWkbTypes

# --- IMPORTANT ---
# Import the dialog class we just created
from .lat_long_dialog import LatLongDigitizerDialog

class LatLongDigitizer:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """
        Constructor.
        :param iface: An interface instance that will be passed to this class
            which is the main interface of QGIS application.
        """
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        self.actions = []
        self.toolbar = self.iface.addToolBar("LatLong Digitizer")
        self.toolbar.setObjectName("LatLongDigitizerToolbar")
        
        # We keep a reference to our dialog so it's not garbage-collected
        self.dlg = None

    def initGui(self):
        """Create the menu entries and toolbar icons."""
        # Add a toolbar button
        icon_path = os.path.join(self.plugin_dir, 'icon.png')
        action = QAction(QIcon(icon_path), 
                         "Digitize Polygon from Lat/Long", 
                         self.iface.mainWindow())
        
        # Connect the button click to our 'run' method
        action.triggered.connect(self.run)
        
        # Add the action to the toolbar and menu
        self.toolbar.addAction(action)
        self.actions.append(action)

    def unload(self):
        """Removes the plugin menu item and icon from QGIS interface."""
        for action in self.actions:
            self.iface.removeToolBarIcon(action)
        del self.toolbar

    def run(self):
        """
        This is the main function that runs when the plugin button is clicked.
        """
        
        # --- 1. Check if a layer is active ---
        layer = self.iface.activeLayer()
        if not layer:
            QMessageBox.warning(self.iface.mainWindow(), 
                                "No Layer", 
                                "Please select a layer in the Layers panel.")
            return

        # --- 2. Check if it's a Polygon layer ---
        if layer.geometryType() != QgsWkbTypes.PolygonGeometry:
            QMessageBox.warning(self.iface.mainWindow(), 
                                "Wrong Layer Type", 
                                "Please select a POLYGON layer.")
            return
            
        # --- 3. Check if layer is in edit mode ---
        if not layer.isEditable():
            QMessageBox.warning(self.iface.mainWindow(), 
                                "Not in Edit Mode", 
                                "Please toggle editing on the selected layer first.")
            return

        # --- 4. If all checks pass, create and show the dialog ---
        
        # Create an instance of our dialog
        # We pass it the layer and the iface
        self.dlg = LatLongDigitizerDialog(layer, self.iface, self.iface.mainWindow())
        
        # Show the dialog
        self.dlg.show()