# Import necessary PyQt and QGIS classes
from qgis.PyQt.QtWidgets import (QDialog, QPushButton, QLineEdit, 
                                 QVBoxLayout, QMessageBox, QLabel)
from qgis.core import (QgsFeature, QgsGeometry, QgsPointXY, QgsProject,
                       QgsCoordinateReferenceSystem, QgsCoordinateTransform)

class LatLongDigitizerDialog(QDialog):
    """
    This is the custom QDialog window for the plugin.
    """

    def __init__(self, layer, iface, parent=None):
        """
        Constructor. Pass in the active layer and the QGIS interface (iface).
        """
        super(LatLongDigitizerDialog, self).__init__(parent)
        
        # Store references to the layer and iface
        self.layer = layer
        self.iface = iface
        
        # This list will store the vertices as QgsPointXY objects
        self.vertices = []  
        self.first_point = None
        
        # --- Define Coordinate Reference Systems (CRS) ---
        
        # 1. CRS for the input coordinates (always WGS 84 - Lat/Lon)
        self.crs_input = QgsCoordinateReferenceSystem("EPSG:4326")
        
        # 2. CRS for the target layer (could be anything)
        self.crs_layer = self.layer.crs()
        
        # 3. Setup the transformation object
        self.transform = QgsCoordinateTransform(self.crs_input, 
                                                self.crs_layer, 
                                                QgsProject.instance())

        # --- Setup the GUI ---
        self.setWindowTitle("Digitize with Lat/Long")
        layout = QVBoxLayout()
        
        # Add a label for instructions
        info_label = QLabel("Enter coordinates as 'Latitude, Longitude'")
        layout.addWidget(info_label)

        # Add the text input box
        self.coord_input = QLineEdit()
        self.coord_input.setPlaceholderText("e.g., 40.7128, -74.0060")
        layout.addWidget(self.coord_input)
        
        # Add the "Add Point" button
        self.btn_add_point = QPushButton("Add Point")
        layout.addWidget(self.btn_add_point)
        
        # Add the "Close Polygon" button
        self.btn_close_polygon = QPushButton("Close Polygon and Save")
        layout.addWidget(self.btn_close_polygon)
        
        # Add the "Cancel" button
        self.btn_cancel = QPushButton("Cancel")
        layout.addWidget(self.btn_cancel)
        
        self.setLayout(layout)
        
        # --- Connect button clicks to methods ---
        self.btn_add_point.clicked.connect(self.add_point)
        self.btn_close_polygon.clicked.connect(self.close_polygon)
        self.btn_cancel.clicked.connect(self.reject) # 'reject' closes the dialog

    def add_point(self):
        """
        Reads text from the QLineEdit, parses it, transforms it,
        and adds it to the vertices list.
        """
        text = self.coord_input.text()
        if not text:
            QMessageBox.warning(self, "Empty Input", "Please enter coordinates.")
            return

        try:
            # Split the input string by comma
            parts = text.split(',')
            if len(parts) != 2:
                raise ValueError("Input must have two parts (lat, lon).")

            # Strip whitespace and convert to float
            lat = float(parts[0].strip())
            lon = float(parts[1].strip())
            
            # --- Coordinate Transformation ---
            # 1. Create the QgsPointXY from input longitude, latitude
            #    Note: QgsPointXY takes (x, y) which is (lon, lat)
            input_point = QgsPointXY(lon, lat)
            
            # 2. Transform the point from EPSG:4326 to the layer's CRS
            transformed_point = self.transform.transform(input_point)
            
            # 3. Store the transformed point
            if not self.first_point:
                self.first_point = transformed_point
                
            self.vertices.append(transformed_point)
            
            # Give user feedback
            QMessageBox.information(self, "Point Added", 
                                    f"Added point {len(self.vertices)}.\n"
                                    f"Original: ({lat}, {lon})\n"
                                    f"Transformed: ({transformed_point.x():.5f}, {transformed_point.y():.5f})")
            
            # Clear the input box for the next point
            self.coord_input.clear()
            self.coord_input.setFocus() # Set cursor back to input box
            
        except Exception as e:
            QMessageBox.critical(self, "Input Error", 
                                 f"Could not parse coordinates. \n"
                                 f"Use 'lat, lon' format. \n\nError: {e}")

    def close_polygon(self):
        """
        Closes the polygon ring, creates the feature, adds it to the layer,
        and closes the dialog.
        """
        if len(self.vertices) < 3:
            QMessageBox.warning(self, "Error", 
                                "You need at least 3 points to make a polygon.")
            return
            
        try:
            # 1. Create a copy of the vertices list
            polygon_ring = list(self.vertices)
            
            # 2. Close the ring by adding the first point to the end
            polygon_ring.append(self.first_point)
            
            # 3. Create the geometry
            #    A polygon is defined by a list of rings.
            #    The first ring is the outer boundary.
            geometry = QgsGeometry.fromPolygonXY([polygon_ring])
            
            # 4. Create the feature and set its geometry
            feature = QgsFeature(self.layer.fields())
            feature.setGeometry(geometry)
            
            # 5. Add the feature to the layer
            self.layer.addFeature(feature)
            
            # 6. Refresh the map canvas to show the new feature
            self.iface.mapCanvas().refresh()
            
            # Give feedback and close the dialog
            QMessageBox.information(self, "Success", "Polygon feature created!")
            self.accept() # 'accept' closes the dialog
            
        except Exception as e:
            QMessageBox.critical(self, "Error", 
                                 f"Could not create polygon. \n\nError: {e}")

    def closeEvent(self, event):
        """
        Called when the user closes the dialog (e.g., clicks 'X' button).
        """
        # Clean up
        self.vertices = []
        self.first_point = None
        event.accept()