"""
/***************************************************************************
 LabelLayer
                                 A QGIS plugin
 Creates a shapefile with user-defined label settings ready to use.
                              -------------------
        begin                : 2012-01-23
        copyright            : (C) 2012 by Vka
        email                : -
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
# Import the PyQt and QGIS libraries
from PyQt4.QtCore import *
from PyQt4.QtGui import *
from qgis.core import *
# Initialize Qt resources from file resources.py
import resources
import shutil
import os
# Import the code for the dialog
from labellayerdialog import LabelLayerDialog

class LabelLayer:

	def __init__(self, iface):
		# Save reference to the QGIS interface
		self.iface = iface
		self.settings = QSettings()

	def initGui(self):
		# Create action that will start plugin configuration
		self.action = QAction(QIcon(":/icon_ll.svg"), \
			"Create labeled layer", self.iface.mainWindow())	
		# connect the action to the run method		
		QObject.connect(self.action, SIGNAL("triggered()"), self.run)

		# Add toolbar button and menu item
		self.iface.addToolBarIcon(self.action)
		self.iface.addPluginToVectorMenu("&Create labeled layer", self.action)

	def unload(self):
		# Remove the plugin menu item and icon
		self.iface.removePluginVectorMenu("&Create labeled layer",self.action)
		self.iface.removeToolBarIcon(self.action)

    # run method that performs all the real work
	def run(self):
		
		self.savelayerPath = ""
	
		# create and show the dialog
		self.dlg = LabelLayerDialog()	
		
		QObject.connect(self.dlg.ui.outputButton, SIGNAL('clicked()'), self.createLayerPath)
		QObject.connect(self.dlg.ui.createButton, SIGNAL('clicked()'), self.createLayer)
		QObject.connect(self.dlg.ui.closeButton, SIGNAL('clicked()'), self.close)
	
		# show the dialog
		self.dlg.show()
		

	def createLayerPath(self):
		self.dlg.ui.output.clear()
		lastDir = self.settings.value("/LabelLayer/lastDir")
		self.savelayerPath = QFileDialog.getSaveFileName(None,"Create output shapefile", lastDir, "*.shp")
		if not self.savelayerPath:
			return
		self.dlg.ui.output.setText(self.savelayerPath)
		
		file_info = QFileInfo(self.savelayerPath)
		self.layer_name = file_info.completeBaseName()
		self.layer_dir = file_info.absolutePath()
		self.settings.setValue("/LabelLayer/lastDir", str(self.layer_dir))
		
	
	def createLayer(self):
	
		if self.savelayerPath == "" or not self.savelayerPath:
			QMessageBox.critical(None, "No output location", "Please choose a destination for the output shapefile.")
			return
		
		crs = self.iface.mapCanvas().mapRenderer().destinationCrs()
		fields = QgsFields()
		fields.append(QgsField("Text", QVariant.String, "text", 100))
		fields.append(QgsField("Size", QVariant.Int, "int", 4))
		fields.append(QgsField("Unit", QVariant.String, "text", 10))
		fields.append(QgsField("Color", QVariant.String, "text",10))
		fields.append(QgsField("Bold", QVariant.Int, "int",2))
		fields.append(QgsField("Italic", QVariant.Int, "int",2))
		fields.append(QgsField("Underline", QVariant.Int, "int",2))
		fields.append(QgsField("Strikeout", QVariant.Int, "int",2))
		fields.append(QgsField("Font", QVariant.String, "text", 50))
		fields.append(QgsField("X", QVariant.Double, "double", 15,3))
		fields.append(QgsField("Y", QVariant.Double, "double",15,3))
		fields.append(QgsField("Rotation", QVariant.Double, "double", 8, 3))
		fields.append(QgsField("UseScale", QVariant.Int, "int",2))
		fields.append(QgsField("ScaleMin",QVariant.Int,"integer",10))
		fields.append(QgsField("ScaleMax",QVariant.Int,"integer",10))
		
		writer = QgsVectorFileWriter(self.savelayerPath, "utf-8", fields, QGis.WKBPoint, crs)
			
		del writer
		
		label_layer = QgsVectorLayer(self.savelayerPath, self.layer_name, "ogr" )
		
		plugin_path = os.path.abspath(__file__)
		plugin_dir = os.path.dirname(plugin_path)
		try:
			shutil.copy(plugin_dir+"/editForm.ui", str(self.layer_dir))
			os.rename(str(self.layer_dir)+"/editForm.ui", str(self.layer_dir)+"/"+str(self.layer_name)+".ui")
		except OSError:
			pass
		except IOError:
			self.iface.messageBar().pushMessage("Attribute edit layout", "Could not set customized edit layout. The standard layout will be used.", level=1)
			
		label_layer.setCustomProperty("labeling", "pal")
		label_layer.setCustomProperty("labeling/enabled", "true")
		label_layer.setCustomProperty("labeling/fontFamily", "Arial")
		label_layer.setCustomProperty("labeling/fontSize", "10")
		label_layer.setCustomProperty("labeling/fieldName", "Text")
		label_layer.setCustomProperty("labeling/dataDefined/Size", "Size")
		label_layer.setCustomProperty("labeling/dataDefined/FontSizeUnit", "Unit")
		label_layer.setCustomProperty("labeling/dataDefined/Bold", "Bold")
		label_layer.setCustomProperty("labeling/dataDefined/Italic", "Italic")
		label_layer.setCustomProperty("labeling/dataDefined/Underline", "Underline")
		label_layer.setCustomProperty("labeling/dataDefined/Color", "Color")
		label_layer.setCustomProperty("labeling/dataDefined/Strikeout", "Strikeout")
		label_layer.setCustomProperty("labeling/dataDefined/Family", "Font")
		label_layer.setCustomProperty("labeling/dataDefined/PositionX", "X")
		label_layer.setCustomProperty("labeling/dataDefined/PositionY", "Y")
		label_layer.setCustomProperty("labeling/dataDefined/Rotation", "Rotation")
		label_layer.setCustomProperty("labeling/dataDefined/ScaleVisibility", "UseScale")
		label_layer.setCustomProperty("labeling/dataDefined/MinScale", "ScaleMin")
		label_layer.setCustomProperty("labeling/dataDefined/MaxScale", "ScaleMax")
		
		label_layer.setEditType(1,QgsVectorLayer.EditRange)
		label_layer.range(1).mMin = 0
		label_layer.range(1).mMax = 1000
		label_layer.range(1).mStep = 1
		label_layer.setFieldEditable(1,True)
		label_layer.setEditType(4,QgsVectorLayer.CheckBox)
		label_layer.setFieldEditable(4,True)
		label_layer.setEditType(5,QgsVectorLayer.CheckBox)
		label_layer.setFieldEditable(5,True)
		label_layer.setEditType(6,QgsVectorLayer.CheckBox)
		label_layer.setFieldEditable(6,True)
		label_layer.setEditType(7,QgsVectorLayer.CheckBox)
		label_layer.setFieldEditable(7,True)
		
		label_layer.setEditorLayout(2)
		label_layer.setEditForm(str(self.layer_dir)+"/"+str(self.layer_name)+".ui")
		label_layer.saveNamedStyle(str(self.layer_dir)+"/"+str(self.layer_name)+".qml")
		
		QgsMapLayerRegistry.instance().addMapLayer(label_layer)
		self.iface.messageBar().pushMessage("Layer created", "Use the 'Add point tool' to add a new label.", level=0, duration=4)
		self.close()
		
	def close(self):
		self.dlg.close()