# -*- coding: utf-8 -*-
"""
***************************************************************************
    las2shp.py
    ---------------------
    Date                 : January 2025
    Copyright            : (c) 2025 by rapidlasso GmbH
    Email                : info near rapidlasso point de
***************************************************************************
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************
"""

__author__ = "rapidlasso"
__date__ = "January 2025"
__copyright__ = "(c) 2025, rapidlasso GmbH"


from qgis.core import QgsProcessingParameterBoolean, QgsProcessingParameterNumber
from qgis.PyQt.QtGui import QIcon

from ..algo import LastoolsAlgorithm
from ..utils import LastoolsUtils, help_string_help, lasgroup_info, lastool_info, licence, paths, readme_url


class Las2Shp(LastoolsAlgorithm):
    TOOL_NAME = "Las2Shp"
    LASTOOL = "las2shp"
    LICENSE = "c"
    LASGROUP = 2
    POINT_Z = "POINT_Z"
    RECORD_SIZE = "RECORD_SIZE"

    def initAlgorithm(self, config=None):
        super().initAlgorithm(config)
        self.add_parameters_point_input_gui()
        self.addParameter(QgsProcessingParameterBoolean(self.POINT_Z, "use PointZ instead of MultiPointZ", False))
        self.addParameter(
            QgsProcessingParameterNumber(
                self.RECORD_SIZE,
                "number of points per record",
                QgsProcessingParameterNumber.Integer,
                1024,
                False,
                0,
                65536,
            )
        )
        self.add_parameters_additional_gui()
        self.add_parameters_verbose_64_gui()
        self.add_parameters_generic_output_gui("Output SHP file", "shp", True)

    def processAlgorithm(self, parameters, context, feedback):
        commands = [self.get_command(parameters, context, feedback)]
        self.add_parameters_point_input_commands(parameters, context, commands)
        if self.parameterAsBool(parameters, self.POINT_Z, context):
            commands.append("-single_points")
        record_size = self.parameterAsInt(parameters, self.RECORD_SIZE, context)
        if record_size != 1024:
            commands.append("-record_size")
            commands.append(str(record_size))
        self.add_parameters_additional_commands(parameters, context, commands)
        self.add_parameters_verbose_64_gui_commands(parameters, context, commands)
        self.add_parameters_generic_output_commands(parameters, context, commands, "-o")
        self.run_lastools(commands, feedback)
        return {"commands": commands}

    def createInstance(self):
        return Las2Shp()

    def name(self):
        return self.TOOL_NAME

    def displayName(self):
        return lastool_info[self.TOOL_NAME]["disp"]

    def group(self):
        return lasgroup_info[self.LASGROUP]["group"]

    def groupId(self):
        return lasgroup_info[self.LASGROUP]["group_id"]

    def helpUrl(self):
        return readme_url(self.LASTOOL)

    def shortHelpString(self):
        return lastool_info[self.TOOL_NAME]["help"] + help_string_help(self.LASTOOL, self.LICENSE)

    def shortDescription(self):
        return lastool_info[self.TOOL_NAME]["desc"]

    def icon(self):
        icon_file = licence[self.LICENSE]["path"]
        return QIcon(f"{paths['img']}{icon_file}")
