# -*- coding: utf-8 -*-
"""
***************************************************************************
    lasoverlap.py
    ---------------------
    Date                 : January 2025
    Copyright            : (c) 2025 by rapidlasso GmbH
    Email                : info near rapidlasso point de
***************************************************************************
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************
"""

__author__ = "rapidlasso"
__date__ = "January 2025"
__copyright__ = "(c) 2025, rapidlasso GmbH"


from qgis.core import QgsProcessingParameterBoolean, QgsProcessingParameterEnum, QgsProcessingParameterNumber
from qgis.PyQt.QtGui import QIcon

from ..algo import LastoolsAlgorithm
from ..utils import LastoolsUtils, help_string_help, lasgroup_info, lastool_info, licence, paths, readme_url


class LasOverlap(LastoolsAlgorithm):
    TOOL_NAME = "LasOverlap"
    LASTOOL = "lasoverlap"
    LICENSE = "c"
    LASGROUP = 6
    CHECK_STEP = "CHECK_STEP"
    ATTRIBUTE = "ATTRIBUTE"
    OPERATION = "OPERATION"
    ATTRIBUTES = ["elevation", "intensity", "number_of_returns", "scan_angle_abs", "density"]
    OPERATIONS = ["lowest", "highest", "average"]
    CREATE_OVERLAP_RASTER = "CREATE_OVERLAP_RASTER"
    CREATE_DIFFERENCE_RASTER = "CREATE_DIFFERENCE_RASTER"

    def initAlgorithm(self, config=None):
        super().initAlgorithm(config)
        self.add_parameters_point_input_gui()
        self.add_parameters_filter1_return_class_flags_gui()
        self.addParameter(
            QgsProcessingParameterNumber(
                self.CHECK_STEP,
                "size of grid used for overlap check",
                QgsProcessingParameterNumber.Double,
                2.0,
                False,
                0.001,
                50.0,
            )
        )
        self.addParameter(QgsProcessingParameterEnum(self.ATTRIBUTE, "attribute to check", self.ATTRIBUTES, False, 0))
        self.addParameter(
            QgsProcessingParameterEnum(self.OPERATION, "operation on attribute per cell", self.OPERATIONS, False, 0)
        )
        self.addParameter(QgsProcessingParameterBoolean(self.CREATE_OVERLAP_RASTER, "create overlap raster", True))
        self.addParameter(
            QgsProcessingParameterBoolean(self.CREATE_DIFFERENCE_RASTER, "create difference raster", True)
        )
        self.add_parameters_additional_gui()
        self.add_parameters_verbose_64_gui()
        self.add_parameters_raster_output_gui()

    def processAlgorithm(self, parameters, context, feedback):
        commands = [self.get_command(parameters, context, feedback)]
        self.add_parameters_verbose_64_gui_commands(parameters, context, commands)
        self.add_parameters_point_input_commands(parameters, context, commands)
        self.add_parameters_filter1_return_class_flags_commands(parameters, context, commands)
        step = self.parameterAsDouble(parameters, self.CHECK_STEP, context)
        if step != 2.0:
            commands.append("-step")
            commands.append(str(step))
        commands.append("-values")
        attribute = self.parameterAsInt(parameters, self.ATTRIBUTE, context)
        if attribute != 0:
            commands.append("-" + self.ATTRIBUTES[attribute])
        operation = self.parameterAsInt(parameters, self.OPERATION, context)
        if operation != 0:
            commands.append("-" + self.OPERATIONS[operation])
        if not self.parameterAsBool(parameters, self.CREATE_OVERLAP_RASTER, context):
            commands.append("-no_over")
        if not self.parameterAsBool(parameters, self.CREATE_DIFFERENCE_RASTER, context):
            commands.append("-no_diff")
        self.add_parameters_raster_output_commands(parameters, context, commands)
        self.add_parameters_additional_commands(parameters, context, commands)
        self.run_lastools(commands, feedback)
        return {"commands": commands}

    def createInstance(self):
        return LasOverlap()

    def name(self):
        return self.TOOL_NAME

    def displayName(self):
        return lastool_info[self.TOOL_NAME]["disp"]

    def group(self):
        return lasgroup_info[self.LASGROUP]["group"]

    def groupId(self):
        return lasgroup_info[self.LASGROUP]["group_id"]

    def helpUrl(self):
        return readme_url(self.LASTOOL)

    def shortHelpString(self):
        return lastool_info[self.TOOL_NAME]["help"] + help_string_help(self.LASTOOL, self.LICENSE)

    def shortDescription(self):
        return lastool_info[self.TOOL_NAME]["desc"]

    def icon(self):
        icon_file = licence[self.LICENSE]["path"]
        return QIcon(f"{paths['img']}{icon_file}")


class LasOverlapPro(LastoolsAlgorithm):
    TOOL_NAME = "LasOverlapPro"
    LASTOOL = "lasoverlap"
    LICENSE = "c"
    LASGROUP = 6
    CHECK_STEP = "CHECK_STEP"
    ATTRIBUTE = "ATTRIBUTE"
    OPERATION = "OPERATION"
    ATTRIBUTES = ["elevation", "intensity", "number_of_returns", "scan_angle_abs", "density"]
    OPERATIONS = ["lowest", "highest", "average"]
    CREATE_OVERLAP_RASTER = "CREATE_OVERLAP_RASTER"
    CREATE_DIFFERENCE_RASTER = "CREATE_DIFFERENCE_RASTER"

    def initAlgorithm(self, config=None):
        super().initAlgorithm(config)
        self.add_parameters_point_input_folder_gui()
        self.add_parameters_files_are_flightlines_gui()
        self.add_parameters_filter1_return_class_flags_gui()
        self.addParameter(
            QgsProcessingParameterNumber(
                self.CHECK_STEP,
                "size of grid used for overlap check",
                QgsProcessingParameterNumber.Double,
                2.0,
                False,
                0.001,
                50.0,
            )
        )
        self.addParameter(QgsProcessingParameterEnum(self.ATTRIBUTE, "attribute to check", self.ATTRIBUTES, False, 0))
        self.addParameter(
            QgsProcessingParameterEnum(self.OPERATION, "operation on attribute per cell", self.OPERATIONS, False, 0)
        )
        self.addParameter(QgsProcessingParameterBoolean(self.CREATE_OVERLAP_RASTER, "create overlap raster", True))
        self.addParameter(
            QgsProcessingParameterBoolean(self.CREATE_DIFFERENCE_RASTER, "create difference raster", True)
        )
        self.add_parameters_additional_gui()
        self.add_parameters_cores_gui()
        self.add_parameters_verbose_64_gui()
        self.add_parameters_output_appendix_gui()
        self.add_parameters_raster_output_gui()
        self.add_parameters_raster_output_format_gui()
        self.add_parameters_output_directory_gui()

    def processAlgorithm(self, parameters, context, feedback):
        commands = [self.get_command(parameters, context, feedback)]
        self.add_parameters_verbose_64_gui_commands(parameters, context, commands)
        self.add_parameters_point_input_folder_commands(parameters, context, commands)
        self.add_parameters_files_are_flightlines_commands(parameters, context, commands)
        self.add_parameters_filter1_return_class_flags_commands(parameters, context, commands)
        step = self.parameterAsDouble(parameters, self.CHECK_STEP, context)
        if step != 2.0:
            commands.append("-step")
            commands.append(str(step))
        commands.append("-values")
        attribute = self.parameterAsInt(parameters, self.ATTRIBUTE, context)
        if attribute != 0:
            commands.append("-" + self.ATTRIBUTES[attribute])
        operation = self.parameterAsInt(parameters, self.OPERATION, context)
        if operation != 0:
            commands.append("-" + self.OPERATIONS[operation])
        if not self.parameterAsBool(parameters, self.CREATE_OVERLAP_RASTER, context):
            commands.append("-no_over")
        if not self.parameterAsBool(parameters, self.CREATE_DIFFERENCE_RASTER, context):
            commands.append("-no_diff")
        self.add_parameters_output_directory_commands(parameters, context, commands)
        self.add_parameters_output_appendix_commands(parameters, context, commands)
        self.add_parameters_raster_output_format_commands(parameters, context, commands)
        self.add_parameters_raster_output_commands(parameters, context, commands)
        self.add_parameters_additional_commands(parameters, context, commands)
        self.add_parameters_cores_commands(parameters, context, commands)
        self.run_lastools(commands, feedback)
        return {"commands": commands}

    def createInstance(self):
        return LasOverlapPro()

    def name(self):
        return self.TOOL_NAME

    def displayName(self):
        return lastool_info[self.TOOL_NAME]["disp"]

    def group(self):
        return lasgroup_info[self.LASGROUP]["group"]

    def groupId(self):
        return lasgroup_info[self.LASGROUP]["group_id"]

    def helpUrl(self):
        return readme_url(self.LASTOOL)

    def shortHelpString(self):
        return lastool_info[self.TOOL_NAME]["help"] + help_string_help(self.LASTOOL, self.LICENSE)

    def shortDescription(self):
        return lastool_info[self.TOOL_NAME]["desc"]

    def icon(self):
        icon_file = licence[self.LICENSE]["path"]
        return QIcon(f"{paths['img']}{icon_file}")
