# -*- coding: utf-8 -*-
"""
/***************************************************************************
 IsometricDensifier
                                 A QGIS plugin
 Densify geometries isometricly adding vertices only where needed.
                              -------------------
        begin                : 2013-10-11
        copyright            : (C) 2013 by Alejandro Aguilar Sierra
        email                : algsierra@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
# Import the PyQt and QGIS libraries
import math
from PyQt4.QtCore import *
from PyQt4.QtGui import *
from qgis.core import *
from qgis.gui import *

# Initialize Qt resources from file resources.py
import resources_rc
# Import the code for the dialog
from isometricdensifierdialog import IsometricDensifierDialog
import os.path

earth_radius_km = 6378.1


class IsometricDensifier:

    def __init__(self, iface):
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value("locale/userLocale")[0:2]
        localePath = os.path.join(self.plugin_dir, 'i18n', 'isometricdensifier_{}.qm'.format(locale))

        if os.path.exists(localePath):
            self.translator = QTranslator()
            self.translator.load(localePath)

            if qVersion() > '4.3.3':
                QCoreApplication.installTranslator(self.translator)

        # Create the dialog (after translation) and keep reference
        self.dlg = IsometricDensifierDialog()

    def initGui(self):
        # Create action that will start plugin configuration
        self.action = QAction(
            QIcon(":/plugins/isometricdensifier/icon.png"),
            u"Isometric geometry densifier", self.iface.mainWindow())
        # connect the action to the run method
        #self.action.triggered.connect(self.run)
        self.action.setEnabled(False)
        self.action.setCheckable(True)
        #Connect to signals for button behaviour
        QObject.connect(self.action, SIGNAL("triggered()"), self.run)
        QObject.connect(self.iface, SIGNAL("currentLayerChanged(QgsMapLayer*)"), self.toggle)
        #QObject.connect(mc, SIGNAL("mapToolSet(QgsMapTool*)"), self.deactivate)

        # Add toolbar button and menu item
        self.iface.addToolBarIcon(self.action)
        self.iface.digitizeToolBar().addAction(self.action)
        self.iface.addPluginToMenu(u"&Isometric geometry densifier", self.action)

    def unload(self):
        # Remove the plugin menu item and icon
        self.iface.removePluginMenu(u"&Isometric geometry densifier", self.action)
        self.iface.removeToolBarIcon(self.action)

    def toggle(self):
        #Decide whether the plugin button/menu is enabled or disabled 
        self.canvas = self.iface.mapCanvas()             
        layer = self.canvas.currentLayer()
        if layer <> None:
            if layer.isEditable():
                self.action.setEnabled(True)
                QObject.connect(layer,SIGNAL("editingStopped()"),self.toggle)
                QObject.disconnect(layer,SIGNAL("editingStarted()"),self.toggle)
            else:
                self.action.setEnabled(False)
                QObject.connect(layer,SIGNAL("editingStarted()"),self.toggle)
                QObject.disconnect(layer,SIGNAL("editingStopped()"),self.toggle)        
                
    def distance2d(self, p, q):
        dlo = math.fabs(p.x() - q.x())
        dla = math.fabs(p.y() - q.y())
        return math.sqrt(dlo*dlo + dla*dla)

    def distance(self, p, q):
        lo1 = math.radians(p.x())
        la1 = math.radians(p.y())
        lo2 = math.radians(q.x())
        la2 = math.radians(q.y())
        dlo = math.fabs(lo2 - lo1)
        r = math.sin(la1)*math.sin(la2) + math.cos(la1)*math.cos(la2)*math.cos(dlo)
        #print "r ",r, "dlo ",dlo, "lo ", lo1, "la ", la1
        return math.acos(r)

    # run method that performs all the real work
    def run(self):
        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result == 1:
            # do something useful (delete the line containing pass and
            # substitute with your code) 
            self.canvas = self.iface.mapCanvas()            
            layer = self.canvas.currentLayer()
            mindis = self.dlg.ui.minDistance.value()
            projenabled = self.dlg.ui.projEnabled.isChecked()
            if projenabled:
                mindis /= earth_radius_km # Convert it to radians
            #else:
            #    mindis = math.degrees(mindis)
            dmin = 1000.0
            dmax = 0.0
            iter = layer.getFeatures()
            dist = QgsDistanceArea()
            dist.setEllipsoidalMode(projenabled)
            print "Mindis ", mindis, "projend ",projenabled
            for feature in iter:
                feaId = feature.id()
                geom = feature.geometry()
                if geom.type() == QGis.Polygon:
                    x = geom.asPolygon()
                    numPts = 0
                    for ring in x:
                        numPts += len(ring)
                        p1 = geom.vertexAt(0)
                        for i in range(numPts-2, -1, -1):
                            p2 = geom.vertexAt(i)
                            if projenabled:
                                d = self.distance(p1, p2)
                            else:
                                d = self.distance2d(p1, p2)
                            #d = dist.measureLine(p1, p2)
                            if d < dmin:
                                dmin = d
                            if d > dmax:
                                dmax = d
                            if d > mindis:
                                n = int(d/mindis) + 1
                                dx = p2.x() - p1.x()
                                dy = p2.y() - p1.y()
                                for j in range(1, n+1):
                                    t = j/float(n+1)
                                    x = p1.x() + dx*t
                                    y = p1.y() + dy*t
                                    layer.insertVertex(x, y, feaId, i+1)
                            p1 = p2
                        #print "Polygon: %d rings with %d points" % (len(x), numPts)
                    print "Distance min %f max %f %f %d %s" % (dmin, dmax, mindis, projenabled, dist.ellipsoid())
                else:
                    QMessageBox.information( self.iface.mainWindow(),"Error", "This is not a polygon vector layer"+str(layer.geometryType())) 
