# -*- coding: utf-8 -*-
"""
/***************************************************************************
 ImportPhotos
                                 A QGIS plugin
 Import photos jpegs
                              -------------------
        begin                : 2018-02-20
        git sha              : $Format:%H$
        copyright            : (C) 2018 by KIOS Research Center
        email                : mariosmsk@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from qgis.PyQt.QtWidgets import QAction, QFileDialog, QMessageBox
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import QSettings, QTranslator, qVersion, QCoreApplication, Qt
from qgis.core import QgsRectangle, QgsVectorFileWriter, QgsCoordinateReferenceSystem, QgsVectorLayer, \
    QgsLayerTreeLayer
# Initialize Qt resources from file resources.py
from . import resources
# Import the code for the dialog
from .ImportPhotos_dialog import ImportPhotosDialog
from .MouseClick import MouseClick
import os.path
import platform
import uuid
import json

try:
    # qgis 3
    import exifread
    from qgis.utils import Qgis
    from qgis.core import QgsProject
except:
    # qgis 2
    try:
        from PIL import Image
        from PIL.ExifTags import TAGS
        from qgis.utils import QGis as Qgis  #  for QGIS 2
        from qgis.core import QgsMapLayerRegistry
    except:
        from PIL import Image
        from PIL.ExifTags import TAGS
        from qgis.utils import Qgis # QGIS 3
        from qgis.core import QgsProject

try:
    import osgeo.ogr as ogr
except:
    import ogr

try:
    import osgeo.osr as osr
except:
    import osr


class ImportPhotos:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'ImportPhotos_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)

            if qVersion() > '4.3.3':
                QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&ImportPhotos')
        # TODO: We are going to let the user set this up in a future iteration
        self.toolbar = self.iface.addToolBar(u'ImportPhotos')
        self.toolbar.setObjectName(u'ImportPhotos')

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('ImportPhotos', message)

    def add_action(
            self,
            icon_path,
            text,
            callback,
            enabled_flag=True,
            add_to_menu=True,
            add_to_toolbar=True,
            status_tip=None,
            whats_this=None,
            parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        # Create the dialog (after translation) and keep reference

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        icon_path = ':/plugins/ImportPhotos/svg/ImportImage.svg'
        self.add_action(
            icon_path,
            text=self.tr(u'Import Photos'),
            callback=self.run,
            parent=self.iface.mainWindow())
        icon_path = ':/plugins/ImportPhotos/svg/SelectImage.svg'
        self.clickPhotos = self.add_action(
            icon_path,
            text=self.tr(u'Click Photos'),
            callback=self.mouseClick,
            parent=self.iface.mainWindow())
        self.dlg = ImportPhotosDialog()
        self.dlg.setWindowFlags(Qt.CustomizeWindowHint | Qt.WindowStaysOnTopHint | Qt.WindowCloseButtonHint)

        self.dlg.ok.clicked.connect(self.ok)
        self.dlg.closebutton.clicked.connect(self.close)
        self.dlg.toolButtonImport.clicked.connect(self.toolButtonImport)
        self.dlg.toolButtonOut.clicked.connect(self.toolButtonOut)

        self.clickPhotos.setCheckable(True)
        self.clickPhotos.setEnabled(True)

        self.listPhotos = []
        self.layernamePhotos = []
        self.canvas = self.iface.mapCanvas()
        self.toolMouseClick = MouseClick(self.canvas, self)

        self.fields = ['ID', 'Name', 'Date', 'Time', 'Lon', 'Lat', 'Altitude', 'North', 'Azimuth', 'Camera Maker',
                       'Camera Model', 'Path']

        self.extension_switch = {
            ".geojson": "GeoJSON",
            ".shp": "ESRI Shapefile",
            ".gpkg":"GPKG",
            ".csv": "CSV",
            ".kml": "KML",
            ".tab": "MapInfo File",
            ".ods": "ODS"
        }

    def mouseClick(self):
        try:
            self.iface.setActiveLayer(self.canvas.layers()[0])
        except:
            pass
        self.canvas.setMapTool(self.toolMouseClick)
        self.clickPhotos.setChecked(True)

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&ImportPhotos'),
                action)
            self.iface.removeToolBarIcon(action)
            # remove the toolbar
        del self.toolbar

    def run(self):
        self.dlg.ok.setEnabled(True)
        self.dlg.closebutton.setEnabled(True)
        self.dlg.toolButtonImport.setEnabled(True)
        self.dlg.toolButtonOut.setEnabled(True)
        self.clickPhotos.setEnabled(True)
        self.dlg.out.setText('')
        self.dlg.imp.setText('')
        self.dlg.progressBar.setValue(0)
        self.dlg.show()

    def close(self):
        self.dlg.close()

    def toolButtonOut(self):
        typefiles = 'GeoJSON (*.geojson *.GEOJSON);; ESRI Shapefile (*.shp *.SHP);; GeoPackage (*.gpkg *.GPKG);; Comma Separated Value (*.csv *.CSV);; Keyhole Markup Language (*.kml *.KML);; Mapinfo TAB (*.tab *.TAB);; Open Document Spreadsheet (*.ods *.ODS)'
        if platform.system() == 'Linux':
            try:
                self.outputPath, self.extension = QFileDialog.getSaveFileNameAndFilter(None, 'Save File', os.path.join(
                    os.path.join(os.path.expanduser('~')),
                    'Desktop'), typefiles)
            except:
                self.outputPath = QFileDialog.getSaveFileName(None, 'Save File', os.path.join(
                    os.path.join(os.path.expanduser('~')),
                    'Desktop'), typefiles)
        else:
            self.outputPath = QFileDialog.getSaveFileName(None, 'Save File', os.path.join(
                os.path.join(os.path.expanduser('~')),
                'Desktop'), typefiles)

        if Qgis.QGIS_VERSION >= '3.0':
            self.outputPath = self.outputPath[0]
        self.dlg.out.setText(self.outputPath)

    def toolButtonImport(self):
        self.directoryPhotos = QFileDialog.getExistingDirectory(None, 'Select a folder:',
                                                                os.path.join(os.path.join(os.path.expanduser('~')),
                                                                             'Desktop'), QFileDialog.ShowDirsOnly)
        self.dlg.imp.setText(self.directoryPhotos)

    def selectDir(self):
        title = 'Warning'
        msg = 'Please select a directory photos.'
        self.showMessage(title, msg, 'Warning')
        return True

    def selectOutp(self):
        title = 'Warning'
        msg = 'Please define output file location.'
        self.showMessage(title, msg, 'Warning')
        return True

    def ok(self):
        if self.dlg.imp.text() == '':
            if self.selectDir():
                return
        if os.path.isdir(self.dlg.imp.text())==False:
            if self.selectDir():
                return
        if self.dlg.out.text() == '':
            if self.selectOutp():
                return
        if os.path.isabs(self.dlg.out.text())==False:
            if self.selectOutp():
                return

        if platform.system()=='Linux':
            lphoto = os.path.basename(self.outputPath)
            try:
                self.extension = '.'+self.extension.split()[-1][2:-1].lower()
            except:
                self.extension = '.shp'
        else:
            tmpbasename, self.extension = os.path.splitext(self.outputPath)
            basename = os.path.basename(self.outputPath)
            lphoto = basename[:-len(self.extension)]

        self.outputPath = self.dlg.out.text()
        self.directoryPhotos = self.dlg.imp.text()
        self.outDirectoryPhotosGeoJSON = self.plugin_dir + '/tmp.geojson'

        self.dlg.ok.setEnabled(False)
        self.dlg.closebutton.setEnabled(False)
        self.dlg.toolButtonImport.setEnabled(False)
        self.dlg.toolButtonOut.setEnabled(False)
        extens = ['jpg', 'jpeg', 'JPG', 'JPEG']
        photos = []
        for root, dirs, files in os.walk(self.directoryPhotos):
            photos.extend(os.path.join(root, name) for name in files
                          if name.lower().endswith(tuple(extens)))
        initphotos = len(photos)

        if initphotos == 0:
            title = 'Warning'
            msg = 'No photos.'
            self.showMessage(title, msg, 'Warning')
            self.dlg.ok.setEnabled(True)
            self.dlg.closebutton.setEnabled(True)
            self.dlg.toolButtonImport.setEnabled(True)
            self.dlg.toolButtonOut.setEnabled(True)
            self.clickPhotos.setChecked(True)
            return

        self.total = 100.0 / initphotos
        self.canvas.setMapTool(self.toolMouseClick)

        self.layernamePhotos.append(lphoto)
        self.truePhotosCount = 0
        self.lon = []
        self.lat = []
        geoPhotos = []

        if Qgis.QGIS_VERSION >= '3.0':
            Qpr_inst = QgsProject.instance()
        else:
            # something for QGIS 2
            Qpr_inst = QgsMapLayerRegistry.instance()
            if platform.system()=='Darwin':
                self.layernamePhotos.append(lphoto+' OGRGeoJSON Point')
            else:
                self.layernamePhotos.append(lphoto)

        if Qgis.QGIS_VERSION >= '3.3':
            # Using ogr
            driver = ogr.GetDriverByName('geojson')
            if os.path.exists(self.outDirectoryPhotosGeoJSON):
                try:
                    os.remove(self.outDirectoryPhotosGeoJSON)
                except:
                    pass
                driver.DeleteDataSource(self.outDirectoryPhotosGeoJSON)

            Shp = driver.CreateDataSource(self.outDirectoryPhotosGeoJSON)

            SR = osr.SpatialReference()
            SR.ImportFromEPSG(4326)
            try:
                os.remove(self.outDirectoryPhotosGeoJSON)
            except:
                pass
            layer = Shp.CreateLayer(os.path.basename(self.outputPath), SR, ogr.wkbPoint)

            # Create attribute fields
            realcolumns = ['Lon', 'Lat', 'Altitude', 'Azimuth']

            # Create ID Field and add it to Layer
            for col in self.fields:
                if col in realcolumns:
                    fieldDefn = ogr.FieldDefn(col, ogr.OFTReal)
                else:
                    fieldDefn = ogr.FieldDefn(col, ogr.OFTString)
                layer.CreateField(fieldDefn)

        for count, imgpath in enumerate(photos):
            try:
                self.dlg.progressBar.setValue(int(count * self.total))
                name = os.path.basename(imgpath)
                imgpath = imgpath.replace('\\', '/')
                if Qgis.QGIS_VERSION >= '3.0':
                    with open(imgpath, 'rb') as imgpathF:
                        tags = exifread.process_file(imgpathF, details=False)
                    if not tags.keys() & {"GPS GPSLongitude", "GPS GPSLatitude"}:
                        continue

                    lat, lon = self.get_exif_location(tags, "lonlat")
                    try:
                        altitude = float(tags["GPS GPSAltitude"].values[0].num) / float(tags["GPS GPSAltitude"].values[0].den)
                    except:
                        altitude = ''
                    uuid_ = str(uuid.uuid4())
                    try:
                        dt1, dt2 = tags["Image DateTime"].values.split()
                        date = dt1.replace(':', '/')
                        time_ = dt2
                    except:
                        try:
                            date = tags["GPS GPSDate"].values.replace(':', '/')
                            tt = [str(i) for i in tags["GPS GPSTimeStamp"].values]
                            time_ = "{:0>2}:{:0>2}:{:0>2}".format(tt[0], tt[1], tt[2])
                        except:
                            date = ''
                            time_ = ''
                    try:
                        azimuth = float(tags["GPS GPSImgDirection"].values[0].num) / float(tags["GPS GPSImgDirection"].values[0].den)
                    except:
                        azimuth = ''
                    try:
                        north = str(tags["GPS GPSImgDirectionRef"].values)
                    except:
                        north = ''
                    try:
                        maker = tags['Image Make']
                    except:
                        maker = ''
                    try:
                        model = tags['Image Model']
                    except:
                        model = ''

                else:
                    # something for QGIS 2
                    a = {}
                    info = Image.open(imgpath)
                    info = info._getexif()

                    if info == None:
                        continue

                    for tag, value in info.items():
                        if TAGS.get(tag, tag) == 'GPSInfo' or TAGS.get(tag, tag) == 'DateTime' or TAGS.get(tag,
                                                                                                           tag) == 'DateTimeOriginal':
                            a[TAGS.get(tag, tag)] = value

                    if a == {}:
                        continue

                    if a['GPSInfo'] != {}:
                        if 1 and 2 and 3 and 4 in a['GPSInfo']:
                            lat = [float(x) / float(y) for x, y in a['GPSInfo'][2]]
                            latref = a['GPSInfo'][1]
                            lon = [float(x) / float(y) for x, y in a['GPSInfo'][4]]
                            lonref = a['GPSInfo'][3]

                            lat = lat[0] + lat[1] / 60 + lat[2] / 3600
                            lon = lon[0] + lon[1] / 60 + lon[2] / 3600

                            if latref == 'S':
                                lat = -lat
                            if lonref == 'W':
                                lon = -lon
                        else:
                            continue

                        uuid_ = str(uuid.uuid4())
                        if 'DateTime' or 'DateTimeOriginal' in a:
                            if 'DateTime' in a:
                                dt1, dt2 = a['DateTime'].split()
                            elif 'DateTimeOriginal' in a:
                                dt1, dt2 = a['DateTimeOriginal'].split()
                            date = dt1.replace(':', '/')
                            time_ = dt2

                        if 6 in a['GPSInfo']:
                            if len(a['GPSInfo'][6]) > 1:
                                mAltitude = float(a['GPSInfo'][6][0])
                                mAltitudeDec = float(a['GPSInfo'][6][1])
                                altitude = str(mAltitude / mAltitudeDec)
                        else:
                            altitude = ''

                        if 16 and 17 in a['GPSInfo']:
                            north = str(a['GPSInfo'][16])
                            azimuth = str(float(a['GPSInfo'][17][0])/float(a['GPSInfo'][17][1]))
                        else:
                            north = ''
                            azimuth = ''

                        maker = ''
                        model = ''

                self.lon.append(lon)
                self.lat.append(lat)
                self.truePhotosCount = self.truePhotosCount + 1

                if Qgis.QGIS_VERSION >= '3.3':
                    # get Layer type
                    featureDefn = layer.GetLayerDefn()

                    # Create Feature
                    feature = ogr.Feature(featureDefn)

                    # Set fields
                    feature.SetField("ID", uuid_)
                    feature.SetField("Name", name)
                    feature.SetField("Date", date)
                    feature.SetField("Time", time_)
                    feature.SetField("Lon", str(lon))
                    feature.SetField("Lat", str(lat))
                    feature.SetField("Altitude", altitude)
                    feature.SetField("North", north)
                    feature.SetField("Azimuth", azimuth)
                    feature.SetField("Camera Maker", str(maker))
                    feature.SetField("Camera Model", str(model))
                    feature.SetField("Path", imgpath)

                    # create the point
                    point = ogr.Geometry(ogr.wkbPoint)
                    point.AddPoint(lon, lat)

                    # add point
                    feature.SetGeometry(point)

                    # add new feature to layer
                    layer.CreateFeature(feature)
                else:
                    geo_info = {"type": "Feature", "properties": {'ID': uuid_, 'Name': name, 'Date': date, 'Time': time_, 'Lon': lon,
                                               'Lat': lat, 'Altitude': altitude, 'North': north, 'Azimuth': azimuth,
                                               'Camera Maker': str(maker), 'Camera Model': str(model), 'Path': imgpath},
                                "geometry": {"coordinates": [lon, lat], "type": "Point"}}
                    geoPhotos.append(geo_info)

            except:
                pass
        # Destroy
        if Qgis.QGIS_VERSION >= '3.3':
            del photos, Shp, point, feature, layer
        else:
            geojson = {"type": "FeatureCollection",
                       "name": lphoto,
                       "crs": {"type": "name", "properties": {"name": "crs:OGC:1.3:CRS84"}},
                       "features": geoPhotos}
            geofile = open(self.outDirectoryPhotosGeoJSON, 'w')
            json.dump(geojson, geofile)
            del geoPhotos, photos
            geofile.close()

        try:
            for layer in self.canvas.layers():
                if layer.publicSource() == self.outputPath:
                    Qpr_inst.instance().removeMapLayer(layer.id())
                    os.remove(self.outputPath)
        except:
            pass

        if platform.system() == 'Linux':
            self.outputPath = self.outputPath + self.extension
            self.extension = self.extension_switch[self.extension]
        else:
            self.extension = self.extension_switch[self.extension.lower()]

        self.layerPhotos = QgsVectorLayer(self.outDirectoryPhotosGeoJSON, 'temp', "ogr")

        QgsVectorFileWriter.writeAsVectorFormat(self.layerPhotos, self.outputPath, "utf-8",
                                                    QgsCoordinateReferenceSystem(self.layerPhotos.crs().authid()),
                                                    self.extension)
        del self.layerPhotos
        self.layerPhotos_final = QgsVectorLayer(self.outputPath, lphoto, "ogr")

        if Qgis.QGIS_VERSION < '3.3':
            # clear temp.geojson file
            f = open(self.outDirectoryPhotosGeoJSON, 'r+')
            f.truncate(0)  # need '0' when using r+

        Qpr_inst.addMapLayers([self.layerPhotos_final])

        try:
            if Qgis.QGIS_VERSION >= '3.0':
                self.layerPhotos_final.loadNamedStyle(self.plugin_dir + "/svg/photos3.qml")
            else:
                self.layerPhotos_final.loadNamedStyle(self.plugin_dir + "/svg/photos2.qml")
        except:
            title = 'Warning'
            msg = 'No geo-tagged images were detected.'
            self.showMessage(title, msg, 'Warning')
            return

        self.layerPhotos_final.setReadOnly(False)
        self.layerPhotos_final.reload()
        self.layerPhotos_final.triggerRepaint()

        try:
            xmin = min(self.lon)
            ymin = min(self.lat)
            xmax = max(self.lon)
            ymax = max(self.lat)
            self.canvas.zoomToSelected(self.layerPhotos_final)
            self.canvas.setExtent(QgsRectangle(xmin, ymin, xmax, ymax))
        except:
            pass

        self.dlg.progressBar.setValue(100)
        self.dlg.progressBar.setValue(0)
        ###########################################
        noLocationPhotosCounter = initphotos - self.truePhotosCount
        if self.truePhotosCount == noLocationPhotosCounter == 0 or self.truePhotosCount == 0:
            title = 'Import Photos'
            msg='Import Completed.\n\nDetails:\n  No new photos were added.'
            self.showMessage(title, msg, 'Information')
        elif (self.truePhotosCount == initphotos) or ((noLocationPhotosCounter + self.truePhotosCount) == initphotos):
            title = 'Import Photos'
            msg='Import Completed.\n\nDetails:\n  ' + str(
                self.truePhotosCount) + ' photo(s) added without error.\n  ' + str(
                noLocationPhotosCounter) + ' photo(s) skipped (because of missing location).'
            self.showMessage(title, msg, 'Information')

        self.dlg.ok.setEnabled(True)
        self.dlg.closebutton.setEnabled(True)
        self.dlg.toolButtonImport.setEnabled(True)
        self.dlg.toolButtonOut.setEnabled(True)
        self.clickPhotos.setChecked(True)

    def refresh(self):  # Deselect features
        mc = self.canvas
        for layer in mc.layers():
            if layer.type() == layer.VectorLayer:
                layer.removeSelection()
        mc.refresh()

    def showMessage(self, title, msg, icon):
        if icon == 'Warning':
            icon = QMessageBox.Warning
        elif icon == 'Information':
            icon = QMessageBox.Information

        msgBox = QMessageBox()
        msgBox.setIcon(icon)
        msgBox.setWindowTitle(title)
        msgBox.setText(msg)
        msgBox.setWindowFlags(Qt.CustomizeWindowHint | Qt.WindowStaysOnTopHint | Qt.WindowCloseButtonHint)
        msgBox.exec_()


######################################################
# based on http://www.codegists.com/snippet/python/exif_gpspy_snakeye_python

    def _get_if_exist(self, data, key):
        if key in data:
            return data[key]

        return None


    def _convert_to_degress(self, value):
        """
        Helper function to convert the GPS coordinates stored in the EXIF to degress in float format

        :param value:
        :type value: exifread.utils.Ratio
        :rtype: float
        """
        d = float(value.values[0].num) / float(value.values[0].den)
        m = float(value.values[1].num) / float(value.values[1].den)
        s = float(value.values[2].num) / float(value.values[2].den)

        return d + (m / 60.0) + (s / 3600.0)


    def get_exif_location(self, exif_data, lonlat):
        """
        Returns the latitude and longitude, if available, from the provided exif_data (obtained through get_exif_data above)
        """

        if lonlat=='lonlat':
            lat = ''
            lon = ''
            gps_latitude = self._get_if_exist(exif_data, 'GPS GPSLatitude')
            gps_latitude_ref = self._get_if_exist(exif_data, 'GPS GPSLatitudeRef')
            gps_longitude = self._get_if_exist(exif_data, 'GPS GPSLongitude')
            gps_longitude_ref = self._get_if_exist(exif_data, 'GPS GPSLongitudeRef')

            if gps_latitude and gps_latitude_ref and gps_longitude and gps_longitude_ref:
                lat = self._convert_to_degress(gps_latitude)
                if gps_latitude_ref.values[0] != 'N':
                    lat = 0 - lat

                lon = self._convert_to_degress(gps_longitude)
                if gps_longitude_ref.values[0] != 'E':
                    lon = 0 - lon

            return lat, lon
