# -*- coding: utf-8 -*-
"""
/***************************************************************************
 IdentifProj

 This QGIS plugin is an easy way to guess which map projection has been used for a location.
 
 The plugin has 3 use cases :
 - type projected coordinates and get all thez possible points all over the world
 - click on a location on the map and find all the possible projected coordinates
 - draw a bbox and find all the projected bboxes
 
 IMPORTANT: at the first start, the plugin will build its CRS database from Qgis CRS list. 
 It can last au couple of minutes but it will only happen one time.
 This plugin has been initially developed during a third year engineering project at ENSG (https://www.ensg.eu)
 
 Licence : open licence 2.0
 
 (C) 2024 by Leonie leroux, Jacques Beilin
 leonie.leroux@ensg.eu, jacques.beilin@ensg.eu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from qgis.PyQt.QtCore import Qt

from PyQt5.QtWidgets import QTreeWidgetItem, QProgressDialog

from qgis.core import QgsCoordinateReferenceSystem
from qgis.core import QgsCoordinateTransform
from qgis.core import QgsCsException
from qgis.core import QgsFeature
from qgis.core import QgsPoint
from qgis.core import QgsProject
from qgis.core import QgsRectangle
from qgis.core import QgsReferencedPointXY
from qgis.core import QgsVectorLayer
from qgis.core import QgsVectorFileWriter

from qgis.gui import QgsMapTool

import processing

# Initialize Qt resources from file resources.py
from .resources import *

import os.path

class Point2Coord(QgsMapTool):
    """
    Contient les méthodes relatives à la fonctionnalité Point2Coord
    
    Cette classe hérite de la classe QgsMapTool pour rendre plus facile la gestion des 
    intéractions sur la carte (notmment utiliser des méthodes déjà implémentées)
    """
    
    def __init__(self, iface, dockwidget, data, config, IdentifProj):
        """
        Constructeur

        iface: QgsInterface; interface du plugin (qui est la même pour toutes les classes)
        dockwidget : Composante graphique de l'interface 
        data : liste de dictonnaire ; correspond au json de configuration chargé dans IdentifProj
        """
        self.iface = iface
        super().__init__(iface.mapCanvas())
        self.canvas = iface.mapCanvas()
        self.p2c_isActive = True
        self.dockwidget = dockwidget
        self.data = data
        self.config = config
        self.script_dir = config["script_dir"]
        self.IdentifProj = IdentifProj
    
    def classif_pt(self, point):
        """
        Méthode qui assigne à un point une partie du monde dans laquelle ce point se situe
        (sur le même principe de que la méthode check_crs_bounds dans IdentifProj)
        
        point : QgsPointXY
        
        primary_region: liste de String ; correspond à la première zone dans laquelle se trouve le point
        sec_region: liste de String ; correspond à la seconde zone dans laquelle se trouve le point
        """
        
        # Définir les rectangles
        west = QgsRectangle(-180.0, -90.0, -60.0, 90.0)
        mid = QgsRectangle(-60.0, -90.0, 60.0, 90.0)
        east = QgsRectangle(60.0, -90.0, 180.0, 90.0)
        
        north = QgsRectangle(-180.0, 0.0, 180.0, 90.0)
        south = QgsRectangle(-180.0, -90.0, 180.0, 0.0)
    
        primary_region = ""
        sec_region = ""
    
        # Vérifier dans quelle région primaire se trouve le point
        if west.contains(point):
            primary_region = "West"
        elif mid.contains(point):
            primary_region = "Middle"
        elif east.contains(point):
            primary_region = "East"
    
        # Vérifier la région secondaire (Nord ou Sud)
        if north.contains(point):
            sec_region = primary_region + " North"
        elif south.contains(point):
            sec_region = primary_region + " South"
    
        return primary_region, sec_region

        
    def canvasPressEvent(self, event):
        """
        Méthode qui détecte un clic sur la carte, affiche les coordonnées et transforme les coordonnées cliquées dans les 
        différentes projections (+ affichage des résultats dans l'interface)
        
        event: évènement à venir détecter (le clic sur la carte)
        
        
        """
        
        skippedCrsTypes =  self.config["skippedCrsTypes"] 
        skippedAutorities = self.config["skippedAutorities"]
        
        ## Définition du crs de la carte (reste la même pendant tout le projet)
        crs_map = QgsCoordinateReferenceSystem("EPSG:4326")
        crs4326 = QgsCoordinateReferenceSystem("EPSG:4326")
        
        ## Pour effacer les résultats du précédent point
        ## "outputFrame" est le tableau de résultat des coordonnées transformées
        self.dockwidget.outputFrame.clear() 
        
        ## Test si la fonctionnalité est activée
        if self.p2c_isActive == True: 
            
            projectCrs = QgsProject.instance().crs()
            print(projectCrs)
            
            trf = QgsCoordinateTransform(projectCrs, crs4326, QgsProject.instance())
            trf.setBallparkTransformsAreAppropriate(True)
            
            ## Récupération des coordonnées du point cliqué + affichage de ses coordonnées
            point = self.toMapCoordinates(event.pos())
            
            point4326 = trf.transform(point)
            pt = QgsReferencedPointXY(point4326, crs_map)
            coords_text = "WGS84 Coordinates of the point : \u03BB = %.5f° \u03C6 = %.5f° " % (point4326.x(), point4326.y())
            self.dockwidget.outputPT.setPlainText(coords_text)
            
            """ Recuperation des CRS intersectes > shapefile intersect.shp """
            
            crs = "epsg:4326"
            lyr = QgsVectorLayer('Point?crs=' + crs, 'points', "memory")
            lyr.startEditing()
            prv = lyr.dataProvider()
            
            point = QgsPoint(point4326.x(), point4326.y())
            print(point)
            ftr = QgsFeature()
            ftr.setGeometry(point)
            prv.addFeatures([ftr])
    
            commited = lyr.commitChanges()
            if commited:
                print("tmp point layer created in memory")
            else:
                print(f'{lyr.commitErrors()}')
                
            try:
                if not os.path.exists(os.path.join(self.script_dir, "tmp")):
                    os.mkdir(os.path.join(self.script_dir, "tmp"))
            except:
                print('Unable to create tmp dir')
                
            # Write to an ESRI Shapefile format dataset using UTF-8 text encoding
            fileTMP = os.path.join(self.script_dir, "tmp", "point.shp")
            save_options = QgsVectorFileWriter.SaveVectorOptions()
            save_options.driverName = "ESRI Shapefile"
            save_options.fileEncoding = "UTF-8"
            transform_context = QgsProject.instance().transformContext()
            QgsVectorFileWriter.writeAsVectorFormatV3(lyr, fileTMP, transform_context, save_options)
    
            bboxWGS84 = os.path.join(os.path.join(self.script_dir, "config"), "bboxWGS84.gpkg")
            self.lyr_bboxWGS84 = QgsVectorLayer(bboxWGS84, 'bboxWGS84', "ogr")
            QgsProject.instance().addMapLayer(self.lyr_bboxWGS84, addToLegend=False)
            
            parameters = {'INPUT':bboxWGS84,
                          'PREDICATE':[0,1],
                          'INTERSECT':fileTMP,
                          'METHOD':0}
            print("native:selectbylocation", parameters)
            ret = processing.run("native:selectbylocation", parameters)
            print(ret)
            
            pathLayerIntersec = os.path.join(self.script_dir, "tmp", "intersec.shp")
            parameters = { 'INPUT' : bboxWGS84, 'OUTPUT' : pathLayerIntersec}
            print('native:saveselectedfeatures', parameters)
            ret = processing.run('native:saveselectedfeatures', parameters) 
            print(ret)   
            
            lyr_intersect = QgsVectorLayer(pathLayerIntersec, 'Candidate CRS', "ogr")
            CandidateCRS = lyr_intersect.getFeatures()

            nCandidateCrs = lyr_intersect.featureCount()
            print("#Candidate Crs : ", nCandidateCrs)

            ## On classe le point en fonction de sa zone primaire et secondaire
            crs_point = pt.crs()
            # prim, sec = self.classif_pt(pt)
            
            # ## Récupération des données de configuration sous la forme d'un dictionnaire
            # data = self.data
            
            # ## On filtre les SRC disponibles pour ne cibler que les SRC valides dans la zone du point (gain de temps)
            # crs_filtre = [item for item in data if sec in item.get("sec region", [])]
            
            ## Créer et configurer la barre de progression
            nMiniCandidateCrs = 0
            if nCandidateCrs > nMiniCandidateCrs:
                progress = QProgressDialog("Calcul en cours...", "Annuler", 0, nCandidateCrs, self.iface.mainWindow())
                progress.setWindowModality(Qt.WindowModal)
                progress.setMinimumDuration(1000)
                progress.setValue(0)
            i = 0 
            
            Lacronym = []
            Loperation = []
            
            ## bouclesur le SRC filtrés
            # for crs in crs_filtre:
            for feature in CandidateCRS:
                
                ## Vérifier si l'utilisateur a annulé l'opération
                if nCandidateCrs > nMiniCandidateCrs:
                    if progress.wasCanceled():
                        break
                
                ## Mettre à jour la barre de progression
                i+=1
                if nCandidateCrs > nMiniCandidateCrs:
                    progress.setValue(i)
                
                ## Récupération des attributs du SRC  
                attrs = feature.attributes()
                
                epsg = attrs[feature.fieldNameIndex("auth_id")]
                
                projectionAcronym = attrs[feature.fieldNameIndex("acronym")]
                operationDescription = attrs[feature.fieldNameIndex("operation")]
                
                areas = str(attrs[feature.fieldNameIndex("areas1")]) \
                    + str(attrs[feature.fieldNameIndex("areas2")]) \
                    + str(attrs[feature.fieldNameIndex("areas3")]) \
                    + str(attrs[feature.fieldNameIndex("areas4")]) \
                    + str(attrs[feature.fieldNameIndex("areas5")]) \
                    + str(attrs[feature.fieldNameIndex("areas6")]) \
                
                # epsg = crs["auth_id"]
                crs_newpt = QgsCoordinateReferenceSystem(epsg)
                try:
                    type_crs_newpt = crs_newpt.type()
                    if type_crs_newpt in skippedCrsTypes:
                        continue
                except:
                    type_crs_newpt= ""
                bounds = crs_newpt.bounds()
                
                ## Vérifier si le point est dans les limites de la projection + si ce n'est pas un SRC ESRI
                if bounds.contains(pt) and not crs_newpt.authid()[:4] in skippedAutorities:
                
                    context = QgsProject.instance().transformContext()
                    transformer = QgsCoordinateTransform(crs_point, crs_newpt, context)
                    
                    Lacronym += [projectionAcronym]
                    Loperation += [operationDescription]
                    
                    try:
                        ## Essayer de transformer le point
                        newpt = transformer.transform(pt)
                        newpt = QgsReferencedPointXY(newpt, crs_newpt)
                        
                        ## Ajouter les coordonnées transformées à l'interface
                        crs_display = str(newpt.crs().authid())
                        crs_name = str(newpt.crs().description())
                        x = f"{newpt.x():.0f}"
                        y = f"{newpt.y():.0f}"
                        item = QTreeWidgetItem([crs_display, x, y])
                        item.setToolTip(0, crs_name)
                        item.setToolTip(1, crs_name)
                        item.setToolTip(2, crs_name)
                        
                        """ add attributes to make filtering possible """
                        item.projectionAcronym = projectionAcronym
                        item.operationDescription = operationDescription
                        item.areas = areas
                        
                        # print(crs_display, areas)
                        
                        self.dockwidget.outputFrame.addTopLevelItem(item)
                
                    except QgsCsException as e:
                        ## Gérer l'erreur de transformation (par exemple, ignorer cette transformation)
                        print(f"Erreur lors de la transformation vers {crs_newpt.authid()}: {e}")
                        pass
                    
                else:
                    pass
                
                self.dockwidget.outputFrame.setSortingEnabled(True)
                self.dockwidget.outputFrame.sortByColumn(0, Qt.AscendingOrder)
                
                
            """ set filter to list of all projection found """
            Lacronym = sorted(list(set(Lacronym)))
            Loperation = sorted(list(set(Loperation)))
            self.dockwidget.filter_Point2Coord_byProjType.clear()
            self.dockwidget.filter_Point2Coord_byProjType.addItem("")
            for operation in Loperation:
                if isinstance(operation, str):
                    self.dockwidget.filter_Point2Coord_byProjType.addItem(operation)
                
            ## Fermer la barre de progression
            if nCandidateCrs > nMiniCandidateCrs:
                progress.close()
                
                
            
            
            
            
            
            
            