# -*- coding: utf-8 -*-
"""
/***************************************************************************
 IdentifProj

 This QGIS plugin is an easy way to guess which map projection has been used for a location.
 
 The plugin has 3 use cases :
 - type projected coordinates and get all thez possible points all over the world
 - click on a location on the map and find all the possible projected coordinates
 - draw a bbox and find all the projected bboxes
 
 IMPORTANT: at the first start, the plugin will build its CRS database from Qgis CRS list. 
 It can last au couple of minutes but it will only happen one time.
 This plugin has been initially developed during a third year engineering project at ENSG (https://www.ensg.eu)
 
 Licence : open licence 2.0
 
 (C) 2024 by Leonie leroux, Jacques Beilin
 leonie.leroux@ensg.eu, jacques.beilin@ensg.eu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from qgis.PyQt.QtCore import Qt

from qgis.core import QgsCsException
from qgis.core import QgsCoordinateReferenceSystem
from qgis.core import QgsCoordinateTransform
from qgis.core import QgsFeature
from qgis.core import QgsGeometry
from qgis.core import QgsPoint
from qgis.core import QgsPointXY
from qgis.core import QgsProject
from qgis.core import QgsRectangle
from qgis.core import QgsReferencedPointXY
from qgis.core import QgsVectorFileWriter
from qgis.core import QgsVectorLayer

from PyQt5.QtWidgets import QTreeWidgetItem

from qgis.gui import QgsMapTool
from qgis.gui import QgsRubberBand
from PyQt5.QtGui import QColor

from qgis._core import Qgis
from PyQt5.QtWidgets import QProgressDialog

import processing

# Initialize Qt resources from file resources.py
from .resources import *

import os

class BBox2Coord(QgsMapTool):
    """
    Contient les méthodes relatives à la fonctionnalité BBox2Coord
    
    Cette classe hérite de la classe QgsMapTool pour rendre plus facile la gestion des 
    intéractions sur la carte (notmment utiliser des méthodes déjà implémentées)
    """
    
    ### Construction de la classe
    def __init__(self, iface, dockwidget, data, config):
        """
        Constructeur

        iface: QgsInterface; interface du plugin (qui est la même pour toutes les classes)
        dockwidget : Composante graphique de l'interface 
        data : liste de dictonnaire ; correspond au json de configuration chargé dans IdentifProj
        """
        
        self.iface = iface
        super().__init__(iface.mapCanvas())
        self.canvas = iface.mapCanvas()
        self.rubberBand = QgsRubberBand(self.canvas, Qgis.GeometryType.Polygon)
        self.rubberBand.setColor(QColor(255, 0, 0, 100))  # Semi-transparent red
        self.rubberBand.setWidth(2)
        self.start_point = None
        self.end_point = None
        self.is_drawing = False  # État pour savoir si on dessine
        self.dockwidget = dockwidget
        self.config = config
        self.script_dir = config["script_dir"]
        self.data = data
        
    ### Méthodes intermédiaire
    def check_polygon_regions(self, polygon):
        """
        Méthode qui assigne à un polygon une (ou plusieurs) partie du monde dans laquelle il se situe
        (sur le même principe de que la méthode check_crs_bounds dans IdentifProj)
        
        polygon : QgsGeometry
        
        primary_regions: liste de String ; correspond aux regions primaires dans lesquelles se trouvent le polygone
        secondary_regions: liste de String ; aux regions secondaire dans lesquelles se trouvent le polygone
        """
            
        ## Définir les rectangles
        west = QgsRectangle(-180.0, -90.0, -60.0, 90.0)
        mid = QgsRectangle(-60.0, -90.0, 60.0, 90.0)
        east = QgsRectangle(60.0, -90.0, 180.0, 90.0)
        
        north = QgsRectangle(-180.0, 0.0, 180.0, 90.0)
        south = QgsRectangle(-180.0, -90.0, 180.0, 0.0)
    
        primary_regions = []
        secondary_regions = []
    
        ## Vérifier les intersections avec les premières et secondes classes
        if polygon.intersects(west):
            primary_regions.append("West")
            if polygon.intersects(north):
                secondary_regions.append("West North")
            if polygon.intersects(south):
                secondary_regions.append("West South")
        
        if polygon.intersects(mid):
            primary_regions.append("Middle")
            if polygon.intersects(north):
                secondary_regions.append("Middle North")
            if polygon.intersects(south):
                secondary_regions.append("Middle South")
        
        if polygon.intersects(east):
            primary_regions.append("East")
            if polygon.intersects(north):
                secondary_regions.append("East North")
            if polygon.intersects(south):
                secondary_regions.append("East South")
        
        return primary_regions, secondary_regions

    ### Gestion de l'interface graphique et méthode principale     
    def canvasPressEvent(self, event):
        """
        Gestion des clics pour l'affichage de la BBox: la BBox est contruite avec deux clics utilisateur
        qui correspondent à deux coins
        
        Pour créer une BBox, il faut faire deux clic gauches, pour supprimer la BBox, utiliser le clic droit
        
        even : évènement clic
        """
        
        ## Gestion des clics gauche
        if event.button() == Qt.LeftButton:
            if not self.is_drawing:
                ## Premier clic : définir le point de départ
                self.start_point = self.toMapCoordinates(event.pos())
                self.is_drawing = True
                self.rubberBand.reset(Qgis.GeometryType.Polygon)
            else:
                ## Deuxième clic : définir le point de fin
                self.end_point = self.toMapCoordinates(event.pos())
                self.is_drawing = False
                self.finalize_bounding_box()
                
        ## Gestion du clic droit       
        elif event.button() == Qt.RightButton:
            ## Clic droit : réinitialiser la bounding box
            self.reset_bounding_box()
    
    def reset_bounding_box(self):
        """
        Méthode attachée au clic droit qui supprimer la BBox affichée en cours
        """
        
        self.rubberBand.reset(Qgis.GeometryType.Polygon)
        self.start_point = None
        self.end_point = None
        self.is_drawing = False
        print("Bounding box réinitialisée.")

    
    def canvasMoveEvent(self, event):
        """
        Méthode pour l'affichage/pré-visualisation de la BBox lorsque l'utilisateur fait glissé la souris
        
        event : évènement sur la map
        """
        
        if self.is_drawing and self.start_point:
            # Mise à jour dynamique du rectangle pendant le déplacement de la souris
            current_point = self.toMapCoordinates(event.pos())
            self.update_rubber_band(self.start_point, current_point)


    
    def update_rubber_band(self, start_point, end_point):
        """
        Met à jour l'affichage du rectangle.
        Rectangle qui s'affiche en fonction de comment l'utilisateur bouge sa souris
        
        start_point: QgsPointXY
        end_point: QgsPointXY
        """
        
        rect = QgsRectangle(start_point, end_point)
        polygon = QgsGeometry.fromPolygonXY([[QgsPointXY(rect.xMinimum(), rect.yMinimum()),
                                              QgsPointXY(rect.xMinimum(), rect.yMaximum()),
                                              QgsPointXY(rect.xMaximum(), rect.yMaximum()),
                                              QgsPointXY(rect.xMaximum(), rect.yMinimum()),
                                              QgsPointXY(rect.xMinimum(), rect.yMinimum())]])
        self.rubberBand.setToGeometry(polygon, QgsProject.instance().crs())
        
    
    def finalize_bounding_box(self):
        """
        Méthode qui calcul les coordonnées de la BBox finale (lorsque l'utilisateur a fait deux clics gauche)
                                                              
                                                            
        """
        
        skippedCrsTypes =  self.config["skippedCrsTypes"] 
        skippedAutorities = self.config["skippedAutorities"]
        
        crs4326 = QgsCoordinateReferenceSystem("EPSG:4326")
        
        Loperation = []
        
        ## Test si l'utilisateur a bien fait deux clics gauche (les deux coins du rectangle)
        if self.start_point and self.end_point:
            
            ## Pour effacer les résultats de la précédente BBox
            ## outputpoly: tableau d'affichage des coordonnées finales
            self.dockwidget.outputPoly.clear() 
            
            ## Créer un rectangle final basé sur les deux points
            self.update_rubber_band(self.start_point, self.end_point)
            
            # txt = f"BBox coordinates (in WGS84): {rect}"#".toString()}"
            
            projectCrs = QgsProject.instance().crs()
            print(projectCrs)
            
            trf = QgsCoordinateTransform(projectCrs, crs4326, QgsProject.instance())
            trf.setBallparkTransformsAreAppropriate(True)
            
            rectProjectCRS = QgsRectangle(self.start_point, self.end_point)
            rect4326 = trf.transformBoundingBox(rectProjectCRS, handle180Crossover=True)
            
            txt = "Bbox (WGS84) : \u03BB = %.5f° \u03C6 = %.5f° to \u03BB = %.5f° \u03C6 = %.5f°" % (rect4326.xMinimum(), rect4326.yMinimum(), rect4326.xMaximum(), rect4326.yMaximum())
            self.dockwidget.polywgs84.setPlainText(txt)
            
            """ Recuperation des CRS intersectes > shapefile intersect.shp """
            
            crs = "epsg:4326"
            lyr = QgsVectorLayer('Polygon?crs=' + crs, 'points', "memory")
            lyr.startEditing()
            prv = lyr.dataProvider()
            
            ftr = QgsFeature()
            geom4326 = QgsGeometry.fromRect(rect4326)
            ftr.setGeometry(geom4326)
            prv.addFeatures([ftr])
    
            commited = lyr.commitChanges()
            if commited:
                print("tmp point layer created in memory")
            else:
                print(f'{lyr.commitErrors()}')
                
            try:
                if not os.path.exists(os.path.join(self.script_dir, "tmp")):
                    os.mkdir(os.path.join(self.script_dir, "tmp"))
            except:
                print('Unable to create tmp dir')
                
            # Write to an ESRI Shapefile format dataset using UTF-8 text encoding
            fileTMP = os.path.join(self.script_dir, "tmp", "point.shp")
            save_options = QgsVectorFileWriter.SaveVectorOptions()
            save_options.driverName = "ESRI Shapefile"
            save_options.fileEncoding = "UTF-8"
            transform_context = QgsProject.instance().transformContext()
            QgsVectorFileWriter.writeAsVectorFormatV3(lyr, fileTMP, transform_context, save_options)

            bboxWGS84 = os.path.join(os.path.join(self.script_dir, "config"), "bboxWGS84.gpkg")
            self.lyr_bboxWGS84 = QgsVectorLayer(bboxWGS84, 'bboxWGS84', "ogr")
            QgsProject.instance().addMapLayer(self.lyr_bboxWGS84, addToLegend=False)
            
            crs_poly = QgsCoordinateReferenceSystem("EPSG:4326")
            
            parameters = {'INPUT':bboxWGS84,
                          'PREDICATE':[0,1],
                          'INTERSECT':fileTMP,
                          'METHOD':0}
            print("native:selectbylocation", parameters)
            ret = processing.run("native:selectbylocation", parameters)
            print(ret)
            
            pathLayerIntersec = os.path.join(self.script_dir, "tmp", "intersec.shp")
            parameters = { 'INPUT' : bboxWGS84, 'OUTPUT' : pathLayerIntersec}
            print('native:saveselectedfeatures', parameters)
            ret = processing.run('native:saveselectedfeatures', parameters) 
            print(ret)   
            
            lyr_intersect = QgsVectorLayer(pathLayerIntersec, 'Candidate CRS', "ogr")
            CandidateCRS = lyr_intersect.getFeatures()

            nCandidateCrs = lyr_intersect.featureCount()
            print("#Candidate Crs : ", nCandidateCrs)
            
            ## Créer et configurer la barre de progression
            progress = QProgressDialog("Calcul en cours...", "Annuler", 0, nCandidateCrs, self.iface.mainWindow())
            progress.setWindowModality(Qt.WindowModal)
            progress.setMinimumDuration(0)
            progress.setValue(0)
            i = 0
            
            ## on boucle sur la liste des crs filtrée
            for feature in CandidateCRS:
                
                ## Vérifier si l'utilisateur a annulé l'opération
                if progress.wasCanceled():
                    break
                ## Mettre à jour la barre de progression
                i+=1
                progress.setValue(i)
                
                ## Récupération des attributs du SRC  
                attrs = feature.attributes()
                
                epsg = attrs[feature.fieldNameIndex("auth_id")]
                operationDescription = attrs[feature.fieldNameIndex("operation")]
                
                # areas = attrs[feature.fieldNameIndex("areas")]
                areas = str(attrs[feature.fieldNameIndex("areas1")]) \
                    + str(attrs[feature.fieldNameIndex("areas2")]) \
                    + str(attrs[feature.fieldNameIndex("areas3")]) \
                    + str(attrs[feature.fieldNameIndex("areas4")]) \
                    + str(attrs[feature.fieldNameIndex("areas5")]) \
                    + str(attrs[feature.fieldNameIndex("areas6")]) \
                
                crs_newpoly = QgsCoordinateReferenceSystem(epsg)
                try:
                    type_crs_newpoly = crs_newpoly.type()
                    if type_crs_newpoly in skippedCrsTypes:
                        continue
                except:
                    type_crs_newpoly = ""
                bounds = crs_newpoly.bounds()
                
                ## Vérifier si le point est dans les limites + si ce n'est pas un crs qu'on ne veut pas lister
                boundsCheck = False
                if bounds.contains(rect4326):
                        boundsCheck = True
                    
                if boundsCheck and not crs_newpoly.authid()[:4] in skippedAutorities:
                
                    context = QgsProject.instance().transformContext()
                    transformer = QgsCoordinateTransform(crs_poly, crs_newpoly, context)
                    
                    Loperation += [operationDescription]
                    
                    new_poly_pt = []
                    
                    for pt in geom4326.vertices():
                    
                        try:
                            ## Essayer de transformer le point
                            new_pt = transformer.transform(pt.x(), pt.y())
                            new_poly_pt.append(new_pt)
                    
                        except QgsCsException as e:
                            ## Gérer l'erreur de transformation (par exemple, ignorer cette transformation)
                            print(f"Erreur lors de la transformation vers {crs_newpoly.authid()}: {e}")
                            pass
                    
                    ## Affichage du polygon sur l'interface résultat
                    polygon = QgsGeometry.fromPolygonXY([new_poly_pt])
                    crs_display = str(crs_newpoly.authid()) 
                    crs_name = str(crs_newpoly.description())
                    poly_display = str(polygon)  
                    item = QTreeWidgetItem([crs_display, crs_name])
                    item.setToolTip(0, poly_display)
                    item.setToolTip(1, poly_display)
                    
                    """ add attributes to make filtering possible """
                    item.operationDescription = operationDescription
                    item.areas = areas
                    
                    # """ add attributes to make filtering possible """
                    # item.projectionAcronym = projectionAcronym
                    # item.operationDescription = operationDescription
                    
                    self.dockwidget.outputPoly.addTopLevelItem(item)
                    
                    
                else: 
                    pass
                
            """ set filter to list of all projection found """
            Loperation = sorted(list(set(Loperation)))
            self.dockwidget.filter_BBox2Coord_byProjType.clear()
            self.dockwidget.filter_BBox2Coord_byProjType.addItem("")
            for operation in Loperation:
                if isinstance(operation, str):
                    self.dockwidget.filter_BBox2Coord_byProjType.addItem(operation)
                
            ## Fermer la barre de progression
            progress.close()
            
            ## Réinitialiser start_point / end_point pour tracer une autre BBox
            self.start_point = None
            self.end_point = None
            
            print("fin du traitement")
            

    







