# -*- coding: utf-8 -*-
"""
/***************************************************************************
 IdentifProj

 This QGIS plugin is an easy way to guess which map projection has been used for a location.
 
 The plugin has 3 use cases :
 - type projected coordinates and get all thez possible points all over the world
 - click on a location on the map and find all the possible projected coordinates
 - draw a bbox and find all the projected bboxes
 
 IMPORTANT: at the first start, the plugin will build its CRS database from Qgis CRS list. 
 It can last au couple of minutes but it will only happen one time.
 This plugin has been initially developed during a third year engineering project at ENSG (https://www.ensg.eu)
 
 Licence : open licence 2.0
 
 (C) 2024 by Leonie leroux, Jacques Beilin
 leonie.leroux@ensg.eu, jacques.beilin@ensg.eu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import json
import os
import re
import tempfile
import shutil
import sqlite3

from qgis.core import QgsGeometry
from qgis.core import QgsPathResolver
from qgis.core import QgsPoint
from qgis.core import QgsPolygon
from qgis.core import QgsLineString
from qgis.core import QgsVectorLayer
from qgis.core import QgsFeature
from qgis.core import QgsProject
from qgis.core import QgsVectorFileWriter
from qgis.core import QgsField

import processing

try:
    from PyQt5.QtCore import QVariant
except:
    pass
    
try:
    from PyQt5.QtCore import QMetaType
except:
    pass

def wrap(s, w):
    return [s[i:i + w] for i in range(0, len(s), w)]

def load_data_crs(filename):
    """

    Parameters
    ----------
    filename : str
        CRS filename.

    Returns
    -------
    data : list of dict
        CRS data

    """
    
    with open(filename, 'rt') as f:
        L = f.readlines()
        s = ""
        for l in L:
            s += l
            
    if s:
        data = json.loads(s)
        
        return data
    return None


def create_geoms(jsonCRS):
    """
    Parameters
    ----------
    jsonCRS : dictionary
        dict produced by calc_BBox.

    Returns
    -------
    PolygonWGS84 : QgsPolygon
        bbox in WGS84
    PolygonCRS : QgsPolygon
        bbox in projected coordinates
    auth_id : str
        CRS authority and id.
    name : str
        CRS name

    """
    
    auth_id = jsonCRS["auth_id"]
    name = jsonCRS["name"]
    projectionAcronym = jsonCRS["projectionAcronym"]
    operationDescription = jsonCRS["operationDescription"]
    areas = jsonCRS["areas"]
    bboxWGS84 = jsonCRS["bounding_box_WGS84"]
    bboxCRS = jsonCRS["transform_bounds"]["coordinates"][0]

    lon = [bboxWGS84["x_min"], bboxWGS84["x_max"], bboxWGS84["x_max"], bboxWGS84["x_min"], bboxWGS84["x_min"]]
    lat = [bboxWGS84["y_max"], bboxWGS84["y_max"], bboxWGS84["y_min"], bboxWGS84["y_min"], bboxWGS84["y_max"]]
    PolygonWGS84 = QgsPolygon(QgsLineString(lon, lat))
    
    P1 = QgsPoint(bboxCRS[0][0], bboxCRS[0][1])
    P2 = QgsPoint(bboxCRS[1][0], bboxCRS[1][1])
    P3 = QgsPoint(bboxCRS[2][0], bboxCRS[2][1])
    P4 = QgsPoint(bboxCRS[3][0], bboxCRS[3][1])
   
    PolygonCRS = QgsPolygon(QgsLineString([P1, P2, P3, P4, P1]))
    
    return PolygonWGS84, PolygonCRS, auth_id, name, projectionAcronym, operationDescription, areas

def getProjTypes(fileIN):
    """
    Get all projection types (or coordinate types)

    Parameters
    ----------
    fileIN : filepath
        file path of json created by calc_BBox

    Returns
    -------
    Lacronym : TYPE
        DESCRIPTION.
    Loperation : TYPE
        DESCRIPTION.

    """
    
    Lacronym = []
    Loperation = []
    
    data = load_data_crs(fileIN)
    
    for jsonCRS in data:
        projectionAcronym = jsonCRS["projectionAcronym"]
        operationDescription = jsonCRS["operationDescription"]
    
        Lacronym += [projectionAcronym]
        Loperation += [operationDescription]
        
    Lacronym = sorted(list(set(Lacronym)))
    Loperation = sorted(list(set(Loperation)))
    
    return Lacronym, Loperation

def getProjDBPath():
    """
    get proj.db file used by Qgis

    Returns
    -------
    pathProjDB : str
        DESCRIPTION.

    """
    import os
    from osgeo import osr
    
    pathProjDB = ""
    paths = osr.GetPROJSearchPaths()
    
    for path in paths:
        if os.path.exists(os.path.join(path, "proj.db")):
            pathProjDB = os.path.join(path, "proj.db")
            break
        
    return pathProjDB

def openProjDB():
    return openDB(getProjDBPath())

def openDB(DBfile):
    """
    

    Parameters
    ----------
    DBfile : TYPE
        DESCRIPTION.

    Returns
    -------
    connection : TYPE
        DESCRIPTION.
    cursor
        DESCRIPTION.

    """
    
    try:
        connection = sqlite3.connect(DBfile)
        cursor = connection.cursor()
    except:
        return None, None
    return connection, cursor

def getCountriesTrigrams(mapFilePath):
    """
    get all countries from Qgis world base map

    Parameters
    ----------
    mapFilePath : str
        file path to world map gpkg.

    Returns
    -------
    countries : list of dictionnaries
        ex : {'name': 'Norway', 'trigram': 'NOR'}
        

    """

    connection = sqlite3.connect(mapFilePath)
    cursor = connection.cursor()
    res = cursor.execute("SELECT * FROM countries")
    countries = res.fetchall()
    connection.close()
    Lcountries = []
    for country in countries:
        if not country[3]==None and not country[5]==None:
            Lcountries += [{"name":country[3], "trigram":country[5]}]
    return Lcountries
 
    
def addTrigramsToProjDB(worldMap, projDB):
    """
    Add trigrams to area list une usage table (proj.db)

    Parameters
    ----------
    worldMap : str
        world map geopackage path
    projDB : str
        proj.db file path

    Returns
    -------
    None.

    """
    trigrams = getCountriesTrigrams(worldMap)
    
    connection = sqlite3.connect(projDB)
    cursor = connection.cursor()
    req1 = "SELECT * FROM extent"
    res1 = cursor.execute(req1)
    extents = res1.fetchall()
    
    for extent in extents:
        auth = extent[0]
        code = extent[1]
        # print(auth, code)
        description = extent[3]
        description2 = ""
        for i in range(len(description)):
            if not description[i] == '"':
                description2 += description[i]
        newdescription = description2
        
        for elem in trigrams:
            if re.search(elem["name"], description, re.IGNORECASE):
                trigram = elem["trigram"]
                if re.search(trigram, newdescription):
                    continue
                newdescription += ';' + trigram
        if not newdescription==description:
            
            req2 = "UPDATE extent SET description = \"%s\" WHERE auth_name=\"%s\" AND code=\"%s\"" % (newdescription, str(auth), str(code))
            # print(req2)
            cursor.execute(req2)
    
    connection.commit()   
    cursor.close()
    connection.close()

def getCRSAreaFromProjDB(projDB):
    """
    Crates dictionnary with 
    - key : auth:code for example 'IGNF:LAMB93'
    - value : liste of areas, for example 'FRANCE METROPOLITAINE (CORSE COMPRISE)'

    Parameters
    ----------

    Returns
    -------
    D : dictionnary
        DESCRIPTION.
        D['IGNF:LAMB93'] = 'FRANCE METROPOLITAINE (CORSE COMPRISE)'

    """
    
    connection, cursor = openDB(projDB)

    req = "SELECT object_auth_name, object_code, description FROM usage INNER JOIN extent ON usage.extent_code = extent.code AND usage.extent_auth_name = extent.auth_name"
    # print(req)
    res = cursor.execute(req)
    # print(res.description)
    des = res.description
    for line in des:
        print(line)
    areas = res.fetchall()
    # print(len(areas))
    L = []
    for area in areas:
        # if str(area[1]) == "3034":
        # print(area)
        
        area1 = area[2]
        t = ""
        for l in area1:
            if l == '.':
                continue
            t += l
        area1 = re.sub('"', "", t)
        area1 = re.sub(':', ";", area1)
        # area1 = re.sub('\.', "", area1)
        
        T2 = area1.split(";")
        for i in range(len(T2)):
            T2[i] = T2[i].strip()
        # print(T2)
        L.append({"epsg": area[0] +':'+str(area[1]), "areas": T2})
        
    connection.close()
        
    D = {}
    for l in L:
        D[l["epsg"]] = l["areas"]
    return D


def getCountryListFromProjDB(auth, code, cursor):
    """
    get country or area list from proj.db database

    Parameters
    ----------
    auth : str
        proj authority
    code : str
        proj code
    cursor : TYPE
        sqlite3 cursor

    Returns
    -------
    L : TYPE
        DESCRIPTION.

    """
    
    print("getCountryListFromProjDB for %s:%s", str(auth), str(code))
    
    connection, cursor = openProjDB()
    
    L = []
    
    req = "SELECT * FROM usage WHERE object_auth_name=\"%s\" AND object_code=\"%s\"" % (auth, code)
    print(req)
    res = cursor.execute(req)
    r = res.fetchall()
    # print(r, type(r))

    L = []
    for y in res.description:
        L += [y[0]]
    y = L.index('extent_code')
    code = r[0][y]

    # print(code)

    res = cursor.execute("SELECT * FROM extent WHERE code=%d" % code)
    r = res.fetchall()
    # print(r, type(r))

    L = []
    for y in res.description:
        L += [y[0]]
    y = L.index('description')
    code = r[0][y]
    
    t = ""
    for l in code:
        if l == "[" or l == '.':
            continue
        t += l
    
    t = re.sub(']', "", t)
    t = re.sub('"', "", t)
    t = re.sub(':', ";", t)
    
    T2 = t.split(";")
    for i in range(len(T2)):
        T2[i] = T2[i].strip()

    
    return T2



def getCountryListFromCrs(crs, dataDir):
    """

    Parameters
    ----------
    crs : QgsCoordinateReferenceSystem

    Returns
    -------
    T2 : list
        List of all countries concerned by crs

    """
    try:
        auth, code = crs.authid().split(':')
        # cursor = openDB(getProjDBPath())
        L = getCountryListFromProjDB(auth, code, cursor)
    except:
        L = []
        
    if len(L)>0:
        return L
    
    crs2 = crs.toGeographicCrs()
    
    T = str(crs2).split(',')
    T2 = []
    for t in T:
        if not re.search("AREA", t):
            continue
        
        L = list(t)
        t = ""
        for l in L:
            if l == "[" or l == '.':
                continue
            t += l
        
        t = re.sub('AREA', "", t)
        t = re.sub(']', "", t)
        t = re.sub('"', "", t)
        t = re.sub(':', ";", t)
        
        T2 = t.split(";")
        
        for i in range(len(T2)):
            T2[i] = T2[i].strip()
        
    if len(T2)==1 and re.search("World", T2[0], re.IGNORECASE):
    
        eps = 1e-3
        
        worldMapFile = os.path.join(dataDir, "world_map.gpkg") 
        worldMap = worldMapFile + '|layername=countries'
        
        if crs.bounds().width() < crs2.bounds().width() - eps and crs.bounds().height() < crs2.bounds().height() - eps:
            lyr = QgsVectorLayer('Polygon?crs=epsg:4326', 'area', "memory")
            lyr.startEditing()
            prv = lyr.dataProvider()
            ftr = QgsFeature()
            ftr.setGeometry(QgsGeometry.fromRect(crs.bounds()))
            prv.addFeatures([ftr])
            
            tmpDir = tempfile.mkdtemp()
            
            fileTMP = os.path.join(tmpDir, "poly.gpkg")
            save_options = QgsVectorFileWriter.SaveVectorOptions()
            save_options.driverName = "gpkg"
            save_options.fileEncoding = "UTF-8"
            transform_context = QgsProject.instance().transformContext()
            QgsVectorFileWriter.writeAsVectorFormatV3(lyr, fileTMP, transform_context, save_options)
            
            fileIntersect = os.path.join(tmpDir, "intersec.gpkg")
            parameters = {'INPUT': worldMap,
                          'PREDICATE': [0],
                          'INTERSECT': fileTMP,
                          'METHOD': 0,
                          'OUTPUT':fileIntersect}
            processing.run("native:extractbylocation", parameters)
            
            lyr = QgsVectorLayer(fileIntersect + "|layername=intersec")
            features = lyr.getFeatures()

            for feature in features:
                attrs = feature.attributes()
                name = attrs[feature.fieldNameIndex("NAME")]   
                if isinstance(name, str):
                    T2 += [name]
                
            # print(lyr.featureCount())
            # print(T2)
            shutil.rmtree(tmpDir)
        
    T2.sort()
    return T2

def json2shpWGS84(fileIN, fileOUT):
    """
    
    Parameters
    ----------
    fileIN : filepath
        file path of json created by calc_BBox
    fileOUT : filepath
        shapefile file path for bboxes in WGS84

    Returns
    -------
    None.

    """
    
    data = load_data_crs(fileIN)
 
    crs = "epsg:4326"
    lyr = QgsVectorLayer('Polygon?crs=' + crs, 'CRS', "memory")
    # Prj = QgsProject.instance()  # ADD THIS LINE
    # Prj.addMapLayers([lyr])  

    lyr.startEditing()
    # Set an object for the data provider for the layer
    prv = lyr.dataProvider()
    
    try:
        fAuthid = QgsField("auth_id",  type=QMetaType.QString, len=254)
        fname = QgsField("name",  type=QMetaType.QString, len=254)
        fProjectionAcronym = QgsField("acronym", type=QMetaType.QString, len=254)
        fOperationDescription = QgsField("operation", type=QMetaType.QString, len=254)
        fAreas1 = QgsField("areas1", type=QMetaType.QString, len=254)
        fAreas2 = QgsField("areas2", type=QMetaType.QString, len=254)
        fAreas3 = QgsField("areas3", type=QMetaType.QString, len=254)
        fAreas4 = QgsField("areas4", type=QMetaType.QString, len=254)
        fAreas5 = QgsField("areas5", type=QMetaType.QString, len=254)
        fAreas6 = QgsField("areas6", type=QMetaType.QString, len=254)

    except:
        fAuthid = QgsField("auth_id",  type=QVariant.String, len=254)
        fname = QgsField("name",  type=QVariant.String, len=254)
        fProjectionAcronym = QgsField("acronym", type=QVariant.String, len=254)
        fOperationDescription = QgsField("operation", type=QVariant.String, len=254)
        fAreas1 = QgsField("areas1", type=QVariant.String, len=254)
        fAreas2 = QgsField("areas2", type=QVariant.String, len=254)
        fAreas3 = QgsField("areas3", type=QVariant.String, len=254)
        fAreas4 = QgsField("areas4", type=QVariant.String, len=254)
        fAreas5 = QgsField("areas5", type=QVariant.String, len=254)
        fAreas6 = QgsField("areas6", type=QVariant.String, len=254)

    
    prv.addAttributes([fAuthid, fname, fProjectionAcronym, fOperationDescription, fAreas1, fAreas2, fAreas3, fAreas4, fAreas5, fAreas6])
    lyr.updateFields()     
    
    for jsonCRS in data:
        PolygonWGS84, PolygonCRS, auth_id, name, projectionAcronym, operationDescription, areas = create_geoms(jsonCRS)

        ftr = QgsFeature()
        ftr.setGeometry(PolygonWGS84)
        
        Lstr = wrap(str(areas), 254)
        for i in range(len(Lstr), 6):
            Lstr.append('')
            
        ftr.setAttributes([auth_id, name, projectionAcronym, operationDescription, Lstr[0], Lstr[1], Lstr[2], Lstr[3], Lstr[4], Lstr[4]])                      
    
        # Add the feature to the layer using this provider (fails)
        prv.addFeatures([ftr])
        
    lyr.updateExtents()

    commited = lyr.commitChanges()
    if commited:
        print("ok")
    else:
        print(f'{lyr.commitErrors()}')
        # return False
        
    # Write to a geopackage format dataset using UTF-8 text encoding
    save_options = QgsVectorFileWriter.SaveVectorOptions()
    save_options.driverName = "gpkg"
    save_options.fileEncoding = "UTF-8"
    transform_context = QgsProject.instance().transformContext()
    error = QgsVectorFileWriter.writeAsVectorFormatV3(lyr, fileOUT, transform_context, save_options)
    

    
def json2shpCRS(fileIN, fileOUT):
    """
    
    Parameters
    ----------
    fileIN : filepath
        file path of json created by calc_BBox
    fileOUT : filepath
        shapefile file path for bboxes in native projection

    Returns
    -------
    None.

    """
    
    data = load_data_crs(fileIN)
 
    crs = "epsg:3857"
    lyr = QgsVectorLayer('Polygon?crs=' + crs, 'CRS', "memory")
    # Prj = QgsProject.instance()  # ADD THIS LINE
    # Prj.addMapLayers([lyr])  

    lyr.startEditing()
    # Set an object for the data provider for the layer
    prv = lyr.dataProvider()
    
    try:
        fAuthid = QgsField("auth_id",  type=QMetaType.QString, len=254)
        fname = QgsField("name",  type=QMetaType.QString, len=254)
        fProjectionAcronym = QgsField("acronym", type=QMetaType.QString, len=254)
        fOperationDescription = QgsField("operation", type=QMetaType.QString, len=254)
        fAreas1 = QgsField("areas1", type=QMetaType.QString, len=254)
        fAreas2 = QgsField("areas2", type=QMetaType.QString, len=254)
        fAreas3 = QgsField("areas3", type=QMetaType.QString, len=254)
        fAreas4 = QgsField("areas4", type=QMetaType.QString, len=254)
        fAreas5 = QgsField("areas5", type=QMetaType.QString, len=254)
        fAreas6 = QgsField("areas6", type=QMetaType.QString, len=254)

    except:
        fAuthid = QgsField("auth_id",  type=QVariant.String, len=254)
        fname = QgsField("name",  type=QVariant.String, len=254)
        fProjectionAcronym = QgsField("acronym", type=QVariant.String, len=254)
        fOperationDescription = QgsField("operation", type=QVariant.String, len=254)
        fAreas1 = QgsField("areas1", type=QVariant.String, len=254)
        fAreas2 = QgsField("areas2", type=QVariant.String, len=254)
        fAreas3 = QgsField("areas3", type=QVariant.String, len=254)
        fAreas4 = QgsField("areas4", type=QVariant.String, len=254)
        fAreas5 = QgsField("areas5", type=QVariant.String, len=254)
        fAreas6 = QgsField("areas6", type=QVariant.String, len=254)
    
    prv.addAttributes([fAuthid, fname, fProjectionAcronym, fOperationDescription, fAreas1, fAreas2, fAreas3, fAreas4, fAreas5, fAreas6])
    lyr.updateFields()     
    
    for jsonCRS in data:
        PolygonWGS84, PolygonCRS, auth_id, name, projectionAcronym, operationDescription, areas = create_geoms(jsonCRS)

        ftr = QgsFeature()
        ftr.setGeometry(PolygonCRS)

        Lstr = wrap(str(areas), 254)
        for i in range(len(Lstr), 6):
            Lstr.append('')
            
        ftr.setAttributes([auth_id, name, projectionAcronym, operationDescription, Lstr[0], Lstr[1], Lstr[2], Lstr[3], Lstr[4], Lstr[4]])                                        
    
        # Add the feature to the layer using this provider (fails)
        prv.addFeatures([ftr])

    commited = lyr.commitChanges()
    if commited:
        print("ok")
    else:
        print(f'{lyr.commitErrors()}')
        # return False
        
    lyr.updateExtents()
        
    # Write to a geopackage format dataset using UTF-8 text encoding
    save_options = QgsVectorFileWriter.SaveVectorOptions()
    save_options.driverName = "gpkg"
    save_options.fileEncoding = "UTF-8"
    transform_context = QgsProject.instance().transformContext()
    error = QgsVectorFileWriter.writeAsVectorFormatV3(lyr, fileOUT, transform_context, save_options)
    
def json2shpWGS84_2(fileIN, fileOUT):
    """
    
    Parameters
    ----------
    fileIN : filepath
        file path of json created by calc_BBox
    fileOUT : filepath
        shapefile file path for bboxes in WGS84

    Returns
    -------
    None.

    """
    
    data = load_data_crs(fileIN)
 
    crs = "epsg:4326"
    lyr = QgsVectorLayer('Polygon?crs=' + crs, 'CRS', "memory")
    # Prj = QgsProject.instance()  # ADD THIS LINE
    # Prj.addMapLayers([lyr])  

    lyr.startEditing()
    # Set an object for the data provider for the layer
    prv = lyr.dataProvider()
    
    try:
        fAuthid = QgsField("auth_id",  type=QMetaType.QString, len=254)
        fname = QgsField("name",  type=QMetaType.QString, len=254)
        fProjectionAcronym = QgsField("acronym", type=QMetaType.QString, len=254)
        fOperationDescription = QgsField("operation", type=QMetaType.QString, len=254)
        fAreas = QgsField("areas", type=QMetaType.QString, len=10000)

    except:
        fAuthid = QgsField("auth_id",  type=QVariant.String, len=254)
        fname = QgsField("name",  type=QVariant.String, len=254)
        fProjectionAcronym = QgsField("acronym", type=QVariant.String, len=254)
        fOperationDescription = QgsField("operation", type=QVariant.String, len=254)
        fAreas = QgsField("areas", type=QVariant.String, len=10000)

    
    prv.addAttributes([fAuthid, fname, fProjectionAcronym, fOperationDescription, fAreas])
    lyr.updateFields()     
    
    for jsonCRS in data:
        PolygonWGS84, PolygonCRS, auth_id, name, projectionAcronym, operationDescription, areas = create_geoms(jsonCRS)
        # print(auth_id, name, projectionAcronym, operationDescription)

        ftr = QgsFeature()
        ftr.setGeometry(PolygonWGS84)
        ftr.setAttributes([auth_id, name, projectionAcronym, operationDescription, str(areas)])                      
    
        # Add the feature to the layer using this provider (fails)
        prv.addFeatures([ftr])
        
    lyr.updateExtents()

    commited = lyr.commitChanges()
    if commited:
        print("ok")
    else:
        print(f'{lyr.commitErrors()}')
        # return False
        
    # Write to a geopackage format dataset using UTF-8 text encoding
    save_options = QgsVectorFileWriter.SaveVectorOptions()
    save_options.driverName = "gpkg"
    save_options.fileEncoding = "UTF-8"
    transform_context = QgsProject.instance().transformContext()
    error = QgsVectorFileWriter.writeAsVectorFormatV3(lyr, fileOUT, transform_context, save_options)
    

    
def json2shpCRS_2(fileIN, fileOUT):
    """
    
    Parameters
    ----------
    fileIN : filepath
        file path of json created by calc_BBox
    fileOUT : filepath
        shapefile file path for bboxes in native projection

    Returns
    -------
    None.

    """
    
    data = load_data_crs(fileIN)
 
    crs = "epsg:3857"
    lyr = QgsVectorLayer('Polygon?crs=' + crs, 'CRS', "memory")
    # Prj = QgsProject.instance()  # ADD THIS LINE
    # Prj.addMapLayers([lyr])  

    lyr.startEditing()
    # Set an object for the data provider for the layer
    prv = lyr.dataProvider()
    
    try:
        fAuthid = QgsField("auth_id",  type=QMetaType.QString, len=254)
        fname = QgsField("name",  type=QMetaType.QString, len=254)
        fProjectionAcronym = QgsField("acronym",  type=QMetaType.QString, len=254)
        fOperationDescription = QgsField("operation",  type=QMetaType.QString, len=254)
        fAreas = QgsField("areas", type=QMetaType.QString, len=10000)

    except:
        fAuthid = QgsField("auth_id",  type=QVariant.String, len=254)
        fname = QgsField("name",  type=QVariant.String, len=254)
        fProjectionAcronym = QgsField("acronym",  type=QVariant.String, len=254)
        fOperationDescription = QgsField("operation",  type=QVariant.String, len=254)
        fAreas = QgsField("areas", type=QVariant.String, len=10000)
    
    prv.addAttributes([fAuthid, fname, fProjectionAcronym, fOperationDescription, fAreas]) 
    lyr.updateFields()     
    
    for jsonCRS in data:
        PolygonWGS84, PolygonCRS, auth_id, name, projectionAcronym, operationDescription, areas = create_geoms(jsonCRS)

        ftr = QgsFeature()
        ftr.setGeometry(PolygonCRS)
        ftr.setAttributes([auth_id, name, projectionAcronym, operationDescription, str(areas)])                      
    
        # Add the feature to the layer using this provider (fails)
        prv.addFeatures([ftr])

    commited = lyr.commitChanges()
    if commited:
        print("ok")
    else:
        print(f'{lyr.commitErrors()}')
        # return False
        
    lyr.updateExtents()
        
    # Write to a geopackage format dataset using UTF-8 text encoding
    save_options = QgsVectorFileWriter.SaveVectorOptions()
    save_options.driverName = "gpkg"
    save_options.fileEncoding = "UTF-8"
    transform_context = QgsProject.instance().transformContext()
    error = QgsVectorFileWriter.writeAsVectorFormatV3(lyr, fileOUT, transform_context, save_options)
    
    
# workingdir = "/home/beilin/.local/share/QGIS/QGIS3/profiles/default/python/plugins/IdentifProj/config/"   
# fileIN = os.path.join(workingdir, "crs_with_bounds.json")

# fileOUT = os.path.join(workingdir, "bboxWGS84.gpkg")
# json2shpWGS84(fileIN, fileOUT)

# fileOUT2 = os.path.join(workingdir, "bboxCRS.gpkg")
# json2shpCRS(fileIN, fileOUT2)