# -*- coding: utf-8 -*-
"""
/***************************************************************************
 IdentifProj

 This QGIS plugin is an easy way to guess which map projection has been used for a location.
 
 The plugin has 3 use cases :
 - type projected coordinates and get all thez possible points all over the world
 - click on a location on the map and find all the possible projected coordinates
 - draw a bbox and find all the projected bboxes
 
 IMPORTANT: at the first start, the plugin will build its CRS database from Qgis CRS list. 
 It can last au couple of minutes but it will only happen one time.
 This plugin has been initially developed during a third year engineering project at ENSG (https://www.ensg.eu)
 
 Licence : open licence 2.0
 
 (C) 2024 by Leonie leroux, Jacques Beilin
 leonie.leroux@ensg.eu, jacques.beilin@ensg.eu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis._core import Qgis

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QTreeWidgetItemIterator

try:
    from PyQt5.QtCore import QVariant
except:
    pass
    
try:
    from PyQt5.QtCore import QMetaType
except:
    pass

from PyQt5.QtWidgets import QProgressDialog, QApplication, QTreeWidget, QTreeWidgetItem, QVBoxLayout, QMainWindow

# from qgis.core import QgsApplication
from qgis.core import QgsPointXY
from qgis.core import QgsCategorizedSymbolRenderer
from qgis.core import QgsCoordinateReferenceSystem
from qgis.core import QgsCoordinateTransform
from qgis.core import QgsCoordinateTransformContext
from qgis.core import QgsCsException
from qgis.core import QgsFeature
from qgis.core import QgsField
from qgis.core import QgsGeometry
from qgis.core import QgsMarkerSymbol
from qgis.core import QgsPathResolver
from qgis.core import QgsPoint
from qgis.core import QgsProject
# from qgis.core import QgsProviderRegistry
from qgis.core import QgsRasterLayer
from qgis.core import QgsRectangle
from qgis.core import QgsRendererCategory
from qgis.core import QgsReferencedPointXY
from qgis.core import QgsVectorLayer
from qgis.core import QgsVectorFileWriter

import processing

# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the DockWidget
from .IdentifProj_dockwidget import IdentifProjDockWidget
import os.path
import json
import re
import shutil

# Import other functions
from .Point2Coord import Point2Coord
from .BBox2Coord import BBox2Coord
from .refactor_data_crs import json2shpWGS84, json2shpCRS, getProjTypes, addTrigramsToProjDB, getCountryListFromCrs, getCRSAreaFromProjDB, getProjDBPath, getCountriesTrigrams

#import config_bbox


class IdentifProj:
    """
    QGIS Plugin Implementation.
    
    Classe qui contient toutes les méthodes d'initialisation, lancement et gestion du plugin.
    
    Contient les méthodes de la fonctionnalités principale: Coord2Point
    
    Contient les méthodes qui permettent de gérer les autres classes/fonctionalités du plugin
    """

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface

        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        self.config_dir = os.path.join(self.plugin_dir, "config")
        self.tmp_dir = os.path.join(self.plugin_dir, "tmp")
        
        self.loadParams()

        # initialize locale
        try:
            locale = self.config["locale"]
        except:
            locale = QSettings().value('locale/userLocale')[0:2]
            
        self.locale = locale
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            '{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&IdentifProj')
        # TODO: We are going to let the user set this up in a future iteration
        self.toolbar = self.iface.addToolBar(u'IdentifProj')
        self.toolbar.setObjectName(u'IdentifProj')
        
        # Initialisation de la couche d'affichage
        self.point_layer = None

        #print "** INITIALIZING IdentifProj"

        self.pluginIsActive = False
        self.dockwidget = None
        
        #Création des attributs nécessaires pour les différentes fonctionnalités 
        self.point_tool = None
        self.p2c_isActive = False
        self.bbox_tool = None
        self.data = None
        
    def loadParams(self):  
        """
        Load json config file

        Returns
        -------
        None.

        """

        try:
            with open(os.path.join(self.plugin_dir,'config','config.json'), "r") as f:  
                L = f.readlines()
                s = ""
                for l in L:
                    s += l
                self.config = json.loads(s)
        except:
            pass

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('IdentifProj', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action


    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/IdentifProj/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u''),
            callback=self.run,
            parent=self.iface.mainWindow())
        
    

    #--------------------------------------------------------------------------
    ### Méthodes de gestion du plugin

    def onClosePlugin(self):
        """Cleanup necessary items here when plugin dockwidget is closed"""

        #print "** CLOSING IdentifProj"

        # disconnects
        self.dockwidget.closingPlugin.disconnect(self.onClosePlugin)

        # remove this statement if dockwidget is to remain
        # for reuse if plugin is reopened
        # Commented next statement since it causes QGIS crashe
        # when closing the docked window:
        # self.dockwidget = None

        self.pluginIsActive = False


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""

        #print "** UNLOAD IdentifProj"

        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&IdentifProj'),
                action)
            self.iface.removeToolBarIcon(action)
        # remove the toolbar
        del self.toolbar
        
    def filterCrsList(self, filterProj, filterCountry, outputFrame):
        """
        filter results on projection type and/or area name

        Parameters
        ----------
        filterProj : TYPE
            DESCRIPTION.
        outputFrame : TYPE
            DESCRIPTION.
        filterCountry : TYPE, optional
            DESCRIPTION. The default is None.

        Returns
        -------
        None.

        """
        
        verbose = False

        currentProjTypeFilter = filterProj.currentText()   
        currentAreaFilter = filterCountry.text().strip()
        if verbose:
            print("Projection type filter : <%s>" % currentProjTypeFilter)
            print("Area filter : <%s>" % currentAreaFilter)

        iterator = QTreeWidgetItemIterator(outputFrame)
        
        while iterator.value():
            item = iterator.value()
            hidden = False
            
            if not currentProjTypeFilter == "":
                if not item.operationDescription == currentProjTypeFilter:
                    hidden = True
                
            if not currentAreaFilter == "":
                if not re.search(currentAreaFilter, item.areas, re.IGNORECASE):
                    hidden = True
                    
            item.setHidden(hidden)
                    
            iterator += 1


    #--------------------------------------------------------------------------
    ### Méthode qui gère les différentes méthodes du plugin
    
    def run(self):
        """Run method that loads and starts the plugin"""

        if not self.pluginIsActive:
            self.pluginIsActive = True

            #print "** STARTING IdentifProj"

            # dockwidget may not exist if:
            #    first run of plugin
            #    removed on close (see self.onClosePlugin method)
            if self.dockwidget == None:
                # Create the dockwidget (after translation) and keep reference
                self.dockwidget = IdentifProjDockWidget() #construction - mettre des actions ici, config
                
                print("Locale : %s" % self.locale)
                
                self.dockwidget.outputPoly.setColumnWidth(0,200)
                self.dockwidget.outputFrame.setColumnWidth(0,200)
                
                ## Gestion du fichier courant
                self.script_dir = os.path.dirname(__file__)
                self.config["script_dir"] = self.script_dir
                self.config["config_dir"] = self.config_dir
                self.config["tmp_dir"] = self.tmp_dir
                self.data_dir = os.path.join(self.script_dir, 'data')
                self.config["data_dir"] = self.data_dir
                
                try:
                    os.mkdir(self.data_dir)
                except:
                    pass
                
                self.originalWorldMapFile = QgsPathResolver().readPath('inbuilt:/data/world_map.gpkg')
                self.worldMapFile = os.path.join(self.data_dir, 'world_map.gpkg')
                try:
                    shutil.copyfile(self.originalWorldMapFile, self.worldMapFile)
                except:
                    pass
                
                self.originalProjDB = getProjDBPath()
                self.projDB = os.path.join(self.data_dir, 'proj.db')
                try:
                    if not os.path.exists(self.projDB):
                        shutil.copyfile(self.originalProjDB, self.projDB)
                except:
                    pass
                
                ## Chargement de la map lors du lancement du plugin - map en WGS84
                root = QgsProject.instance().layerTreeRoot()
                layerName = "Countries"
                layerFound = False
                for node in root.children():
                    if node.name() == layerName:
                        layerFound = True
                        self.worldMapLayer = node
                if (layerFound == False):
                    self.loadWorldMap(QgsPathResolver().readPath('inbuilt:/data/world_map.gpkg'))
                    
                # crs = QgsProject.instance().crs()
                # if crs.authid() == "EPSG:4326":
                #     print("setCRS")
                #     crs3857 = QgsCoordinateReferenceSystem("EPSG:3857")
                #     QgsProject.instance().setCrs(crs3857)
                    
                #     print(f"Project crs set to manually '{crs3857.authid()}'.")
                #     print(f"Project's crs is '{QgsProject.instance().crs().authid()}'")

                #     # Mimic the situation events are processed at some time
                #     QApplication.instance().processEvents()
                #     print("Queued events processed at some time.")
                        

                
                ## Création du fichier json de configuration lors du premier lancement du plugin
                json_file_path = os.path.join(self.config_dir, "crs_with_bounds.json")
                if not os.path.exists(json_file_path):
                    ret = self.calc_BBox(json_file_path, self.config["skippedCrsTypes"])
                    
                    try:
                        self.worldMapLayer.removeSelection()
                    except:
                        pass
                    
                fileBBoxWGS84 = os.path.join(self.config_dir, "bboxWGS84.gpkg")  
                fileBBoxCrs = os.path.join(self.config_dir, "bboxCRS.gpkg")
                if os.path.exists(json_file_path):
                    if not os.path.exists(fileBBoxWGS84):
                        json2shpWGS84(json_file_path, fileBBoxWGS84)
    
                    if not os.path.exists(fileBBoxCrs):
                        json2shpCRS(json_file_path, fileBBoxCrs)
                        
                if not os.path.exists(fileBBoxWGS84) or  not os.path.exists(fileBBoxCrs):
                    self.pluginIsActive = False
                    self.dockwidget = None
                    return
                
                ## Chargement du fichier json de configuration dans une variable utilisée dans les différentes fonctionnalités
                self.data = self.load_crs_data(json_file_path)
                
                ## Lancement de la fonction Coord2Point (avec appui sur le pushbutton dans l'interface)
                self.dockwidget.btGetCoord.clicked.connect(lambda: self.handleGetCoord(self.data))

                ## Gestion fonctionnalité Point2Coord: active/désactive la fonction de récupération des click sur la map
                self.dockwidget.act_p2c.clicked.connect(self.activate_canvasPressEvent)
                self.dockwidget.des_p2c.clicked.connect(self.deactivate_canvasPressEvent)
                
                ## Gestion fonctionnalité BBox2Coord: active/désactive la création de BBox sur la map
                self.dockwidget.createBBox.clicked.connect(self.activate_BBox)
                self.dockwidget.desBBox.clicked.connect(self.deactivate_BBox)
                
                self.dockwidget.filter_Point2Coord_byProjType.currentIndexChanged.connect(lambda: self.filterCrsList(self.dockwidget.filter_Point2Coord_byProjType, self.dockwidget.filter_Point2Coord_byCountry, self.dockwidget.outputFrame))
                self.dockwidget.filter_BBox2Coord_byProjType.currentIndexChanged.connect(lambda: self.filterCrsList(self.dockwidget.filter_BBox2Coord_byProjType, self.dockwidget.filter_BBox2Coord_byCountry, self.dockwidget.outputPoly))
                
                self.dockwidget.filter_Point2Coord_byCountry.textChanged.connect(lambda: self.filterCrsList(self.dockwidget.filter_Point2Coord_byProjType, self.dockwidget.filter_Point2Coord_byCountry, self.dockwidget.outputFrame))
                self.dockwidget.filter_BBox2Coord_byCountry.textChanged.connect(lambda: self.filterCrsList(self.dockwidget.filter_BBox2Coord_byProjType, self.dockwidget.filter_BBox2Coord_byCountry, self.dockwidget.outputPoly))

    
            # connect to provide cleanup on closing of dockwidget
            self.dockwidget.closingPlugin.connect(self.onClosePlugin)

            # show the dockwidget
            # TODO: fix to allow choice of dock location
            self.iface.addDockWidget(Qt.RightDockWidgetArea, self.dockwidget)
            self.dockwidget.show()
            
    #--------------------------------------------------------------------------
    ### Méthodes qui gèrent la fonctionnalité Coord2Point
    
    def identifProj(self, crs_dic):
        """
        Fonction principale de la classe (et du plugin) qui correspond à la fonctionnalité Coord2Point
        Récupère les coordonées entrées par l'utilisateur pour trouver les projections correspondant à ces coordonnées
        
        crs_dic : liste de dictionnaires ; chaque dictionnaire correspond à la description d'une projection
        Cette liste de dictionnaires est obtenue après le chargement du fichier json de configuration
        
        list_valid_pt : liste de QgsReferencePointXY ; correspond aux points avec les coordonnées entrées pas l'utilisateur dans 
        les différentes projections
        list_valid_crs : liste de QgsCoordinateReferenceSystem ; correspond aux projections valides pour le point entré
        
        """
        
        print("Coord2Point")
        
        ## Récupération des coordonnées entrées par l'utilisateur
        x = self.dockwidget.x.text()
        y = self.dockwidget.y.text()
        
        ## Vérifier que l'utilisateur entre des coordonnées valides
        try:
            X = float(x)
            Y = float(y)
        
        except ValueError:
            print("Erreur : les coordonnées doivent être des nombres valides.")
            return
        
        """ Recuperation des CRS intersectes > shapefile intersect.shp """
        point = QgsPoint(X,Y)
        ftr = QgsFeature()
        ftr.setGeometry(point)
        
        crs = "epsg:3857"
        lyr = QgsVectorLayer('Point?crs=' + crs, 'CRS', "memory")
        lyr.startEditing()
        prv = lyr.dataProvider()
        
        prv.addFeatures([ftr])

        commited = lyr.commitChanges()
        if commited:
            print("ok")
        else:
            print(f'{lyr.commitErrors()}')
            
        
        try:
            if not os.path.exists(self.tmp_dir):
                os.mkdir(self.tmp_dir)
                
        except:
            print('Unable to create tmp dir')
            
        # Write to an ESRI Shapefile format dataset using UTF-8 text encoding
        fileTMP = os.path.join(self.script_dir, "tmp", "point.shp")
        save_options = QgsVectorFileWriter.SaveVectorOptions()
        save_options.driverName = "ESRI Shapefile"
        save_options.fileEncoding = "UTF-8"
        transform_context = QgsProject.instance().transformContext()
        QgsVectorFileWriter.writeAsVectorFormatV3(lyr, fileTMP, transform_context, save_options)

        bboxCRS = os.path.join(self.plugin_dir, "config", "bboxCRS.gpkg")
        self.lyr_bboxCRS = QgsVectorLayer(bboxCRS, 'bboxCRS', "ogr")
        QgsProject.instance().addMapLayer(self.lyr_bboxCRS, addToLegend=False)
        parameters = {'INPUT':bboxCRS,
                      'PREDICATE':[0,1],
                      'INTERSECT':fileTMP,
                      'METHOD':0}
        processing.run("native:selectbylocation", parameters)

        try:
            os.mkdir(os.path.join(self.script_dir, "tmp"))
        except:
            print('Unable to create tmp dir')
        pathLayerIntersec = os.path.join(self.script_dir, "tmp", "intersec.shp")
        parameters = { 'INPUT' : bboxCRS, 'OUTPUT' : pathLayerIntersec}
        processing.run('native:saveselectedfeatures', parameters) 
        
        lyr_intersect = QgsVectorLayer(pathLayerIntersec, 'Candidate CRS', "ogr")
        CandidateCRS = lyr_intersect.getFeatures()

        nCandidateCrs = lyr_intersect.featureCount()
        print("#Candidate Crs : ", nCandidateCrs)
        
        
        point = QgsPointXY(X,Y)
        
        list_valid_crs = []
        list_valid_pt = []
        
        nMiniCandidateCrs = 150
        if nCandidateCrs > nMiniCandidateCrs:
            ## Créer et configurer la barre de progression pour faire patienter l'utilisateur
            progress = QProgressDialog("Processing...", "Cancel", 0, nCandidateCrs, self.iface.mainWindow())
            progress.setWindowModality(Qt.WindowModal)
            progress.setMinimumDuration(1000)
            progress.setValue(0)
        i = 0
        
        ## Boucle sur la liste des projections disponibles pour tester les différentes projections
        # for crs in crs_dic:
        for feature in CandidateCRS:
                
            # Vérifier si l'utilisateur a annulé l'opération
            if nCandidateCrs > nMiniCandidateCrs:
                if progress.wasCanceled():
                    break
            
            ## Gestion de la progessbar
            i+=1
            if nCandidateCrs > nMiniCandidateCrs:
                progress.setValue(i)
                
            ## Récupération des attributs du SRC  
            attrs = feature.attributes()
            epsg = attrs[feature.fieldNameIndex("auth_id")]
            
            crs_v = QgsCoordinateReferenceSystem(epsg)
            pt = QgsReferencedPointXY(point, crs_v)
            list_valid_pt.append(pt)
            list_valid_crs.append(crs_v)
            
            ## Récupérer le polygon de zone de validité de la projection dans les coordonnées de la projections
            # coordinates = crs['transform_bounds']['coordinates'][0]
            # points = [QgsPointXY(coord[0], coord[1]) for coord in coordinates]
            # polygon = QgsGeometry.fromPolygonXY([points])
            
            ## Test pour savoir si les coordonnées entrées se situent dans l'emprise de la projection
            # if polygon.contains(point):
            #     epsg = crs['auth_id']
            #     crs_v = QgsCoordinateReferenceSystem(epsg)
            #     pt = QgsReferencedPointXY(point, crs_v)
            #     list_valid_pt.append(pt)
            #     list_valid_crs.append(crs_v)
                
        if nCandidateCrs > nMiniCandidateCrs:
            # Fermer la barre de progression
            progress.close()
        
        return list_valid_pt, list_valid_crs, X, Y
    
    
    def displayPoints(self, points, x, y):
        """
        Cette fonction s'occupe de l'affichage des points dans les différentes projections (avec les coordonées entrées par l'utilisateur')
        Ajoute une liste de points à la couche vectorielle.
        
        points: liste de QgsReferencedPointXY ; liste des points à afficher
        """
        
        colors = ["#000000","#FF0000","#800000","#FFFF00","#808000","#00FF00","#008000","#00FFFF","#008080","#0000FF","#000080","#FF00FF","#800080","#C0C0C0","#808080"]
        figures = ["circle", "square", "diamond", "pentagon", "hexagon", "triangle", "star_diamond", "asterisk_fill", "octagon", "trapazoid", "parallelogram_left", "parallelogram_right"]
        
        ## Créer une couche vectorielle en WGS84 (pour correspondre au fond de carte/map)
        point_layer = QgsVectorLayer(
            "Point?crs=EPSG:4326",  
            "Projection(s) for (E=%.0fm, N=%.0fm)" % (x,y),          
            "memory"               
        )
        
        Loperation = []
        
        
        try:
            fAuthid = QgsField("epsg",  type=QMetaType.QString, len=254)
            fdescription = QgsField("description",  type=QMetaType.QString, len=254)
            fX = QgsField("X", type=QMetaType.Double)
            fY = QgsField("Y", type=QMetaType.Double)
            fOperation = QgsField("operation", type=QMetaType.QString, len=254)
        except:
            fAuthid = QgsField("auth_id",  type=QVariant.String, len=254)
            fdescription = QgsField("description",  type=QVariant.String, len=254)
            fX = QgsField("X", type=QVariant.Double)
            fY = QgsField("Y", type=QVariant.Double)  
            fOperation = QgsField("operation", type=QVariant.String, len=254)

        ## Ajouter des champs (attributs) à la couche - ces champs décrivent le point affiché (notamment sa projection)
        point_layer.dataProvider().addAttributes([fAuthid, fdescription, fX, fY, fOperation])
        point_layer.updateFields()

        ## Récupérer le fournisseur de données de la couche
        data_provider = point_layer.dataProvider()
        crs_WGS = QgsCoordinateReferenceSystem('EPSG:4326')
        
        ## Créer et configurer la barre de progression
        progress = QProgressDialog("Affichage des points en cours...", "Annuler", 0, len(points), self.iface.mainWindow())
        progress.setWindowModality(Qt.WindowModal)
        progress.setMinimumDuration(0)
        progress.setValue(0)
        i = 0

        ## Créer des entités pour chaque point
        features = []
        for point in points:
            
            ## Gestion de la progessbar
            i+=1
            progress.setValue(i)
            
            ## Transformation des coordonnées dans la projection de la carte (pour l'affichage)
            try:
                x = point.x()
                y = point.y()
                
                crs_point = point.crs()
                epsg = crs_point.authid()
                description = crs_point.description()
                try:
                    operationDescription = crs_point.operation().description()
                except:
                    operationDescription = ""
                    
                Loperation += [operationDescription]
                
                context = QgsProject.instance().transformContext()
                transformer = QgsCoordinateTransform(crs_point, crs_WGS, context)
                pt = transformer.transform(point)
                
            except QgsCsException as e:
                print("Erreur de transformation pour %s " % str(e))
                continue
            
            feature = QgsFeature()
            feature.setGeometry(QgsGeometry.fromPointXY(pt))
            feature.setAttributes([epsg, description, x, y, operationDescription])
            features.append(feature)

        ## Ajouter les entités à la couche
        data_provider.addFeatures(features)
        
        Loperation = sorted(list(set(Loperation)))
        markers = []
        Noperation = len(Loperation)
        
        for i in range(Noperation):
            ncolor = i % len(colors)
            nfigure = i // len(colors)
            
            symbol = QgsMarkerSymbol.createSimple({'color': colors[ncolor], 'size': 2, 'outline_color': 'black', 'name' : figures[nfigure]})
            markers += [QgsRendererCategory(Loperation[i], symbol, Loperation[i], True)]
            
        renderer = QgsCategorizedSymbolRenderer("operation", markers)
        point_layer.setRenderer(renderer)
        point_layer.triggerRepaint()

        ## Mettre à jour la couche pour afficher les points
        point_layer.updateExtents()
        # Ajouter la couche au projet QGIS
        QgsProject.instance().addMapLayer(point_layer)
        
        ## Fermer la barre de progression
        progress.close()
        
    def handleGetCoord(self, crs_dic):
        """
        Gestionnaire pour le clic du bouton: appelle identifProj pour récupérer les points et les afficher sur la carte.
        
        crs_dic : liste de dictionnaires ; chaque dictionnaire correspond à la description d'une projection
        Cette liste de dictionnaires est obtenue après le chargement du fichier json de configuration
        """
        # Appel à identifProj pour récupérer la liste des points
        ret = self.identifProj(crs_dic)
        
        try:
            points = ret[0]
            x = ret[2]
            y = ret[3]
        
            # Afficher les points sur la carte
            self.displayPoints(points, x, y)
        except:
            pass
    
    #--------------------------------------------------------------------------
    ### Méthodes de configuration des autres éléments du plugin
        
    def check_crs_bounds(self, crs, bounds):
        """
        Méthode qui assigne à chaque projection une (ou plusieurs) partie du monde pour laquelle cette projection est valide
        
        crs : QgsCoordinateReferenceSystem
        
        primary_regions: liste de String ; correspond aux premières zones de validité de la projection
        secondary_regions: liste de String ; correspond aux secondes zones de validité de la projection
        """
        
        ## Définir les différentes parties du monde sous la forme de QgsRectangle
        west = QgsRectangle(-180.0, -90.0, -60.0, 90.0)
        mid = QgsRectangle(-60.0, -90.0, 60.0, 90.0)
        east = QgsRectangle(60.0, -90.0, 180.0, 90.0)
        
        north = QgsRectangle(-180.0, 0.0, 180.0, 90.0)
        south = QgsRectangle(-180.0, -90.0, 180.0, 0.0)
    
        primary_regions = []
        secondary_regions = []
    
        ## Vérifier les intersections avec les premières et secondes classes
        if bounds.intersects(west):
            primary_regions.append("West")
            if bounds.intersects(north):
                secondary_regions.append("West North")
            if bounds.intersects(south):
                secondary_regions.append("West South")
        
        if bounds.intersects(mid):
            primary_regions.append("Middle")
            if bounds.intersects(north):
                secondary_regions.append("Middle North")
            if bounds.intersects(south):
                secondary_regions.append("Middle South")
        
        if bounds.intersects(east):
            primary_regions.append("East")
            if bounds.intersects(north):
                secondary_regions.append("East North")
            if bounds.intersects(south):
                secondary_regions.append("East South")
        
        return primary_regions, secondary_regions
    
    
    def calc_BBox(self, json_file_path, skippedCrsTypes):
        """
        Méthode qui génère le fichier de configuration json. Ce fichier renseigne pour chaque projection dans la base de données QGIS:
        une description, son code EPSG, ses zones de validité (définie avec la fonction  check_crs_bounds) et les limites de validités
        de la projection dans son propre src
        
        json_file_path: string ; correspond à l'endroit où le fichier json est enregistré
        """
        
        ## Liste pour stocker les différents SRC dans QGIS
        crs_WGS = QgsCoordinateReferenceSystem('EPSG:4326')
        crs = QgsCoordinateReferenceSystem()
        liste_crs = crs.validSrsIds()
        crs_json = []
        
        ## Créer et configurer la barre de progression - pour faire patienter l'utilisateur
        progress = QProgressDialog("Building CRS database...\nThis task will be done only one time, at first start of the plugin", "Cancel", 0, len(liste_crs), self.iface.mainWindow())
        progress.setWindowModality(Qt.WindowModal)
        progress.setMinimumDuration(0)
        # progress.setCancelButton(None)
        progress.setValue(0)
        i = 0
        
        log = os.path.join(self.tmp_dir, "crs_database.log")
        Llog = []
        
        # trigrams = getCountriesTrigrams(self.worldMapFile)
        
        addTrigramsToProjDB(self.worldMapFile, self.projDB)
        
        areaDict = getCRSAreaFromProjDB(self.projDB)

        # for index, (key, value) in enumerate(areaDict.items()):
        #     for j in range(len(trigrams)):
        #         for k in range(len(value)):
        #             if re.search(trigrams[j]["name"], value[k], re.IGNORECASE):
        #                 areaDict[key] += [trigrams[j]["trigram"]]
                        
        # print(areaDict["EPSG:3034"])
        
        # context = QgsProject.instance().transformContext()
        context = QgsCoordinateTransformContext()
        
        ## Boucle sur les projections disponibles dans QGIS pour recalculer les limites de validité
        for crs_id in liste_crs:
            
            crs = crs.fromSrsId(crs_id);
            try:
                crsType = crs.type()
                if crsType in skippedCrsTypes:
                    continue
            except:
                crsType = ""
            
            ## Gestion de la progessbar
            i+=1
            progress.setValue(max(2,i))  
            
            if progress.wasCanceled():
                return False
            
            if crs.isValid():
                bounds = crs.bounds()  # Bounding box en WGS 84
                
                if not bounds.isNull():
                    
                    try:
                        areaList = areaDict[crs.authid().upper()]
                    except Exception as err:
                        print("Error while creating country list for %s :" % crs.authid().upper(), err)
                        areaList = getCountryListFromCrs(crs, self.data_dir)
                        
                    # context.allowFallbackTransform(crs_WGS, crs)
                    
                    transformer = QgsCoordinateTransform(crs_WGS, crs, context)
                    transformer.setBallparkTransformsAreAppropriate(True)
                    
                    ## On va chercher les coins de la bounding box - 2 par définition
                    x_min = bounds.xMinimum()
                    y_min = bounds.yMinimum()
                    x_max = bounds.xMaximum()
                    y_max = bounds.yMaximum()
                    
                    ## On cherche ensuite les 4 coins de la bbox (car les bounds transformées formeront un polygone plus ou moins
                    ## rectangle en fonction de la projection) + on les transforme dans la projection de la boucle
                    try:
                        c_ll = transformer.transform(QgsPointXY(x_min,y_min))
                        c_lu = transformer.transform(QgsPointXY(x_min,y_max))
                        c_uu = transformer.transform(QgsPointXY(x_max,y_max))
                        c_ul = transformer.transform(QgsPointXY(x_max,y_min))
                    except QgsCsException as e:
                        Llog.append(f"Transformation error for EPSG:{crs} : {e}")
                        continue
                    
                    coord_poly = [[c_ll.x(), c_ll.y()], [c_lu.x(), c_lu.y()], [c_uu.x(), c_uu.y()], [c_ul.x(), c_ul.y()], [c_ll.x(), c_ll.y()]]
                    
                    ## On cherche les zones de validés grossière de la projection
                    prim, sec = self.check_crs_bounds(crs, bounds)
                    
                    try:
                        projectionAcronym = crs.projectionAcronym()
                    except:
                        projectionAcronym = ""
                        
                    try:
                        operationDescription = crs.operation().description()
                    except:
                        operationDescription = ""
                    
                    ## On rempli les différents du json pour chaque projection
                    crs_info = {
                        "auth_id": crs.authid(),
                        "name": crs.description(),
                        "projectionAcronym": projectionAcronym,
                        "operationDescription": operationDescription,
                        "type": crsType,
                        "primary region": prim,
                        "sec region": sec,
                        "areas": areaList,
                        "bounding_box_WGS84": {
                            "x_min": bounds.xMinimum(),
                            "y_min": bounds.yMinimum(),
                            "x_max": bounds.xMaximum(),
                            "y_max": bounds.yMaximum()
                        },
                        "transform_bounds": {
                            "type": "Polygon",
                            "coordinates": [coord_poly]  # GeoJSON utilise une liste imbriquée
                            }
                    }
                    crs_json.append(crs_info)
                    
                    # try:
                    #     s = json.dumps({"areas": areaList})

                    # except Exception as err:
                    #     print(err)
                    #     print(areaList)
                    # if i > 10:
                    #     break
                
                    
        # Fermer la barre de progression
        progress.close()
        
        ## Exporter en fichier JSON
        script_dir = os.path.dirname(__file__)
        
        ## Définir le chemin du dossier 'config'
        config_dir = os.path.join(script_dir, "config")
    
        ## Vérifier si le dossier 'config' existe, sinon le créer
        if not os.path.exists(config_dir):
            os.makedirs(config_dir)
            print(f"'config' directory created : {config_dir}")
        else:
            print(f"'config' directory already exists at : {config_dir}")
    
        ## Définir le chemin du fichier JSON dans le dossier 'config' + exporter
        output_file = json_file_path
        with open(output_file, "w", encoding="utf-8") as f:
            json.dump(crs_json, f, ensure_ascii=False, indent=4)
        
        print(f"Json created at : {output_file}")
        
        
    def load_crs_data(self, filename):
        """
        Charge le fichier JSON de configuration et le convertit en dictionnaire Python.
        
        filename: Chemin vers le fichier JSON
        
        return: liste de dictionnaires contenant les données du JSON (les descriptions des projections)
        """
        
        try:
            with open(filename, "r", encoding="utf-8") as f:
                data = json.load(f)  # Charge le contenu du fichier en un dictionnaire
            return data
        
        except FileNotFoundError:
            print(f"Error : unable to find {filename}")
        except json.JSONDecodeError as e:
            print(f"Error : unable to parse {filename} : {e}")
        except Exception as e:
            print(f"Error : {e}")
        return {}
    
    
    def loadRasterMap(self, filename):
        """
        Charge la map/ fond de carte (format tiff) du plugin
        
        filename: Chemin vers le fichier tiff
        """

        ## Charger le fichier GeoTIFF comme couche raster
        layer = QgsRasterLayer(filename, "Map")

        if not layer.isValid():
            ## Si le fichier n'est pas valide, afficher une erreur
            print(f"Error : unable to load '{filename}'.")
            return

        ## Ajouter la couche au projet actuel
        QgsProject.instance().addMapLayer(layer)
        print(f"'{filename}' loaded")
        
    def loadVectorMap(self, fileName, layerName=""):
        """
        
        Charge la map/ fond de carte (format vectoriel) du plugin
        
        filename: Chemin vers le fichier
        """
        vlayer = QgsVectorLayer(fileName, layerName, "ogr")
        if not vlayer.isValid():
            print("Layer failed to load!")
        else:
            QgsProject.instance().addMapLayer(vlayer)
            return vlayer
        
    def loadWorldMap(self, map_file_path):
        """
        Loads default Qgis map

        Parameters
        ----------
        map_file_path : str
            Qgis world default map path

        Returns
        -------
        None.

        """
        
        self.worldMapLayer = self.loadVectorMap(map_file_path + "|layername=countries", "Countries")
        self.worldMapLayer_states_provinces = self.loadVectorMap(map_file_path + "|layername=states_provinces", "State provincies")
        self.worldMapLayer_disputed_borders = self.loadVectorMap(map_file_path + "|layername=disputed_borders", "Disputed Borders")
        self.worldMapLayer.loadNamedStyle(os.path.join(self.data_dir, 'world_map.qml'), True)
        self.worldMapLayer.triggerRepaint()
        
        
    #--------------------------------------------------------------------------
    ### Méthodes qui gèrent les autres fonctionnalités du plugin 
    
    #Point2Coord 
        
    def activate_canvasPressEvent(self):
        """Active l'outil de récupération des coordonnées."""
        
        if not hasattr(self, 'point_tool') or self.point_tool is None:
            ## Crée l'outil Point2Coord uniquement s'il n'existe pas
            self.point_tool = Point2Coord(self.iface, self.dockwidget, self.data, self.config, self)
        
        ## Active l'outil pour récupérer les coordonnées sur la carte
        self.iface.mapCanvas().setMapTool(self.point_tool)
        print("Outil de récupération des coordonnées activé.")
        
        
        
    def deactivate_canvasPressEvent(self):
        """Désactive l'outil de récupération des coordonnées."""
        
        if hasattr(self, 'point_tool') and self.point_tool is not None:
            ## Réinitialise l'outil de la carte à l'outil par défaut (outil de navigation)
            self.iface.mapCanvas().unsetMapTool(self.point_tool)
            self.point_tool = None  # Libère l'instance pour éviter des conflits futurs
            print("Outil de récupération des coordonnées désactivé.")
        else:
            print("Aucun outil de récupération des coordonnées actif à désactiver.")
            
    
    #BBox2Coord      
    
    def activate_BBox(self):
        """Active l'outil de récupération des coordonnées de la BBox"""
        
        if not hasattr(self, 'bbox_tool') or self.bbox_tool is None:
            ## Crée l'outil BBox2Coord uniquement s'il n'existe pas
            self.bbox_tool = BBox2Coord(self.iface, self.dockwidget, self.data, self.config)
        
        ## Active l'outil pour récupérer les coordonnées sur la carte
        self.iface.mapCanvas().setMapTool(self.bbox_tool)
        print("Outil de récupération bbox activé.")
        
    def deactivate_BBox(self):
        """Désactive l'outil de récupération des coordonnées de la BBox"""
        
        if hasattr(self, 'bbox_tool') and self.bbox_tool is not None:
            ## Désactive l'outil en réinitialisant le map tool
            self.iface.mapCanvas().unsetMapTool(self.bbox_tool)
            print("Outil de récupération bbox désactivé.")
