# -*- coding: utf-8 -*-
"""
/***************************************************************************
 IPP
                                 A QGIS plugin
 This plugin calculates IPP
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-05-16
        git sha              : $Format:%H$
        copyright            : (C) 2023 by Truong Anh Huy LE
        email                : huymop.lee@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import *
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QFileDialog, QMessageBox
import os
from qgis.core import *
from qgis.gui import *
from qgis.utils import iface
import pandas as pd
import processing
import tempfile


# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .IPP_dialog import IPPDialog
import os.path


class IPP:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'IPP_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&IPP')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('IPP', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToVectorMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/IPP/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'IPP'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginVectorMenu(
                self.tr(u'&IPP'),
                action)
            self.iface.removeToolBarIcon(action)

    def select_output_file(self):
        filename, _filter = QFileDialog.getSaveFileName(
            self.dlg, "Select output file ", "", '*.shp')
        self.dlg.lineEdit.setText(filename)

    def select_population_file(self):
        filename, _filter = QFileDialog.getSaveFileName(
            self.dlg, "Select output file ", "", '*.shp')
        self.dlg.lineEdit_2.setText(filename)


    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = IPPDialog()
            self.dlg.pushButton.clicked.connect(self.select_output_file)
            self.dlg.pushButton_2.clicked.connect(self.select_population_file)
        layers = QgsProject.instance().layerTreeRoot().children()
        self.dlg.comboBox.clear()
        self.dlg.comboBox.addItems([layer.name() for layer in layers])
        self.dlg.comboBox_2.clear()
        self.dlg.comboBox_2.addItems([layer.name() for layer in layers])

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:

            #Read input concentration layer
            selectedLayerIndex = self.dlg.comboBox.currentIndex()
            Input = layers[selectedLayerIndex].layer()
            Input_path = Input.dataProvider().dataSourceUri()
            concentration = Input_path

            #Read Input bande layer
            selectedLayerIndex1 = self.dlg.comboBox_2.currentIndex()
            Bande = layers[selectedLayerIndex1].layer()
            Bande_path = Bande.dataProvider().dataSourceUri()

            #Read Input population layer
            population = self.dlg.lineEdit_2.text()

            #Read Output layer
            exit = self.dlg.lineEdit.text()

            #Create temporary layer
            temp_file_path = tempfile.NamedTemporaryFile(suffix='.shp').name

            #Cut concentration layer over bande d'étude layer
            processing.run("native:clip", {'INPUT' : concentration,'OVERLAY' : Bande_path, 'OUTPUT':temp_file_path})

            #Overlay population layer on cut concentration layer
            processing.run('qgis:intersection',{'INPUT':temp_file_path,'FIELD' : ['MAILLE', 'OUTPUTNO2.'], 'OUTPUT':exit,'OVERLAY': population,'OVERLAY_FIELDS' : ['Idcar_200m', 'Ind'], 'OVERLAY_FIELDS_PREFIX' : ''})


            #Calculate
            layer = QgsVectorLayer(exit, '', 'ogr')
            pv = layer.dataProvider()
            fields = layer.fields()
            idx = fields.indexFromName('CONCAN')

            #Create new fields
            pv.addAttributes([QgsField('Surface', QVariant.Double), QgsField('Population', QVariant.Double), QgsField('Concen', QVariant.Double), QgsField('IPP', QVariant.Double)])
            layer.updateFields()

            #Define equations
            expression1 = QgsExpression('$area') #Surface
            expression2 = QgsExpression('"Ind"*"Surface"/40000') #Population
            expression3 = QgsExpression('"CONCAN"') #Concentration
            expression4 = QgsExpression('"Concen" * "Population"') #IPP

            #Define calcul context
            context = QgsExpressionContext()
            context.appendScopes(QgsExpressionContextUtils.globalProjectLayerScopes(layer))

            #Calculate new fields
            #Area
            with edit(layer):
                for f in layer.getFeatures():
                    context.setFeature(f)
                    f['Surface'] = expression1.evaluate(context)
                    layer.updateFeature(f)

            #Population
            with edit(layer):
                for f in layer.getFeatures():
                    context.setFeature(f)
                    f['Population'] = expression2.evaluate(context)
                    layer.updateFeature(f)

            #Concentration
            with edit(layer):
                for f in layer.getFeatures():
                    context.setFeature(f)
                    f['Concen'] = expression3.evaluate(context)
                    layer.updateFeature(f)

            #IPP
            with edit(layer):
                for f in layer.getFeatures():
                    context.setFeature(f)
                    f['IPP'] = expression4.evaluate(context)
                    layer.updateFeature(f)

            #Add output layer in Qgis
            iface.addVectorLayer(exit, '', 'ogr')

            #Import IPP data to dataframe and show result
            Concen = []
            Population = []
            IPP = []
            for f in layer.getFeatures():
                Concen.append(float(f['Concen']))
                Population.append(float(f['Population']))
                IPP.append(float(f['IPP']))
            df = pd.DataFrame({'NO2' : Concen, 'Population': Population, 'IPP': IPP})

            IPP_cumul = df['IPP'].sum()
            Pop_tot = df['Population'].sum()

            msg_box = QMessageBox()
            msg_box.setIcon(QMessageBox.Information)
            msg_box.setWindowTitle('IPP cumumlé')

            msg_box.setText('IPP cumulé: ' + str(round(IPP_cumul,2)) + '\n' + 'Population totale: ' + str(round(Pop_tot,2)))
            msg_box.addButton(QMessageBox.Ok)
            msg_box.setDefaultButton(QMessageBox.Ok)

            msg_box.exec()
            self.iface.messageBar().pushMessage( "Success", "Output file written at "+ exit,level = Qgis.Success, duration = 3)


