# -*- coding: utf-8 -*-

"""
/***************************************************************************
 buildings2sewert
                                 A QGIS plugin
 Connect buildings to the sewer system
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2021-06-15
        copyright            : (C) 2021 by Josep Pueyo-Ros, ICRA
        email                : jpueyo@icra.cat
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Josep Pueyo-Ros, ICRA'
__date__ = '2021-06-15'
__copyright__ = '(C) 2021 by Josep Pueyo-Ros, ICRA'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import *
from PyQt5.QtGui import QIcon
from qgis.core import *
import qgis.utils
import processing
import os
from .utils.check_extent import check_extent
from .utils.z_sampling import z_sampling

pluginPath = os.path.dirname(__file__)


class buildings2sewertAlgorithm(QgsProcessingAlgorithm):
    """
    This is an example algorithm that takes a vector layer and
    creates a new identical one.

    It is meant to be used as an example of how to create your own
    algorithms and explain methods and variables used to do it. An
    algorithm like this will be available in all elements, and there
    is not need for additional work.

    All Processing algorithms should extend the QgsProcessingAlgorithm
    class.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    OUTPUT = 'OUTPUT'
    INPUT = 'INPUT'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector features source. It can have any kind of
        # geometry.
        self.addParameter(
            QgsProcessingParameterVectorLayer(
                'INPUT',
                self.tr('Buildings layer'),
                [QgsProcessing.TypeVectorAnyGeometry]
            )
        )

        self.addParameter(
            QgsProcessingParameterField(
                'INPUT_Z',
                self.tr('Field with buildings altitude'),
                parentLayerParameterName='INPUT',
                allowMultiple=False,
                optional=True
            )
        )

        self.addParameter(
            QgsProcessingParameterVectorLayer(
                'MANHOLES',
                self.tr('Manholes layer'),
                [QgsProcessing.TypeVectorPoint]
            )
        )

        self.addParameter(
            QgsProcessingParameterField(
                'NODE_ID',
                self.tr('Field with manholes id'),
                parentLayerParameterName='MANHOLES',
                allowMultiple=False
            )
        )

        self.addParameter(
            QgsProcessingParameterField(
                'NODE_Z',
                self.tr('Field with manholes altitude'),
                parentLayerParameterName='MANHOLES',
                allowMultiple=False,
                optional=True
            )
        )

        self.addParameter(
            QgsProcessingParameterRasterLayer(
                'DEM',
                self.tr('Elevations raster'),
                optional=True
            )
        )

        self.addParameter(
            QgsProcessingParameterDistance(
                'MAX_DIST',
                self.tr('Maximum distance')
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                'Z_TOL',
                self.tr('Altitude tolerance'),
                type=QgsProcessingParameterNumber.Integer
            )
        )

        self.addParameter(
            QgsProcessingParameterBoolean(
                'LINES_BOOL',
                self.tr('Create connection lines')
            )
        )

        self.addParameter(
            QgsProcessingParameterFeatureSink(
                'LINES',
                self.tr('Connection lines layer'),
                type=QgsProcessing.TypeVectorLine
            )
        )

        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Output layer'),
                type=QgsProcessing.TypeVectorPoint
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        # Retrieve the feature source and sink. The 'dest_id' variable is used
        # to uniquely identify the feature sink, and must be included in the
        # dictionary returned by the processAlgorithm function.

        # load layers and parameters
        parcels_o = self.parameterAsVectorLayer(parameters, 'INPUT', context)
        nodes_o = self.parameterAsVectorLayer(parameters, 'MANHOLES', context)
        mde = self.parameterAsRasterLayer(parameters, 'DEM', context)

        # load parameters
        max_dist = self.parameterAsDouble(parameters, 'MAX_DIST', context)
        max_z_tol = self.parameterAsInt(parameters, 'Z_TOL', context)
        node_id = self.parameterAsString(parameters, 'NODE_ID', context)
        node_z_field = self.parameterAsString(parameters, 'NODE_Z', context)
        parcel_z_field = self.parameterAsString(parameters, 'INPUT_Z', context)
        connection_lines = self.parameterAsBool(parameters, 'LINES_BOOL', context)

        feedback.setProgressText("mde: {}".format(mde is None))
        feedback.setProgressText("parcel_z_field: {}".format(parcel_z_field == ''))
        feedback.setProgressText("node_z_field: {}".format(node_z_field == ''))


        if mde is None and (parcel_z_field == '' or node_z_field == ''):
            feedback.reportError("Elevation raster or altitude fields are needed if altitudes not provided in the layers")
            return {}
        if mde is not None:
            if not check_extent(parcels_o, mde) or not check_extent(nodes_o, mde):
                feedback.reportError("Some of the layers are out of DEM")
                return {}

        # convert parcels to centroids
        parcels = processing.run("native:centroids", {'INPUT': parcels_o, 'ALL_PARTS': False, 'OUTPUT': 'memory:'})[
            'OUTPUT']

        # copy nodes_o
        features = nodes_o.getFeatures()
        nodes = QgsVectorLayer("Point", "duplicated_layer", "memory")
        nodes_data = nodes.dataProvider()
        attr = nodes_o.dataProvider().fields().toList()
        nodes_data.addAttributes(attr)
        nodes.updateFields()
        nodes_data.addFeatures(features)

        # class to calculate distance between points
        d = QgsDistanceArea()

        if parcel_z_field == '':
            feedback.setProgressText("Calculating altitudes of parcels...")
            parcels = z_sampling(parcels, mde, feedback)
            parcel_z_field = 'z'
        if node_z_field == '':
            feedback.setProgressText("Calculating altitudes of nodes...")
            nodes = z_sampling(nodes, mde, feedback)
            node_z_field = 'z'

        feedback.setProgressText("Calculation of altitudes finished")
        feedback.setProgressText("Calculating connections...")

        # get indexs for nodes fields
        node_idx = nodes.fields().indexFromName(node_id)
        node_z = nodes.fields().indexFromName(node_z_field)

        # check if fields exist and delete if
        if parcels.fields().indexFromName("manhole_id") != -1:
            idx = parcels.fields().indexFromName("manhole_id")
            parcels.dataProvider().deleteAttributes([idx])
            parcels.updateFields()
        if parcels.fields().indexFromName('dist') != -1:
            idx = parcels.fields().indexFromName('dist')
            parcels.dataProvider().deleteAttributes([idx])
            parcels.updateFields()
        if parcels.fields().indexFromName('z_diff') != -1:
            idx = parcels.fields().indexFromName('z_diff')
            parcels.dataProvider().deleteAttributes([idx])
            parcels.updateFields()

        # create fields to save data
        parcels.dataProvider().addAttributes([
            QgsField("manhole_id", QVariant.String),
            QgsField("dist", QVariant.Double),
            QgsField("z_diff", QVariant.Double)
        ])
        parcels.updateFields()

        # get index of created fields
        parcel_node = parcels.fields().indexFromName("manhole_id")
        parcel_dist = parcels.fields().indexFromName("dist")
        parcel_z_diff = parcels.fields().indexFromName("z_diff")
        parcel_z = parcels.fields().indexFromName(parcel_z_field)

        request = QgsFeatureRequest().setSubsetOfAttributes([parcel_node, parcel_dist, parcel_z_diff, parcel_z])

        total = 100.0 / parcels.featureCount() if parcels.featureCount() else 0
        features = parcels.getFeatures(request)

        # destination layer
        (sink, dest_id) = self.parameterAsSink(parameters, self.OUTPUT,
                                               context, parcels.fields(), parcels.wkbType(), parcels_o.sourceCrs())

        # connection lines layer
        if connection_lines:
            line_fields = QgsFields()
            line_fields.append(QgsField("dist", QVariant.Double))
            line_fields.append(QgsField("z_diff", QVariant.Double))

            (lines, lines_id) = self.parameterAsSink(parameters, 'LINES',
                                                     context, line_fields, Qgis.WkbType.LineString, parcels_o.sourceCrs())

        # iterate points
        for current, parcel in enumerate(features):
            if feedback.isCanceled():
                break

            z_tol = 0

            # create pointXY
            p_geom = parcel.geometry().asPoint()

            # set min_dist as infinite
            min_dist = float('inf')

            # burn variables in case a feature is not valid
            closest = None
            z_diff = None

            # loop until min dist or z-tol reach threshold
            while (min_dist >= max_dist) and (z_tol <= max_z_tol):
                # filter lower nodes
                exp = QgsExpression('{} <= {} + {}'.format(node_z_field, parcel[parcel_z_field], z_tol))
                request = QgsFeatureRequest(exp).setSubsetOfAttributes([node_idx, node_z])

                # if no nodes are found, increase z_tol
                if len(list(nodes.getFeatures(request))) == 0:
                    z_tol += 1
                    continue

                # search closest node
                for node in nodes.getFeatures(request):
                    n_geom = node.geometry().asPoint()
                    dist = d.measureLine(p_geom, n_geom)

                    if dist < min_dist and dist > 0:
                        min_dist = dist
                        closest = node[node_idx]
                        z_diff = node[node_z] - parcel[parcel_z]
                        if connection_lines:
                            closest_geom = node.geometry().asPoint()
                z_tol += 1
            parcel[parcel_node] = closest
            parcel[parcel_dist] = min_dist
            parcel[parcel_z_diff] = z_diff
            # feedback.setProgressText(str(parcel.attributes()))
            sink.addFeature(parcel, QgsFeatureSink.FastInsert)

            # add connection line
            if connection_lines and closest is not None:
                line = QgsFeature(line_fields)
                line.setAttributes([min_dist, z_diff])
                line.setGeometry(QgsGeometry.fromPolylineXY([p_geom, closest_geom]))
                lines.addFeature(line, QgsFeatureSink.FastInsert)

            # Update the progress bar
            feedback.setProgress(int(current * total))

        # Return the results of the algorithm. In this case our only result is
        # the feature sink which contains the processed features, but some
        # algorithms may return multiple feature sinks, calculated numeric
        # statistics, etc. These should all be included in the returned
        # dictionary, with keys matching the feature corresponding parameter
        # or output names.
        if connection_lines:
            return {self.OUTPUT: dest_id, 'LINES': lines_id}
        else:
            return {self.OUTPUT: dest_id}

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'buildings2sewer'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    # def group(self):
    #     """
    #     Returns the name of the group this algorithm belongs to. This string
    #     should be localised.
    #     """
    #     return self.tr(self.groupId())
    #
    # def groupId(self):
    #     """
    #     Returns the unique ID of the group this algorithm belongs to. This
    #     string should be fixed for the algorithm, and must not be localised.
    #     The group id should be unique within each provider. Group id should
    #     contain lowercase alphanumeric characters only and no spaces or other
    #     formatting characters.
    #     """
    #     return 'Sewer system'

    def icon(self):
        return QIcon(os.path.join(pluginPath, '..', 'icons', 'buildings2sewer.png'))

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return buildings2sewertAlgorithm()

    def shortHelpString(self):
        return "<p>This algorithm connects the buildings of a city to the manholes of the sewer system. It connects each building (using the centroid as a departure point) to the closest manhole that is in the same or in a lower altitude.</p>" \
               "<p>It returns the centroids of the buildings with the following fields:" \
               "<ul><li>Id of the manhole which the building is connected</li><li>Altitude of the building's centroid</li><li>Distance between the building's centroid and the connected manhole</li>" \
               "<li>Altitude difference between the building's centroid and the connected manhole</li></ul>" \
               "If \"Create connection lines is checked\", it also returns a layer with the lines showing each connection.</p>" \
               "<p>Two parameters can be adjusted:<ul>" \
               "<li> <b>Maximum distance:</b> If a connection is larger than the maximum distance, the algorithm searches manholes 1 meter above. This iteration is repeated until the connection is shorter than the maximum distance or until the altitude tolerance is reached </li>" \
               "<li> <b>Altitude tolerance:</b> It determines how many meters upper a manhole can be regarding the building to connect. The altitude tolerance is only used when the maximum distance is surpassed</li>" \
               "</ul></p>"
