import os
import sys
import time
import csv
import itertools
import uuid

from PyQt5 import uic
from PyQt5.QtWidgets import *
from PyQt5.QtGui import QIcon

FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'ui_files', 'ui_lith_s.ui'))

class LithSolid(QDialog, FORM_CLASS):
    def __init__(self, lookup_type_csv_path, lookup_struc_csv_path, layer_dict):
        # setup UI and connect the buttons
        #QDialog.__init__(self)

        super(LithSolid, self).__init__()
        self.setupUi(self)

        ################################
        ##### DEFINE SUB FUNCTIONS #####
        ################################

        def insert_lookup_tables(in_dict):
            ## set number of rows
            self.tbl_type_look.setRowCount(len(in_dict['type']['type']))
            self.tbl_struc_look.setRowCount(len(in_dict['structure']['struc']))

            ## insert into table
            for i in range(len(in_dict['type']['type'])):
                type_input = QTableWidgetItem(in_dict['type']['type'][i])
                self.tbl_type_look.setItem(i, 0, type_input)
                
            for j in range(len(in_dict['structure']['struc'])):
                struc_input = QTableWidgetItem(in_dict['structure']['struc'][j])
                self.tbl_struc_look.setItem(j, 0, struc_input)

        ## get current directory
        current_dir = os.path.dirname(__file__)
        self.selected_layer = None

        ## change layer dictionary to a class level variable
        self.layer_dict = layer_dict
        
        ## fill layer table if having previous layer/lithology associations
        if len(self.layer_dict) > 0:
            self.insert_layer()

        ## set up a lookup dictionary
        ## read values from csv files
        type_csv_s = csv.DictReader(open(lookup_type_csv_path, 'r'), delimiter=';')
        struc_csv_s = csv.DictReader(open(lookup_struc_csv_path, 'r'), delimiter=';')
        lookup_dict = {'type':{'type':[], 'score':[]}, 'structure':{'struc':[], 'score':[]}}

        for line in type_csv_s:
            lookup_dict['type']['type'].append(line['Type'])
            lookup_dict['type']['score'].append(line['Score'])

        for line in struc_csv_s:
            lookup_dict['structure']['struc'].append(line['Structure'])
            lookup_dict['structure']['score'].append(line['Score'])


        insert_lookup_tables(lookup_dict)
        self.tbl_type_look.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.tbl_struc_look.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.tbl_layer_s.setSelectionBehavior(QAbstractItemView.SelectRows)

        ## add icons
        self.btn_add.setIcon(QIcon(os.path.join(current_dir, 'icons', 'add.png')))
        self.btn_rem.setIcon(QIcon(os.path.join(current_dir, 'icons', 'remove.png')))
        self.btn_save.setIcon(QIcon(os.path.join(current_dir, 'icons', 'save.png')))
        self.btn_load.setIcon(QIcon(os.path.join(current_dir, 'icons', 'load.png')))

        ## set up layer table
        ## define column behavior to adapt to the column content
        self.tbl_layer_s.setColumnWidth(1, 150)
        self.tbl_layer_s.setColumnWidth(2, 50)
        self.tbl_layer_s.horizontalHeader().setSectionResizeMode(0, QHeaderView.Stretch)
        self.tbl_layer_s.horizontalHeader().setSectionResizeMode(1, QHeaderView.ResizeToContents)

        ##########################
        ##### MANAGE BUTTONS #####
        ##########################

        ## connect buttons to signals
        self.btn_apply.setEnabled(False)
        self.btn_add.clicked.connect(lambda: self.add_layer(current_dir))
        self.btn_apply.clicked.connect(lambda: self.apply(lookup_dict))
        self.btn_rem.clicked.connect(self.remove)
        self.btn_save.clicked.connect(lambda: self.save(current_dir))
        self.btn_load.clicked.connect(lambda: self.load(current_dir))
        self.btn_accept.clicked.connect(self.accept)
        self.btn_clear_all.clicked.connect(self.clear_layers)
        self.btn_clear_asso.clicked.connect(self.clear_asso)

        ## define table click behavior
        self.tbl_layer_s.cellClicked.connect(lambda: self.clicked(lookup_dict))
        self.tbl_type_look.cellClicked.connect(lambda: self.clicked(lookup_dict))
        self.tbl_type_look.cellDoubleClicked.connect(lambda: self.apply(lookup_dict))
        self.tbl_struc_look.cellClicked.connect(lambda: self.clicked(lookup_dict))
        self.tbl_struc_look.cellDoubleClicked.connect(lambda: self.apply(lookup_dict))


    def add_layer(self, current_dir):
        ## open a file dialog
        ## result is a two element tuple, containing the list of selected layers at index 0
        selected_files = QFileDialog.getOpenFileNames(self, 'Lithology input file', current_dir, 'Images (*.tif *.grd)')[0]

        if len(selected_files) > 0:
            for layer in selected_files:
                ## get path and file name from layer path
                path, filename = os.path.split(layer)

                ## get file name and extension
                layer_s = str(os.path.splitext(filename)[0])

                counter = len(self.layer_dict)

                ## add layers to a dictionary
                self.layer_dict[str(counter)] = {'path':layer, 'name':layer_s, 'lith_type':None, 'lith_struc': None, 'score_type':None, 'score_struc':None}

            self.insert_layer()

        else:
            pass

    def insert_layer(self):
        ## set number of rows
        self.tbl_layer_s.setRowCount(len(self.layer_dict))

        ## insert layer into table
        for index, key in enumerate(self.layer_dict):
            ## get corresponding table widget item
            layer_input = QTableWidgetItem(self.layer_dict[key]['name'])
            self.tbl_layer_s.setItem(index, 0, layer_input)

            if self.layer_dict[key]['lith_type'] != None and self.layer_dict[key]['lith_struc'] != None:
                lith_input = QTableWidgetItem(self.layer_dict[key]['lith_type'] + ',' + self.layer_dict[key]['lith_struc'])
                self.tbl_layer_s.setItem(index, 1, lith_input)
   
    def clicked(self, lookup_dict):
        ## get list with coded indices of all selected rows
        self.selected_layer = self.tbl_layer_s.selectionModel().selectedRows()
        self.selected_type = self.tbl_type_look.selectionModel().selectedRows()
        self.selected_struc = self.tbl_struc_look.selectionModel().selectedRows()
        self.selection_list = None

        ## must have selection from both tables
        ## one or both list elements equal 0 if not
        if len(self.selected_layer) > 0 and len(self.selected_type) > 0 and len(self.selected_struc) > 0:
            self.selection_list = [self.selected_layer, self.selected_type, self.selected_struc]
            self.btn_apply.setEnabled(True)

    def apply(self, lookup_dict):
        if self.selection_list and self.selected_layer:
            ## insert lookup information into layer table
            for i in self.selected_layer:
                for j in self.selected_type:
                    self.layer_dict[str(i.row())]['lith_type'] = lookup_dict['type']['type'][j.row()]
                    self.layer_dict[str(i.row())]['score_type'] = lookup_dict['type']['score'][j.row()]

                    for k in self.selected_struc:
                        lith_input = QTableWidgetItem(lookup_dict['type']['type'][j.row()] + ',' + lookup_dict['structure']['struc'][k.row()])
                        self.tbl_layer_s.setItem(i.row(), 1, lith_input)
                        self.layer_dict[str(i.row())]['lith_struc'] = lookup_dict['structure']['struc'][k.row()]
                        self.layer_dict[str(i.row())]['score_struc'] = lookup_dict['structure']['score'][k.row()]
                        
        self.selection_list = None
        self.selected_layer = None
        self.selected_type = None
        self.selected_struc = None

        self.tbl_layer_s.clearSelection()
        self.tbl_type_look.clearSelection()
        self.tbl_struc_look.clearSelection()

        self.btn_apply.setEnabled(False)

    def remove(self):
        ## loop through selection
        ## remove row at selected table index
        ## remove entry from layer dictionary
        if self.selected_layer:
            for i in self.selected_layer:
                self.tbl_layer_s.removeRow(i.row())
                self.layer_dict.pop(str(i.row()), None)
                
                ## repolutate dictionary keys
                ## NOTE: removing from table changes row numbers, but does not change dicitonary keys
                for j in range(i.row(), len(self.layer_dict)):
                    self.layer_dict[str(j)] = self.layer_dict.pop(str(j+1))

            self.selected_layer = None
            self.tbl_layer_s.clearSelection()

    def save(self, current_dir):
        save_name = QFileDialog.getSaveFileName(self, 'Consolidated Lithology Configuration Save', current_dir, 'All Files (*);;Text Files (*.txt)')[0]

        if save_name:
            with open(save_name, 'w+') as s:
                for i in range(len(self.layer_dict)):
                    s.write(self.layer_dict[str(i)]['path'] + ';' + \
                            self.layer_dict[str(i)]['name'] + ';' + \
                            self.layer_dict[str(i)]['lith_type'] + ';' + \
                            self.layer_dict[str(i)]['lith_struc'] + ';' + \
                            self.layer_dict[str(i)]['score_type'] + ';' + \
                            self.layer_dict[str(i)]['score_struc'] + '\n')

    def load(self, current_dir):
        load_name = QFileDialog.getOpenFileName(self, 'Consolidated Lithology Configuration Load', current_dir, 'All Files (*);;Text Files (*.txt)')[0]

        if load_name:
            self.clear_layers()

            try:
                with open(load_name, 'r') as l:
                    lines = l.read().splitlines()

                    for line in lines:
                        if len(line.split(';')) != 6:
                            QMessageBox.information(None, 'Wrong Input', 'Your selected file does not match the required entries. You probably loaded a loose lithology input file.')
                            break

                        if len(line.split(';')) == 6:
                            filepath, layer_name, lith_type, lith_struc, score_type, score_struc = line.split(';')

                            counter = len(self.layer_dict)

                            ## add layers to a dictionary
                            self.layer_dict[str(counter)] = {'path':filepath, 'name':layer_name, 'lith_type':lith_type, 'lith_struc':lith_struc, 'score_type':score_type, 'score_struc':score_struc}

                self.insert_layer()

            except Exception as e:
                print(e)
                pass
                    

    def clear_layers(self):
        ## empty entire layers table
        self.tbl_layer_s.setRowCount(0)
        self.layer_dict = {}

    def clear_asso(self):
        ## empty current lithology association
        if self.selected_layer:
            for i in self.selected_layer: 
                self.layer_dict[str(i.row())]['lith_type'] = None
                self.layer_dict[str(i.row())]['lith_struc'] = None
                self.layer_dict[str(i.row())]['score_type'] = None
                self.layer_dict[str(i.row())]['score_struc'] = None
                lith_input = QTableWidgetItem('')
                self.tbl_layer_s.setItem(i.row(), 1, lith_input)

        self.selected_layer = None
        self.selection_list = None

    def accept(self):
        check_state = self.check()

        if check_state == True:
            self.hide()

        else:
            pass

    def check(self):
        ## get row count of layer table
        all_layer_rows = self.tbl_layer_s.rowCount()

        ## create empty lists to store row number of mistaken rows
        missing_layer_paths = []
        missing_layer_liths = []

        for row in range(0, all_layer_rows):
            ## check, if layer path exists on the system
            if not os.path.exists(self.layer_dict[str(row)]['path']):
                missing_layer_paths.append(row+1)

            ## check, if layer has a lookup lithology
            if self.tbl_layer_s.item(row, 1).text() == '':
                missing_layer_liths.append(row+1)

            ## check, if association is complete (two items, separated by comma)
            if ',' not in self.tbl_layer_s.item(row, 1).text():
                missing_layer_liths.append(row+1)

        ## show message box with row number if one or two layer paths are missing
        if len(missing_layer_paths) > 0:
            if len(missing_layer_paths) == 1:
                QMessageBox.information(None, 'Info', 'The layer in line %s does not exist.' % (missing_layer_paths[0]))

            elif len(missing_layer_paths) > 1:
                QMessageBox.information(None, 'Info', 'The layers in the lines %s do not exist.' % format([miss for miss in missing_layer_paths]).replace('[','').replace(']',''))

        ## show message box with row number if one or two layer paths have missing lookup lithology
        if len(missing_layer_liths) > 0:
            if len(missing_layer_liths) == 1:
                QMessageBox.information(None, 'Info', 'The layer in line %s does not have an assiocated lookup lithology.' % (missing_layer_liths[0]))

            elif len(missing_layer_liths) > 1:
                QMessageBox.information(None, 'Info', 'The layers in the lines %s do not have associated lookup lithologies.' % format([miss for miss in missing_layer_liths]).replace('[','').replace(']',''))

        if len(missing_layer_paths) > 0 or len(missing_layer_liths) > 0:
            return False

        else:
            return True
