import os
import sys
import time
import csv

from PyQt5 import uic, QtCore
from PyQt5.QtWidgets import *
from PyQt5.QtGui import QIcon

FORM_CLASS, _ = uic.loadUiType(os.path.join(os.path.dirname(__file__), 'ui_files', 'ui_lith_l.ui'))

class LithLoose(QDialog, FORM_CLASS):
    def __init__(self, lookup_csv_path, layer_dict):
        # setup UI and connect the buttons
        #QDialog.__init__(self)

        super(LithLoose, self).__init__()
        self.setupUi(self)

        ################################
        ##### DEFINE SUB FUNCTIONS #####
        ################################

        def insert_lookup_tables(in_dict):
            ## set number of rows
            self.tbl_lith_look.setRowCount(len(in_dict['short']))

            ## insert into table
            for i in range(len(in_dict['short'])):
                short_input = QTableWidgetItem(in_dict['short'][i])
                self.tbl_lith_look.setItem(i, 0, short_input)

                long_input = QTableWidgetItem(in_dict['long'][i])
                self.tbl_lith_look.setItem(i, 1, long_input)

        ## get current directory
        current_dir = os.path.dirname(__file__)
        self.selected_layer = None

        ## change layer dictionary to a class level variable
        self.layer_dict = layer_dict
        
        ## fill layer table if having previous layer/lithology associations
        if len(self.layer_dict) > 0:
            self.insert_layer()

        ## set up a lookup dictionary
        ## read values from csv files
        lith_csv_l = csv.DictReader(open(lookup_csv_path, 'r'), delimiter=';')
        lookup_dict = {'short':[], 'long':[], 'score':[]}

        for line in lith_csv_l:
            lookup_dict['short'].append(line['Short'])
            lookup_dict['long'].append(line['Name'])
            lookup_dict['score'].append(line['Score'])

        insert_lookup_tables(lookup_dict)
        self.tbl_lith_look.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.tbl_layer_l.setSelectionBehavior(QAbstractItemView.SelectRows)

        ## add icons
        self.btn_add.setIcon(QIcon(os.path.join(current_dir, 'icons', 'add.png')))
        self.btn_rem.setIcon(QIcon(os.path.join(current_dir, 'icons', 'remove.png')))
        self.btn_save.setIcon(QIcon(os.path.join(current_dir, 'icons', 'save.png')))
        self.btn_load.setIcon(QIcon(os.path.join(current_dir, 'icons', 'load.png')))

        ## set up layer table
        ## define column behavior to adapt to the column content
        self.tbl_layer_l.setColumnWidth(1, 150)
        self.tbl_layer_l.setColumnWidth(2, 50)
        self.tbl_layer_l.horizontalHeader().setSectionResizeMode(0, QHeaderView.Stretch)
        self.tbl_layer_l.horizontalHeader().setSectionResizeMode(1, QHeaderView.ResizeToContents)
        self.tbl_layer_l.horizontalHeader().setSectionResizeMode(2, QHeaderView.ResizeToContents)

        ##########################
        ##### MANAGE BUTTONS #####
        ##########################

        ## connect buttons to signals
        self.btn_apply.setEnabled(False)
        self.btn_add.clicked.connect(lambda: self.add_layer(current_dir))
        self.btn_apply.clicked.connect(lambda: self.apply(lookup_dict))
        self.btn_rem.clicked.connect(self.remove)
        self.btn_save.clicked.connect(lambda: self.save(current_dir))
        self.btn_load.clicked.connect(lambda: self.load(current_dir))
        self.btn_accept.clicked.connect(self.accept)
        self.btn_clear_all.clicked.connect(self.clear_layers)
        self.btn_clear_asso.clicked.connect(self.clear_asso)

        ## define table click behavior
        self.tbl_layer_l.cellClicked.connect(lambda: self.clicked(lookup_dict))
        self.tbl_lith_look.cellClicked.connect(lambda: self.clicked(lookup_dict))
        self.tbl_lith_look.cellDoubleClicked.connect(lambda: self.apply(lookup_dict))

    def add_layer(self, current_dir):
        ## open a file dialog
        ## result is a two element tuple, containing the list of selected layers at index 0
        selected_files = QFileDialog.getOpenFileNames(self, 'Lithology input file', current_dir, 'Images (*.tif *.grd)')[0]

        if len(selected_files) > 0:
            for layer in selected_files:
                ## get path and file name from layer path
                path, filename = os.path.split(layer)

                ## get file name and extension
                layer_l = str(os.path.splitext(filename)[0])

                counter = len(self.layer_dict)

                ## add layers to a dictionary
                self.layer_dict[str(counter)] = {'path':layer, 'name':layer_l, 'lith_short':None, 'lith_long': None, 'org':None, 'score':None}

            self.insert_layer()

        else:
            pass

    def insert_layer(self):
        ## set number of rows
        self.tbl_layer_l.setRowCount(len(self.layer_dict))
        cmb_entries = ['not set', 'yes', 'no']

        ## insert layer into table
        for index, key in enumerate(self.layer_dict):
            ## get corresponding table widget item
            layer_input = QTableWidgetItem(self.layer_dict[key]['name'])
            lith_input = QTableWidgetItem(self.layer_dict[key]['lith_long'])

            ## set corresponding table widget item
            self.tbl_layer_l.setItem(index, 0, layer_input)
            self.tbl_layer_l.setItem(index, 1, lith_input)

            self.cmb_layer_org = QComboBox()
            self.cmb_layer_org.addItems(cmb_entries)
            self.tbl_layer_l.setCellWidget(index, 2, self.cmb_layer_org)

            ## if organic content has not been set
            ## find text in list and corresponding index in combobox
            ## set combobox to index
            if self.layer_dict[key]['org'] not in cmb_entries[1:]:
                index = self.cmb_layer_org.findText(cmb_entries[0], QtCore.Qt.MatchFixedString)
                self.cmb_layer_org.setCurrentIndex(index)

            else:
                for entry in cmb_entries[1:]:
                    if self.layer_dict[key]['org'] == entry:
                        index = self.cmb_layer_org.findText(entry, QtCore.Qt.MatchFixedString)
                        self.cmb_layer_org.setCurrentIndex(index)

    def clicked(self, lookup_dict):
        #self.btn_rem.setEnabled(True)
        self.selected_layer = self.tbl_layer_l.selectionModel().selectedRows()
        self.selected_lookup = self.tbl_lith_look.selectionModel().selectedRows()
        self.selection_list = None


        ## get list with coded indices of all selected rows
        #selected_layer = self.tbl_layer_l.selectedIndexes()
        #selected_lookup = self.tbl_lith_look.selectedIndexes()

        ## if to remove a layer from the table, only take this one selection
        #if len(selected_layer) > 0 and len(selected_lookup) == 0:
        #    self.selected_layer = selected_layer
        #    self.selection_list = None

        ## must have selection from both tables
        ## one or both list elements equal 0 if not
        if len(self.selected_layer) > 0 and len(self.selected_lookup) > 0:
            self.selection_list = [self.selected_layer, self.selected_lookup]
            self.btn_apply.setEnabled(True)

        ## set self.selection_list to None
        ## enables latter check to have both tables selected
        #else:
        #    self.selection_list = None
        
    def apply(self, lookup_dict):
        if self.selection_list and self.selected_layer:
            ## take selection from both tables
            #selected_layer, selected_lookup = self.selection_list
            
            ## insert lookup information into layer table
            for i in self.selected_layer:
                for j in self.selected_lookup:
                    lith_input = QTableWidgetItem(lookup_dict['long'][j.row()])
                    self.tbl_layer_l.setItem(i.row(), 1, lith_input)
                    self.layer_dict[str(i.row())]['lith_short'] = lookup_dict['short'][j.row()]
                    self.layer_dict[str(i.row())]['lith_long'] = lookup_dict['long'][j.row()]
                    self.layer_dict[str(i.row())]['org'] = 'not set'
                    self.layer_dict[str(i.row())]['score'] = lookup_dict['score'][j.row()]

        self.selection_list = None
        self.selected_layer = None

        self.tbl_layer_l.clearSelection()
        self.tbl_lith_look.clearSelection()

        self.btn_apply.setEnabled(False)

    def remove(self):
        ## loop through selection
        ## remove row at selected table index
        ## remove entry from layer dictionary
        if self.selected_layer:
            for i in self.selected_layer:
                self.tbl_layer_l.removeRow(i.row())
                self.layer_dict.pop(str(i.row()), None)

                ## repolutate dictionary keys
                ## NOTE: removing from table changes row numbers, but does not change dicitonary keys
                for j in range(i.row(), len(self.layer_dict)):
                    self.layer_dict[str(j)] = self.layer_dict.pop(str(j+1))

            self.selected_layer = None
            self.tbl_layer_l.clearSelection()

    def save(self, current_dir):
        save_name = QFileDialog.getSaveFileName(self, 'Unconsolidated Lithology Configuration Save', current_dir, 'All Files (*);;Text Files (*.txt)')[0]

        if save_name:
            with open(save_name, 'w+') as s:
                for i in range(len(self.layer_dict)):
                    self.layer_dict[str(i)]['org'] = self.tbl_layer_l.cellWidget(i, 2).currentText()

                    s.write(self.layer_dict[str(i)]['path'] + ';' + \
                            self.layer_dict[str(i)]['name'] + ';' + \
                            self.layer_dict[str(i)]['lith_short'] + ';' + \
                            self.layer_dict[str(i)]['lith_long'] + ';' + \
                            self.layer_dict[str(i)]['org'] + ';' + \
                            self.layer_dict[str(i)]['score'] + '\n')

    def load(self, current_dir):
        load_name = QFileDialog.getOpenFileName(self, 'Unconsolidated Lithology Configuration Load', current_dir, 'All Files (*);;Text Files (*.txt)')[0]

        if load_name:
            self.clear_layers()

            try:
                with open(load_name, 'r') as l:
                    lines = l.read().splitlines()

                    for line in lines:
                        if len(line.split(';')) != 6:
                            QMessageBox.information(None, 'Wrong Input', 'Your selected file does not match the required entries. You probably loaded a solid lithology input file.')
                            break

                        if len(line.split(';')) == 6:
                            filepath, layer_name, lith_short, lith_long, lith_org, lith_score = line.split(';')

                            counter = len(self.layer_dict)

                            ## add layers to a dictionary
                            self.layer_dict[str(counter)] = {'path':filepath, 'name':layer_name, 'lith_short':lith_short, 'lith_long':lith_long, 'org':lith_org, 'score':lith_score}

                self.insert_layer()

            except Exception as e:
                print(e)
                pass
                    

    def clear_layers(self):
        ## empty entire layers table
        self.tbl_layer_l.setRowCount(0)
        self.layer_dict = {}

    def clear_asso(self):
        ## empty current lithology association
        if self.selected_layer:
            for i in self.selected_layer: 
                self.layer_dict[str(i.row())]['lith_short'] = None
                self.layer_dict[str(i.row())]['lith_long'] = None
                self.layer_dict[str(i.row())]['score'] = None
                #self.layer_dict.pop(str(i.row()), None)
                lith_input = QTableWidgetItem('')
                self.tbl_layer_l.setItem(i.row(), 1, lith_input)

        self.selected_layer = None
        self.selection_list = None

    def accept(self):
        check_state = self.check()

        if check_state == True:
            self.hide()

        else:
            pass

    def check(self):
        ## get row count of layer table
        all_layer_rows = self.tbl_layer_l.rowCount()

        ## create empty lists to store row number of mistaken rows
        missing_layer_paths = []
        missing_layer_liths = []
        missing_layer_org = []

        for row in range(0, all_layer_rows):
            ## check if layer path exists on the system
            if not os.path.exists(self.layer_dict[str(row)]['path']):
                missing_layer_paths.append(row+1)

            ## check if layer has a lookup lithology
            if self.tbl_layer_l.item(row, 1).text() == '':
                missing_layer_liths.append(row+1)

            ## check if organics have been set
            if self.tbl_layer_l.cellWidget(row, 2).currentText() == 'not set':
                missing_layer_org.append(row+1)

        ## show message box with row number if one or two layer paths are missing
        if len(missing_layer_paths) > 0:
            if len(missing_layer_paths) == 1:
                QMessageBox.information(None, 'Info', 'The layer in line %s does not exist.' % (missing_layer_paths[0]))

            elif len(missing_layer_paths) > 1:
                QMessageBox.information(None, 'Info', 'The layers in the lines %s do not exist.' % format([miss for miss in missing_layer_paths]).replace('[','').replace(']',''))

        ## show message box with row number if one or two layer paths have missing lookup lithology
        if len(missing_layer_liths) > 0:
            if len(missing_layer_liths) == 1:
                QMessageBox.information(None, 'Info', 'The layer in line %s does not have an assiocated lookup lithology.' % (missing_layer_liths[0]))

            elif len(missing_layer_liths) > 1:
                QMessageBox.information(None, 'Info', 'The layers in the lines %s do not have associated lookup lithologies.' % format([miss for miss in missing_layer_liths]).replace('[','').replace(']',''))

        if len(missing_layer_org) > 0:
            if len(missing_layer_org) == 1:
                QMessageBox.information(None, 'Info', 'The layer in line %s does not have information about organic content.' % (missing_layer_org[0]))

            elif len(missing_layer_org) > 1:
                QMessageBox.information(None, 'Info', 'The layers in the lines %s do not have information about organic content.' % format([miss for miss in missing_layer_org]).replace('[','').replace(']',''))

        if len(missing_layer_paths) > 0 or len(missing_layer_liths) > 0 or len(missing_layer_org) > 0:
            return False

        else:
            return True
