"""
car_PreparaCarKml
***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************
"""
__author__ = 'Maik Rodrigues'
__date__ = '2025-02-27'
__copyright__ = '(C) 2025 by Maik Rodrigues'
__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import QCoreApplication
from qgis.PyQt.QtWidgets import QMessageBox
from ..geocapt.imgs import Imgs
from qgis.core import (QgsProject,
                       QgsMapLayer,
                       QgsVectorLayer,
                       QgsVectorFileWriter,
                       QgsProcessingException,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterFolderDestination)
from qgis.PyQt.QtGui import QIcon
import os
from geocar.images.Imgs import *

class prepareCAR(QgsProcessingAlgorithm):
    OUTPUT_FOLDER = 'OUTPUT_FOLDER'

    # Define o caminho padrão
    CAMINHO_PADRAO = r"C:\Projetos CAR KML"

    def tr(self, string, string_pt=None):
        if string_pt:
            return QCoreApplication.translate('Processing', string_pt)  # Return the Portuguese translation
        return QCoreApplication.translate('Processing', string)  # Default to English

    def createInstance(self):
        return prepareCAR()

    def name(self):
        return 'preparaCAR'

    def displayName(self):
        return self.tr('Prepara KML para o CAR')

    def group(self):
        return self.tr(self.groupId())

    def groupId(self):
        return ''

    def icon(self):
        return QIcon(os.path.join(os.path.dirname(os.path.dirname(__file__)), 'images/geocar.png'))
    txt_en = '''Export vector layers that are selected from the QGIS project to KML files in a destination folder.'''
    txt_pt = '''Exportar camadas vetoriais que estão selecionadas do projeto QGIS para arquivos KML em uma pasta de destino.'''

    figure1 = 'images\illustration\PrepareCAR.png'
    figure2 = 'images\modelo-logos-parcerias-geo.png'
	
    def shortHelpString(self):
        
        social_BW = Imgs().social_BW
        footer = '''<div style="text-align: right;">
                      <img src="'''+ os.path.join(os.path.dirname(os.path.dirname(__file__)),self.figure1) +'''"><br>
                      <img src="'''+ os.path.join(os.path.dirname(os.path.dirname(__file__)),self.figure2) +'''">
                      </div>
                      <div align="right">
                      <p align="right">
                      <a href="https://georondonia.ifro.edu.br/"><span style="font-weight: bold;">Conheça o projeto GeoRondônia!</span></a>
                      </p>'''+ social_BW + '''</div>
                    </div>'''
        return self.tr(self.txt_en, self.txt_pt) + footer

    def initAlgorithm(self, config=None):
        # Define the folder destination parameter
        self.addParameter(
            QgsProcessingParameterFolderDestination(
                self.OUTPUT_FOLDER,
                'Pasta para gravar os arquivos KML (Arquivos para o CAR)',
                defaultValue=self.CAMINHO_PADRAO  # Usa o caminho padrão
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        output_folder = self.parameterAsString(parameters, self.OUTPUT_FOLDER, context)
        

        # Verifica se o caminho existe; se não existir, cria-o
        if not os.path.exists(output_folder):
            try:
                os.makedirs(output_folder)
                feedback.pushInfo(f"Pasta '{output_folder}' criada com sucesso.")
            except Exception as e:
                raise QgsProcessingException(f"Erro ao criar a pasta '{output_folder}': {str(e)}")

        # Continua com o restante do processamento
        caminho = output_folder
        
        # Obter o projeto atual
        project = QgsProject.instance()

        # Obter a raiz da árvore de camadas
        root = project.layerTreeRoot()

        # Filtrar as camadas selecionadas
        camadas_selecionadas = [
            project.mapLayer(node.layerId()) for node in root.findLayers() if node.isVisible()
        ]

        if not camadas_selecionadas:
            raise QgsProcessingException("Nenhuma camada selecionada. Por favor, selecione as camadas na árvore de camadas.")

        # Inicia a exportação
        feedback.pushInfo(f"{len(camadas_selecionadas)} camada(s) selecionada(s). Iniciando a exportação...")

        # Lista para armazenar camadas exportadas com sucesso
        camadas_exportadas = []

        total = 100.0 / len(camadas_selecionadas) if camadas_selecionadas else 1
        for current, c in enumerate(camadas_selecionadas):
            if feedback.isCanceled():
                break

            try:
                # Verifica se é uma camada vetorial
                if not isinstance(c, QgsVectorLayer):
                    feedback.reportError(f"Camada {c.name()} ignorada (não é uma camada vetorial).")
                    continue

                # Verifica se a camada tem feições
                if c.featureCount() == 0:
                    feedback.reportError(f"Camada {c.name()} ignorada (não contém feições).")
                    continue

                # Define o caminho+nome do arquivo KML
                nome_arquivo = os.path.join(caminho, f"{c.name()}.kml")

                # Exporta a camada para KML
                error = QgsVectorFileWriter.writeAsVectorFormat(
                    c, nome_arquivo, "UTF-8", c.crs(), "KML"
                )

                if error[0] == QgsVectorFileWriter.NoError:
                    feedback.pushInfo(f"Camada >>{c.name()}<< exportada com sucesso para KML.")
                    camadas_exportadas.append(c.name())
                else:
                    feedback.reportError(f"Erro ao exportar a camada {c.name()}: {error[0]}")

            except Exception as e:
                feedback.reportError(f"Erro inesperado ao processar a camada {c.name()}: {str(e)}")

            feedback.setProgress(int(current * total))

        # Mensagem final
        if camadas_exportadas:
            feedback.pushInfo("Exportação concluída!")
            feedback.pushInfo("_____________________________")
            feedback.pushInfo("Camadas Exportadas para KML! ")
            feedback.pushInfo("_____________________________")
            feedback.reportError(f"Veja os arquivos na pasta: {caminho}")

        else:
            feedback.reportError("Nenhuma camada foi exportada.")

        return {"CAMADAS_EXPORTADAS": camadas_exportadas}