from shapely.geometry import shape
import pyproj
import pandas as pd
import requests
import re
from osgeo import gdal, ogr
from concurrent.futures import ThreadPoolExecutor
import time

from ..utils import config as cfg


# Diccionario de mapeo de atributos 
mapeo = {

    "municipio": "county",
    "corregimiento_": "municipality",
    "ap_nombre": "locality",
    #"tipo_ecosis": "habitat",
    #"diseno_monitoreo": "samplingProtocol",
    "temporada_m": "eventRemarks",
    "id_transecto": "eventID",
    #"id_indiv": "organismID",
    "familia_av": "family",
    "genero_av": "genus",  
    "nomb_cientif": "scientificName", 
    "nomb_com": "vernacularName",
    "canonicalName": "acceptedNameUsage",
    "scientificNameAuthorship": "scientificNameAuthorship",
    "taxonomicStatus": "taxonomicStatus",     
    "taxonRank": "taxonRank",
    "comentarios":"occurrenceRemarks",
    "nom_res": "recordedBy",
    "latitud_1": "decimalLatitude",
    "longitud_1": "decimalLongitude",
    "num_indiv": "individualCount",
    "clas_edad": "lifeStage",
    "cob_sustrato": "habitat",
    "estado": "vitality"

    
    

}



# Lista completa de atributos en el archivo final
lista_atributos = [
    "id", "type", "language", "institutionID", "institutionCode", "datasetName",
    "basisOfRecord", "occurrenceID", "recordNumber", "individualCount",
    "occurrenceStatus", "occurrenceRemarks", "lifeStage", "recordedBy", "organismRemarks", "eventID", "organismID",
    "parentEventID", "fieldNumber", "eventDate", "year", "month", "day",
    "verbatimEventDate", "habitat", "samplingProtocol", "sampleSizeValue", "sampleSizeUnit",
    "eventRemarks", "continent", "waterBody", "country",
    "countryCode", "stateProvince", "county", "municipality", "locality",
    "minimumElevationInMeters", "maximumElevationInMeters", "verbatimElevation",
    "locationRemarks", "decimalLatitude", "decimalLongitude", "geodeticDatum",
    "verbatimCoordinates", "verbatimLatitude", "verbatimLongitude",
    "verbatimCoordinateSystem", "verbatimSRS", "footprintWKT", "footprintSRS",
    "verbatimIdentification", "identificationQualifier", "scientificName",
    "acceptedNameUsage", "higherClassification", "kingdom", "phylum", "class",
    "order", "family", "genus", "specificEpithet", "infraspecificEpithet",
    "taxonRank", "verbatimTaxonRank", "scientificNameAuthorship", "vernacularName", "vitality",
    "taxonomicStatus", "permitText"
]


def consultar_taxon_rank(nomb_cientif):
    url = "https://api.gbif.org/v1/species/match"
    params = {"name": nomb_cientif}
    
    try:
        time.sleep(1) 
        response = requests.get(url, params=params)
        if response.status_code == 200:
            data = response.json()
            scientific_name = data.get("scientificName", "")
            canonical_name = data.get("canonicalName", "")
            
            # Extraer la autoría eliminando el nombre científico del nombre canónico
            if canonical_name and scientific_name and scientific_name != canonical_name:
                scientific_name_authorship = scientific_name.replace(canonical_name, "").strip()
            else:
                scientific_name_authorship = None

            return {
                "taxonRank": data.get("rank", None),
                "canonicalName": canonical_name,
                "scientificNameAuthorship": scientific_name_authorship,
                "taxonomicStatus": data.get("status", None)
            }
        else:

            return None
    except Exception as e:

        return None

# Función para agregar los datos de la API al archivo Excel existente
def agregar_datos_api_a_excel(ruta_excel_flora_pnn):
    try:

        df = pd.read_excel(ruta_excel_flora_pnn)

        if "nomb_cientif" not in df.columns:
            raise ValueError("❌ ERROR: La columna 'nomb_cientif' no está en el archivo Excel.")


        especies = df["nomb_cientif"].tolist()

        # Ejecutar consultas a la API en paralelo
        with ThreadPoolExecutor(max_workers=50) as executor:
            resultados = list(executor.map(consultar_taxon_rank, especies))

        # Convertir a DataFrame
        datos_api = pd.DataFrame([r if isinstance(r, dict) else {} for r in resultados])

        # Agregar columnas nuevas al dataframe original
        df["taxonRank"] = datos_api["taxonRank"]
        df["canonicalName"] = datos_api["canonicalName"]
        df["scientificNameAuthorship"] = datos_api["scientificNameAuthorship"]
        df["taxonomicStatus"] = datos_api["taxonomicStatus"]

        df.to_excel(ruta_excel_flora_pnn, index=False)


    except Exception as e:
        return None

# Función para extraer coordenadas
def extraer_coordenadas(layer, epsg_origen):
    resultados = []
    transformer = pyproj.Transformer.from_crs( epsg_origen, cfg.EPSG_DESTINO, always_xy=True)

    for feature in layer:
        geom = feature.GetGeometryRef()
        if geom and geom.GetGeometryType() == ogr.wkbPoint:
            lon, lat = transformer.transform(geom.GetX(), geom.GetY())

            atributos = feature.items()
            atributos["latitud_1"] = lat
            atributos["longitud_1"] = lon
            resultados.append(atributos)

    return resultados


# Función para exportar a Excel
def exportar_excel(dataframe, ruta_salida):
    try:

        dataframe.to_excel(ruta_salida, index=False)

    except Exception as e:
        return None


# Función para procesar campos específicos
def procesar_campos_especificos(df):
    """Calcula los campos en orden secuencial asegurando que cada campo esté disponible antes de ser usado."""

    if df is None or df.empty:

        return df  

    # Mapear elevacion_m a verbatimElevation
    if "elevacion_m" in df.columns:
        df["verbatimElevation"] = df["elevacion_m"]
    else:

        df["verbatimElevation"] = None

     # Mapear latitud a verbatimLatitude
    if "latitud" in df.columns:
        df["verbatimLatitude"] = df["latitud"]
    else:

        df["verbatimLatitude"] = None
    
    # Mapear fecha_da a verbatimEventDate
    if "fecha_da" in df.columns:
        df["verbatimEventDate"] = df["fecha_da"]
    else:

        df["verbatimEventDate"] = None

    # Calcular `eventDate`, `year`, `month`, `day` y conservamos `verbatimEventDate`
    if "verbatimEventDate" in df.columns:

        df["eventDate"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.strftime('%Y-%m-%d')
        df["year"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.year
        df["month"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.month
        df["day"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.day
    
    else:

        df["eventDate"] = None
        df["year"] = None
        df["month"] = None
        df["day"] = None

    # Mapear longitud a verbatimLongitude
    if "longitud" in df.columns:
        df["verbatimLongitude"] = df["longitud"]
    else:

        df["verbatimLongitude"] = None

    # Se calcula  `recordNumber`
    if "object_id" in df.columns:
        df["recordNumber"] = df["object_id"]
    else:

        df["recordNumber"] = None


    # Calcular `occurrenceID`
    if "territorial" in df.columns and "recordNumber" in df.columns:
        df["occurrenceID"] = df.apply(
            lambda row: re.sub(r'[^A-Za-z0-9]', '', row["territorial"]) + ":Fauna densidad caracol pala:" + str(row["recordNumber"]),
            axis=1
        )
    else:

        df["occurrenceID"] = None

     # Asignar `occurrenceID` a `id`
    if "occurrenceID" in df.columns:
        df["id"] = df["occurrenceID"]
    else:

        df["id"] = None


    # Asignar `verbatimElevation` a `minimumElevationInMeters` y `maximumElevationInMeters`
    if "verbatimElevation" in df.columns:
        df["minimumElevationInMeters"] = df["verbatimElevation"]
        df["maximumElevationInMeters"] = df["verbatimElevation"]
    else:

        df["minimumElevationInMeters"] = None
        df["maximumElevationInMeters"] = None

    # Concatenar `COOR_ESTE` y `COOR_NORTE` para `verbatimCoordinates`
    if "verbatimLongitude" in df.columns and "verbatimLatitude" in df.columns:
        df["verbatimCoordinates"] = df.apply(
            lambda row: str(row["verbatimLongitude"]).replace(",", ".") + ", " + str(row["verbatimLatitude"]).replace(",", "."),
            axis=1
        )

    else:

        df["verbatimCoordinates"] = None


    # Asignar `nomb_cientif` a `verbatimIdentification`
    if "nomb_cientif" in df.columns:
        df["verbatimIdentification"] = df["nomb_cientif"]
    else:

        df["verbatimIdentification"] = None

    if "profundidad" in df.columns and "unid_prof" in df.columns:
        df["depth"] = df.apply(
            lambda r: f"{r['profundidad']} {r['unid_prof']}",
            axis=1
        )
    else:
        df["depth"] = None


    


    df["measurementValue_1"] = df["long_t"] if "long_t" in df.columns else None
    df["measurementType_1"] = "Longitud total del individuo encontrado" if "long_t" in df.columns else None
    df["measurementUnit_1"] = df["unid_longit"] if "long_t" in df.columns else None

    df["measurementValue_2"] = df["gros_labio"] if "gros_labio" in df.columns else None
    df["measurementType_2"] = "Grosor del labio del individuo" if "gros_labio" in df.columns else None
    df["measurementUnit_2"] = df["unid_grosor"] if "gros_labio" in df.columns else None

    return df





# Función principal para procesar 
def procesar_fauna_caracol_pnn(ruta_gdb, capa_flora_pnn,  ruta_excel_flora_pnn, archivo_entrada_flora_pnn, archivo_salida_fauna_caracol_pnn, epsg_origen):
    try:
        # Abrir la Geodatabase
        gdb = gdal.OpenEx(ruta_gdb, gdal.OF_VECTOR)
        if not gdb:
            raise RuntimeError(f"❌ No se pudo abrir la GDB en {ruta_gdb}")

        global EPSG_Runtime
        EPSG_Runtime = epsg_origen
        # Extraer coordenadas de la capa 
        datos_capa = extraer_coordenadas(gdb.GetLayerByName(capa_flora_pnn), epsg_origen)


        resultado = pd.DataFrame(datos_capa)


        if resultado is None or resultado.empty:
            return
        
        resultado = resultado.applymap(
        lambda x: None
        if pd.notna(x) and str(x).strip().lower() in ["sin informacion", "sin información"]
        else x
        )


        # Exportar el resultado del join a un archivo Excel intermedio
        exportar_excel(resultado, ruta_excel_flora_pnn)

        # Leer el archivo Excel intermedio y agregar taxonRank
        agregar_datos_api_a_excel(ruta_excel_flora_pnn)

        # Leer el archivo Excel con taxonRank agregado
        df_intermedio = pd.read_excel(ruta_excel_flora_pnn)

        # Procesar campos específicos
        df_intermedio = procesar_campos_especificos(df_intermedio)

        # Crear DataFrame final con todos los atributos de lista_atributos
        df_final = pd.DataFrame(columns=lista_atributos)

        # Mapear los datos del DataFrame intermedio al DataFrame final
        for columna_intermedia, columna_final in mapeo.items():
            if columna_intermedia in df_intermedio.columns:
                df_final[columna_final] = df_intermedio[columna_intermedia]


        crs = pyproj.CRS.from_user_input(epsg_origen)
        if crs.is_geographic:
            coord_system = "Coordenadas geográficas"
        else:
            coord_system = "Coordenadas proyectadas"

        # Valores constantes 
        valores_constantes = {
            #"occurrenceStatus": "present",
            "language": "es",
            "continent": "América del Sur",
            "country": "Colombia",
            "countryCode": "CO",
            "geodeticDatum": "WGS84",
            "verbatimSRS": epsg_origen,
            "verbatimCoordinateSystem": coord_system
        }


        # Agregar valores constantes
        for clave, valor in valores_constantes.items():
            df_final[clave] = valor

        # Agregar los campos calculados
        
        df_final["recordNumber"] = df_intermedio["recordNumber"]
        df_final["eventDate"] = df_intermedio["eventDate"]
        df_final["year"] = df_intermedio["year"]
        df_final["month"] = df_intermedio["month"]
        df_final["day"] = df_intermedio["day"]
        df_final["verbatimEventDate"] = df_intermedio["verbatimEventDate"]
        df_final["verbatimElevation"] = df_intermedio["verbatimElevation"]
        df_final["minimumElevationInMeters"] = df_intermedio["minimumElevationInMeters"]
        df_final["maximumElevationInMeters"] = df_intermedio["maximumElevationInMeters"]
        df_final["verbatimCoordinates"] = df_intermedio["verbatimCoordinates"]
        df_final["verbatimIdentification"] = df_intermedio["verbatimIdentification"]
        df_final["occurrenceID"] = df_intermedio["occurrenceID"]
        df_final["id"] = df_intermedio["id"]
        df_final["verbatimLatitude"] = df_intermedio["verbatimLatitude"]
        df_final["verbatimLongitude"] = df_intermedio["verbatimLongitude"]
        df_final["depth"] = df_intermedio["depth"]
        

        df_final["measurementValue_1"] = df_intermedio["measurementValue_1"]
        df_final["measurementType_1"] = df_intermedio["measurementType_1"]
        df_final["measurementUnit_1"] = df_intermedio["measurementUnit_1"]
        df_final["measurementValue_2"] = df_intermedio["measurementValue_2"]
        df_final["measurementType_2"] = df_intermedio["measurementType_2"]
        df_final["measurementUnit_2"] = df_intermedio["measurementUnit_2"]
       
        measurement_columns_to_drop  = []

        for col in df_final.columns:
            if col.startswith("measurementValue_"):
                suf = col.split("_")[-1] 
                v = f"measurementValue_{suf}"
                t = f"measurementType_{suf}"
                u = f"measurementUnit_{suf}"

                serie = df_final[v].astype("string")

                vacio = (
                    serie.isna() |
                    (serie.str.strip() == "")
                ).all()

                if vacio:
                    measurement_columns_to_drop .extend([v, t, u])

       
        measurement_columns_to_drop  = [c for c in measurement_columns_to_drop  if c in df_final.columns]
        if measurement_columns_to_drop :
            df_final.drop(columns=measurement_columns_to_drop , inplace=True)


        


        # Exportar el DataFrame final a un archivo Excel
        exportar_excel(df_final, archivo_salida_fauna_caracol_pnn)

    except Exception as e:
        return None