from shapely.geometry import shape
import pyproj
import pandas as pd
import requests
import re
from osgeo import gdal, ogr
from concurrent.futures import ThreadPoolExecutor
import time

from ..utils import config as cfg


# Diccionario de mapeo de atributos 
mapeo = {

    "municipio": "county",
    "departamento": "stateProvince",
    "corregimiento_": "municipality",
    #"ap_nombre": "locality",
    #"ecosiste": "habitat",
    "tipo_muestreo": "samplingProtocol",
    #"temporada_m": "eventRemarks",
    "id_transecto": "eventID",
    "id_marca_ante": "organismID",
    #"superfamilia_": "superfamily",
    #"orden_mi": "order",
    "familia": "family",
    "genero": "genus",  
    "nomb_cientif": "scientificName", 
    "grupo_funcion": "vernacularName",
    "canonicalName": "acceptedNameUsage",
    "scientificNameAuthorship": "scientificNameAuthorship",
    "taxonomicStatus": "taxonomicStatus",     
    "taxonRank": "taxonRank",
    "comentarios":"occurrenceRemarks",
    #"nom_resp": "recordedBy",
    "identifica": "identifiedBy",
    "latitud_1": "decimalLatitude",
    "longitud_1": "decimalLongitude",
    "id_muestra": "materialSampleID",
    "profundidad_": "verbatimDepth",
    "num_indiv": "individualCount",
    "estado_desa": "lifeStage",
    "sexo": "sex",
    "und_muestreo": "parentEventID",
    "pendiente_grad": "inclinationInDegrees",
    "tipo_registro": "type",
    "comportamiento": "behavior",
    "recaptura": "organismRemarks"
     
    

}



# Lista completa de atributos en el archivo final
lista_atributos = [
    "id", "type", "language", "institutionID", "institutionCode", "datasetName",
    "basisOfRecord", "occurrenceID", "recordNumber", "individualCount",
    "occurrenceStatus", "behavior","occurrenceRemarks", "lifeStage", "recordedBy", "identifiedBy", "organismRemarks", "eventID", "materialSampleID","organismID",
    "parentEventID", "fieldNumber", "eventDate", "year", "month", "day", "eventTime",
    "verbatimEventDate", "habitat", "samplingProtocol", "sampleSizeValue", "sampleSizeUnit",
    "eventRemarks", "continent", "waterBody", "country",
    "countryCode", "stateProvince", "county", "municipality", "locality",
    "minimumElevationInMeters", "maximumElevationInMeters", "verbatimElevation", "inclinationInDegrees",
    "locationRemarks", "decimalLatitude", "decimalLongitude", "geodeticDatum",
    "verbatimCoordinates", "verbatimLatitude", "verbatimLongitude",
    "verbatimCoordinateSystem", "verbatimSRS", "footprintWKT", "footprintSRS",
    "verbatimIdentification", "sex", "identificationQualifier", "scientificName",
    "acceptedNameUsage", "higherClassification", "kingdom", "phylum", "class",
    "order", "superfamily", "family", "genus", "specificEpithet", "infraspecificEpithet",
    "taxonRank", "verbatimTaxonRank", "scientificNameAuthorship", "vernacularName", "vitality",
    "taxonomicStatus", "permitText"
]


def consultar_taxon_rank(nomb_cientif):
    url = "https://api.gbif.org/v1/species/match"
    params = {"name": nomb_cientif}
    
    try:
        time.sleep(1) 
        response = requests.get(url, params=params)
        if response.status_code == 200:
            data = response.json()
            scientific_name = data.get("scientificName", "")
            canonical_name = data.get("canonicalName", "")
            
            # Extraer la autoría eliminando el nombre científico del nombre canónico
            if canonical_name and scientific_name and scientific_name != canonical_name:
                scientific_name_authorship = scientific_name.replace(canonical_name, "").strip()
            else:
                scientific_name_authorship = None

            return {
                "taxonRank": data.get("rank", None),
                "canonicalName": canonical_name,
                "scientificNameAuthorship": scientific_name_authorship,
                "taxonomicStatus": data.get("status", None)
            }
        else:

            return None
    except Exception as e:

        return None

# Función para agregar los datos de la API al archivo Excel existente
def agregar_datos_api_a_excel(ruta_excel_flora_pnn):
    try:

        df = pd.read_excel(ruta_excel_flora_pnn)

        if "nomb_cientif" not in df.columns:
            raise ValueError("❌ ERROR: La columna 'nomb_cientif' no está en el archivo Excel.")


        especies = df["nomb_cientif"].tolist()

        # Ejecutar consultas a la API en paralelo
        with ThreadPoolExecutor(max_workers=50) as executor:
            resultados = list(executor.map(consultar_taxon_rank, especies))

        # Convertir a DataFrame
        datos_api = pd.DataFrame([r if isinstance(r, dict) else {} for r in resultados])

        # Agregar columnas nuevas al dataframe original
        df["taxonRank"] = datos_api["taxonRank"]
        df["canonicalName"] = datos_api["canonicalName"]
        df["scientificNameAuthorship"] = datos_api["scientificNameAuthorship"]
        df["taxonomicStatus"] = datos_api["taxonomicStatus"]

        df.to_excel(ruta_excel_flora_pnn, index=False)


    except Exception as e:
        return None

# Función para extraer coordenadas
def extraer_coordenadas(layer, epsg_origen):
    resultados = []
    transformer = pyproj.Transformer.from_crs( epsg_origen, cfg.EPSG_DESTINO, always_xy=True)

    for feature in layer:
        geom = feature.GetGeometryRef()
        if geom and geom.GetGeometryType() == ogr.wkbPoint:
            lon, lat = transformer.transform(geom.GetX(), geom.GetY())

            atributos = feature.items()
            atributos["latitud_1"] = lat
            atributos["longitud_1"] = lon
            resultados.append(atributos)

    return resultados


# Función para exportar a Excel
def exportar_excel(dataframe, ruta_salida):
    try:

        dataframe.to_excel(ruta_salida, index=False)

    except Exception as e:
        return None


# Función para procesar campos específicos
def procesar_campos_especificos(df):
    """Calcula los campos en orden secuencial asegurando que cada campo esté disponible antes de ser usado."""

    if df is None or df.empty:

        return df  

    # Mapear elevacion_m a verbatimElevation
    if "elevacion_m" in df.columns:
        df["verbatimElevation"] = df["elevacion_m"]
    else:

        df["verbatimElevation"] = None

     # Mapear latitud a verbatimLatitude
    if "latitud" in df.columns:
        df["verbatimLatitude"] = df["latitud"]
    else:

        df["verbatimLatitude"] = None
    
    # Mapear fecha_da a verbatimEventDate
    if "fecha_da" in df.columns:
        df["verbatimEventDate"] = df["fecha_da"]
    else:

        df["verbatimEventDate"] = None

    # Calcular `eventDate`, `year`, `month`, `day` y conservamos `verbatimEventDate`
    if "verbatimEventDate" in df.columns:

        df["eventDate"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.strftime('%Y-%m-%d')
        df["year"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.year
        df["month"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.month
        df["day"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.day
    
    else:

        df["eventDate"] = None
        df["year"] = None
        df["month"] = None
        df["day"] = None

    # Mapear longitud a verbatimLongitude
    if "longitud" in df.columns:
        df["verbatimLongitude"] = df["longitud"]
    else:

        df["verbatimLongitude"] = None

    # Se calcula  `recordNumber`
    if "object_id" in df.columns:
        df["recordNumber"] = df["object_id"]
    else:

        df["recordNumber"] = None


    # Calcular `occurrenceID`
    if "territorial" in df.columns and "recordNumber" in df.columns:
        df["occurrenceID"] = df.apply(
            lambda row: re.sub(r'[^A-Za-z0-9]', '', row["territorial"]) + ":Fauna ballenas abundancia:" + str(row["recordNumber"]),
            axis=1
        )
    else:

        df["occurrenceID"] = None

     # Asignar `occurrenceID` a `id`
    if "occurrenceID" in df.columns:
        df["id"] = df["occurrenceID"]
    else:

        df["id"] = None


    # Asignar `verbatimElevation` a `minimumElevationInMeters` y `maximumElevationInMeters`
    if "verbatimElevation" in df.columns:
        df["minimumElevationInMeters"] = df["verbatimElevation"]
        df["maximumElevationInMeters"] = df["verbatimElevation"]
    else:

        df["minimumElevationInMeters"] = None
        df["maximumElevationInMeters"] = None

    # Concatenar `COOR_ESTE` y `COOR_NORTE` para `verbatimCoordinates`
    if "verbatimLongitude" in df.columns and "verbatimLatitude" in df.columns:
        df["verbatimCoordinates"] = df.apply(
            lambda row: str(row["verbatimLongitude"]).replace(",", ".") + ", " + str(row["verbatimLatitude"]).replace(",", "."),
            axis=1
        )

    else:

        df["verbatimCoordinates"] = None


    # Asignar `nomb_cientif` a `verbatimIdentification`
    if "nomb_cientif" in df.columns:
        df["verbatimIdentification"] = df["nomb_cientif"]
    else:

        df["verbatimIdentification"] = None

    # Mapear higherClassification 
    campos_clasificacion = [
        "familia", "genero"
    ]

    for col in campos_clasificacion:
        if col not in df.columns:
            df[col] = None

    df["higherClassification"] = df[campos_clasificacion]\
        .applymap(lambda x: x if pd.notna(x) and str(x).strip().lower() not in ["", "none", "nan", "<null>"] else pd.NA)\
        .apply(lambda row: " | ".join(row.dropna().astype(str)), axis=1)
    
    # Mapear a scientificName con validación
    df["scientificName"] = df.apply(
        lambda row: (
            row["nomb_cientif"]
            if pd.notna(row["nomb_cientif"]) and str(row["nomb_cientif"]).strip().lower() not in ["", "none", "nan", "<null>"]
            else (
                row["higherClassification"].split(" | ")[-1]
                if pd.notna(row["higherClassification"]) and " | " in row["higherClassification"]
                else row["higherClassification"]
            )
        ),
        axis=1
    )


    # concatenar ap_nombre y sector para locality
    if "sector" in df.columns and "ap_nombre" in df.columns:
        df["locality"] = df.apply(
            lambda row: f"{row['ap_nombre']}" if pd.notna(row["ap_nombre"]) and pd.isna(row["sector"]) else
                        f"{row['sector']}" if pd.notna(row["sector"]) and pd.isna(row["ap_nombre"]) else
                        f"{row['ap_nombre']}, {row['sector']}" if pd.notna(row["ap_nombre"]) and pd.notna(row["sector"]) else None,
                        
            axis=1
        )

    # concatenar temporadas para eventRemarks
    if "temporada_clim" in df.columns and "temporada_m" in df.columns:
        df["eventRemarks"] = df.apply(
            lambda row: f"{row['temporada_m']}" if pd.notna(row["temporada_m"]) and pd.isna(row["temporada_clim"]) else
                        f"{row['temporada_clim']}" if pd.notna(row["temporada_clim"]) and pd.isna(row["temporada_m"]) else
                        f"{row['temporada_m']}, {row['temporada_clim']}" if pd.notna(row["temporada_m"]) and pd.notna(row["temporada_clim"]) else None,
                        
            axis=1
        )
    else:
        df["eventRemarks"] = None






    # Mapear hora_ini a eventTime 
    if "hora_ini" in df.columns:
        df["eventTime"] = df["hora_ini"].apply(
            lambda x: pd.to_datetime(
                str(x)
                .lower()
                .replace("a. m.", "am")
                .replace("p. m.", "pm")
                .replace("a.m.", "am")
                .replace("p.m.", "pm")
                .strip(),
                format="%I:%M:%S %p",
                errors="coerce"
            ).strftime("%H:%M:%S")
            if pd.notna(x)
            and str(x).strip().lower() not in ["", "nan", "none", "sin informacion"]
            else None
        )
    else:
        df["eventTime"] = None



    # concatenar acompanantes y nom_resp para recordedBy
    if "nom_resp" in df.columns and "acompanantes" in df.columns:
        df["recordedBy"] = df.apply(
            lambda row: f"{row['acompanantes']}" if pd.notna(row["acompanantes"]) and pd.isna(row["nom_resp"]) else
                        f"{row['nom_resp']}" if pd.notna(row["nom_resp"]) and pd.isna(row["acompanantes"]) else
                        f"{row['acompanantes']}, {row['nom_resp']}" if pd.notna(row["acompanantes"]) and pd.notna(row["nom_resp"]) else None,
                        
            axis=1
        )
    

    # Agregar cada campo de medición individualmente
    df["measurementValue_1"] = df["condicion"] if "condicion" in df.columns else None
    df["measurementType_1"] = "Este campo indica la condicion de los individuos de la especie al momento del registro del dato" if "condicion" in df.columns else None

    df["measurementValue_2"] = df["id_marca_der_nueva"] if "id_marca_der_nueva" in df.columns else None
    df["measurementType_2"] = "Codigo de la marca nueva que se le puso en la parte derecha al individuo" if "id_marca_der_nueva" in df.columns else None
    
    df["measurementValue_3"] = df["id_marca_izq_nueva"] if "id_marca_izq_nueva" in df.columns else None
    df["measurementType_3"] = "Codigo de la marca nueva que se le puso en la parte izquierda al individuo" if "id_marca_izq_nueva" in df.columns else None

    df["measurementValue_4"] = df["posicion_marcas"] if "posicion_marcas" in df.columns else None
    df["measurementType_4"] = "Sitio donde se ubican las marcas" if "posicion_marcas" in df.columns else None

    df["measurementValue_5"] = df["long_plastron"] if "long_plastron" in df.columns else None
    df["measurementType_5"] = "Longitud del plastron" if "long_plastron" in df.columns else None

    df["measurementValue_6"] = df["long_cloa_cola"] if "long_cloa_cola" in df.columns else None
    df["measurementType_6"] = "Longitud de la cloaca  a la cola" if "long_cloa_cola" in df.columns else None

    df["measurementValue_7"] = df["long_cola_cap"] if "long_cola_cap" in df.columns else None
    df["measurementType_7"] = "Longitud de la cola al caparazon" if "long_cola_cap" in df.columns else None

    df["measurementValue_8"] = df["long_largo_cur_cap"] if "long_largo_cur_cap" in df.columns else None
    df["measurementType_8"] = "Longitud largo curvo caparazon" if "long_largo_cur_cap" in df.columns else None

    df["measurementValue_9"] = df["long_rect_capa"] if "long_rect_capa" in df.columns else None
    df["measurementType_9"] = "Longitud recto caparazon" if "long_rect_capa" in df.columns else None

    df["measurementValue_10"] = df["long_cabeza"] if "long_cabeza" in df.columns else None
    df["measurementType_10"] = "Longitud de la cabeza" if "long_cabeza" in df.columns else None
    df["measurementUnit_10"] = df["unid_long"] if "long_cabeza" in df.columns else None

    df["measurementValue_11"] = df["anch_plastron"] if "anch_plastron" in df.columns else None
    df["measurementType_11"] = "Ancho del plastron" if "anch_plastron" in df.columns else None

    df["measurementValue_12"] = df["anch_cabeza"] if "anch_cabeza" in df.columns else None
    df["measurementType_12"] = "Ancho de la cabeza" if "anch_cabeza" in df.columns else None

    df["measurementValue_13"] = df["anch_rect_capa"] if "anch_rect_capa" in df.columns else None
    df["measurementType_13"] = "Ancho recto caparazon" if "anch_rect_capa" in df.columns else None

    df["measurementValue_14"] = df["anch_curvo_capa"] if "anch_curvo_capa" in df.columns else None
    df["measurementType_14"] = "Ancho curvo caparazon" if "anch_curvo_capa" in df.columns else None
    df["measurementUnit_14"] = df["unid_ancho"] if "anch_curvo_capa" in df.columns else None

    df["measurementValue_15"] = df["peso"] if "peso" in df.columns else None
    df["measurementType_15"] = "Peso" if "peso" in df.columns else None
    df["measurementUnit_15"] = df["unid_peso"] if "peso" in df.columns else None

    df["measurementValue_16"] = df["epibiontes_"] if "epibiontes_" in df.columns else None
    df["measurementType_16"] = "Epibiontes" if "epibiontes_" in df.columns else None

    df["measurementValue_17"] = df["visibilidad"] if "visibilidad" in df.columns else None
    df["measurementType_17"] = "Hace referencia  a la estimación cualitativa de que tanto se puede observar a la distancia en el momento del muestreo" if "visibilidad" in df.columns else None

    df["measurementValue_18"] = df["oleaje"] if "oleaje" in df.columns else None
    df["measurementType_18"] = "Estimación cualitativa de la intensidad del oleaje en el momento del muestreo" if "oleaje" in df.columns else None

    df["measurementValue_19"] = df["marea"] if "marea" in df.columns else None
    df["measurementType_19"] = "Estimación cualitativa de la marea en el momento del muestreo" if "marea" in df.columns else None

    df["measurementValue_20"] = df["viento"] if "viento" in df.columns else None
    df["measurementType_20"] = "Estimación cualitativa de la intensidad del viento en el momento del muestreo" if "viento" in df.columns else None

    
    





    return df





# Función principal para procesar 
def procesar_abundancia_tortugas_pnn(ruta_gdb, capa_flora_pnn,  ruta_excel_flora_pnn, archivo_entrada_flora_pnn, archivo_salida_abundancia_tortugas_pnn, epsg_origen):
    try:
        # Abrir la Geodatabase
        gdb = gdal.OpenEx(ruta_gdb, gdal.OF_VECTOR)
        if not gdb:
            raise RuntimeError(f"❌ No se pudo abrir la GDB en {ruta_gdb}")

        global EPSG_Runtime
        EPSG_Runtime = epsg_origen
        # Extraer coordenadas de la capa 
        datos_capa = extraer_coordenadas(gdb.GetLayerByName(capa_flora_pnn), epsg_origen)


        resultado = pd.DataFrame(datos_capa)


        if resultado is None or resultado.empty:
            return
        
        resultado = resultado.applymap(
        lambda x: None
        if pd.notna(x) and str(x).strip().lower() in ["sin informacion", "sin información"]
        else x
        )


        # Exportar el resultado del join a un archivo Excel intermedio
        exportar_excel(resultado, ruta_excel_flora_pnn)

        # Leer el archivo Excel intermedio y agregar taxonRank
        agregar_datos_api_a_excel(ruta_excel_flora_pnn)

        # Leer el archivo Excel con taxonRank agregado
        df_intermedio = pd.read_excel(ruta_excel_flora_pnn)

        # Procesar campos específicos
        df_intermedio = procesar_campos_especificos(df_intermedio)

        # Crear DataFrame final con todos los atributos de lista_atributos
        df_final = pd.DataFrame(columns=lista_atributos)

        # Mapear los datos del DataFrame intermedio al DataFrame final
        for columna_intermedia, columna_final in mapeo.items():
            if columna_intermedia in df_intermedio.columns:
                df_final[columna_final] = df_intermedio[columna_intermedia]


        crs = pyproj.CRS.from_user_input(epsg_origen)
        if crs.is_geographic:
            coord_system = "Coordenadas geográficas"
        else:
            coord_system = "Coordenadas proyectadas"

        # Valores constantes 
        valores_constantes = {
            #"occurrenceStatus": "present",
            "language": "es",
            "continent": "América del Sur",
            "country": "Colombia",
            "countryCode": "CO",
            "geodeticDatum": "WGS84",
            "verbatimSRS": epsg_origen,
            "verbatimCoordinateSystem": coord_system
        }


        # Agregar valores constantes
        for clave, valor in valores_constantes.items():
            df_final[clave] = valor

        # Agregar los campos calculados
        
        df_final["recordNumber"] = df_intermedio["recordNumber"]
        df_final["eventDate"] = df_intermedio["eventDate"]
        df_final["year"] = df_intermedio["year"]
        df_final["month"] = df_intermedio["month"]
        df_final["day"] = df_intermedio["day"]
        df_final["verbatimEventDate"] = df_intermedio["verbatimEventDate"]
        df_final["verbatimElevation"] = df_intermedio["verbatimElevation"]
        df_final["minimumElevationInMeters"] = df_intermedio["minimumElevationInMeters"]
        df_final["maximumElevationInMeters"] = df_intermedio["maximumElevationInMeters"]
        df_final["verbatimCoordinates"] = df_intermedio["verbatimCoordinates"]
        df_final["verbatimIdentification"] = df_intermedio["verbatimIdentification"]
        df_final["occurrenceID"] = df_intermedio["occurrenceID"]
        df_final["id"] = df_intermedio["id"]
        df_final["verbatimLatitude"] = df_intermedio["verbatimLatitude"]
        df_final["verbatimLongitude"] = df_intermedio["verbatimLongitude"] 
        df_final["locality"] = df_intermedio["locality"]
        df_final["eventTime"] = df_intermedio["eventTime"]
        df_final["higherClassification"] = df_intermedio["higherClassification"]
        df_final["scientificName"] = df_intermedio["scientificName"]
        df_final["eventRemarks"] = df_intermedio["eventRemarks"]
        df_final["recordedBy"] = df_intermedio["recordedBy"]
       

        df_final["measurementValue_1"] = df_intermedio["measurementValue_1"]
        df_final["measurementType_1"] = df_intermedio["measurementType_1"]
        df_final["measurementValue_2"] = df_intermedio["measurementValue_2"]
        df_final["measurementType_2"] = df_intermedio["measurementType_2"]
        df_final["measurementValue_3"] = df_intermedio["measurementValue_3"]
        df_final["measurementType_3"] = df_intermedio["measurementType_3"]
        df_final["measurementValue_4"] = df_intermedio["measurementValue_4"]
        df_final["measurementType_4"] = df_intermedio["measurementType_4"]
        df_final["measurementValue_5"] = df_intermedio["measurementValue_5"]
        df_final["measurementType_5"] = df_intermedio["measurementType_5"]
        df_final["measurementValue_6"] = df_intermedio["measurementValue_6"]
        df_final["measurementType_6"] = df_intermedio["measurementType_6"]
        df_final["measurementValue_7"] = df_intermedio["measurementValue_7"]
        df_final["measurementType_7"] = df_intermedio["measurementType_7"]
        df_final["measurementValue_8"] = df_intermedio["measurementValue_8"]
        df_final["measurementType_8"] = df_intermedio["measurementType_8"]
        df_final["measurementValue_9"] = df_intermedio["measurementValue_9"]
        df_final["measurementType_9"] = df_intermedio["measurementType_9"]
        df_final["measurementValue_10"] = df_intermedio["measurementValue_10"]
        df_final["measurementType_10"] = df_intermedio["measurementType_10"]
        df_final["measurementUnit_10"] = df_intermedio["measurementUnit_10"]
        df_final["measurementValue_11"] = df_intermedio["measurementValue_11"]
        df_final["measurementType_11"] = df_intermedio["measurementType_11"]
        df_final["measurementValue_12"] = df_intermedio["measurementValue_12"]
        df_final["measurementType_12"] = df_intermedio["measurementType_12"]
        df_final["measurementValue_13"] = df_intermedio["measurementValue_13"]
        df_final["measurementType_13"] = df_intermedio["measurementType_13"]
        df_final["measurementValue_14"] = df_intermedio["measurementValue_14"]
        df_final["measurementType_14"] = df_intermedio["measurementType_14"]
        df_final["measurementUnit_14"] = df_intermedio["measurementUnit_14"]
        df_final["measurementValue_15"] = df_intermedio["measurementValue_15"]
        df_final["measurementType_15"] = df_intermedio["measurementType_15"]
        df_final["measurementUnit_15"] = df_intermedio["measurementUnit_15"]
        df_final["measurementValue_16"] = df_intermedio["measurementValue_16"]
        df_final["measurementType_16"] = df_intermedio["measurementType_16"]
        df_final["measurementValue_17"] = df_intermedio["measurementValue_17"]
        df_final["measurementType_17"] = df_intermedio["measurementType_17"]
        df_final["measurementValue_18"] = df_intermedio["measurementValue_18"]
        df_final["measurementType_18"] = df_intermedio["measurementType_18"]
        df_final["measurementValue_19"] = df_intermedio["measurementValue_19"]
        df_final["measurementType_19"] = df_intermedio["measurementType_19"]
        df_final["measurementValue_20"] = df_intermedio["measurementValue_20"]
        df_final["measurementType_20"] = df_intermedio["measurementType_20"]  

        measurement_columns_to_drop  = []

        for col in df_final.columns:
            if col.startswith("measurementValue_"):
                suf = col.split("_")[-1] 
                v = f"measurementValue_{suf}"
                t = f"measurementType_{suf}"
                u = f"measurementUnit_{suf}"

                serie = df_final[v].astype("string")

                vacio = (
                    serie.isna() |
                    (serie.str.strip() == "")
                ).all()

                if vacio:
                    measurement_columns_to_drop .extend([v, t, u])

        
        measurement_columns_to_drop  = [c for c in measurement_columns_to_drop  if c in df_final.columns]
        if measurement_columns_to_drop :
            df_final.drop(columns=measurement_columns_to_drop , inplace=True)

              


        # Exportar el DataFrame final a un archivo Excel
        exportar_excel(df_final, archivo_salida_abundancia_tortugas_pnn)

    except Exception as e:
        print("\n❌ ERROR en procesar_abundancia_tortugas_pnn:", str(e), "\n")
        return None