from shapely.geometry import shape, Point, MultiLineString
from shapely.ops import transform
from shapely.wkt import loads as wkt_loads
import pyproj
import pandas as pd
import requests
import re
from osgeo import ogr, osr
from concurrent.futures import ThreadPoolExecutor
import time

from ..utils.codigos import departamentos, municipios
from ..utils.config import EPSG_DESTINO
import json

# Diccionario de mapeo de atributos para fauna marina
mapeo = {
    "OPERADOR": "institutionCode",
    "ABUND_ABS": "individualCount",
    "ID_MUEST_T": "eventID",
    "NOMBRE": "fieldNumber",
    "HABITAT": "habitat",
    #"DESCRIP": "samplingProtocol",
    "CUERPO_AGU": "waterBody",
    "VEREDA": "locality",
    "latitud": "decimalLatitude",
    "longitud": "decimalLongitude",
    "COOR_NORTE": "verbatimLatitude",
    "COOR_ESTE": "verbatimLongitude",
    "ESPECIE": "scientificName", 
    "canonicalName": "acceptedNameUsage",
    "scientificNameAuthorship": "scientificNameAuthorship",
    "taxonomicStatus": "taxonomicStatus",
    "DIVISION": "phylum",
    "CLASE": "class",
    "ORDEN": "order",
    "FAMILIA": "family",
    "GENERO": "genus",
    "taxonRank": "taxonRank",
    #"N_COMUN": "vernacularName",
    #"PROF_MUES": "verbatimDepth",
    "OBSERV": "eventRemarks",
    "T_ESF_MUEST": "samplingEffort",
    "LONGITUD_m": "sampleSizeValue",
    "PROYECTO": "datasetName"
}



# Lista completa de atributos en el archivo final
lista_atributos = [
    "id", "type", "language", "institutionID", "institutionCode", "datasetName",
    "basisOfRecord", "occurrenceID", "recordNumber", "individualCount",
    "occurrenceStatus", "occurrenceRemarks", "organismRemarks", "eventID",
    "parentEventID", "fieldNumber", "eventDate", "year", "month", "day",
    "verbatimEventDate", "habitat", "samplingProtocol", "sampleSizeValue", "sampleSizeUnit",
    "eventRemarks", "continent", "waterBody", "country",
    "countryCode", "stateProvince", "county", "municipality", "locality",
    "verbatimDepth", "samplingEffort",
    "locationRemarks", "decimalLatitude", "decimalLongitude", "geodeticDatum",
    "verbatimCoordinates", "verbatimLatitude", "verbatimLongitude",
    "verbatimCoordinateSystem", "verbatimSRS", "footprintWKT", "footprintSRS",
    "verbatimIdentification", "identificationQualifier", "scientificName",
    "acceptedNameUsage", "higherClassification", "kingdom", "phylum", "class",
    "order", "family", "genus", "specificEpithet", "infraspecificEpithet",
    "taxonRank", "verbatimTaxonRank", "scientificNameAuthorship", "vernacularName",
    "taxonomicStatus"
]


def consultar_taxon_rank(especie):
    url = f"http://www.marinespecies.org/rest/AphiaRecordsByMatchNames?scientificnames[]={especie}&marine_only=false"

    try:
        time.sleep(1)  
        response = requests.get(url)

        if response.status_code == 200:
            data = response.json()
            especie_data = data[0][0] if data and data[0] else {}

            return {
                "taxonRank": especie_data.get("rank", None),
                "canonicalName": especie_data.get("valid_name", None),
                "scientificNameAuthorship": especie_data.get("authority", None),
                "taxonomicStatus": especie_data.get("status", None),
                "scientificNameID": f"https://www.marinespecies.org/aphia.php?p=taxdetails&id={especie_data.get('AphiaID')}" if especie_data.get("AphiaID") else None,
                "nameAccordingTo": "World Register of Marine Species",
                "nameAccordingToID": "https://www.marinespecies.org"
            }

        else:

            return None

    except Exception as e:

        return None


def agregar_datos_api_a_excel(ruta_excel_transecto_fm):
    try:

     
        df = pd.read_excel(ruta_excel_transecto_fm)

        if "ESPECIE" not in df.columns:
            raise ValueError("❌ ERROR: La columna 'ESPECIE' no está en el archivo Excel.")


        especies = df["ESPECIE"].tolist()

        # Ejecutar en paralelo las consultas a la API
        with ThreadPoolExecutor(max_workers=10) as executor:
            resultados = list(executor.map(consultar_taxon_rank, especies))

        # Convertir a DataFrame
        datos_api = pd.DataFrame([r if isinstance(r, dict) else {} for r in resultados])

        # Agregar columnas nuevas al DataFrame original
        df["taxonRank"] = datos_api["taxonRank"]
        df["canonicalName"] = datos_api["canonicalName"]
        df["scientificNameAuthorship"] = datos_api["scientificNameAuthorship"]
        df["taxonomicStatus"] = datos_api["taxonomicStatus"]
        df["scientificNameID"] = datos_api.get("scientificNameID")
        df["nameAccordingTo"] = datos_api.get("nameAccordingTo")
        df["nameAccordingToID"] = datos_api.get("nameAccordingToID")


        df.to_excel(ruta_excel_transecto_fm, index=False)


    except Exception as e:
        return None


# Función para convertir geometría de OGR a Shapely
def ogr_to_shapely(feature):
    geom_ogr = feature.GetGeometryRef()  # Obtener geometría OGR
    if geom_ogr is None:
        return None
    return shape(wkt_loads(geom_ogr.ExportToWkt()))  # Convertir OGR → Shapely directamente


# Función para extraer coordenadas y calcular WKT, centroide, y vertice
def extraer_coordenadas(capa, epsg_origen):
    resultados = []
    try:
        transformer = pyproj.Transformer.from_crs(epsg_origen, EPSG_DESTINO, always_xy=True)

        for feature in capa:
            geom = ogr_to_shapely(feature)  # Convertimos OGR → Shapely

            if geom is None:

                continue

            # Transformar la geometría a EPSG:4326
            geom_4326 = transform(transformer.transform, geom)

            # Convertir la geometría transformada a WKT
            wkt = geom_4326.wkt

            centroid = geom.centroid

            # Transformar el centroide a EPSG:4326
            centroid_x, centroid_y = transformer.transform(centroid.x, centroid.y)

            # Manejar geometrías múltiples
            coords = []
            if isinstance(geom, MultiLineString):
                for line in geom.geoms:  # Iterar sobre las sub-geometrías
                    coords.extend(line.coords)
            else:
                coords = geom.coords

            # Encontrar el vértice más cercano al centroide
            nearest_vertex = None
            min_distance = float('inf')
            for coord in coords:
                point = Point(coord)
                distance = centroid.distance(point)
                if distance < min_distance:
                    min_distance = distance
                    nearest_vertex = coord

            # Transformar el vértice más cercano a EPSG:4326
            nearest_vertex_x, nearest_vertex_y = transformer.transform(nearest_vertex[0], nearest_vertex[1])

            atributos = {field: feature.GetField(field) for field in feature.keys()}
            atributos["footprintWKT"] = wkt
            atributos["decimalLatitude"] = nearest_vertex_y
            atributos["decimalLongitude"] = nearest_vertex_x
            resultados.append(atributos)


    except Exception as e:
        return None

    return resultados


# Función para realizar el join
def realizar_join(capa, tabla, enlace_transecto_fm):
    try:


        capa_df = pd.DataFrame(capa)
        tabla_df = pd.DataFrame(tabla)

        if capa_df.empty or tabla_df.empty:

            return None

        resultado_df = pd.merge(capa_df, tabla_df, on=enlace_transecto_fm, how="inner")

        if resultado_df.empty:

            return None

        return resultado_df

    except Exception as e:

        return None

# Función para exportar a Excel
def exportar_excel(dataframe, ruta_salida):
    try:

        dataframe.to_excel(ruta_salida, index=False)

    except Exception as e:
        return None


# Función para extraer los nombres de los departamentos y los municipios
def convertir_codigos_nombres(df):
    """Convierte los códigos de departamentos y municipios a nombres."""
    if "DEPTO" in df.columns:

        df["stateProvince"] = df["DEPTO"].apply(lambda x: departamentos.get(x, "Desconocido"))
    else:
        df["stateProvince"] = None

    if "MUNICIPIO" in df.columns:

        df["county"] = df["MUNICIPIO"].apply(lambda x: municipios.get(x, "Desconocido"))
    else:
        df["county"] = None

    return df

# Función para procesar campos específicos
def procesar_campos_especificos(df):
    """Calcula los campos en orden secuencial asegurando que cada campo esté disponible antes de ser usado."""

    if df is None or df.empty:

        return df  

    # Mapear FECHA_MFA a verbatimEventDate
    if "FECHA_MFA" in df.columns:
        df["verbatimEventDate"] = df["FECHA_MFA"]
    else:

        df["verbatimEventDate"] = None


    # Se calcula `recordNumber`
    if "OBJECTID" in df.columns:
        df["recordNumber"] = df["OBJECTID"]
    else:

        df["recordNumber"] = None

    # Se calcula `type`, ya que otros campos dependerán de él
    if "DETERM" in df.columns:
        df["DETERM"] = df["DETERM"].fillna(0).astype(float).astype(int).astype(str).replace({"0": ""})
        mapeo_determ = {
            411: "Event",
            413: "Event",
            414: "PhysicalObject",
            415: "Sound",
            416: "PhysicalObject",
            417: "PhysicalObject",
            418: "PhysicalObject",
            419: ""
        }
        df["type"] = df["DETERM"].map(mapeo_determ).fillna("")
    else:

        df["type"] = None

    # Calcular `occurrenceID`
    if "OPERADOR" in df.columns and "recordNumber" in df.columns:
        df["occurrenceID"] = df.apply(
            lambda row: re.sub(r'[^A-Za-z0-9]', '', row["OPERADOR"]) + ":transecto_fm:" + str(row["recordNumber"]),
            axis=1
        )
    else:

        df["occurrenceID"] = None

     # Asignar `occurrenceID` a `id`
    if "occurrenceID" in df.columns:
        df["id"] = df["occurrenceID"]
    else:

        df["id"] = None

    # Calcular `basisOfRecord`, que depende de `type`
    if "type" in df.columns:
        mapeo_basis = {
            "Event": "HumanObservation",
            "StillImage": "MachineObservation",
            "PhysicalObject": "MaterialSample",
            "Sound": "HumanObservation"
        }
        df["basisOfRecord"] = df["type"].map(mapeo_basis).fillna("Desconocido")
    else:

        df["basisOfRecord"] = None

    # Calcular `occurrenceRemarks`
    if "DETERM" in df.columns in df.columns:
        df["DETERM"] = df["DETERM"].fillna(0).astype(float).astype(int).astype(str).replace({"0": ""})
        mapeo_determ = {
            411: "Captura de individuos",
            413: "Observación",
            414: "Marcas de Individuos",
            415: "Detección auditiva",
            416: "Huellas",
            417: "Heces",
            418: "Pelos",
            419: "Otro"
        }
        df["temp_occurrenceRemarks"] = df["DETERM"].map(mapeo_determ).fillna("")
    else:

        df["temp_occurrenceRemarks"] = None

    if "temp_occurrenceRemarks" in df.columns and "OBSERV" in df.columns:
        df["occurrenceRemarks"] = df.apply(
            lambda row: (
                str(row["temp_occurrenceRemarks"]) if pd.notna(row["temp_occurrenceRemarks"]) and pd.isna(row["OBSERV"]) else
                str(row["OBSERV"]) if pd.notna(row["OBSERV"]) and pd.isna(row["temp_occurrenceRemarks"]) else
                f"{row['temp_occurrenceRemarks']} | {row['OBSERV']}" if pd.notna(row["temp_occurrenceRemarks"]) and pd.notna(row["OBSERV"]) else
                None
            ),
            axis=1
        )


    # Calcular `eventDate`, `year`, `month`, `day` y conservamos `verbatimEventDate`
    if "verbatimEventDate" in df.columns:

        df["eventDate"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.strftime('%Y-%m-%d')
        df["year"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.year
        df["month"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.month
        df["day"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.day

    else:

        df["eventDate"] = None
        df["year"] = None
        df["month"] = None
        df["day"] = None


    # Asignar `ESPECIE` a `verbatimIdentification`
    if "ESPECIE" in df.columns:
        df["verbatimIdentification"] = df["ESPECIE"]
    else:

        df["verbatimIdentification"] = None

    # se mapea a dynamicProperties
    if "ESTACIONAL" in df.columns:
        df["ESTACIONAL"] = df["ESTACIONAL"].fillna(0).astype(float).astype(int).astype(str).replace({"0": ""})
        mapeo_estacional = {
            "401": "Seca",
            "402": "Lluvias",
            "403": "Transición",
            "404": "Veranillo de San Juan"           
        }
        df["temp_estacional"] = df["ESTACIONAL"].map(mapeo_estacional).fillna("")
    else:

        df["temp_estacional"] = None

    if "FASE" in df.columns:
        df["FASE"] = df["FASE"].fillna(0).astype(float).astype(int).astype(str).replace({"0": ""})
        mapeo_fase = {
            "401": "Seca",
            "402": "Lluvias",
            "403": "Transición",
            "404": "Veranillo de San Juan"           
        }
        df["temp_fase"] = df["FASE"].map(mapeo_fase).fillna("")
    else:

        df["temp_fase"] = None

    campos_clasificacion = [
        "DESCRIP", "temp_fase", "temp_estacional", "DESCR_EPOC"
    ]

    for col in campos_clasificacion:
        if col not in df.columns:
            df[col] = None

    # Generar JSON como string en dynamicProperties
    df["dynamicProperties"] = df[campos_clasificacion].apply(
        lambda row: json.dumps({k: v for k, v in row.items() if pd.notna(v) and str(v).strip().lower() not in ["", "none", "nan", "<null>"]}),
        axis=1
    )



    # Se mapea a samplingProtocol
    if "INS_MUEST" in df.columns:
        df["INS_MUEST"] = df["INS_MUEST"].fillna(0).astype(float).astype(int).astype(str).replace({"0": ""})
        mapeo_ins = {
            "501": "ADCP",
            "502": "Botella Go Flo",
            "503": "Botella Nansen",
            "504": "Botella Niskin",
            "505": "Botella Routner",
            "506": "Box corer",
            "507": "Censo visual",
            "508": "CTD",
            "509": "CTDO",
            "510": "Cuadrante",
            "511": "Draga",
            "512": "Draga Van Veen",
            "513": "HADCP",
            "514": "LADCP",
            "515": "Nasas",
            "516": "Nucleador (Corazonador)",
            "517": "Observación ",
            "518": "Otro",
            "519": "Palangre horizontal ",
            "520": "Palangre vertical",
            "521": "Piston corer",
            "522": "Recolección directa",
            "523": "Red cónica",
            "524": "Red de arrastre",
            "525": "Red de encierro",
            "526": "Red de enmalle",
            "527": "Transecto con punto intercepto",
            "528": "Transectos con cadena intercepto",
            "529": "Transectos en línea"          
            }
        df["samplingProtocol"] = df["INS_MUEST"].map(mapeo_ins).fillna("")
    else:

        df["samplingProtocol"] = None


    # Mapeamos a verbatimDepth    
    if "PROFUN_R" in df.columns and "PROF_MUES" in df.columns:
        df["verbatimDepth"] = df.apply(
            lambda row: (
                f"{int(row['PROFUN_R'])} adultos" if pd.notna(row["PROFUN_R"]) and pd.isna(row["PROF_MUES"]) else
                f"{int(row['PROF_MUES'])} juveniles" if pd.notna(row["PROF_MUES"]) and pd.isna(row["PROFUN_R"]) else
                f"{int(row['PROFUN_R'])} adultos | {int(row['PROF_MUES'])} juveniles" if pd.notna(row["PROFUN_R"]) and pd.notna(row["PROF_MUES"]) else
                None
            ),
            axis=1
        )

    # Mapeamos a vernacularName    
    if "GRUPO" in df.columns and "N_COMUN" in df.columns:
        df["vernacularName"] = df.apply(
            lambda row: (
                f"{int(row['GRUPO'])} adultos" if pd.notna(row["GRUPO"]) and pd.isna(row["N_COMUN"]) else
                f"{int(row['N_COMUN'])} juveniles" if pd.notna(row["N_COMUN"]) and pd.isna(row["GRUPO"]) else
                f"{int(row['GRUPO'])} adultos | {int(row['N_COMUN'])} juveniles" if pd.notna(row["GRUPO"]) and pd.notna(row["N_COMUN"]) else
                None
            ),
            axis=1
        )



    return df



# Función principal para procesar transecto_fm
def procesar_transecto_fm(ruta_gdb, capa_transecto_fm, tabla_transecto_fm, enlace_transecto_fm, ruta_excel_transecto_fm, archivo_entrada_transecto_fm, archivo_salida_transecto_fm, epsg_origen):
    try:
         # 🔹 Abrir la geodatabase con GDAL
        driver = ogr.GetDriverByName("OpenFileGDB")
        gdb = driver.Open(ruta_gdb, 0)
        if gdb is None:

            return

        # 🔹 Cargar la capa de fauna transecto con GDAL
        capa = gdb.GetLayerByName(capa_transecto_fm)
        if capa is None:

            return

        datos_capa = extraer_coordenadas(capa, epsg_origen)  # Llamar a la función

        # 🔹 Cargar la tabla de fauna con GDAL
        tabla = gdb.GetLayerByName(tabla_transecto_fm)
        if tabla is None:

            return

        datos_tabla = [registro.items() for registro in tabla]



        resultado = realizar_join(datos_capa, datos_tabla, enlace_transecto_fm)

        if resultado is None or resultado.empty:

            return

        # Exportar el resultado del join a un archivo Excel intermedio
        exportar_excel(resultado, ruta_excel_transecto_fm)

        # Leer el archivo Excel intermedio y agregar taxonRank
        agregar_datos_api_a_excel(ruta_excel_transecto_fm)

        # Leer el archivo Excel con taxonRank agregado
        df_intermedio = pd.read_excel(ruta_excel_transecto_fm)

        # Procesar campos específicos
        df_intermedio = procesar_campos_especificos(df_intermedio)

        # Crear DataFrame final con todos los atributos de lista_atributos
        df_final = pd.DataFrame(columns=lista_atributos)

        # Mapear los datos del DataFrame intermedio al DataFrame final
        for columna_intermedia, columna_final in mapeo.items():
            if columna_intermedia in df_intermedio.columns:
                df_final[columna_final] = df_intermedio[columna_intermedia]

        crs = pyproj.CRS.from_user_input(epsg_origen)
        if crs.is_geographic:
            coord_system = "Coordenadas geográficas"
        else:
            coord_system = "Coordenadas proyectadas"
        # Valores constantes para fauna marina
        valores_constantes = {
            "occurrenceStatus": "present",
            "language": "es",
            "continent": "América del Sur",
            "country": "Colombia",
            "countryCode": "CO",
            "geodeticDatum": "WGS84",
            "verbatimSRS": epsg_origen,
            "verbatimCoordinateSystem": coord_system
        }
        # Agregar valores constantes
        for clave, valor in valores_constantes.items():
            df_final[clave] = valor

        # Agregar los campos calculados
        df_final["type"] = df_intermedio["type"]
        df_final["recordNumber"] = df_intermedio["recordNumber"]
        df_final["basisOfRecord"] = df_intermedio["basisOfRecord"]
        df_final["occurrenceRemarks"] = df_intermedio["occurrenceRemarks"]
        df_final["eventDate"] = df_intermedio["eventDate"]
        df_final["year"] = df_intermedio["year"]
        df_final["month"] = df_intermedio["month"]
        df_final["day"] = df_intermedio["day"]
        df_final["verbatimEventDate"] = df_intermedio["verbatimEventDate"]
        #df_final["verbatimElevation"] = df_intermedio["verbatimElevation"]
        #df_final["minimumElevationInMeters"] = df_intermedio["minimumElevationInMeters"]
        #df_final["maximumElevationInMeters"] = df_intermedio["maximumElevationInMeters"]
        #df_final["verbatimCoordinates"] = df_intermedio["verbatimCoordinates"]
        df_final["verbatimIdentification"] = df_intermedio["verbatimIdentification"]
        df_final["occurrenceID"] = df_intermedio["occurrenceID"]
        df_final["id"] = df_intermedio["id"]
        #df_final["stateProvince"] = df_intermedio["stateProvince"]
        #df_final["county"] = df_intermedio["county"]
        df_final["dynamicProperties"] = df_intermedio["dynamicProperties"]
        df_final["samplingProtocol"] = df_intermedio["samplingProtocol"]
        df_final["footprintWKT"] = df_intermedio["footprintWKT"]
        df_final["decimalLatitude"] = df_intermedio["decimalLatitude"]
        df_final["decimalLongitude"] = df_intermedio["decimalLongitude"]
        df_final["verbatimDepth"] = df_intermedio["verbatimDepth"]
        df_final["vernacularName"] = df_intermedio["vernacularName"]


        # Exportar el DataFrame final a un archivo Excel
        exportar_excel(df_final, archivo_salida_transecto_fm)

    except Exception as e:
        return None