import pyproj
import pandas as pd
import requests
import re
from osgeo import gdal, ogr
from concurrent.futures import ThreadPoolExecutor
import time

from ..utils.codigos import departamentos, municipios
from ..utils.config import EPSG_DESTINO

# Diccionario de mapeo de atributos para tortugas_OFM
mapeo = {
    "OPERADOR": "institutionCode",
    #"NOMBRE": "fieldNumber",
    "latitud": "decimalLatitude",
    "longitud": "decimalLongitude",
    "ID_INGRESO": "eventID",
    "ID_AVISTAM": "fieldNumber",
    "FECHA": "verbatimEventDate",
    "PROFUND_m": "verbatimDepth",
    #"ORDEN": "order",
    #"FAMILIA": "family",
    #"GENERO": "genus",   
    "especie": "scientificName", 
    #"NOMB_COMUN": "vernacularName",
    "taxonRank": "taxonRank",
    "canonicalName": "acceptedNameUsage",
    "scientificNameAuthorship": "scientificNameAuthorship",
    "taxonomicStatus": "taxonomicStatus",
    "scientificNameID": "scientificNameID",
    "nameAccordingTo": "nameAccordingTo",
    "nameAccordingToID": "nameAccordingToID",
    "NUM_ANIMAL": "individualCount",
    "PROYECTO": "datasetName"
    
}


# Lista completa de atributos en el archivo final
lista_atributos = [
    "id", "type", "language", "institutionID", "institutionCode", "datasetName", "basisOfRecord", "occurrenceID", "recordedBy", 
    "individualCount", "lifeStage", "organismQuantity", "organismQuantityType", "behavior", "occurrenceStatus", "occurrenceRemarks", 
    "eventID", "parentEventID", "fieldNumber", "eventDate", "eventTime", "year", "month", "day", "verbatimEventDate", "samplingProtocol", 
    "samplingEffort", "eventRemarks", "locationID", "continent", "waterBody", "country", "countryCode", "stateProvince", "minimumDepthInMeters", "maximumDepthInMeters",
    "verbatimDepth", "decimalLatitude", "decimalLongitude", "geodeticDatum", "verbatimCoordinates", "verbatimLatitude", "verbatimLongitude", "verbatimCoordinateSystem",
    "verbatimSRS", "verbatimIdentification", "identificationQualifier", "dateIdentified", "scientificNameID", "nameAccordingToID", "scientificName", "acceptedNameUsage", 
    "nameAccordingTo", "higherClassification", "kingdom", "phylum", "class", "order", "family", "genus", "specificEpithet", "taxonRank", "verbatimTaxonRank", 
    "scientificNameAuthorship", "vernacularName", "taxonomicStatus", "identificationVerificationStatus", "measurementValue", "measurementType", 
    "measurementValue_1", "measurementType_1", "measurementValue_2", "measurementType_2", "measurementUnit_2", "measurementValue_3", "measurementType_3", "measurementUnit_3"
]


def consultar_taxon_rank(especie):
    url = f"http://www.marinespecies.org/rest/AphiaRecordsByMatchNames?scientificnames[]={especie}&marine_only=false"

    try:
        time.sleep(1)  
        response = requests.get(url)

        if response.status_code == 200:
            data = response.json()
            especie_data = data[0][0] if data and data[0] else {}

            return {
                "taxonRank": especie_data.get("rank", None),
                "canonicalName": especie_data.get("valid_name", None),
                "scientificNameAuthorship": especie_data.get("authority", None),
                "taxonomicStatus": especie_data.get("status", None),
                "scientificNameID": f"https://www.marinespecies.org/aphia.php?p=taxdetails&id={especie_data.get('AphiaID')}" if especie_data.get("AphiaID") else None,
                "nameAccordingTo": "World Register of Marine Species",
                "nameAccordingToID": "https://www.marinespecies.org"
            }

        else:

            return None

    except Exception as e:

        return None


def agregar_datos_api_a_excel(ruta_excel_tortugas_OFM):
    try:

     
        df = pd.read_excel(ruta_excel_tortugas_OFM)

        if "especie" not in df.columns:
            raise ValueError("❌ ERROR: La columna 'ESPEC_NOMB' no está en el archivo Excel.")


        especies = df["especie"].tolist()

        # Ejecutar en paralelo las consultas a la API
        with ThreadPoolExecutor(max_workers=10) as executor:
            resultados = list(executor.map(consultar_taxon_rank, especies))

        # Convertir a DataFrame
        datos_api = pd.DataFrame([r if isinstance(r, dict) else {} for r in resultados])

        # Agregar columnas nuevas al DataFrame original
        df["taxonRank"] = datos_api["taxonRank"]
        df["canonicalName"] = datos_api["canonicalName"]
        df["scientificNameAuthorship"] = datos_api["scientificNameAuthorship"]
        df["taxonomicStatus"] = datos_api["taxonomicStatus"]
        df["scientificNameID"] = datos_api.get("scientificNameID")
        df["nameAccordingTo"] = datos_api.get("nameAccordingTo")
        df["nameAccordingToID"] = datos_api.get("nameAccordingToID")


        df.to_excel(ruta_excel_tortugas_OFM, index=False)


    except Exception as e:
        return None

# Función para extraer coordenadas
def extraer_coordenadas(layer, epsg_origen):
    resultados = []
    transformer = pyproj.Transformer.from_crs(epsg_origen, EPSG_DESTINO, always_xy=True)

    for feature in layer:
        geom = feature.GetGeometryRef()
        if geom and geom.GetGeometryType() == ogr.wkbPoint:
            lon, lat = transformer.transform(geom.GetX(), geom.GetY())

            atributos = feature.items()
            atributos["latitud"] = lat
            atributos["longitud"] = lon
            resultados.append(atributos)

    return resultados


# Función para realizar el join
def realizar_join(capa, tabla, enlace_tortugas_OFM):
    try:


        capa_df = pd.DataFrame(capa)
        tabla_df = pd.DataFrame(tabla)

        if capa_df.empty or tabla_df.empty:

            return None

        resultado_df = pd.merge(capa_df, tabla_df, on=enlace_tortugas_OFM, how="inner")

        if resultado_df.empty:

            return None

        return resultado_df

    except Exception as e:

        return None


# Función para mapear especie
def mapear_espec_nomb(df):
    if "ESPEC_NOMB" in df.columns:
        df["ESPEC_NOMB"] = df["ESPEC_NOMB"].fillna(0).astype(float).astype(int).astype(str).replace({"0", ""})

    mapa_espec_nomb = {
        "4401": "Carettacaretta",
        "4402": "Cheloniaagassizii",
        "4403": "Cheloniamydas",
        "4404": "Dermochelyscoriacea",
        "4405": "Eretmochelysimbricata",
        "4406": "Lepidochelyskempii",
        "4407": "LepidochelysLepidochelys",
        "4408": "Otro"    
    }
    if "ESPEC_NOMB" in df.columns:
        df["especie"] = df["ESPEC_NOMB"].map(mapa_espec_nomb).fillna(df["ESPEC_NOMB"])
    else:

        df["especie"] = None
    return df



# Función para exportar a Excel
def exportar_excel(dataframe, ruta_salida):
    try:

        dataframe.to_excel(ruta_salida, index=False)

    except Exception as e:
        return None


# Función para procesar campos específicos
def procesar_campos_especificos(df):
    """Calcula los campos en orden secuencial asegurando que cada campo esté disponible antes de ser usado."""

    if df is None or df.empty:

        return df  

     # Mapear COOR_NORTE a verbatimLatitude
    if "COOR_NORTE" in df.columns:
        df["verbatimLatitude"] = df["COOR_NORTE"]
    else:

        df["verbatimLatitude"] = None

    # Mapear COOR_ESTE a verbatimLongitude
    if "COOR_ESTE" in df.columns:
        df["verbatimLongitude"] = df["COOR_ESTE"]
    else:

        df["verbatimLongitude"] = None

    # Se calcula `recordNumber`
    if "OBJECTID" in df.columns:
        df["recordNumber"] = df["OBJECTID"]
    else:

        df["recordNumber"] = None

    # Calcular `occurrenceID`
    if "OPERADOR" in df.columns and "recordNumber" in df.columns:
        df["occurrenceID"] = df.apply(
            lambda row: re.sub(r'[^A-Za-z0-9]', '', row["OPERADOR"]) + ":tortugasOFM:" + str(row["recordNumber"]),
            axis=1
        )
    else:

        df["occurrenceID"] = None

     # Asignar `occurrenceID` a `id`
    if "occurrenceID" in df.columns:
        df["id"] = df["occurrenceID"]
    else:

        df["id"] = None


    # Concatenar `COOR_ESTE` y `COOR_NORTE` para `verbatimCoordinates`
    if "verbatimLongitude" in df.columns and "verbatimLatitude" in df.columns:
        df["verbatimCoordinates"] = df.apply(
            lambda row: str(row["verbatimLongitude"]).replace(",", ".") + ", " + str(row["verbatimLatitude"]).replace(",", "."),
            axis=1
        )
    else:

        df["verbatimCoordinates"] = None


    # se mapea a variables temporales para despues sacar el type

    if "EVENTTYPE" in df.columns:
        mapeo_type = {
            1101: "Event",
            1102: "Event",
            1103: "Event",
            1104: "Event",
            1105: "PhysicalObject",
            1106: "Event"            
        }
        df["temp_type"] = df["EVENTTYPE"].map(mapeo_type).fillna("")
    else:

        df["temp_type"] = None

    if "EVIDENCIA" in df.columns:
        mapeo_evidencia = {
            181: "StillImage",
            182: "MovingImage",
            183: "MovingImage",
            184: "Event"           
        }
        df["temp_type_2"] = df["EVIDENCIA"].map(mapeo_evidencia).fillna("")
    else:

        df["temp_type_2"] = None

    if "temp_type" in df.columns and "temp_type_2" in df.columns:
        df["type"] = df.apply(
            lambda row: (
                str(row["temp_type"]) if pd.notna(row["temp_type"]) and pd.isna(row["temp_type_2"]) else
                str(row["temp_type_2"]) if pd.notna(row["temp_type_2"]) and pd.isna(row["temp_type"]) else
                f"{row['temp_type']} | {row['temp_type_2']}" if pd.notna(row["temp_type"]) and pd.notna(row["temp_type_2"]) else
                None
            ),
            axis=1
        )



    # Calcular `basisOfRecord`, que depende de `type`
    if "temp_type" in df.columns:
        mapeo_basis = {
            "Event": "HumanObservation",
            "PhysicalObject": "MaterialSample"
        }
        df["temp_basisOfRecord"] = df["temp_type"].map(mapeo_basis).fillna("")
    else:

        df["temp_basisOfRecord"] = None

    if "temp_type_2" in df.columns:
        mapeo_basis_2 = {
            "StillImage": "MachineObservation",
            "MovingImage": "MachineObservation",
            "Event": "HumanObservation"
        }
        df["temp_basisOfRecord_2"] = df["temp_type_2"].map(mapeo_basis_2).fillna("")
    else:

        df["temp_basisOfRecord_2"] = None
    
    
    if "temp_basisOfRecord" in df.columns and "temp_basisOfRecord_2" in df.columns:
        df["basisOfRecord"] = df.apply(
            lambda row: (
                str(row["temp_basisOfRecord"]) if pd.notna(row["temp_basisOfRecord"]) and pd.isna(row["temp_basisOfRecord_2"]) else
                str(row["temp_basisOfRecord_2"]) if pd.notna(row["temp_basisOfRecord_2"]) and pd.isna(row["temp_basisOfRecord"]) else
                f"{row['temp_basisOfRecord']} | {row['temp_basisOfRecord_2']}" if pd.notna(row["temp_basisOfRecord"]) and pd.notna(row["temp_basisOfRecord_2"]) else
                None
            ),
            axis=1
        )
    
    
    
    # Calcular `occurrenceRemarks`
    if "EVENTTYPE" in df.columns:
        mapeo_eventtype = {
            1101: "Afloat",
            1102: "Boat strike",
            1103: "Entanglement",
            1104: "Sighting",
            1105: "Skeletal Remains",
            1106: "Stranding"
        }
        df["temp_occurrenceRemarks"] = df["EVENTTYPE"].map(mapeo_eventtype).fillna("")
    else:

        df["temp_occurrenceRemarks"] = None

    if "EVIDENCIA" in df.columns:
        mapeo_evidencia = {
            181: "Fotografía",
            182: "video",
            183: "Video y fotografía",
            184: "Solo observación" 
        }
        df["temp_occurrenceRemarks_2"] = df["EVIDENCIA"].map(mapeo_evidencia).fillna("")
    else:

        df["temp_occurrenceRemarks_2"] = None
    
    
    #  Mapear occurrenceRemarks 
    campos_occurrenceRemarks = [
        "temp_occurrenceRemarks", "temp_occurrenceRemarks_2", "OBSERVACIO_y"
    ]
    

    for col in campos_occurrenceRemarks:
        if col not in df.columns:
            df[col] = None

  
    df["occurrenceRemarks"] = df[campos_occurrenceRemarks]\
        .applymap(lambda x: x if pd.notna(x) and str(x).strip().lower() not in ["", "none", "nan", "<null>"] else pd.NA)\
        .apply(lambda row: " | ".join(row.dropna().astype(str)), axis=1)



    # Mapear TIEMPO_I y TIEMPO_F a eventTime con formato HH:MM:SS
    if "TIEMPO_I" in df.columns or "TIEMPO_F" in df.columns:
        def formatear_hora(horas):
            tiempo = pd.to_timedelta(horas, unit="h")
            return f"{int(tiempo.components.hours):02}:{int(tiempo.components.minutes):02}:{int(tiempo.components.seconds):02}"
        

        df["eventTime"] = df.apply(
            lambda row: (
                formatear_hora(row.get("TIEMPO_I")) if pd.notna(row.get("TIEMPO_I")) and pd.isna(row.get("TIEMPO_F")) else
                formatear_hora(row.get("TIEMPO_F")) if pd.notna(row.get("TIEMPO_F")) and pd.isna(row.get("TIEMPO_I")) else
                f"{formatear_hora(row.get('TIEMPO_I'))} / {formatear_hora(row.get('TIEMPO_F'))}"
                if pd.notna(row.get("TIEMPO_I")) and pd.notna(row.get("TIEMPO_F")) else None
            ),
            axis=1
        )



    # Mapeamos a order
    if "ORDEN" in df.columns:
        mapeo_orden = {
            4101: "Testudines",
            4102: "Otro"        
        }
        df["order"] = df["ORDEN"].map(mapeo_orden).fillna("")
    else:

        df["order"] = None


    # Mapeamos SUBORDEN a una variable temporal
    if "SUBORDEN" in df.columns:
        df["temp_suborden"] = df["SUBORDEN"]
    else:

        df["temp_suborden"] = None

    # Mapeamos a family
    if "FAMILIA" in df.columns:
        mapeo_familia = {
            4201: "Cheloniidae",
            4202: "Dermochelyidae",
            403: "otro"
        }
        df["family"] = df["FAMILIA"].map(mapeo_familia).fillna("")
    else:

        df["family"] = None


    # Mapeamos a genus
    if "GENERO" in df.columns:
        mapeo_genero = {
            4301: "Carettasp.",
            4302: "Cheloniasp.",
            4303: "Dermochelyssp.",
            4304: "Eretmochelyssp.",
            4305: "Lepidochelyssp.",
            4306: "Otro"

        }
        df["genus"] = df["GENERO"].map(mapeo_genero).fillna("")
    else:

        df["genus"] = None
    
    # Mapear higherClassification 
    campos_clasificacion = [
        "order", "temp_suborden", "family", "genus"
    ]

    for col in campos_clasificacion:
        if col not in df.columns:
            df[col] = None

    df["higherClassification"] = df[campos_clasificacion]\
        .applymap(lambda x: x if pd.notna(x) and str(x).strip().lower() not in ["", "none", "nan", "<null>"] else pd.NA)\
        .apply(lambda row: " | ".join(row.dropna().astype(str)), axis=1)

    # Mapear a scientificName con validación
    df["scientificName"] = df.apply(
        lambda row: (
            row["especie"]
            if pd.notna(row["especie"]) and str(row["especie"]).strip().lower() not in ["", "none", "nan", "<null>", "Otros", "Otro", "otro","Otra","Otras"]
            else (
                row["higherClassification"].split(" | ")[-1]
                if pd.notna(row["higherClassification"]) and " | " in row["higherClassification"]
                else row["higherClassification"]
            )
        ),
        axis=1
    )



    # Mapeamos NOMB_COMUN
    if "NOMB_COMUN" in df.columns:
        df["NOMB_COMUN"] = df["NOMB_COMUN"].fillna(0).astype(float).astype(int).astype(str).replace({"0": ""})

        mapeo_nombre = {
            "4501": "Caguamo",
            "4502": "Tortuga carey",
            "4503": "Tortuga verde",
            "4504": "Otro"
     }
        df["vernacularName"] = df["NOMB_COMUN"].map(mapeo_nombre).fillna("")
    else:

        df["vernacularName"] = None

   

    # Mapear FECHA_MFL a verbatimEventDate
    if "FECHA" in df.columns:
        df["verbatimEventDate"] = df["FECHA"]
    else:

        df["verbatimEventDate"] = None


    # Calcular `eventDate`, `year`, `month`, `day` y conservamos `verbatimEventDate`
    if "verbatimEventDate" in df.columns:

        df["eventDate"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.strftime('%Y-%m-%d')
        df["year"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.year
        df["month"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.month
        df["day"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.day

    else:

        df["eventDate"] = None
        df["year"] = None
        df["month"] = None
        df["day"] = None

    if "DIRE_BARCO" in df.columns:
        df["DIRE_BARCO"] = df["DIRE_BARCO"].fillna(0).astype(float).astype(int).astype(str).replace({"0": ""})
        mapa_dire = {
        "161": "N (Norte)",
        "162": "S (Sur)",
        "163": "E (Este)",
        "164": "O (Oeste)",
        "165": "NE (Nor-Este)",
        "166": "SE (Sur-Este)",
        "167": "SO (Sur-Oeste)",
        "168": "NO (Nor-Oeste)",
        "169": "Plataforma fija (no aplica)"
        }
        df["temp_measurementValue"] = df["DIRE_BARCO"].map(mapa_dire).fillna("")
    else:

        df["temp_measurementValue"] = None


    if "temp_measurementValue" in df.columns and df["temp_measurementValue"].dropna().empty is False:
        df["measurementValue"] = df["temp_measurementValue"]
        df["measurementType"] = "Dirección del barco"
    else:

        df["measurementValue"] = None
        df["measurementType"] = None


    if "DIRE_GRUPO" in df.columns:
        df["DIRE_GRUPO"] = df["DIRE_BADIRE_GRUPORCO"].fillna(0).astype(float).astype(int).astype(str).replace({"0": ""})
        mapa_dire_gr = {
        "161": "N (Norte)",
        "162": "S (Sur)",
        "163": "E (Este)",
        "164": "O (Oeste)",
        "165": "NE (Nor-Este)",
        "166": "SE (Sur-Este)",
        "167": "SO (Sur-Oeste)",
        "168": "NO (Nor-Oeste)",
        "169": "Plataforma fija (no aplica)"
        }
        df["temp_measurementValue_1"] = df["DIRE_GRUPO"].map(mapa_dire_gr).fillna("")
    else:

        df["temp_measurementValue_1"] = None

    if "temp_measurementValue_1" in df.columns and df["temp_measurementValue_1"].dropna().empty is False:
        df["measurementValue_1"] = df["temp_measurementValue_1"]
        df["measurementType_1"] = "Comportamiento de viaje de los individuos respecto a la plataforma de observación"
    else:

        df["measurementValue_1"] = None
        df["measurementType_1"] = None


    # Verificar si "ANG_OBSERV" existe y tiene al menos un valor no vacío
    if "ANG_OBSERV" in df.columns and df["ANG_OBSERV"].dropna().empty is False:
        df["measurementValue_2"] = df["ANG_OBSERV"]
        df["measurementType_2"] = "Dirección del animal respecto a la plataforma"
        df["measurementUnit_2"] = "grados"
    else:

        df["measurementValue_2"] = None
        df["measurementType_2"] = None
        df["measurementUnit_2"] = None


    # Verificar si "DISTAN_b_m" existe y tiene al menos un valor no vacío
    if "DISTAN_b_m" in df.columns and df["DISTAN_b_m"].dropna().empty is False:
        df["measurementValue_3"] = df["DISTAN_b_m"]
        df["measurementType_3"] = "Distancia de la plataforma al animal"
        df["measurementUnit_3"] = "m"
    else:

        df["measurementValue_3"] = None
        df["measurementType_3"] = None
        df["measurementUnit_3"] = None

    return df



# Función principal para procesar tortugas_OFM
def procesar_tortugas_OFM(ruta_gdb, capa_tortugas_OFM, tabla_tortugas_OFM, enlace_tortugas_OFM, ruta_excel_tortugas_OFM, archivo_entrada_tortugas_OFM, archivo_salida_tortugas_OFM, epsg_origen):
    try:
        # Abrir la Geodatabase
        gdb = gdal.OpenEx(ruta_gdb, gdal.OF_VECTOR)
        if not gdb:
            raise RuntimeError(f"❌ No se pudo abrir la GDB en {ruta_gdb}")

        # Extraer coordenadas de la capa de tortugas_OFM
        datos_capa = extraer_coordenadas(gdb.GetLayerByName(capa_tortugas_OFM), epsg_origen)

        # Extraer atributos de la tabla de tortugas_OFM
        datos_tabla = []
        layer = gdb.GetLayerByName(tabla_tortugas_OFM)
        for feature in layer:
            datos_tabla.append(feature.items())  

        resultado = realizar_join(datos_capa, datos_tabla, enlace_tortugas_OFM)

        if resultado is None or resultado.empty:

            return

        # Primero mapear especie
        resultado = mapear_espec_nomb(resultado)

        # Exportar el resultado del join a un archivo Excel intermedio
        exportar_excel(resultado, ruta_excel_tortugas_OFM)

        # Leer el archivo Excel intermedio y agregar taxonRank
        agregar_datos_api_a_excel(ruta_excel_tortugas_OFM)

        # Leer el archivo Excel con taxonRank agregado
        df_intermedio = pd.read_excel(ruta_excel_tortugas_OFM)
    
        # Procesar campos específicos
        df_intermedio = procesar_campos_especificos(df_intermedio)

        # Crear DataFrame final con todos los atributos de lista_atributos
        df_final = pd.DataFrame(columns=lista_atributos)

        # Mapear los datos del DataFrame intermedio al DataFrame final
        for columna_intermedia, columna_final in mapeo.items():
            if columna_intermedia in df_intermedio.columns:
                df_final[columna_final] = df_intermedio[columna_intermedia]

        crs = pyproj.CRS.from_user_input(epsg_origen)
        if crs.is_geographic:
            coord_system = "Coordenadas geográficas"
        else:
            coord_system = "Coordenadas proyectadas"
        # Valores constantes para tortugas_OFM
        valores_constantes = {
            "occurrenceStatus": "present",
            "language": "es",
            "continent": "América del Sur",
            "country": "Colombia",
            "countryCode": "CO",
            "geodeticDatum": "WGS84",
            "verbatimSRS": epsg_origen,
            "verbatimCoordinateSystem": coord_system
        }

        # Agregar valores constantes
        for clave, valor in valores_constantes.items():
            df_final[clave] = valor

        # Agregar los campos calculados
        
        df_final["verbatimCoordinates"] = df_intermedio["verbatimCoordinates"]
        #df_final["verbatimIdentification"] = df_intermedio["verbatimIdentification"]
        df_final["occurrenceID"] = df_intermedio["occurrenceID"]
        df_final["type"] = df_intermedio["type"]
        df_final["basisOfRecord"] = df_intermedio["basisOfRecord"]
        df_final["occurrenceRemarks"] = df_intermedio["occurrenceRemarks"]
        df_final["eventTime"] = df_intermedio["eventTime"]
        df_final["order"] = df_intermedio["order"]
        #df_final["temp_suborden"] = df_intermedio["temp_suborden"]
        df_final["family"] = df_intermedio["family"]
        df_final["genus"] = df_intermedio["genus"]
        df_final["higherClassification"] = df_intermedio["higherClassification"]
        df_final["scientificName"] = df_intermedio["scientificName"]
        df_final["vernacularName"] = df_intermedio["vernacularName"]
        df_final["measurementValue"] = df_intermedio["measurementValue"]
        df_final["measurementType"] = df_intermedio["measurementType"]
        df_final["measurementValue_1"] = df_intermedio["measurementValue_1"]
        df_final["measurementType_1"] = df_intermedio["measurementType_1"]
        df_final["measurementValue_2"] = df_intermedio["measurementValue_2"]
        df_final["measurementType_2"] = df_intermedio["measurementType_2"]
        df_final["measurementUnit_2"] = df_intermedio["measurementUnit_2"]
        df_final["measurementValue_3"] = df_intermedio["measurementValue_3"]
        df_final["measurementType_3"] = df_intermedio["measurementType_3"]
        df_final["measurementUnit_3"] = df_intermedio["measurementUnit_3"]
        df_final["eventDate"] = df_intermedio["eventDate"]
        df_final["year"] = df_intermedio["year"]
        df_final["month"] = df_intermedio["month"]
        df_final["day"] = df_intermedio["day"]
        df_final["verbatimEventDate"] = df_intermedio["verbatimEventDate"]
      
        measurement_columns_to_drop  = []

        for col in df_final.columns:
            if col.startswith("measurementValue_"):
                suf = col.split("_")[-1] 
                v = f"measurementValue_{suf}"
                t = f"measurementType_{suf}"
                u = f"measurementUnit_{suf}"

                serie = df_final[v].astype("string")

                vacio = (
                    serie.isna() |
                    (serie.str.strip() == "")
                ).all()

                if vacio:
                    measurement_columns_to_drop .extend([v, t, u])

        
        measurement_columns_to_drop  = [c for c in measurement_columns_to_drop  if c in df_final.columns]
        if measurement_columns_to_drop :
            df_final.drop(columns=measurement_columns_to_drop , inplace=True)

        # Exportar el DataFrame final a un archivo Excel
        exportar_excel(df_final, archivo_salida_tortugas_OFM)

    except Exception as e:
        return None