import pyproj
import pandas as pd
import requests
import re
from osgeo import gdal, ogr
from concurrent.futures import ThreadPoolExecutor
import time 

from ..utils.codigos import departamentos, municipios
from ..utils.config import EPSG_DESTINO

# Diccionario de mapeo de atributos para peces_OFM
mapeo = {
    "OPERADOR": "institutionCode",
    #"NOMBRE": "fieldNumber",
    "latitud": "decimalLatitude",
    "longitud": "decimalLongitude",
    "ID_INGRESO": "eventID",
    "CERTE_DATO": "identificationVerificationStatus",
    "ID_AVISTAM": "fieldNumber",
    "FECHA": "verbatimEventDate",
    "PROFUND_m": "verbatimDepth",
    #"ORDEN": "order",
    #"FAMILIA": "family",
    #"GENERO": "genus",   
    "especie": "scientificName", 
    #"NOMB_COMUN": "vernacularName",
    "taxonRank": "taxonRank",
    "canonicalName": "acceptedNameUsage",
    "scientificNameAuthorship": "scientificNameAuthorship",
    "taxonomicStatus": "taxonomicStatus",
    "scientificNameID": "scientificNameID",
    "nameAccordingTo": "nameAccordingTo",
    "nameAccordingToID": "nameAccordingToID",
    "NUM_ANIMAL": "individualCount",
    "COMPORTAMI": "behavior",
    "PROYECTO": "datasetName"
    
}



# Lista completa de atributos en el archivo final
lista_atributos = [
    "id", "type", "language", "institutionID", "institutionCode", "datasetName", "basisOfRecord", "occurrenceID", "recordedBy", 
    "individualCount", "lifeStage", "organismQuantity", "organismQuantityType", "behavior", "occurrenceStatus", "occurrenceRemarks", 
    "eventID", "parentEventID", "fieldNumber", "eventDate", "eventTime", "year", "month", "day", "verbatimEventDate", "samplingProtocol", 
    "samplingEffort", "eventRemarks", "locationID", "continent", "waterBody", "country", "countryCode", "stateProvince", "minimumDepthInMeters", "maximumDepthInMeters",
    "verbatimDepth", "decimalLatitude", "decimalLongitude", "geodeticDatum", "verbatimCoordinates", "verbatimLatitude", "verbatimLongitude", "verbatimCoordinateSystem",
    "verbatimSRS", "verbatimIdentification", "identificationQualifier", "dateIdentified", "scientificNameID", "nameAccordingToID", "scientificName", "acceptedNameUsage", 
    "nameAccordingTo", "higherClassification", "kingdom", "phylum", "class", "order", "family", "genus", "specificEpithet", "taxonRank", "verbatimTaxonRank", 
    "scientificNameAuthorship", "vernacularName", "taxonomicStatus", "identificationVerificationStatus", "measurementValue", "measurementType", "measurementUnit", 
    "measurementValue_1", "measurementType_1"
]


def consultar_taxon_rank(especie):
    url = f"http://www.marinespecies.org/rest/AphiaRecordsByMatchNames?scientificnames[]={especie}&marine_only=false"

    try:
        time.sleep(1)  
        response = requests.get(url)

        if response.status_code == 200:
            data = response.json()
            especie_data = data[0][0] if data and data[0] else {}

            return {
                "taxonRank": especie_data.get("rank", None),
                "canonicalName": especie_data.get("valid_name", None),
                "scientificNameAuthorship": especie_data.get("authority", None),
                "taxonomicStatus": especie_data.get("status", None),
                "scientificNameID": f"https://www.marinespecies.org/aphia.php?p=taxdetails&id={especie_data.get('AphiaID')}" if especie_data.get("AphiaID") else None,
                "nameAccordingTo": "World Register of Marine Species",
                "nameAccordingToID": "https://www.marinespecies.org"
            }

        else:

            return None

    except Exception as e:

        return None


def agregar_datos_api_a_excel(ruta_excel_peces_OFM):
    try:

        df = pd.read_excel(ruta_excel_peces_OFM)

        if "especie" not in df.columns:
            raise ValueError("❌ ERROR: La columna 'ESPEC_NOMB' no está en el archivo Excel.")


        especies = df["especie"].tolist()

        # Ejecutar en paralelo las consultas a la API
        with ThreadPoolExecutor(max_workers=10) as executor:
            resultados = list(executor.map(consultar_taxon_rank, especies))

        # Convertir a DataFrame
        datos_api = pd.DataFrame([r if isinstance(r, dict) else {} for r in resultados])

        # Asegurar que todas las columnas necesarias existan aunque estén vacías
        columnas_requeridas = [
            "taxonRank", "canonicalName", "scientificNameAuthorship",
            "taxonomicStatus", "scientificNameID", "nameAccordingTo", "nameAccordingToID"
        ]
        for col in columnas_requeridas:
            if col not in datos_api.columns:
                datos_api[col] = None

        # Agregar columnas nuevas al DataFrame original
        df["taxonRank"] = datos_api["taxonRank"]
        df["canonicalName"] = datos_api["canonicalName"]
        df["scientificNameAuthorship"] = datos_api["scientificNameAuthorship"]
        df["taxonomicStatus"] = datos_api["taxonomicStatus"]
        df["scientificNameID"] = datos_api.get("scientificNameID")
        df["nameAccordingTo"] = datos_api.get("nameAccordingTo")
        df["nameAccordingToID"] = datos_api.get("nameAccordingToID")


        df.to_excel(ruta_excel_peces_OFM, index=False)


    except Exception as e:
        return None


# Función para extraer coordenadas
def extraer_coordenadas(layer, epsg_origen):
    resultados = []
    transformer = pyproj.Transformer.from_crs(epsg_origen, EPSG_DESTINO, always_xy=True)

    for feature in layer:
        geom = feature.GetGeometryRef()
        if geom and geom.GetGeometryType() == ogr.wkbPoint:
            lon, lat = transformer.transform(geom.GetX(), geom.GetY())

            atributos = feature.items()
            atributos["latitud"] = lat
            atributos["longitud"] = lon
            resultados.append(atributos)

    return resultados


# Función para realizar el join
def realizar_join(capa, tabla, enlace_peces_OFM):
    try:


        capa_df = pd.DataFrame(capa)
        tabla_df = pd.DataFrame(tabla)

        if capa_df.empty or tabla_df.empty:

            return None

        resultado_df = pd.merge(capa_df, tabla_df, on=enlace_peces_OFM, how="inner")

        if resultado_df.empty:

            return None

        return resultado_df

    except Exception as e:

        return None


# Función para mapear especie
def mapear_espec_nomb(df):
    if "ESPEC_NOMB" in df.columns:
        df["ESPEC_NOMB"] = df["ESPEC_NOMB"].fillna(0).astype(float).astype(int).astype(str).replace({"0": ""})
    else:

        df["ESPEC_NOMB"] = None

    mapa_espec_nomb = {
        "5101": "Ablenneshians",
        "5102": "Abomaetheostoma",
        "5103": "Abramiteseques",
        "5104": "Abudefdufconcolor",
        "5105": "Abudefdufsaxatilis",
        "5106": "Abudefduftaurus",
        "5107": "Abudefduftroschelii",
        "5108": "Acanthemblemariaaspera",
        "5109": "Acanthemblemariabetinensis",
        "5110": "Acanthemblemariagreenfieldi",
        "5111": "Acanthemblemariahancocki",
        "5112": "Acanthemblemariamaria",
        "5113": "Acanthemblemariarivasi",
        "5114": "Acanthemblemariaspinosa",
        "5115": "Acanthocybiumsolandri",
        "5116": "Acanthostracionpolygonia",
        "5117": "Acanthostracionquadricornis",
        "5118": "Acanthurusbahianus",
        "5119": "Acanthuruschirurgus",
        "5120": "Acanthuruscoeruleus",
        "5121": "Acanthurusnigricans",
        "5122": "Acanthurustriostegus",
        "5123": "Acanthurusxanthopterus",
        "5124": "Achirusachirus",
        "5125": "Achirusklunzingeri",
        "5126": "Achiruslineatus",
        "5127": "Achirusmazatlanus",
        "5128": "Achirusnovoae",
        "5129": "Achirusscutum",
        "5130": "Acyrtopsberyllinus",
        "5131": "Acyrtusartius",
        "5132": "Acyrtusrubiginosus",
        "5133": "Aequidenspulcher",
        "5134": "Ageneiosuspardalis",
        "5135": "Agonostomusmonticola",
        "5136": "Ahliaegmontis",
        "5137": "Akkodionaea",
        "5138": "Albulanemoptera",
        "5139": "Albulavulpes",
        "5140": "Alectisciliaris",
        "5141": "Alphestesafer",
        "5142": "Alphestesimmaculatus",
        "5143": "Aluterusheudelotii",
        "5144": "Aluterusmonoceros",
        "5145": "Aluterusschoepfi",
        "5146": "Aluterusscriptus",
        "5147": "Amblycirrhituspinos",
        "5148": "Amphelikturusdendriticus",
        "5149": "Amphichthyscryptocentrus",
        "5150": "Anarchiasgalapagensis",
        "5151": "Anchoacayorum",
        "5152": "Anchoacolonensis",
        "5153": "Anchoacubana",
        "5154": "Anchoafilifera",
        "5155": "Anchoahepsetus",
        "5156": "Anchoalamprotaenia",
        "5157": "Anchoalucida",
        "5158": "Anchoalyolepis",
        "5159": "Anchoapanamensis",
        "5160": "Anchoaparva",
        "5161": "Anchoaspinifer",
        "5162": "Anchoatrinitatis",
        "5163": "Anchoawalkeri",
        "5164": "Anchoviaclupeoides",
        "5165": "Anchoviellaelongata",
        "5166": "Anchoviellalepidentostole",
        "5167": "Anchoviellaperfasciata",
        "5168": "Ancylopsettacycloidea",
        "5169": "Ancylopsettadilecta",
        "5170": "Ancylopsettakumperae",
        "5171": "Ancylopsettaommata",
        "5172": "Anguillaclepsydra",
        "5173": "Anguillalentiginosa",
        "5174": "Anguillamiliaris",
        "5175": "Anguillarobusta",
        "5176": "Anguillarostrata",
        "5177": "Anisotremuscaesius",
        "5178": "Anisotremusinterruptus",
        "5179": "Anisotremusmoricandi",
        "5180": "Anisotremussurinamensis",
        "5181": "Anisotremustaeniatus",
        "5182": "Anisotremusvirginicus",
        "5183": "Antennariusavalonis",
        "5184": "Antennariusbermudensis",
        "5185": "Antennariuscommerson",
        "5186": "Antennariusmultiocellatus",
        "5187": "Antennariusocellatus",
        "5188": "Antennariuspauciradiatus",
        "5189": "Antennariusradiosus",
        "5190": "Antennariussanguineus",
        "5191": "Antennariusstriatus",
        "5192": "Antennatusstrigatus",
        "5193": "Anthiastenuis",
        "5194": "Antigoniacapros",
        "5195": "Antigoniacombatia",
        "5196": "Aplatophischauliodus",
        "5197": "Apogonaffinis",
        "5198": "Apogonatradorsatus",
        "5199": "Apogonaurolineatus",
        "5200": "Apogonbinotatus",
        "5201": "Apogondovii",
        "5202": "Apogonlachneri",
        "5203": "Apogonleptocaulus",
        "5204": "Apogonmaculatus",
        "5205": "Apogonpacificus",
        "5206": "Apogonphenax",
        "5207": "Apogonpillionatus",
        "5208": "Apogonplanifrons",
        "5209": "Apogonpseudomaculatus",
        "5210": "Apogonquadrisquamatus",
        "5211": "Apogonrobinsi",
        "5212": "Apogontownsendi",
        "5213": "Apsilusdentatus",
        "5214": "Archosargusprobatocephalus",
        "5215": "Archosargusrhomboidalis",
        "5216": "Arcosmacrophthalmus",
        "5217": "Argentinabrucei",
        "5218": "Argentinastriata",
        "5219": "Argyripnusatlanticus",
        "5220": "Argyropelecusaculeatus",
        "5221": "Argyropelecuslychnus",
        "5222": "Ariommabondi",
        "5223": "Ariommaregulus",
        "5224": "Ariopsisbonillai",
        "5225": "Ariopsisseemanni",
        "5226": "Ariosomaanale",
        "5227": "Ariosomabalearicum",
        "5228": "Ariosomagilberti",
        "5229": "Ariosomaprorigerum",
        "5230": "Arothronhispidus",
        "5231": "Arothronmeleagris",
        "5232": "Astrapogonalutus",
        "5233": "Astrapogonpuncticulatus",
        "5234": "Astrapogonstellatus",
        "5235": "Astroscopusy-graecum",
        "5236": "Astyanaxcaucanus",
        "5237": "Astyanaxfasciatus",
        "5238": "Atherinellabeani",
        "5239": "Atherinellablackburni",
        "5240": "Atherinellabrasiliensis",
        "5241": "Atherinomorusstipes",
        "5242": "Atractodenchelysphrix",
        "5243": "Aulostomuschinensis",
        "5244": "Aulostomusmaculatus",
        "5245": "Auxisrochei",
        "5246": "Auxisthazard",
        "5247": "Awaoustajasica",
        "5248": "Axoclinusrubinoffi",
        "5249": "Bagrebagre",
        "5250": "Bagremarinus",
        "5251": "Bagrepanamensis",
        "5252": "Bagrepinnimaculatus",
        "5253": "Bairdiellaensifera",
        "5254": "Bairdiellaronchus",
        "5255": "Bairdiellasanctaeluciae",
        "5256": "Balistescapriscus",
        "5257": "Balistespolylepis",
        "5258": "Balistesvetula",
        "5259": "Barbuliferceuthoecus",
        "5260": "Bathyanthiascubensis",
        "5261": "Bathyanthiasmexicana",
        "5262": "Bathyclupeaargentea",
        "5263": "Bathycongrusbullisi",
        "5264": "Bathycongrusdubius",
        "5265": "Bathycongrusmacrurus",
        "5266": "Bathycongrusvaridens",
        "5267": "Bathycongrusvicinalis",
        "5268": "Bathygadusmacrops",
        "5269": "Bathygobiusandrei",
        "5270": "Bathygobiuscuracao",
        "5271": "Bathygobiusmystacium",
        "5272": "Bathygobiusramosus",
        "5273": "Bathygobiussoporator",
        "5274": "Bathypteroisbigelowi",
        "5275": "Bathyurocongervicinus",
        "5276": "Batrachoidesboulengeri",
        "5277": "Batrachoidesmanglae",
        "5278": "Batrachoidespacifici",
        "5279": "Batrachoidessurinamensis",
        "5280": "Bellatorbrachychir",
        "5281": "Bellatoregretta",
        "5282": "Bellatorgymnostethus",
        "5283": "Bellatorribeiroi",
        "5284": "Bellatorxenisma",
        "5285": "Bembropsanatirostris",
        "5286": "Bembropsgobioides",
        "5287": "Bembropsocellatus",
        "5288": "Bembropssp.1",
        "5289": "Bembropssp.2",
        "5290": "Benthodesmussimonyi",
        "5291": "Benthodesmustenuis",
        "5292": "Bodianusdiplotaenia",
        "5293": "Bodianuspulchellus",
        "5294": "Bodianusrufus",
        "5295": "Bollmanniaboqueronensis",
        "5296": "Bollmanniachlamydes",
        "5297": "Bollmanniaeigenmanni",
        "5298": "Bollmanniagomezi",
        "5299": "Bollmannialitura",
        "5300": "Bothuslunatus",
        "5301": "Bothusmaculiferus",
        "5302": "Bothusocellatus",
        "5303": "Bramacaudatus",
        "5304": "Bregmacerosatlanticus",
        "5305": "Bregmacerosbathymaster",
        "5306": "Bregmaceroscantori",
        "5307": "Bregmaceroshoudei",
        "5308": "Brotulabarbata",
        "5309": "Brotulaclarkae",
        "5310": "Brotulaordwayi",
        "5311": "Bryconsinuensis",
        "5312": "Bryxdunckeri",
        "5313": "Bryxrandalli",
        "5314": "Bufoceratiaswedli",
        "5315": "Calamopteryxgoslinei",
        "5316": "Calamopteryxrobinsorum",
        "5317": "Calamusbajonado",
        "5318": "Calamuscalamus",
        "5319": "Calamuscervigoni",
        "5320": "Calamuspenna",
        "5321": "Calamuspennatula",
        "5322": "Callechelysbilinearis",
        "5323": "Cantherhinesdumerilii",
        "5324": "Cantherhinesmacrocerus",
        "5325": "Cantherhinespullus",
        "5326": "Canthidermismaculata",
        "5327": "Canthidermissufflamen",
        "5328": "Canthigasterjanthinoptera",
        "5329": "Canthigasterpunctatissima",
        "5330": "Canthigasterrostrata",
        "5331": "Caquetaiakraussii",
        "5332": "Caranxbartholomaei",
        "5333": "Caranxcaballus",
        "5334": "Caranxcaninus",
        "5335": "Caranxcrysos",
        "5336": "Caranxhippos",
        "5337": "Caranxlatus",
        "5338": "Caranxlugubris",
        "5339": "Caranxmelampygus",
        "5340": "Caranxotrynter",
        "5341": "Caranxruber",
        "5342": "Caranxsexfasciatus",
        "5343": "Carapusbermudensis",
        "5344": "Catesbyapseudomuraena",
        "5345": "Cathoropsmapale",
        "5346": "Cathoropsmultiradiatus",
        "5347": "Cathoropsraredonae",
        "5348": "Cathoropsspixii",
        "5349": "Cathoropssteindachneri",
        "5350": "Caulolatilusaffinis",
        "5351": "Caulolatiluscyanops",
        "5352": "Caulolatilusguppyi",
        "5353": "Caulolatilusintermedius",
        "5354": "Caulolatiluswilliamsi",
        "5355": "Centropomusensiferus",
        "5356": "Centropomusmexicanus",
        "5357": "Centropomusparallelus",
        "5358": "Centropomuspectinatus",
        "5359": "Centropomusundecimalis",
        "5360": "Centropygeargi",
        "5361": "Cephalopholiscruentata",
        "5362": "Cephalopholisfulva",
        "5363": "Cephalopholispanamensis",
        "5364": "Cerdalefloridana",
        "5365": "Cetengraulisedentulus",
        "5366": "Cetengraulismysticetus",
        "5367": "Chaenopsislimbaughi",
        "5368": "Chaenopsisocellata",
        "5369": "Chaenopsisresh",
        "5370": "Chaetodipterusfaber",
        "5371": "Chaetodipteruszonatus",
        "5372": "Chaetodonaculeatus",
        "5373": "Chaetodoncapistratus",
        "5374": "Chaetodonhumeralis",
        "5375": "Chaetodonocellatus",
        "5376": "Chaetodonsedentarius",
        "5377": "Chaetodonstriatus",
        "5378": "Channomuraenavittata",
        "5379": "Chanoschanos",
        "5380": "Chauliodusbarbatus",
        "5381": "Chauliodussloani",
        "5382": "Chaunaxpictus",
        "5383": "Chaunaxsuttkusi",
        "5384": "Cheilopogoncyanopterus",
        "5385": "Cheilopogonfurcatus",
        "5386": "Cheilopogonheterurus",
        "5387": "Cherublemmaemmelas",
        "5388": "Chilomycterusantennatus",
        "5389": "Chilomycterusantillarum",
        "5390": "Chilomycterusatringa",
        "5391": "Chilomycterusreticulatus",
        "5392": "Chilorhinussuensonii",
        "5393": "Chirocentrodonbleekerianus",
        "5394": "Chlopsisdentatus",
        "5395": "Chlorophthalmusagassizi",
        "5396": "Chlorophthalmusmento",
        "5397": "Chloroscombruschrysurus",
        "5398": "Chloroscombrusorqueta",
        "5399": "Chromisatrilobata",
        "5400": "Chromiscyanea",
        "5401": "Chromisenchrysura",
        "5402": "Chromisinsolata",
        "5403": "Chromismultilineata",
        "5404": "Chromisscotti",
        "5405": "Cirrhitichthysoxycephalus",
        "5406": "Cirrhitusrivulatus",
        "5407": "Citharichthysarenaceus",
        "5408": "Citharichthyscornutus",
        "5409": "Citharichthysdinoceros",
        "5410": "Citharichthysgilberti",
        "5411": "Citharichthysgymnorhinus",
        "5412": "Citharichthysmacrops",
        "5413": "Citharichthysminutus",
        "5414": "Citharichthysplatophrys",
        "5415": "Citharichthysspilopterus",
        "5416": "Citharichthysstigmaeus",
        "5417": "Citharichthysvaldezi",
        "5418": "Clepticusparrae",
        "5419": "Coelorinchuscaelorhincus",
        "5420": "Coelorinchuscanus",
        "5421": "Coelorinchuscaribbaeus",
        "5422": "Colocongermeadi",
        "5423": "Colomesuspsittacus",
        "5424": "Congeresculentus",
        "5425": "Congertriporiceps",
        "5426": "Conodonnobilis",
        "5427": "Conodonserrifer",
        "5428": "Cookeolusjaponicus",
        "5429": "Coralliozetuscardonae",
        "5430": "Corydorasesperanze",
        "5431": "Corydorasseptentrionalis",
        "5432": "Coryphaenaequiselis",
        "5433": "Coryphaenahippurus",
        "5434": "Coryphaenoidescapito",
        "5435": "Coryphaenoideszaniophorus",
        "5436": "Coryphopterusalloides",
        "5437": "Coryphopterusdicrus",
        "5438": "Coryphopteruseidolon",
        "5439": "Coryphopterusglaucofraenum",
        "5440": "Coryphopterushyalinus",
        "5441": "Coryphopteruslipernes",
        "5442": "Coryphopteruspersonatus",
        "5443": "Coryphopterusthrix",
        "5444": "Coryphopterustortugae",
        "5445": "Coryphopterusurospilus",
        "5446": "Cosmocampusalbirostris",
        "5447": "Cosmocampusbrachycephalus",
        "5448": "Cosmocampuselucens",
        "5449": "Crossoloricariavariegata",
        "5450": "Cryptotomusroseus",
        "5451": "Ctenochaetusmarginatus",
        "5452": "Ctenogobiusboleosoma",
        "5453": "Ctenogobiussaepapallens",
        "5454": "Ctenogobiusstigmaticus",
        "5455": "Ctenogobiusstigmaturus",
        "5456": "Ctenoluciushujeta",
        "5457": "Ctenosciaenagracilicirrhus",
        "5458": "Ctenosciaenaperuviana",
        "5459": "Cubicepsgracilis",
        "5460": "Curimatamivartii",
        "5461": "Cyclopsettachittendeni",
        "5462": "Cyclopsettafimbriata",
        "5463": "Cyclopsettapanamensis",
        "5464": "Cyclopsettaquerna",
        "5465": "Cynoponticusconiceps",
        "5466": "Cynoponticussavanna",
        "5467": "Cynopotamusmagdalenae",
        "5468": "Cynoscionacoupa",
        "5469": "Cynoscionjamaicensis",
        "5470": "Cynoscionleiarchus",
        "5471": "Cynoscionmicrolepidotus",
        "5472": "Cynoscionnannus",
        "5473": "Cynoscionphoxocephalus",
        "5474": "Cynoscionreticulatus",
        "5475": "Cynoscionsimilis",
        "5476": "Cynoscionvirescens",
        "5477": "Cyphocharaxmagdalenae",
        "5478": "Cyprinodondearborni",
        "5479": "Cyprinodonvariegatus",
        "5480": "Cypseluruscallopterus",
        "5481": "Cypselurusmelanurus",
        "5482": "Cyttopsisrosea",
        "5483": "Dactylopterusvolitans",
        "5484": "Dactyloscopuscrossotus",
        "5485": "Dactyloscopuspoeyi",
        "5486": "Dactyloscopustridigitatus",
        "5487": "Daectordowi",
        "5488": "Decapterusmacarellus",
        "5489": "Decapterusmacrosoma",
        "5490": "Decapteruspunctatus",
        "5491": "Decapterustabl",
        "5492": "Decodonmelasma",
        "5493": "Decodonpuellaris",
        "5494": "Dermatolepisdermatolepis",
        "5495": "Dermatolepisinermis",
        "5496": "Diaphusdumerilii",
        "5497": "Diaphusgarmani",
        "5498": "Diaphuslucidus",
        "5499": "Diaphusperspicillatus",
        "5500": "Diapterusauratus",
        "5501": "Diapterusaureolus",
        "5502": "Diapterusperuvianus",
        "5503": "Diapterusrhombeus",
        "5504": "Dibranchusatlanticus",
        "5505": "Dibranchushystrix",
        "5506": "Diodonholocanthus",
        "5507": "Diodonhystrix",
        "5508": "Diplacanthopomabrachysoma",
        "5509": "Diplectrumbivittatum",
        "5510": "Diplectrumeumelum",
        "5511": "Diplectrumeuryplectrum",
        "5512": "Diplectrumformosum",
        "5513": "Diplectrummacropoma",
        "5514": "Diplectrummaximum",
        "5515": "Diplectrumpacificum",
        "5516": "Diplectrumradiale",
        "5517": "Diplodusargenteus",
        "5518": "Diplogrammuspauciradiatus",
        "5519": "Diplospinusmultistriatus",
        "5520": "Doratonotusmegalepis",
        "5521": "Dormitatorlatifrons",
        "5522": "Dormitatormaculatus",
        "5523": "Doryrhamphusexcisus",
        "5524": "Dysommaanguillare",
        "5525": "Dysomminarugosa",
        "5526": "Echeneisnaucrates",
        "5527": "Echeneisneucratoides",
        "5528": "Echidnacatenata",
        "5529": "Echidnanebulosa",
        "5530": "Echiophispunctifer",
        "5531": "Ekemblemarianigra",
        "5532": "Elacatinusdigueti",
        "5533": "Elacatinusdilepsis",
        "5534": "Elacatinusevelynae",
        "5535": "Elacatinushorsti",
        "5536": "Elacatinusillecebrosum",
        "5537": "Elacatinusillecebrosus",
        "5538": "Elacatinuslouisae",
        "5539": "Elacatinusmacrodon",
        "5540": "Elacatinusmultifasciatus",
        "5541": "Elacatinuspallens",
        "5542": "Elacatinuspuncticulatus",
        "5543": "Elacatinussaucrus",
        "5544": "Elacatinustenox",
        "5545": "Elacatinusxanthiprora",
        "5546": "Elagatisbipinnulata",
        "5547": "Eleotrisamblyopsis",
        "5548": "Eleotrispicta",
        "5549": "Eleotrispisonis",
        "5550": "Eleotrissmaragdus",
        "5551": "Elopsaffinis",
        "5552": "Elopssaurus",
        "5553": "Emblemariabiocellata",
        "5554": "Emblemariacaldwelli",
        "5555": "Emblemariacaycedoi",
        "5556": "Emblemariadiphyodontis",
        "5557": "Emblemariapandionis",
        "5558": "Emblemariopsisbahamensis",
        "5559": "Emblemariopsisleptocirris",
        "5560": "Emblemariopsisoccidentalis",
        "5561": "Emblemariopsisrandalli",
        "5562": "Emblemariopsissignifera",
        "5563": "Emblemariopsistayrona",
        "5564": "Emmelichthyopsatlanticus",
        "5565": "Enchelycorecarychroa",
        "5566": "Enchelycorenigricans",
        "5567": "Engrauliseurystole",
        "5568": "Engyophryssanctilaurentii",
        "5569": "Engyophryssenta",
        "5570": "Enneanectesaltivelis",
        "5571": "Enneanectesatrorus",
        "5572": "Enneanectesboehlkei",
        "5573": "Enneanectespectoralis",
        "5574": "Entomacrodusnigricans",
        "5575": "Entomacrodusteevani",
        "5576": "Epigonusoccidentalis",
        "5577": "Epigonuspandionis",
        "5578": "Epinephelusadscensionis",
        "5579": "Epinephelusanalogus",
        "5580": "Epinephelusfasciatus",
        "5581": "Epinephelusguttatus",
        "5582": "Epinephelusitajara",
        "5583": "Epinepheluslabriformis",
        "5584": "Epinephelusmorio",
        "5585": "Epinephelusmystacinus",
        "5586": "Epinephelusstriatus",
        "5587": "Equetuslanceolatus",
        "5588": "Equetuspunctatus",
        "5589": "Erythroclesmonodi",
        "5590": "Etelisoculatus",
        "5591": "Etropuscrossotus",
        "5592": "Etrumeusteres",
        "5593": "Eucinostomusargenteus",
        "5594": "Eucinostomusgracilis",
        "5595": "Eucinostomusgula",
        "5596": "Eucinostomusharengulus",
        "5597": "Eucinostomushavana",
        "5598": "Eucinostomusjonesii",
        "5599": "Eucinostomusmelanopterus",
        "5600": "Eugerresbrasilianus",
        "5601": "Eugerresplumieri",
        "5602": "Euleptorhamphusvelox",
        "5603": "Euleptorhamphusviridis",
        "5604": "Eumegistusbrevorti",
        "5605": "Euthynnusalletteratus",
        "5606": "Euthynnuslineatus",
        "5607": "Evermannichthysmetzelaari",
        "5608": "Evorthoduslyricus",
        "5609": "Evoxymetopontaeniatus",
        "5610": "Exocoetusmonocirrhus",
        "5611": "Exocoetusobtusirostris",
        "5612": "Exocoetusvolitans",
        "5613": "Facciolellasp.",
        "5614": "Fistulariacommersonii",
        "5615": "Fistulariacorneta",
        "5616": "Fistulariapetimba",
        "5617": "Fistulariatabacaria",
        "5618": "Forcipigerflavissimus",
        "5619": "Gadellaimberbis",
        "5620": "Gambusiaaffinis",
        "5621": "Gambusialemaitrei",
        "5622": "Gambusiameadi",
        "5623": "Gambusianicaraguensis",
        "5624": "Gastropsettafrontalis",
        "5625": "Gempylusserpens",
        "5626": "Geophagussteindachneri",
        "5627": "Gephyroberyxdarwini",
        "5628": "Gerrescinereus",
        "5629": "Gilbertolusalatus",
        "5630": "Gillellusuranidea",
        "5631": "Ginsburgellusnovemlineatus",
        "5632": "Gnathanodonspeciosus",
        "5633": "Gnatholepisthompsoni",
        "5634": "Gnathophismystax",
        "5635": "Gobiesoxadustus",
        "5636": "Gobiesoxnudus",
        "5637": "Gobiesoxpunctulatus",
        "5638": "Gobiesoxstrumosus",
        "5639": "Gobioidesbroussonnetii",
        "5640": "Gobiomorusdormitor",
        "5641": "Gobionellushastatus",
        "5642": "Gobionellusoceanicus",
        "5643": "Gobiosomapallens",
        "5644": "Gobiosomaspes",
        "5645": "Gobiosomaspilotum",
        "5646": "Gobulushancocki",
        "5647": "Gobulusmyersi",
        "5648": "Gonioplectrushispanus",
        "5649": "Gonostomaatlanticum",
        "5650": "Grammaloreto",
        "5651": "Grammamelacara",
        "5652": "Guavinaguavina",
        "5653": "Gunterichthyslongipenis",
        "5654": "Gymnachirusmelas",
        "5655": "Gymnachirusnudus",
        "5656": "Gymnomuraenazebra",
        "5657": "Gymnothoraxcastaneus",
        "5658": "Gymnothoraxconspersus",
        "5659": "Gymnothoraxdovii",
        "5660": "Gymnothoraxelegans",
        "5661": "Gymnothoraxequatorialis",
        "5662": "Gymnothoraxfunebris",
        "5663": "Gymnothoraxhubbsi",
        "5664": "Gymnothoraxmiliaris",
        "5665": "Gymnothoraxmoringa",
        "5666": "Gymnothoraxocellatus",
        "5667": "Gymnothoraxpanamensis",
        "5668": "Gymnothoraxphalarus",
        "5669": "Gymnothoraxpolygonius",
        "5670": "Gymnothoraxundulatus",
        "5671": "Gymnothoraxvicinus",
        "5672": "Haemulonalbum",
        "5673": "Haemulonaurolineatum",
        "5674": "Haemulonbonariense",
        "5675": "Haemulonboschmae",
        "5676": "Haemuloncarbonarium",
        "5677": "Haemulonchrysargyreum",
        "5678": "Haemulonflaviguttatum",
        "5679": "Haemulonflavolineatum",
        "5680": "Haemulonmacrostomum",
        "5681": "Haemulonmaculicauda",
        "5682": "Haemulonmelanurum",
        "5683": "Haemulonparra",
        "5684": "Haemulonplumierii",
        "5685": "Haemulonsciurus",
        "5686": "Haemulonscudderii",
        "5687": "Haemulonsexfasciatum",
        "5688": "Haemulonsteindachneri",
        "5689": "Haemulonstriatum",
        "5690": "Haemulopsiscorvinaeformis",
        "5691": "Haemulopsisleuciscus",
        "5692": "Halicampuscrinitus",
        "5693": "Halichoeresbathyphilus",
        "5694": "Halichoeresbivittatus",
        "5695": "Halichoerescaudalis",
        "5696": "Halichoereschierchiae",
        "5697": "Halichoerescyanocephalus",
        "5698": "Halichoeresdiscolor",
        "5699": "Halichoeresdispilus",
        "5700": "Halichoeresgarnoti",
        "5701": "Halichoeresmaculipinna",
        "5702": "Halichoeresmalpelo",
        "5703": "Halichoeresmelanotis",
        "5704": "Halichoeresnicholsi",
        "5705": "Halichoeresnotospilus",
        "5706": "Halichoerespictus",
        "5707": "Halichoerespoeyi",
        "5708": "Halichoeresradiatus",
        "5709": "Halieutichthysaculeatus",
        "5710": "Halosaurusguentheri",
        "5711": "Halosaurusovenii",
        "5712": "Halosaurusradiatus",
        "5713": "Harengulaclupeola",
        "5714": "Harengulahumeralis",
        "5715": "Harengulajaguana",
        "5716": "Harengulathrissina",
        "5717": "Helicolenusdactylopterus",
        "5718": "Hemanthiasaureorubens",
        "5719": "Hemanthiasperuanus",
        "5720": "Hemanthiassignifer",
        "5721": "Hemanthiasvivanus",
        "5722": "Hemiancistruswilsoni",
        "5723": "Hemibryconjabonero",
        "5724": "Hemibrycontolimae",
        "5725": "Hemicaranxamblyrhynchus",
        "5726": "Hemicaranxzelotes",
        "5727": "Hemiemblemariasimulus",
        "5728": "Hemiramphusbalao",
        "5729": "Hemiramphusbrasiliensis",
        "5730": "Hemiramphussaltator",
        "5731": "Heterocongerdigueti",
        "5732": "Heterocongerlongissimus",
        "5733": "Heteropriacanthuscruentatus",
        "5734": "Hippocampuserectus",
        "5735": "Hippocampusingens",
        "5736": "Hippocampusreidi",
        "5737": "Hippoglossinatetrophthalma",
        "5738": "Hirundichthysaffinis",
        "5739": "Hirundichthysrondeletii",
        "5740": "Hirundichthysspeculiger",
        "5741": "Histriohistrio",
        "5742": "Holacanthusciliaris",
        "5743": "Holacanthuspasser",
        "5744": "Holacanthustricolor",
        "5745": "Holanthiasmartinicensis",
        "5746": "Hollardiasp.",
        "5747": "Holocentrusadscensionis",
        "5748": "Holocentrusrufus",
        "5749": "Hopliasmalabaricus",
        "5750": "Hoplopagrusguentherii",
        "5751": "Hoplostethusmediterraneus",
        "5752": "Hoplostethusoccidentalis",
        "5753": "Hoplunnismacrura",
        "5754": "Hoplunnismegista",
        "5755": "Hoplunnispunctata",
        "5756": "Hoplunnistenuis",
        "5757": "Hymenocephalusgracilis",
        "5758": "Hymenocephalusitalicus",
        "5759": "Hypleurochilusaequipinnis",
        "5760": "Hypleurochilusspringeri",
        "5761": "Hypoatherinaharringtonensis",
        "5762": "Hypoatherinaharringtonensis",
        "5763": "Hypoplectrusaberrans",
        "5764": "Hypoplectruschlorurus",
        "5765": "Hypoplectrusgummigutta",
        "5766": "Hypoplectrusguttavarius",
        "5767": "Hypoplectrusindigo",
        "5768": "Hypoplectrusnigricans",
        "5769": "Hypoplectrusprovidencianus",
        "5770": "Hypoplectruspuella",
        "5771": "Hypoplectrusunicolor",
        "5772": "Hyporhamphusgilli",
        "5773": "Hyporhamphusroberti",
        "5774": "Hyporhamphusunifasciatus",
        "5775": "Hyporthodusacanthistius",
        "5776": "Hypsoblenniusbrevipinnis",
        "5777": "Hypsoblenniusinvemar",
        "5778": "Ichthyapusophioneus",
        "5779": "Ijimaiaantillarum",
        "5780": "Ilishafuerthii",
        "5781": "Inermiavittata",
        "5782": "Isopisthusparvipinnis",
        "5783": "Isopisthusremifer",
        "5784": "Istiophorusplatypterus",
        "5785": "Japonocongercaribbeus",
        "5786": "Jenkinsialamprotaenia",
        "5787": "Johnrandallianigrirostris",
        "5788": "Joturuspichardi",
        "5789": "Kathetostomacubana",
        "5790": "Katsuwonuspelamis",
        "5791": "Kaupichthyshyoproroides",
        "5792": "Kaupichthysnuchalis",
        "5793": "Kuhliamugil",
        "5794": "Kyphosusanalogus",
        "5795": "Kyphosuselegans",
        "5796": "Kyphosusincisor",
        "5797": "Kyphosussectator",
        "5798": "Labichthyscarinatus",
        "5799": "Labrisomusalbigenys",
        "5800": "Labrisomusbucciferus",
        "5801": "Labrisomusdendriticus",
        "5802": "Labrisomusgobio",
        "5803": "Labrisomusguppyi",
        "5804": "Labrisomushaitiensis",
        "5805": "Labrisomuskalisherae",
        "5806": "Labrisomusnigricinctus",
        "5807": "Labrisomusnuchipinnis",
        "5808": "Lachnolaimusmaximus",
        "5809": "Lactophrystrigonus",
        "5810": "Laemonemabarbatula",
        "5811": "Laemonemagoodebeanorum",
        "5812": "Lagocephaluslaevigatus",
        "5813": "Lagocephaluslagocephalus",
        "5814": "Lampadenaluminosa",
        "5815": "Lamprisguttatus",
        "5816": "Larimusacclivis",
        "5817": "Larimusargenteus",
        "5818": "Larimusbreviceps",
        "5819": "Larimuseffulgens",
        "5820": "Lepidonectesbimaculata",
        "5821": "Lepophidiumaporrhox",
        "5822": "Lepophidiumbrevibarbe",
        "5823": "Lepophidiumnegropinna",
        "5824": "Lepophidiumpheromystax",
        "5825": "Lepophidiumprofundorum",
        "5826": "Lepophidiumprorates",
        "5827": "Lepophidiumsp.",
        "5828": "Leporinusmuyscorum",
        "5829": "Lestrolepisintermedia",
        "5830": "Liopropomacarmabi",
        "5831": "Liopropomamultilineatum",
        "5832": "Liopropomarubre",
        "5833": "Lipogrammaevides",
        "5834": "Lipogrammatrilineata",
        "5835": "Lobotespacificus",
        "5836": "Lobotessurinamensis",
        "5837": "Lonchopisthushigmani",
        "5838": "Lonchopisthuslemur",
        "5839": "Lonchopisthusmicrognathus",
        "5840": "Lonchopisthussinuscalifornicus",
        "5841": "Lonchopisthusvanderbilti",
        "5842": "Lonchuruslanceolatus",
        "5843": "Lophiodescaulinaris",
        "5844": "Lophiodesmonodi",
        "5845": "Lophiodesreticulatus",
        "5846": "Lophiusamericanus",
        "5847": "Lophiusgastrophysus",
        "5848": "Lophogobiuscyprinoides",
        "5849": "Lucayablenniuszingaro",
        "5850": "Lupinoblenniusvinctus",
        "5851": "Lutjanusanalis",
        "5852": "Lutjanusapodus",
        "5853": "Lutjanusaratus",
        "5854": "Lutjanusargentiventris",
        "5855": "Lutjanusbuccanella",
        "5856": "Lutjanuscolorado",
        "5857": "Lutjanuscyanopterus",
        "5858": "Lutjanusgriseus",
        "5859": "Lutjanusguttatus",
        "5860": "Lutjanusinermis",
        "5861": "Lutjanusjocu",
        "5862": "Lutjanusjordani",
        "5863": "Lutjanusmahogoni",
        "5864": "Lutjanusnovemfasciatus",
        "5865": "Lutjanusperu",
        "5866": "Lutjanuspurpureus",
        "5867": "Lutjanussynagris",
        "5868": "Lutjanusviridis",
        "5869": "Lutjanusvivanus",
        "5870": "Lycengraulisgrossidens",
        "5871": "Lythrypnuscrocodilus",
        "5872": "Lythrypnusdalli",
        "5873": "Lythrypnuselasson",
        "5874": "Lythrypnusgilberti",
        "5875": "Lythrypnusheterochroma",
        "5876": "Lythrypnusminimus",
        "5877": "Lythrypnusmowbrayi",
        "5878": "Lythrypnusnesiotes",
        "5879": "Lythrypnusokapia",
        "5880": "Lythrypnuspulchellus",
        "5881": "Lythrypnusrhizophora",
        "5882": "Lythrypnussolanensis",
        "5883": "Lythrypnusspilus",
        "5884": "Lythrypnuszebra",
        "5885": "Macrodonancylodon",
        "5886": "Macrodonmordax",
        "5887": "Makairanigricans",
        "5888": "Malacanthusbrevirostris",
        "5889": "Malacanthusplumieri",
        "5890": "Malacocephalusoccidentalis",
        "5891": "Malacocephalusokamurai",
        "5892": "Malacoctenusaurolineatus",
        "5893": "Malacoctenusboehlkei",
        "5894": "Malacoctenusdelalandii",
        "5895": "Malacoctenuserdmani",
        "5896": "Malacoctenusgilli",
        "5897": "Malacoctenusmacropus",
        "5898": "Malacoctenustriangulatus",
        "5899": "Malacoctenusversicolor",
        "5900": "Malthopsisgnoma",
        "5901": "Maurolicusmuelleri",
        "5902": "Megalechisthoracatum",
        "5903": "Megalopsatlanticus",
        "5904": "Melichthysniger",
        "5905": "Melichthysvidua",
        "5906": "Membrasanalis",
        "5907": "Membrasargentea",
        "5908": "Menticirrhusamericanus",
        "5909": "Menticirrhuselongatus",
        "5910": "Menticirrhuslittoralis",
        "5911": "Menticirrhuspanamensis",
        "5912": "Merlucciusalbidus",
        "5913": "Merlucciusangustimanus",
        "5914": "Microdesmuscarri",
        "5915": "Microgobiuscarri",
        "5916": "Microgobiusmeeki",
        "5917": "Microgobiussignatus",
        "5918": "Microphisbrachyurus",
        "5919": "Micropogoniasfurnieri",
        "5920": "Micropogoniasundulatus",
        "5921": "Microspathodonbairdii",
        "5922": "Microspathodonchrysurus",
        "5923": "Microspathodondorsalis",
        "5924": "Molamola",
        "5925": "Monacanthusciliatus",
        "5926": "Monacanthushispidus",
        "5927": "Monacanthustuckeri",
        "5928": "Monoleneasaedai",
        "5929": "Monolenemaculipinna",
        "5930": "Monolenemegalepis",
        "5931": "Monolenesessilicauda",
        "5932": "Monopenchelysacuta",
        "5933": "Moringuaedwardsi",
        "5934": "Mugilcephalus",
        "5935": "Mugilcurema",
        "5936": "Mugilhospes",
        "5937": "Mugilincilis",
        "5938": "Mugilliza",
        "5939": "Mugiltrichodon",
        "5940": "Mulloidichthysdentatus",
        "5941": "Mulloidichthysmartinicus",
        "5942": "Mulloidichthysvanicolensis",
        "5943": "Mullusauratus",
        "5944": "Mycteropercaacutirostris",
        "5945": "Mycteropercabonaci",
        "5946": "Mycteropercacidi",
        "5947": "Mycteropercainterstitialis",
        "5948": "Mycteropercaolfax",
        "5949": "Mycteropercaphenax",
        "5950": "Mycteropercatigris",
        "5951": "Mycteropercavenenosa",
        "5952": "Mycteropercaxenarcha",
        "5953": "Myctophumaffine",
        "5954": "Myctophumasperum",
        "5955": "Myctophumnitidulum",
        "5956": "Myctophumobtusirostre",
        "5957": "Myctophumselenops",
        "5958": "Myrichthysbreviceps",
        "5959": "Myrichthysocellatus",
        "5960": "Myripristisberndti",
        "5961": "Myripristisjacobus",
        "5962": "Myripristisleiognathus",
        "5963": "Myrophisanterodorsalis",
        "5964": "Myrophisplatyrhynchus",
        "5965": "Myrophisplumbeus",
        "5966": "Myrophispunctatus",
        "5967": "Naucratesductor",
        "5968": "Nealotustripes",
        "5969": "Nebrismicrops",
        "5970": "Nematistiuspectoralis",
        "5971": "Nemichthysscolopaceus",
        "5972": "Neobythitesgilli",
        "5973": "Neobythitesmarginatus",
        "5974": "Neobythitesocellatus",
        "5975": "Neocongermucronatus",
        "5976": "Neoepinnulaamericana",
        "5977": "Neoepinnulaorientalis",
        "5978": "Neomerinthebeanorum",
        "5979": "Neoniphonmarianus",
        "5980": "Neoscombropsatlanticus",
        "5981": "Neoscopelusmacrolepidotus",
        "5982": "Neoscopelusmicrochir",
        "5983": "Neslongus",
        "5984": "Nettastomamelanura",
        "5985": "Nezumiaaequalis",
        "5986": "Nicholsinadenticulata",
        "5987": "Nicholsinausta",
        "5988": "Nomeusgronovii",
        "5989": "Notacanthusspinosus",
        "5990": "Notariusarmbrusteri",
        "5991": "Notariusbonillai",
        "5992": "Notariuscookei",
        "5993": "Notariusgrandicassis",
        "5994": "Notariuskessleri",
        "5995": "Notariusneogranatensis",
        "5996": "Notariustroschelii",
        "5997": "Novaculichthystaeniourus",
        "5998": "Ocyuruschrysurus",
        "5999": "Odontognathuscompressus",
        "6000": "Odontognathusmucronatus",
        "6001": "Odontosciondentex",
        "6002": "Odontoscionxanthops",
        "6003": "Ogcocephaluscorniger",
        "6004": "Ogcocephalusnasutus",
        "6005": "Ogcocephalusnotatus",
        "6006": "Ogcocephalusparvus",
        "6007": "Ogcocephaluspumilus",
        "6008": "Ogcocephalusradiatus",
        "6009": "Ogcocephalusrostellum",
        "6010": "Ogcocephalusvespertilio",
        "6011": "Ogilbiacayorum",
        "6012": "Oligoplitesaltus",
        "6013": "Oligoplitespalometa",
        "6014": "Oligoplitesrefulgens",
        "6015": "Oligoplitessaliens",
        "6016": "Oligoplitessaurus",
        "6017": "Omobranchuspunctatus",
        "6018": "Ophichthuscruentifer",
        "6019": "Ophichthuscylindroideus",
        "6020": "Ophichthusgomesii",
        "6021": "Ophichthushyposagmatus",
        "6022": "Ophichthusophis",
        "6023": "Ophichthuspuncticeps",
        "6024": "Ophichthusremiger",
        "6025": "Ophichthusspinicauda",
        "6026": "Ophichthuszophochir",
        "6027": "Ophidionfulvum",
        "6028": "Ophidionholbrookii",
        "6029": "Ophioblenniusmacclurei",
        "6030": "Ophioblenniussteindachneri",
        "6031": "Ophioscionpunctatissimus",
        "6032": "Ophioscionscierus",
        "6033": "Ophioscionstrabo",
        "6034": "Ophiosciontypicus",
        "6035": "Opisthonemalibertate",
        "6036": "Opisthonemaoglinum",
        "6037": "Opisthopterusdovii",
        "6038": "Opisthopterusequatorialis",
        "6039": "Opistognathusaurifrons",
        "6040": "Opistognathusfenmutis",
        "6041": "Opistognathuslonchurus",
        "6042": "Opistognathusmacrognathus",
        "6043": "Opistognathusmaxillosus",
        "6044": "Opistognathuspanamaensis",
        "6045": "Opistognathusscops",
        "6046": "Opistognathuswhitehursti",
        "6047": "Oreochromisniloticus",
        "6048": "Orthopristisruber",
        "6049": "Ostichthystrachypoma",
        "6050": "Ostracionmeleagris",
        "6051": "Otophidiumchickcharney",
        "6052": "Oxycirrhitestypus",
        "6053": "Oxyporhamphusmicropterus",
        "6054": "Oxyurichthysstigmalophius",
        "6055": "Pagruspagrus",
        "6056": "Parabathymyrusoregoni",
        "6057": "Parablenniusmarmoreus",
        "6058": "Paraclinusbarbatus",
        "6059": "Paraclinusfasciatus",
        "6060": "Paraclinusmarmoratus",
        "6061": "Paraclinusnaeorhegmis",
        "6062": "Paraclinusnigripinnis",
        "6063": "Paracongercaudilimbatus",
        "6064": "Paradiplogrammusbairdi",
        "6065": "Parahollardiaschmidti",
        "6066": "Paralichthystropicus",
        "6067": "Paralonchurusbrasiliensis",
        "6068": "Paralonchuruselegans",
        "6069": "Paralonchurusgoodei",
        "6070": "Paralonchuruspetersii",
        "6071": "Paralonchurusrathbuni",
        "6072": "Paranthiascolonus",
        "6073": "Paranthiasfurcifer",
        "6074": "Parapsettuspanamensis",
        "6075": "Parasphyraenopsincisus",
        "6076": "Parasudistruculenta",
        "6077": "parazenpacificus",
        "6078": "Parequesacuminatus",
        "6079": "Parequesiwamotoi",
        "6080": "Parexocoetusbrachypterus",
        "6081": "Parophidionschmidti",
        "6082": "Pellonaharroweri",
        "6083": "Pempherispoeyi",
        "6084": "Pempherisschomburgkii",
        "6085": "Penetopteryxnanus",
        "6086": "Peprilusparu",
        "6087": "Peprilussnyderi",
        "6088": "Peristedionbarbiger",
        "6089": "Peristedionbrevirostre",
        "6090": "Peristedioncrustosum",
        "6091": "Peristedionecuadorense",
        "6092": "Peristediongracile",
        "6093": "Peristediongreyae",
        "6094": "Peristedionlongispatha",
        "6095": "Peristedionminiatum",
        "6096": "Peristedionthompsoni",
        "6097": "Peristediontruncatum",
        "6098": "Petrotyxsanguineus",
        "6099": "Phaeoptyxconklini",
        "6100": "Phaeoptyxpigmentaria",
        "6101": "Phaeoptyxxenus",
        "6102": "Phenacoscorbisnebris",
        "6103": "Phtheirichthyslineatus",
        "6104": "Physiculusfulvus",
        "6105": "Physiculusnematopus",
        "6106": "Physiculusrastrelliger",
        "6107": "Pimelodusblochii",
        "6108": "Pisodonophisdaspilotus",
        "6109": "Plagioscionmagdalenae",
        "6110": "Plagiotremusazaleus",
        "6111": "Platybeloneargalus",
        "6112": "Platygillellusrubrocinctus",
        "6113": "Plectranthiasgarrupellus",
        "6114": "Plectrypopsretrospinis",
        "6115": "Poeciliacaucana",
        "6116": "Poeciliagillii",
        "6117": "Poeciliamexicana",
        "6118": "Poeciliaorri",
        "6119": "Poeciliasphenops",
        "6120": "Poecilopsettainermis",
        "6121": "Pollichthysmauli",
        "6122": "Polydactylusapproximans",
        "6123": "Polydactylusoligodon",
        "6124": "Polydactylusopercularis",
        "6125": "Polydactylusvirginicus",
        "6126": "Polyipnusasteroides",
        "6127": "Polyipnusclarus",
        "6128": "Polymetmecorythaeola",
        "6129": "Polymixialowei",
        "6130": "Polymixianobilis",
        "6131": "Pomacanthusarcuatus",
        "6132": "Pomacanthusparu",
        "6133": "Pomacanthuszonipectus",
        "6134": "Pomadasysbranickii",
        "6135": "Pomadasyscorvinaeformis",
        "6136": "Pomadasyscrocro",
        "6137": "Pomadasyspanamensis",
        "6138": "Pomatomussaltatrix",
        "6139": "Pontinusclemensi",
        "6140": "Pontinusfurcirhinus",
        "6141": "Pontinusinesae",
        "6142": "Pontinuslongispinis",
        "6143": "Pontinusnematophthalmus",
        "6144": "Pontinusrathbuni",
        "6145": "Porichthysbathoiketes",
        "6146": "Porichthysmargaritatus",
        "6147": "Porichthysoculellus",
        "6148": "Porichthyspauciradiatus",
        "6149": "Porichthysplectrodon",
        "6150": "Porichthysporosissimus",
        "6151": "Priacanthusarenatus",
        "6152": "Priacanthuscruentatus",
        "6153": "Priolepishipoliti",
        "6154": "Priolepisrobinsi",
        "6155": "Prionotusalbirostris",
        "6156": "Prionotusbeanii",
        "6157": "Prionotushorrens",
        "6158": "Prionotuslongispinosus",
        "6159": "Prionotusophryas",
        "6160": "Prionotuspunctatus",
        "6161": "Prionotusroseus",
        "6162": "Prionotusrubio",
        "6163": "Prionotusstearnsi",
        "6164": "Prionotusstephanophrys",
        "6165": "Prionotusteaguei",
        "6166": "Prionuruslaticlavius",
        "6167": "Prionuruspunctatus",
        "6168": "Pristigenysalta",
        "6169": "Pristigenysserrula",
        "6170": "Pristipomoides",
        "6171": "Pristipomoidesaquilonaris",
        "6172": "Pristipomoidesfreemani",
        "6173": "Prochilodusmagdalenae",
        "6174": "Prochilodusreticulatus",
        "6175": "Prognathodesaculeatus",
        "6176": "Prognathodesfalcifer",
        "6177": "Prognichthysgibbifrons",
        "6178": "Promethichthysprometheus",
        "6179": "Pronotogrammuseos",
        "6180": "Protosciaenabathytatos",
        "6181": "Protosciaenatrewavasae",
        "6182": "Psenescyanophrys",
        "6183": "Pseudobalistesnaufragium",
        "6184": "Pseudogrammagregoryi",
        "6185": "Pseudophallusmindii",
        "6186": "Pseudophichthyssplendens",
        "6187": "Pseudoplatystomafasciatum",
        "6188": "Pseudupeneusgrandisquamis",
        "6189": "Pseudupeneusmaculatus",
        "6190": "Psilotrisbatrachodes",
        "6191": "Psilotriscelsus",
        "6192": "Ptereleotrishelenae",
        "6193": "Ptereleotrishelenae",
        "6194": "Pteroisvolitans",
        "6195": "Pterygoplichthysundecimalis",
        "6196": "Pycnommaroosevelti",
        "6197": "Pythonichthyssanguineus",
        "6198": "Rachycentroncanadum",
        "6199": "Regalecusglesne",
        "6200": "Remoraremora",
        "6201": "Remorinaalbescens",
        "6202": "Rhamdiaquelen",
        "6203": "Rhamdiasebae",
        "6204": "Rhinesomusbicaudalis",
        "6205": "Rhinesomustriqueter",
        "6206": "Rhomboplitesaurorubens",
        "6207": "Rhynchocongerflavus",
        "6208": "Rhynchocongerguppyi",
        "6209": "Risorruber",
        "6210": "Roeboidesdayi",
        "6211": "Rypticusbicolor",
        "6212": "Rypticusbistrispinus",
        "6213": "Rypticusbornoi",
        "6214": "Rypticusnigripinnis",
        "6215": "Rypticusrandalli",
        "6216": "Rypticussaponaceus",
        "6217": "Rypticussubbifrenatus",
        "6218": "Saccogastermelanomycter",
        "6219": "Salminusaffinis",
        "6220": "Sardasarda",
        "6221": "Sardinellaaurita",
        "6222": "Sardinellajaneiro",
        "6223": "Sargocentronbullisi",
        "6224": "Sargocentroncoruscus",
        "6225": "Sargocentronsuborbitalis",
        "6226": "Sargocentronvexillarium",
        "6227": "Sauridabrasiliensis",
        "6228": "Sauridacaribbaea",
        "6229": "Sauridanormani",
        "6230": "Sauridasuspicio",
        "6231": "Scartellacristata",
        "6232": "Scaruscoelestinus",
        "6233": "Scaruscoeruleus",
        "6234": "Scaruscompressus",
        "6235": "Scaruscroicensis",
        "6236": "Scarusghobban",
        "6237": "Scarusguacamaia",
        "6238": "Scarusiseri",
        "6239": "Scarusperrico",
        "6240": "Scarustaeniopterus",
        "6241": "Scarusvetula",
        "6242": "Sciadesdowii",
        "6243": "Sciadesherzbergii",
        "6244": "Sciadesproops",
        "6245": "Sciadestroschelii",
        "6246": "Scomberjaponicus",
        "6247": "Scomberomorusbrasiliensis",
        "6248": "Scomberomoruscavalla",
        "6249": "Scomberomorusregalis",
        "6250": "Scomberomorussierra",
        "6251": "Scorpaenaafuerae",
        "6252": "Scorpaenaagassizii",
        "6253": "Scorpaenaalbifimbria",
        "6254": "Scorpaenabergii",
        "6255": "Scorpaenabrachyptera",
        "6256": "Scorpaenabrasiliensis",
        "6257": "Scorpaenacalcarata",
        "6258": "Scorpaenadispar",
        "6259": "Scorpaenagrandicornis",
        "6260": "Scorpaenahistrio",
        "6261": "Scorpaenainermis",
        "6262": "Scorpaenaisthmensis",
        "6263": "Scorpaenamelasma",
        "6264": "Scorpaenamystes",
        "6265": "ScorpaenaNuevoScorpaena",
        "6266": "Scorpaenaplumieri",
        "6267": "Scorpaenarussula",
        "6268": "Scorpaenodes",
        "6269": "Scorpaenodescaribbaeus",
        "6270": "Scorpaenodesxyris",
        "6271": "Sectatorocyurus",
        "6272": "Selarcrumenophthalmus",
        "6273": "Selenebrevoortii",
        "6274": "Selenebrownii",
        "6275": "Seleneperuviana",
        "6276": "Selenesetapinnis",
        "6277": "Selenespixii",
        "6278": "Selenevomer",
        "6279": "Serioladumerili",
        "6280": "Seriolafasciata",
        "6281": "Seriolalalandi",
        "6282": "Seriolaperuana",
        "6283": "Seriolarivoliana",
        "6284": "Seriolazonata",
        "6285": "Serraniculuspumilio",
        "6286": "Serranusannularis",
        "6287": "Serranusatrobranchus",
        "6288": "Serranusbaldwini",
        "6289": "Serranuschionaraia",
        "6290": "Serranusflaviventris",
        "6291": "Serranusnotospilus",
        "6292": "Serranusphoebe",
        "6293": "Serranuspsittacinus",
        "6294": "Serranussp.",
        "6295": "Serranustabacarius",
        "6296": "Serranustigrinus",
        "6297": "Serranustortugarum",
        "6298": "Setarchesguentheri",
        "6299": "Sicydiumantillarum",
        "6300": "Sigmopselongatum",
        "6301": "Soleonasusfinis",
        "6302": "Sorubimcuspicaudus",
        "6303": "Sorubimlima",
        "6304": "Sparisomaatomarium",
        "6305": "Sparisomaaurofrenatum",
        "6306": "Sparisomachrysopterum",
        "6307": "Sparisomaradians",
        "6308": "Sparisomarubripinne",
        "6309": "Sparisomaviride",
        "6310": "Sphoeroidesannulatus",
        "6311": "Sphoeroidesdorsalis",
        "6312": "Sphoeroideseulepidotus",
        "6313": "Sphoeroidesgeorgemilleri",
        "6314": "Sphoeroidesgreeleyi",
        "6315": "Sphoeroideslobatus",
        "6316": "Sphoeroidesmaculatus",
        "6317": "Sphoeroidesnephelus",
        "6318": "Sphoeroidespachygaster",
        "6319": "Sphoeroidesspengleri",
        "6320": "Sphoeroidestestudineus",
        "6321": "Sphoeroidestyleri",
        "6322": "Sphoeroidesyergeri",
        "6323": "Sphyraenabarracuda",
        "6324": "Sphyraenaensis",
        "6325": "Sphyraenagenie",
        "6326": "Sphyraenaguachancho",
        "6327": "Sphyraenaidiastes",
        "6328": "Sphyraenalucasana",
        "6329": "Sphyraenapicudilla",
        "6330": "Starksiaatlantica",
        "6331": "Starksiafasciata",
        "6332": "Starksiahassi",
        "6333": "Starksialepicoelia",
        "6334": "Starksiananodes",
        "6335": "Starksiasluiteri",
        "6336": "Starksiavariabilis",
        "6337": "Starksiay-lineata",
        "6338": "Stathmonotusgymnodermis",
        "6339": "Stathmonotusstahli",
        "6340": "Stegastesacapulcoensis",
        "6341": "Stegastesarcifrons",
        "6342": "Stegastesdiencaeus",
        "6343": "Stegastesdorsopunicans",
        "6344": "Stegastesflavilatus",
        "6345": "Stegastesfuscus",
        "6346": "Stegastesleucorus",
        "6347": "Stegastesleucostictus",
        "6348": "Stegastesmellis",
        "6349": "Stegastesotophorus",
        "6350": "Stegastespartitus",
        "6351": "Stegastesplanifrons",
        "6352": "Stegastesrectifraenum",
        "6353": "Stegastesvariabilis",
        "6354": "Steindachneriaargentea",
        "6355": "Stelliferbrasiliensis",
        "6356": "Stelliferchaoi",
        "6357": "Stellifercolonensis",
        "6358": "Stellifergriseus",
        "6359": "Stelliferlanceolatus",
        "6360": "Stellifermancorensis",
        "6361": "Stellifermicrops",
        "6362": "Stellifernaso",
        "6363": "Stelliferrastrifer",
        "6364": "Stelliferstellifer",
        "6365": "Stellifervenezuelae",
        "6366": "Stemonosudisgracilis",
        "6367": "Stemonosudisrothschildi",
        "6368": "Stephanolepishispidus",
        "6369": "Stephanolepissetifer",
        "6370": "Sternoptyxdiaphana",
        "6371": "Sternopygusmacrurus",
        "6372": "Stethojulisbandanensis",
        "6373": "Stomiasaffinis",
        "6374": "Strongyluramarina",
        "6375": "Strongyluranotata",
        "6376": "Strongylurascapularis",
        "6377": "Strongyluratimucu",
        "6378": "Stygnobrotulalatebricola",
        "6379": "Sufflamenverres",
        "6380": "Syaciumgunteri",
        "6381": "Syaciumlatifrons",
        "6382": "Syaciummicrurum",
        "6383": "Syaciumovale",
        "6384": "Syaciumpapillosum",
        "6385": "Syaciumsp.",
        "6386": "Symphurusarawak",
        "6387": "Symphuruscallopterus",
        "6388": "Symphuruscaribbeanus",
        "6389": "Symphuruschabanaudi",
        "6390": "Symphurusdiomedianus",
        "6391": "Symphuruselongatus",
        "6392": "Symphurusginsburgi",
        "6393": "Symphurusgorgonae",
        "6394": "Symphurushernandezi",
        "6395": "Symphurusleei",
        "6396": "Symphurusmarginatus",
        "6397": "Symphurusn.sp.",
        "6398": "Symphurusoligomerus",
        "6399": "Symphurusparvus",
        "6400": "Symphuruspelicanus",
        "6401": "Symphuruspiger",
        "6402": "Symphurusplagusia",
        "6403": "Symphuruspusillus",
        "6404": "Symphurustessellatus",
        "6405": "Symphysanodonberryi",
        "6406": "Synagropsbellus",
        "6407": "Synagropspseudomicrolepis",
        "6408": "Synagropsspinosus",
        "6409": "Synagropstrispinosus",
        "6410": "Synbranchusmarmoratus",
        "6411": "Syngnathuscaribbaeus",
        "6412": "Syngnathusdawsoni",
        "6413": "Syngnathuspelagicus",
        "6414": "Syngnathusrandalli",
        "6415": "Synodontisclarias",
        "6416": "Synodusevermanni",
        "6417": "Synodusfoetens",
        "6418": "Synodusintermedius",
        "6419": "Synoduslacertinus",
        "6420": "Synoduspoeyi",
        "6421": "Synodussaurus",
        "6422": "Synodusscituliceps",
        "6423": "Synodussechurae",
        "6424": "Synodussynodus",
        "6425": "Taractesrubescens",
        "6426": "Taractichthyslongipinnis",
        "6427": "Tetrapturusalbidus",
        "6428": "Thalassomabifasciatum",
        "6429": "Thalassomagrammaticum",
        "6430": "Thalassomalucasanum",
        "6431": "Thalassophrynemaculosa",
        "6432": "Thalassophrynemegalops",
        "6433": "Thalassophrynenattereri",
        "6434": "Thunnusalalunga",
        "6435": "Thunnusalbacares",
        "6436": "Thunnusatlanticus",
        "6437": "Thunnusobesus",
        "6438": "Thunnusthynnus",
        "6439": "Tigrigobiusgemmatum",
        "6440": "Tomicodonfasciatus",
        "6441": "Trachelyopterusinsignis",
        "6442": "Trachinocephalusmyops",
        "6443": "Trachinotuscarolinus",
        "6444": "Trachinotusfalcatus",
        "6445": "Trachinotusgoodei",
        "6446": "Trachinotuskennedyi",
        "6447": "Trachinotusrhodopus",
        "6448": "Trachinotusstilbe",
        "6449": "Trachuruslathami",
        "6450": "Trichiuruslepturus",
        "6451": "Trichogasterpectoralis",
        "6452": "Trichopsettacaribbaea",
        "6453": "Trichopsettamelasma",
        "6454": "Trinectesfonsecensis",
        "6455": "Trinectesinscriptus",
        "6456": "Trinectesmaculatus",
        "6457": "Trinectespaulistanus",
        "6458": "Trinectessp.",
        "6459": "Triplophoshemingi",
        "6460": "Triportheusmagdalenae",
        "6461": "Tylosurusacus",
        "6462": "Tylosuruscrocodilus",
        "6463": "Umbrinabroussonnetii",
        "6464": "Umbrinacoroides",
        "6465": "Umbrinamilliae",
        "6466": "Upeneusparvus",
        "6467": "Uraspissecunda",
        "6468": "Urocongersyringinus",
        "6469": "Uropterygiusmacrocephalus",
        "6470": "Uropterygiusmacularius",
        "6471": "Veneficaprocera",
        "6472": "Xanthichthysmento",
        "6473": "Xanthichthysringens",
        "6474": "Xenomystaxatrarius",
        "6475": "Xenomystaxaustrinus",
        "6476": "Xenomystaxbidentatus",
        "6477": "Xenomystaxcongroides",
        "6478": "Xenophthalmichthysdanae",
        "6479": "Xiphiasgladius",
        "6480": "Xyrichthysmartinicensis",
        "6481": "Xyrichthysnovacula",
        "6482": "Xyrichthyssplendens",
        "6483": "Zalieuteselater",
        "6484": "Zalieutesmcgintyi",
        "6485": "Zancluscornutus",
        "6486": "Zenionhololepis",
        "6487": "Otro"    
        }
    if "ESPEC_NOMB" in df.columns:
        df["especie"] = df["ESPEC_NOMB"].map(mapa_espec_nomb).fillna(df["ESPEC_NOMB"])
    else:

        df["especie"] = None
    return df



# Función para exportar a Excel
def exportar_excel(dataframe, ruta_salida):
    try:

        dataframe.to_excel(ruta_salida, index=False)

    except Exception as e:
        return None


# 🔹 Función para procesar campos específicos
def procesar_campos_especificos(df):
    """Calcula los campos en orden secuencial asegurando que cada campo esté disponible antes de ser usado."""

    if df is None or df.empty:

        return df  

     # Mapear COOR_NORTE a verbatimLatitude
    if "COOR_NORTE" in df.columns:
        df["verbatimLatitude"] = df["COOR_NORTE"]
    else:

        df["verbatimLatitude"] = None

    # Mapear COOR_ESTE a verbatimLongitude
    if "COOR_ESTE" in df.columns:
        df["verbatimLongitude"] = df["COOR_ESTE"]
    else:

        df["verbatimLongitude"] = None

    # Se calcula  `recordNumber`
    if "OBJECTID" in df.columns:
        df["recordNumber"] = df["OBJECTID"]
    else:

        df["recordNumber"] = None

    # Calcular `occurrenceID`
    if "OPERADOR" in df.columns and "recordNumber" in df.columns:
        df["occurrenceID"] = df.apply(
            lambda row: re.sub(r'[^A-Za-z0-9]', '', row["OPERADOR"]) + ":pecesOFM:" + str(row["recordNumber"]),
            axis=1
        )
    else:

        df["occurrenceID"] = None

     # Asignar `occurrenceID` a `id`
    if "occurrenceID" in df.columns:
        df["id"] = df["occurrenceID"]
    else:

        df["id"] = None


    # Concatenar `COOR_ESTE` y `COOR_NORTE` para `verbatimCoordinates`
    if "verbatimLongitude" in df.columns and "verbatimLatitude" in df.columns:
        df["verbatimCoordinates"] = df.apply(
            lambda row: str(row["verbatimLongitude"]).replace(",", ".") + ", " + str(row["verbatimLatitude"]).replace(",", "."),
            axis=1
        )
    else:

        df["verbatimCoordinates"] = None


    # se mapea a variables temporales para despues sacar el type

    if "EVIDENCIA" in df.columns:
        mapeo_evidencia = {
            181: "StillImage",
            182: "MovingImage",
            183: "MovingImage",
            184: "Event"           
        }
        df["type"] = df["EVIDENCIA"].map(mapeo_evidencia).fillna("")
    else:

        df["type"] = None

    # Calcular `basisOfRecord`, que depende de `type`

    if "type" in df.columns:
        mapeo_basis_2 = {
            "StillImage": "MachineObservation",
            "MovingImage": "MachineObservation",
            "Event": "HumanObservation"
        }
        df["basisOfRecord"] = df["type"].map(mapeo_basis_2).fillna("")
    else:

        df["basisOfRecord"] = None
      
    
    # Calcular `occurrenceRemarks`
    if "EVIDENCIA" in df.columns:
        mapeo_evidencia = {
            181: "Fotografía",
            182: "video",
            183: "Video y fotografía",
            184: "Solo observación" 
        }
        df["temp_occurrenceRemarks"] = df["EVIDENCIA"].map(mapeo_evidencia).fillna("")
    else:

        df["temp_occurrenceRemarks"] = None

    if "temp_occurrenceRemarks" in df.columns and "OBSERVACIO_y" in df.columns:
        df["occurrenceRemarks"] = df.apply(
            lambda row: (
                str(row["temp_occurrenceRemarks"]) if pd.notna(row["temp_occurrenceRemarks"]) and pd.isna(row["OBSERVACIO_y"]) else
                str(row["OBSERVACIO_y"]) if pd.notna(row["OBSERVACIO_y"]) and pd.isna(row["temp_occurrenceRemarks"]) else
                f"{row['temp_occurrenceRemarks']} | {row['OBSERVACIO_y']}" if pd.notna(row["temp_occurrenceRemarks"]) and pd.notna(row["OBSERVACIO_y"]) else
                None
            ),
            axis=1
        )
    
    

    # Mapear HORA a eventTime (formato HH:MM:SS)
    if "HORA" in df.columns:
        df["eventTime"] = pd.to_timedelta(df["HORA"], unit="h").dt.components.apply(
            lambda x: f"{int(x.hours):02}:{int(x.minutes):02}:{int(x.seconds):02}", axis=1
        )
    else:

        df["eventTime"] = None




    # Mapeamos a order
    if "ORDEN" in df.columns:
        df["ORDEN"] = df["ORDEN"].fillna(0).astype(float).astype(int).astype(str).replace({"0": ""})
    else:

        df["ORDEN"] = None

    mapeo_orden = {
        "3101": "Albuliformes",
        "3102": "Anguilliformes",
        "3103": "Ateleopodiformes",
        "3104": "Atheriniformes",
        "3105": "Aulopiformes",
        "3106": "Batrachoidiformes",
        "3107": "Beloniformes",
        "3108": "Beryciformes",
        "3109": "Characiformes",
        "3110": "Clupeiformes",
        "3111": "Cypriniformes",
        "3112": "Cyprinodontiformes",
        "3113": "Elopiformes",
        "3114": "Gadiformes",
        "3115": "Gasterosteiformes",
        "3116": "Gonorhynchiformes",
        "3117": "Gymnotiformes",
        "3118": "Lampridiformes",
        "3119": "Lophiiformes",
        "3120": "Mugiliformes",
        "3121": "Myctophiformes",
        "3122": "Ophidiiformes",
        "3123": "Osmeriformes",
        "3124": "Osteoglossiformes",
        "3125": "Perciformes",
        "3126": "Pleuronectiformes",
        "3127": "Polymixiiformes",
        "3128": "Saccopharyngiformes",
        "3129": "Salmoniformes",
        "3130": "Scorpaeniformes",
        "3131": "Semionotiformes",
        "3132": "Siluriformes",
        "3133": "Stephanoberyciformes",
        "3134": "Stomiiformes",
        "3135": "Synbranchiformes",
        "3136": "Tetraodontiformes",
        "3137": "Zeiformes",
        "3138": "Otro"        
        }
    
    df["order"] = df["ORDEN"].map(mapeo_orden).fillna("")


    # Mapeamos SUBORDEN a una variable temporal
    if "SUBORDEN" in df.columns:
        df["SUBORDEN"] = df["SUBORDEN"].fillna(0).astype(float).astype(int).astype(str).replace({"0": ""})
    else:

        df["SUBORDEN"] = None

    mapeo_suborden = {
        "3201": "Acanthuroidei",
        "3202": "Anabanthoidei",
        "3203": "Anguilloidei",
        "3204": "Blennioidei",
        "3205": "Callionymoidei",
        "3206": "Congroidei",
        "3207": "Cottoidei",
        "3208": "Dactylopteroidei",
        "3209": "Gobiesocoidei",
        "3210": "Gobioidei",
        "3211": "Gymnotoidei",
        "3212": "Labroidei",
        "3213": "Muraenoidei",
        "3214": "Normanichtyioidei",
        "3215": "Percoidei",
        "3216": "Platycephaloidei",
        "3217": "Scombroidei",
        "3218": "Scombrolabracoidei",
        "3219": "Scorpaenoidei",
        "3220": "Sternopygoidei",
        "3221": "Stromateoidei",
        "3222": "Tetraodontoidei",
        "3223": "Trachinoidei",
        "3224": "Triacanthoidei",
        "3225": "Otro"
    }
    df["temp_suborden"] = df["SUBORDEN"].map(mapeo_suborden).fillna("")

    # Mapeamos a family
    if "FAMILIA" in df.columns:
        df["FAMILIA"] = df["FAMILIA"].fillna(0).astype(float).astype(int).astype(str).replace({"0": ""})
    else:

        df["FAMILIA"] = None

    mapeo_familia = {
        "3301": "Acanthuridae",
        "3302": "Achiridae",
        "3303": "Achiropsettidae",
        "3304": "Acropomatidae",
        "3305": "Ageneiosidae",
        "3306": "Albulidae",
        "3307": "Alepisauridae",
        "3308": "Alepocephalidae",
        "3309": "Amarsipidae",
        "3310": "Ammodytidae",
        "3311": "Anablebidae",
        "3312": "Anguillidae",
        "3313": "Anoplogastridae",
        "3314": "Anostomidae",
        "3315": "Anotopteridae",
        "3316": "Antennariidae",
        "3317": "Aphyonidae",
        "3318": "Aplocheilidae",
        "3319": "Apogonidae",
        "3320": "Apteronotidae",
        "3321": "Argentinidae",
        "3322": "Ariidae",
        "3323": "Ariommatidae",
        "3324": "Aspredinidae",
        "3325": "Astroblepidae",
        "3326": "Ateleopodidae",
        "3327": "Atherinidae",
        "3328": "Atherinopsidae",
        "3329": "Auchenipteridae",
        "3330": "Aulopodidae",
        "3331": "Aulostomidae",
        "3332": "Balistidae",
        "3333": "Barbourisiidae",
        "3334": "Bathyclupeidae",
        "3335": "Bathyclupeidae",
        "3336": "Bathylagidae",
        "3337": "Batrachoididae",
        "3338": "Belonidae",
        "3339": "Belontiidae",
        "3340": "Bembridae",
        "3341": "Berycidae",
        "3342": "Blenniidae",
        "3343": "Bothidae",
        "3344": "Bramidae",
        "3345": "Bregmacerotidae",
        "3346": "Bythitidae",
        "3347": "Callanthiidae",
        "3348": "Callichthyidae",
        "3349": "Callionymidae",
        "3350": "Caproidae",
        "3351": "Caracanthidae",
        "3352": "Carangidae",
        "3353": "Carapidae",
        "3354": "Caristiidae",
        "3355": "Caulophrynidae",
        "3356": "Centrolophidae",
        "3357": "Centrophrynidae",
        "3358": "Centropomidae",
        "3359": "Cepolidae",
        "3360": "Ceratiidae",
        "3361": "Cetomimidae",
        "3362": "Cetopsidae",
        "3363": "Chaenopsidae",
        "3364": "Chaetodontidae",
        "3365": "Chanidae",
        "3366": "Characidae",
        "3367": "Chaunacidae",
        "3368": "Cheilodactylidae",
        "3369": "Chiasmodontidae",
        "3370": "Chlopsidae",
        "3371": "Chlorophthalmidae",
        "3372": "Cichlidae",
        "3373": "Cirrhitidae",
        "3374": "Clupeidae",
        "3375": "Colocongridae",
        "3376": "Congiopodidae",
        "3377": "Congridae",
        "3378": "Coryphaenidae",
        "3379": "Ctenoluciidae",
        "3380": "Curimatidae",
        "3381": "Cyematidae",
        "3382": "Cynodontidae",
        "3383": "Cynoglossidae",
        "3384": "Cyprinidae",
        "3385": "Cyprinodontidae",
        "3386": "Dactylopteridae",
        "3387": "Dactyloscopidae",
        "3388": "Derichthyidae",
        "3389": "Diceratiidae",
        "3390": "Diodontidae",
        "3391": "Diretmidae",
        "3392": "Doradidae",
        "3393": "Echeneidae",
        "3394": "Electrophoridae",
        "3395": "Eleotridae",
        "3396": "Elopidae",
        "3397": "Emmelichthyidae",
        "3398": "Engraulidae",
        "3399": "Ephippidae",
        "3400": "Epigonidae",
        "3401": "Erythrinidae",
        "3402": "Eurypharyngidae",
        "3403": "Evermannellidae",
        "3404": "Exocoetidae",
        "3405": "Fistulariidae",
        "3406": "Fundulidae",
        "3407": "Gadidae",
        "3408": "Gasteropelecidae",
        "3409": "Gempylidae",
        "3410": "Gerreidae",
        "3411": "Gibberichthyidae",
        "3412": "Gigantactinidae",
        "3413": "Giganturidae",
        "3414": "Gobiesocidae",
        "3415": "Gobiidae",
        "3416": "Gonostomatidae",
        "3417": "Grammatidae",
        "3418": "Grammicolepididae",
        "3419": "Gymnotidae",
        "3420": "Haemulidae",
        "3421": "Halosauridae",
        "3422": "Helogeneidae",
        "3423": "Hemiodontidae",
        "3424": "Hemiramphidae",
        "3425": "Heptapteridae",
        "3426": "Heterenchelyidae",
        "3427": "Himantholophidae",
        "3428": "Holocentridae",
        "3429": "Hypophthalmidae",
        "3430": "Hypopomidae",
        "3431": "Inermiidae",
        "3432": "Ipnopidae",
        "3433": "Kuhliidae",
        "3434": "Kyphosidae",
        "3435": "Labridae",
        "3436": "Labrisomidae",
        "3437": "Lampridae",
        "3438": "Latridae",
        "3439": "Lebiasinidae",
        "3440": "Lepisosteidae",
        "3441": "Leptochilichthyidae",
        "3442": "Linophrynidae",
        "3443": "Liparidae",
        "3444": "Lobotidae",
        "3445": "Lophiidae",
        "3446": "Lophotidae",
        "3447": "Loricariidae",
        "3448": "Lutjanidae",
        "3449": "Luvaridae",
        "3450": "Macroramphosidae",
        "3451": "Macrouridae",
        "3452": "Macrurocyttidae",
        "3453": "Macruronidae",
        "3454": "Malacanthidae",
        "3455": "Megalomycteridae",
        "3456": "Megalopidae",
        "3457": "Melamphaidae",
        "3458": "Melanocetidae",
        "3459": "Melanonidae",
        "3460": "Merlucciidae",
        "3461": "Microdesmidae",
        "3462": "Microstomatidae",
        "3463": "Mirapinnidae",
        "3464": "Molidae",
        "3465": "Monacanthidae",
        "3466": "Monocentridae",
        "3467": "Monognathidae",
        "3468": "Moridae",
        "3469": "Moringuidae",
        "3470": "Mugilidae",
        "3471": "Mullidae",
        "3472": "Muraenesocidae",
        "3473": "Muraenidae",
        "3474": "Muraenolepididae",
        "3475": "Myctophidae",
        "3476": "Myrocongridae",
        "3477": "Nandidae",
        "3478": "Nematistiidae",
        "3479": "Nemichthyidae",
        "3480": "Neoceratiidae",
        "3481": "Neoscopelidae",
        "3482": "Nettastomatidae",
        "3483": "Nomeidae",
        "3484": "Normanichthyidae",
        "3485": "Notacanthidae",
        "3486": "Notosudidae",
        "3487": "Ogcocephalidae",
        "3488": "Omosudidae",
        "3489": "Oneirodidae",
        "3490": "Ophichthidae",
        "3491": "Ophidiidae",
        "3492": "Opisthoproctidae",
        "3493": "Opistognathidae",
        "3494": "Oplegnathidae",
        "3495": "Oreosomatidae",
        "3496": "Osteoglossidae",
        "3497": "Ostraciidae",
        "3498": "Parabrotulidae",
        "3499": "Paralepididae",
        "3500": "Paralichthyidae",
        "3501": "Parazenidae",
        "3502": "Pempheridae",
        "3503": "Percichthyidae",
        "3504": "Percophidae",
        "3505": "Peristediidae",
        "3506": "Peristediidae",
        "3507": "Phosichthyidae",
        "3508": "Phycidae",
        "3509": "Pimelodidae",
        "3510": "Pinguipedidae",
        "3511": "Platytroctidae",
        "3512": "Pleuronectidae",
        "3513": "Poeciliidae",
        "3514": "Polymixiidae",
        "3515": "Polynemidae",
        "3516": "Pomacanthidae",
        "3517": "Pomacentridae",
        "3518": "Pomatomidae",
        "3519": "Priacanthidae",
        "3520": "Pristigasteridae",
        "3521": "Prochilodontidae",
        "3522": "Profundulidae",
        "3523": "Psychrolutidae",
        "3524": "Rachycentridae",
        "3525": "Regalecidae",
        "3526": "Rhamphichthyidae",
        "3527": "Rondeletiidae",
        "3528": "Saccopharyngidae",
        "3529": "Salmonidae",
        "3530": "Scaridae",
        "3531": "Schindleriidae",
        "3532": "Sciaenidae",
        "3533": "Scoloplacidae",
        "3534": "Scomberesocidae",
        "3535": "Scombridae",
        "3536": "Scombrolabracidae",
        "3537": "Scopelarchidae",
        "3538": "Scophthalmidae",
        "3539": "Scorpaenidae",
        "3540": "Serranidae",
        "3541": "Serrivomeridae",
        "3542": "Soleidae",
        "3543": "Sparidae",
        "3544": "Sphyraenidae",
        "3545": "Steindachneriidae",
        "3546": "Stephanoberycidae",
        "3547": "Sternoptychidae",
        "3548": "Sternopygidae",
        "3549": "Stomiidae",
        "3550": "Stromateidae",
        "3551": "Stylephoridae",
        "3552": "Synaphobranchidae",
        "3553": "Synbranchidae",
        "3554": "Syngnathidae",
        "3555": "Synodontidae",
        "3556": "Tetragonuridae",
        "3557": "Tetraodontidae",
        "3558": "Thaumatichthyidae",
        "3559": "Trachichthyidae",
        "3560": "Trachipteridae",
        "3561": "Triacanthodidae",
        "3562": "Trichiuridae",
        "3563": "Trichomycteridae",
        "3564": "Triglidae",
        "3565": "Tripterygiidae",
        "3566": "Uranoscopidae",
        "3567": "Xiphiidae",
        "3568": "Zanclidae",
        "3569": "Zeidae",
        "3570": "Otro"
        }
    df["family"] = df["FAMILIA"].map(mapeo_familia).fillna("")


    # Mapeamos SUBFAMILIA a una variable temporal
    if "SUBFAMILIA" in df.columns:
        df["SUBFAMILIA"] = df["SUBFAMILIA"].fillna(0).astype(float).astype(int).astype(str).replace({"0": ""})
    else:

        df["SUBFAMILIA"] = None
    
    mapeo_subfamilia = {
        "3601": "Alosinae",
        "3602": "Anostominae",
        "3603": "Anthiinae",
        "3604": "Antigoniinae",
        "3605": "Apsilinae",
        "3606": "Bathygadinae",
        "3607": "Bathymyrinae",
        "3608": "Batrachoidinae",
        "3609": "Brosmophycinae",
        "3610": "Brotulinae",
        "3611": "Brotulotaeniinae",
        "3612": "Bythitinae",
        "3613": "Chilodontinae",
        "3614": "Chrominae",
        "3615": "Clupeinae",
        "3616": "Congrinae",
        "3617": "Dussumieriinae",
        "3618": "Epinephelinae",
        "3619": "Etelinae",
        "3620": "Grammistinae",
        "3621": "Harpadontinae",
        "3622": "Heterocongrinae",
        "3623": "Heterotidinae",
        "3624": "Hippocampinae",
        "3625": "Holocentrinae",
        "3626": "Ilyophinae",
        "3627": "Istiophorinae",
        "3628": "Lampanyctinae",
        "3629": "Latilinae",
        "3630": "Lebiasininae",
        "3631": "Lipropomatinae",
        "3632": "Lutjaninae",
        "3633": "Macrourinae",
        "3634": "Macrouroidinae",
        "3635": "Malacanthinae",
        "3636": "Maurolicinae",
        "3637": "Muraeninae",
        "3638": "Myripristinae",
        "3639": "Myrophinae",
        "3640": "Neobythitinae",
        "3641": "Ophichthinae",
        "3642": "Ophidiinae",
        "3643": "Osteoglossinae",
        "3644": "Paradichthyinae",
        "3645": "Pelloninae",
        "3646": "Peristediinae",
        "3647": "Poecilopsettinae",
        "3648": "Pomacentrinae",
        "3649": "Porichthyinae",
        "3650": "Prionurinae",
        "3651": "Pristigasterinae",
        "3652": "Pyrrhulininae",
        "3653": "Scarinae",
        "3654": "Scorpaeninae",
        "3655": "Sebastinae",
        "3656": "Serraninae",
        "3657": "Setarchinae",
        "3658": "Sparisomatinae",
        "3659": "Sternoptychinae",
        "3660": "Stomiinae",
        "3661": "Symphurinae",
        "3662": "Syngnathinae",
        "3663": "Thalassophryninae",
        "3664": "Trachyrincinae",
        "3665": "Triglinae",
        "3666": "Uropterygiinae",
        "3667": "Xiphiinae",
        "3668": "Zeniinae",
        "3669": "Otro"
    }
    df["temp_subfamily"] = df["SUBFAMILIA"].map(mapeo_subfamilia).fillna("")


    # Mapeamos a genus
    if "GENERO" in df.columns:
        df["GENERO"] = df["GENERO"].fillna(0).astype(float).astype(int).astype(str).replace({"0": ""})
    else:

        df["GENERO"] = None
    
    mapeo_genero = {
        "4101": "Ablennessp.",
        "4102": "Abomasp.",
        "4103": "Abramitessp.",
        "4104": "Abudefdufsp.",
        "4105": "Acanthemblemariasp.",
        "4106": "Acanthocybiumsp.",
        "4107": "Acanthostracionsp.",
        "4108": "Acanthurussp.",
        "4109": "Achirussp.",
        "4110": "Acyrtopssp.",
        "4111": "Acyrtussp.",
        "4112": "Aequidenssp.",
        "4113": "Ageneiosussp.",
        "4114": "Agonostomussp.",
        "4115": "Ahliasp.",
        "4116": "Akkosp.",
        "4117": "Albulasp.",
        "4118": "Alectissp.",
        "4119": "Alphestessp.",
        "4120": "Aluterussp.",
        "4121": "Amblycirrhitussp.",
        "4122": "Amphelikturussp.",
        "4123": "Amphichthyssp.",
        "4124": "Anarchiassp.",
        "4125": "Anchoasp.",
        "4126": "Anchoviasp.",
        "4127": "Anchoviellasp.",
        "4128": "Ancylopsettasp.",
        "4129": "Anguillasp.",
        "4130": "Anisotremussp.",
        "4131": "Antennariussp.",
        "4132": "Antennatussp.",
        "4133": "Anthiassp.",
        "4134": "Antigoniasp.",
        "4135": "Aplatophissp.",
        "4136": "Apogonsp.",
        "4137": "Apsilussp.",
        "4138": "Archosargussp.",
        "4139": "Arcossp.",
        "4140": "Argentinasp.",
        "4141": "Argyripnussp.",
        "4142": "Argyropelecussp.",
        "4143": "Ariommasp.",
        "4144": "Ariopsissp.",
        "4145": "Ariosomasp.",
        "4146": "Arothronsp.",
        "4147": "Astrapogonsp.",
        "4148": "Astroscopussp.",
        "4149": "Astyanaxsp.",
        "4150": "Atherinellasp.",
        "4151": "Atherinomorussp.",
        "4152": "Atractodenchelyssp.",
        "4153": "Aulostomussp.",
        "4154": "Auxissp.",
        "4155": "Awaoussp.",
        "4156": "Axoclinussp.",
        "4157": "Bagresp.",
        "4158": "Bairdiellasp.",
        "4159": "Balistessp.",
        "4160": "Barbulifersp.",
        "4161": "Bascanichthyssp.",
        "4162": "Bathyanthiassp.",
        "4163": "Bathyclupeasp.",
        "4164": "Bathycongrussp.",
        "4165": "Bathygadussp.",
        "4166": "Bathygobiussp.",
        "4167": "Bathypteroissp.",
        "4168": "Bathyurocongersp.",
        "4169": "Batrachoidessp.",
        "4170": "Bellatorsp.",
        "4171": "Bembropssp.",
        "4172": "Benthocometessp.",
        "4173": "Benthodesmussp.",
        "4174": "Bodianussp.",
        "4175": "Bollmanniasp.",
        "4176": "Bothussp.",
        "4177": "Bramasp.",
        "4178": "Bregmacerossp.",
        "4179": "Brotulasp.",
        "4180": "Bryconsp.",
        "4181": "Bryxsp.",
        "4182": "Bufoceratiassp.",
        "4183": "Calamopteryxsp.",
        "4184": "Calamussp.",
        "4185": "Callechelyssp.",
        "4186": "Callionymussp.",
        "4187": "Cantherhinessp.",
        "4188": "Canthidermissp.",
        "4189": "Canthigastersp.",
        "4190": "Caquetaiasp.",
        "4191": "Caranxsp.",
        "4192": "Carapussp.",
        "4193": "Catesbyasp.",
        "4194": "Cathoropssp.",
        "4195": "Caulolatilussp.",
        "4196": "Centropomussp.",
        "4197": "Centropygesp.",
        "4198": "Cephalopholissp.",
        "4199": "Cerdalesp.",
        "4200": "Cetengraulissp.",
        "4201": "Chaenopsissp.",
        "4202": "Chaetodipterussp.",
        "4203": "Chaetodonsp.",
        "4204": "Channomuraenasp.",
        "4205": "Chanossp.",
        "4206": "Charaxsp.",
        "4207": "Chascanopsettasp.",
        "4208": "Chauliodussp.",
        "4209": "Chaunaxsp.",
        "4210": "Cheilopogonsp.",
        "4211": "Cherublemmasp.",
        "4212": "Chilomycterussp.",
        "4213": "Chilorhinussp.",
        "4214": "Chirocentrodonsp.",
        "4215": "Chlopsissp.",
        "4216": "Chlorophthalmussp.",
        "4217": "Chloroscombrussp.",
        "4218": "Chromissp.",
        "4219": "Cirrhitichthyssp.",
        "4220": "Cirrhitussp.",
        "4221": "Citharichthyssp.",
        "4222": "Clepticussp.",
        "4223": "Coelorinchussp.",
        "4224": "Colocongersp.",
        "4225": "Colomesussp.",
        "4226": "Congersp.",
        "4227": "Conodonsp.",
        "4228": "Cookeolussp.",
        "4229": "Coralliozetussp.",
        "4230": "Corydorassp.",
        "4231": "Coryphaenasp.",
        "4232": "Coryphaenoidessp.",
        "4233": "Coryphopterussp.",
        "4234": "Cosmocampussp.",
        "4235": "Creagrutussp.",
        "4236": "Crossoloricariasp.",
        "4237": "Cryptotomussp.",
        "4238": "Ctenochaetussp.",
        "4239": "Ctenogobiussp.",
        "4240": "Ctenoluciussp.",
        "4241": "Ctenosciaenasp.",
        "4242": "Cubicepssp.",
        "4243": "Curimatasp.",
        "4244": "Cyclopsettasp.",
        "4245": "Cynoponticussp.",
        "4246": "Cynopotamussp.",
        "4247": "Cynoscionsp.",
        "4248": "Cyphocharaxsp.",
        "4249": "Cyprinodonsp.",
        "4250": "Cypselurussp.",
        "4251": "Cyttopsissp.",
        "4252": "Dactylopterussp.",
        "4253": "Dactyloscopussp.",
        "4254": "Daectorsp.",
        "4255": "Decapterussp.",
        "4256": "Decodonsp.",
        "4257": "Dermatolepissp.",
        "4258": "Diaphussp.",
        "4259": "Diapterussp.",
        "4260": "Dibranchussp.",
        "4261": "Dicrolenesp.",
        "4262": "Diodonsp.",
        "4263": "Diplacanthopomasp.",
        "4264": "Diplectrumsp.",
        "4265": "Diplodussp.",
        "4266": "Diplogrammussp.",
        "4267": "Diplospinussp.",
        "4268": "Doratonotussp.",
        "4269": "Dormitatorsp.",
        "4270": "Doryrhamphussp.",
        "4271": "Dysommasp.",
        "4272": "Dysomminasp.",
        "4273": "Echeneissp.",
        "4274": "Echidnasp.",
        "4275": "Echiodonsp.",
        "4276": "Echiophissp.",
        "4277": "Ekemblemariasp.",
        "4278": "Elacatinussp.",
        "4279": "Elagatissp.",
        "4280": "Eleotrissp.",
        "4281": "Elopssp.",
        "4282": "Emblemariasp.",
        "4283": "Emblemariopsissp.",
        "4284": "Emmelichthyopssp.",
        "4285": "Enchelycoresp.",
        "4286": "Engraulissp.",
        "4287": "Engyophryssp.",
        "4288": "Enneanectessp.",
        "4289": "Entomacrodussp.",
        "4290": "Epigonussp.",
        "4291": "Epinephelussp.",
        "4292": "Equetussp.",
        "4293": "Erythroclessp.",
        "4294": "Etelissp.",
        "4295": "Etropussp.",
        "4296": "Etrumeussp.",
        "4297": "Eucinostomussp.",
        "4298": "Eugerressp.",
        "4299": "Euleptorhamphussp.",
        "4300": "Eumegistussp.",
        "4301": "Euthynnussp.",
        "4302": "Evermannichthyssp.",
        "4303": "Evorthodussp.",
        "4304": "Evoxymetoponsp.",
        "4305": "Exocoetussp.",
        "4306": "Facciolellasp.",
        "4307": "Fistulariasp.",
        "4308": "Foetorepussp.",
        "4309": "Forcipigersp.",
        "4310": "Gadellasp.",
        "4311": "Gambusiasp.",
        "4312": "Gastropsettasp.",
        "4313": "Gempylussp.",
        "4314": "Geophagussp.",
        "4315": "Gephyroberyxsp.",
        "4316": "Gerressp.",
        "4317": "Gilbertolussp.",
        "4318": "Gillellussp.",
        "4319": "Ginsburgellussp.",
        "4320": "Gnathanodonsp.",
        "4321": "Gnatholepissp.",
        "4322": "Gnathophissp.",
        "4323": "Gobiesoxsp.",
        "4324": "Gobioidessp.",
        "4325": "Gobiomorussp.",
        "4326": "Gobionellussp.",
        "4327": "Gobiosomasp.",
        "4328": "Gobulussp.",
        "4329": "Gonioplectrussp.",
        "4330": "Gonostomasp.",
        "4331": "Grammasp.",
        "4332": "Guavinasp.",
        "4333": "Gunterichthyssp.",
        "4334": "Gymnachirussp.",
        "4335": "Gymnomuraenasp.",
        "4336": "Gymnothoraxsp.",
        "4337": "Haemulonsp.",
        "4338": "Haemulopsissp.",
        "4339": "Halicampussp.",
        "4340": "Halichoeressp.",
        "4341": "Halieutichthyssp.",
        "4342": "Halosaurussp.",
        "4343": "Harengulasp.",
        "4344": "Helicolenussp.",
        "4345": "Hemanthiassp.",
        "4346": "Hemiancistrussp.",
        "4347": "Hemibryconsp.",
        "4348": "Hemicaranxsp.",
        "4349": "Hemiemblemariasp.",
        "4350": "Hemiramphussp.",
        "4351": "Heterocongersp.",
        "4352": "Heteropriacanthussp.",
        "4353": "Hippocampussp.",
        "4354": "Hippoglossinasp.",
        "4355": "Hirundichthyssp.",
        "4356": "Histriosp.",
        "4357": "Holacanthussp.",
        "4358": "Holanthiassp.",
        "4359": "Hollardiasp.",
        "4360": "Holocentrussp.",
        "4361": "Hopliassp.",
        "4362": "Hoplopagrussp.",
        "4363": "Hoplostethussp.",
        "4364": "Hoplunnissp.",
        "4365": "Hymenocephalussp.",
        "4366": "Hypleurochilussp.",
        "4367": "Hypoatherinasp.",
        "4368": "Hypoplectrussp.",
        "4369": "Hyporhamphussp.",
        "4370": "Hyporthodussp.",
        "4371": "Hypsoblenniussp.",
        "4372": "Ichthyapussp.",
        "4373": "Ijimaiasp.",
        "4374": "Ilishasp.",
        "4375": "Ilyophissp.",
        "4376": "Inermiasp.",
        "4377": "Isopisthussp.",
        "4378": "Istiophorussp.",
        "4379": "Japonocongersp.",
        "4380": "Jenkinsiasp.",
        "4381": "Johnrandalliasp.",
        "4382": "Joturussp.",
        "4383": "Kathetostomasp.",
        "4384": "Katsuwonussp.",
        "4385": "Kaupichthyssp.",
        "4386": "Kuhliasp.",
        "4387": "Kyphosussp.",
        "4388": "Labichthyssp.",
        "4389": "Labrisomussp.",
        "4390": "Lachnolaimussp.",
        "4391": "Lactophryssp.",
        "4392": "Laemonemasp.",
        "4393": "Lagocephalussp.",
        "4394": "Lampadenasp.",
        "4395": "Lamprissp.",
        "4396": "Larimussp.",
        "4397": "Lebiasinasp.",
        "4398": "Lepidonectessp.",
        "4399": "Lepidophanessp.",
        "4400": "Lepophidiumsp.",
        "4401": "Leporinussp.",
        "4402": "Lestrolepissp.",
        "4403": "Liopropomasp.",
        "4404": "Lipogrammasp.",
        "4405": "Lobotessp.",
        "4406": "Lonchopisthussp.",
        "4407": "Lonchurussp.",
        "4408": "Lophiodessp.",
        "4409": "Lophiussp.",
        "4410": "Lophogobiussp.",
        "4411": "Lucayablenniussp.",
        "4412": "Lupinoblenniussp.",
        "4413": "Lutjanussp.",
        "4414": "Lycengraulissp.",
        "4415": "Lythrypnussp.",
        "4416": "Macrodonsp.",
        "4417": "Makairasp.",
        "4418": "Malacanthussp.",
        "4419": "Malacocephalussp.",
        "4420": "Malacoctenussp.",
        "4421": "Malacosteussp.",
        "4422": "Malthopsissp.",
        "4423": "Maurolicussp.",
        "4424": "Megalechissp.",
        "4425": "Megalopssp.",
        "4426": "Melichthyssp.",
        "4427": "Membrassp.",
        "4428": "Menticirrhussp.",
        "4429": "Merlucciussp.",
        "4430": "Microdesmussp.",
        "4431": "Microgobiussp.",
        "4432": "Microphissp.",
        "4433": "Micropogoniassp.",
        "4434": "Microspathodonsp.",
        "4435": "Molasp.",
        "4436": "Monacanthussp.",
        "4437": "Monolenesp.",
        "4438": "Monomitopussp.",
        "4439": "Monopenchelyssp.",
        "4440": "Moringuasp.",
        "4441": "Mugilsp.",
        "4442": "Mulloidichthyssp.",
        "4443": "Mullussp.",
        "4444": "Mycteropercasp.",
        "4445": "Myctophumsp.",
        "4446": "Myrichthyssp.",
        "4447": "Myripristissp.",
        "4448": "Myrophissp.",
        "4449": "Naucratessp.",
        "4450": "Nealotussp.",
        "4451": "Nebrissp.",
        "4452": "Nematistiussp.",
        "4453": "Nemichthyssp.",
        "4454": "Neobythitessp.",
        "4455": "Neocongersp.",
        "4456": "Neoepinnulasp.",
        "4457": "Neomerinthesp.",
        "4458": "Neoniphonsp.",
        "4459": "Neoscombropssp.",
        "4460": "Neoscopelussp.",
        "4461": "Nessp.",
        "4462": "Nettastomasp.",
        "4463": "Nezumiasp.",
        "4464": "Nicholsinasp.",
        "4465": "Nomeussp.",
        "4466": "Notacanthussp.",
        "4467": "Notariussp.",
        "4468": "Novaculichthyssp.",
        "4469": "Ocyurussp.",
        "4470": "Odontognathussp.",
        "4471": "Odontoscionsp.",
        "4472": "Ogcocephalussp.",
        "4473": "Ogilbiasp.",
        "4474": "Oligoplitessp.",
        "4475": "Omobranchussp.",
        "4476": "Ophichthussp.",
        "4477": "Ophidionsp.",
        "4478": "Ophioblenniussp.",
        "4479": "Ophioscionsp.",
        "4480": "Opisthonemasp.",
        "4481": "Opisthopterussp.",
        "4482": "Opistognathussp.",
        "4483": "Oreochromissp.",
        "4484": "Orthopristissp.",
        "4485": "Ostichthyssp.",
        "4486": "Ostracionsp.",
        "4487": "Otophidiumsp.",
        "4488": "Oxycirrhitessp.",
        "4489": "Oxyporhamphussp.",
        "4490": "Oxyurichthyssp.",
        "4491": "Pagrussp.",
        "4492": "Palatogobiussp.",
        "4493": "Parabathymyrussp.",
        "4494": "Parablenniussp.",
        "4495": "Paraclinussp.",
        "4496": "Paracongersp.",
        "4497": "Paradiplogrammussp.",
        "4498": "Parahollardiasp.",
        "4499": "Paralichthyssp.",
        "4500": "Paralonchurussp.",
        "4501": "Paranthiassp.",
        "4502": "Parapsettussp.",
        "4503": "Parasphyraenopssp.",
        "4504": "Parasudissp.",
        "4505": "parazensp.",
        "4506": "Parequessp.",
        "4507": "Parexocoetussp.",
        "4508": "Parophidionsp.",
        "4509": "Pellonasp.",
        "4510": "Pempherissp.",
        "4511": "Penetopteryxsp.",
        "4512": "Peprilussp.",
        "4513": "Peristedionsp.",
        "4514": "Petrotyxsp.",
        "4515": "Phaeoptyxsp.",
        "4516": "Phenacoscorbissp.",
        "4517": "Phtheirichthyssp.",
        "4518": "Physiculussp.",
        "4519": "Pimelodussp.",
        "4520": "Pisodonophissp.",
        "4521": "Plagioscionsp.",
        "4522": "Plagiotremussp.",
        "4523": "Platybelonesp.",
        "4524": "Platygillellussp.",
        "4525": "Plectranthiassp.",
        "4526": "Plectrypopssp.",
        "4527": "Poeciliasp.",
        "4528": "Poecilopsettasp.",
        "4529": "Pollichthyssp.",
        "4530": "Polydactylussp.",
        "4531": "Polyipnussp.",
        "4532": "Polymetmesp.",
        "4533": "Polymixiasp.",
        "4534": "Pomacanthussp.",
        "4535": "Pomadasyssp.",
        "4536": "Pomatomussp.",
        "4537": "Pontinussp.",
        "4538": "Porichthyssp.",
        "4539": "Priacanthussp.",
        "4540": "Priolepissp.",
        "4541": "Prionotussp.",
        "4542": "Prionurussp.",
        "4543": "Pristigenyssp.",
        "4544": "Pristipomoidessp.",
        "4545": "Prochilodussp.",
        "4546": "Prognathodessp.",
        "4547": "Prognichthyssp.",
        "4548": "Promethichthyssp.",
        "4549": "Pronotogrammussp.",
        "4550": "Protosciaenasp.",
        "4551": "Psenessp.",
        "4552": "Pseudobalistessp.",
        "4553": "Pseudogrammasp.",
        "4554": "Pseudomyrophissp.",
        "4555": "Pseudophallussp.",
        "4556": "Pseudophichthyssp.",
        "4557": "Pseudoplatystomasp.",
        "4558": "Pseudupeneussp.",
        "4559": "Psilotrissp.",
        "4560": "Ptereleotrissp.",
        "4561": "Ptereleotrissp.",
        "4562": "Pteroissp.",
        "4563": "Pterygoplichthyssp.",
        "4564": "Pycnommasp.",
        "4565": "Pythonichthyssp.",
        "4566": "Rachycentronsp.",
        "4567": "Regalecussp.",
        "4568": "Remorasp.",
        "4569": "Remorinasp.",
        "4570": "Rhamdiasp.",
        "4571": "Rhinesomussp.",
        "4572": "Rhomboplitessp.",
        "4573": "Rhynchocongersp.",
        "4574": "Risorsp.",
        "4575": "Roeboidessp.",
        "4576": "Rypticussp.",
        "4577": "Saccogastersp.",
        "4578": "Salminussp.",
        "4579": "Sanopussp.",
        "4580": "Sardasp.",
        "4581": "Sardinellasp.",
        "4582": "Sargocentronsp.",
        "4583": "Sauridasp.",
        "4584": "Scartellasp.",
        "4585": "Scarussp.",
        "4586": "Sciadessp.",
        "4587": "Scomberomorussp.",
        "4588": "Scombersp.",
        "4589": "Scopelarchussp.",
        "4590": "Scorpaenasp.",
        "4591": "Scorpaenodessp.",
        "4592": "Sectatorsp.",
        "4593": "Selarsp.",
        "4594": "Selenesp.",
        "4595": "Seriolasp.",
        "4596": "Serraniculussp.",
        "4597": "Serranussp.",
        "4598": "Setarchessp.",
        "4599": "Sicydiumsp.",
        "4600": "Sigmopssp.",
        "4601": "Soleonasussp.",
        "4602": "Sorubimsp.",
        "4603": "sp1sp.",
        "4604": "Sparisomasp.",
        "4605": "Sphoeroidessp.",
        "4606": "Sphyraenasp.",
        "4607": "Starksiasp.",
        "4608": "Stathmonotussp.",
        "4609": "Stegastessp.",
        "4610": "Steindachneriasp.",
        "4611": "Stellifersp.",
        "4612": "Stemonosudissp.",
        "4613": "Stephanolepissp.",
        "4614": "Sternoptyxsp.",
        "4615": "Sternopygussp.",
        "4616": "Stethojulissp.",
        "4617": "Stomiassp.",
        "4618": "Strongylurasp.",
        "4619": "Stygnobrotulasp.",
        "4620": "Sufflamensp.",
        "4621": "Syaciumsp.",
        "4622": "Symbolophorussp.",
        "4623": "Symphurussp.",
        "4624": "Symphysanodonsp.",
        "4625": "Synagropssp.",
        "4626": "Synaphobranchussp.",
        "4627": "Synbranchussp.",
        "4628": "Syngnathussp.",
        "4629": "Synodontissp.",
        "4630": "Synodussp.",
        "4631": "Taractessp.",
        "4632": "Taractichthyssp.",
        "4633": "Tetrapturussp.",
        "4634": "Thalassomasp.",
        "4635": "Thalassophrynesp.",
        "4636": "Thunnussp.",
        "4637": "Tigrigobiussp.",
        "4638": "Tomicodonsp.",
        "4639": "Trachelyopterussp.",
        "4640": "Trachinocephalussp.",
        "4641": "Trachinotussp.",
        "4642": "Trachurussp.",
        "4643": "Trichiurussp.",
        "4644": "Trichogastersp.",
        "4645": "Trichopsettasp.",
        "4646": "Trinectessp.",
        "4647": "Triplophossp.",
        "4648": "Triportheussp.",
        "4649": "Tylosurussp.",
        "4650": "Umbrinasp.",
        "4651": "Upeneussp.",
        "4652": "Uraspissp.",
        "4653": "Urocongersp.",
        "4654": "Uropterygiussp.",
        "4655": "Varicussp.",
        "4656": "Veneficasp.",
        "4657": "Ventrifossasp.",
        "4658": "Vinciguerriasp.",
        "4659": "Xanthichthyssp.",
        "4660": "Xenolepidichthyssp.",
        "4661": "Xenomystaxsp.",
        "4662": "Xenophthalmichthyssp.",
        "4663": "Xiphiassp.",
        "4664": "Xyrichthyssp.",
        "4665": "Zalieutessp.",
        "4666": "Zanclussp.",
        "4667": "Zenionsp.",
        "4668": "Otro"
        }
    df["genus"] = df["GENERO"].map(mapeo_genero).fillna("")

    
    # Mapear higherClassification 
    campos_clasificacion = [
        "order", "temp_suborden", "family", "temp_subfamily", "genus"
    ]

    for col in campos_clasificacion:
        if col not in df.columns:
            df[col] = None

    df["higherClassification"] = df[campos_clasificacion]\
        .applymap(lambda x: x if pd.notna(x) and str(x).strip().lower() not in ["", "none", "nan", "<null>"] else pd.NA)\
        .apply(lambda row: " | ".join(row.dropna().astype(str)), axis=1)


  # Mapear a scientificName con validación
    df["scientificName"] = df.apply(
        lambda row: (
            row["especie"]
            if pd.notna(row["especie"]) and str(row["especie"]).strip().lower() not in ["", "none", "nan", "<null>","Otro","Otros","otro","Otra","Otras"]
            else (
                row["higherClassification"].split(" | ")[-1]
                if pd.notna(row["higherClassification"]) and " | " in row["higherClassification"]
                else row["higherClassification"]
            )
        ),
        axis=1
    )


    # Mapeamos NOMB_COMUN
    if "NOMB_COMUN" in df.columns:
        df["NOMB_COMUN"] = df["NOMB_COMUN"].fillna(0).astype(float).astype(int).astype(str).replace({"0": ""})
    else:
        df["NOMB_COMUN"] = None

    mapeo_nombre = {
        "6501": "Abado blanco",
        "6502": "Acapulco major",
        "6503": "african pomapano",
        "6504": "Agallona",
        "6505": "Aguada",
        "6506": "Aguapuro",
        "6507": "Aguijón sable",
        "6508": "Aguja azul",
        "6509": "Aguja blanca",
        "6510": "Aguja corbata",
        "6511": "Agujeta",
        "6512": "Agujeta alargada.",
        "6513": "Agujeta balajú, medio pico",
        "6514": "Agujeta blanca",
        "6515": "Agujeta blanca.",
        "6516": "Agujeta brasoleña, medio pico",
        "6517": "Agujeta voladora",
        "6518": "Agujeto",
        "6519": "Agujón del Atlántico",
        "6520": "Agujon needlefish",
        "6521": "Agujón negro",
        "6522": "Agujón verde",
        "6523": "Albacora",
        "6524": "Almaco Jack",
        "6525": "American sackfish",
        "6526": "American spiny flounder",
        "6527": "Anchoa",
        "6528": "Anchoa bocona",
        "6529": "Anchoa chiquita",
        "6530": "Anchoa cubana",
        "6531": "Anchoa de banda estrecha",
        "6532": "Anchoa de cayo",
        "6533": "Anchoa de fondo",
        "6534": "Anchoa de hebra",
        "6535": "Anchoa dentona",
        "6536": "Anchoa legítima",
        "6537": "Anchoa machete",
        "6538": "Anchoa naranja",
        "6539": "Anchoa ñata",
        "6540": "Anchoa ojitos",
        "6541": "Anchoa ojo gordo",
        "6542": "Anchoa parva",
        "6543": "Anchoa persistente",
        "6544": "Anchoa trompalarga",
        "6545": "Anchoita negra",
        "6546": "Anchova de banco",
        "6547": "Anchoveta",
        "6548": "Anchoveta chuchuelo",
        "6549": "Anchoveta rabiamarilla",
        "6550": "Anchovieta alargada",
        "6551": "Anchovieta cubana",
        "6552": "Ángel rey",
        "6553": "Anglefin whiff",
        "6554": "Anguila americana",
        "6555": "Anguila de lodo",
        "6556": "Apricot bass",
        "6557": "Arcoiris de Cortés",
        "6558": "Arenca",
        "6559": "Arenero",
        "6560": "Arenquillo dentón",
        "6561": "Arenquillo navaja",
        "6562": "Arepa",
        "6563": "Arepa criolla",
        "6564": "Arepa de bajo",
        "6565": "Arepa de canal",
        "6566": "Arepa de cola manchada",
        "6567": "Arepa de cuatro manchas",
        "6568": "Arepa de tres manchas",
        "6569": "Arepa del Caribe",
        "6570": "Arepa manchada",
        "6571": "Arepa oscura",
        "6572": "Arepa playera",
        "6573": "Argentina de Bruce",
        "6574": "Argentina estriada",
        "6575": "Armored searobin",
        "6576": "Arrow blenny",
        "6577": "Arrowtooth Eel",
        "6578": "Asaedae flounder",
        "6579": "Atlantic batfish",
        "6580": "Atlantic fangjaw",
        "6581": "Atlantic lizardfish",
        "6582": "Atlantic midshipman",
        "6583": "Atlantic needlefish",
        "6584": "Atlantic sabretooth anchovy",
        "6585": "Atún aletiamarillo",
        "6586": "Atún aletiazul",
        "6587": "Atún aletinegro",
        "6588": "Atún blanco",
        "6589": "Atún ojo gordo",
        "6590": "Atún ojón",
        "6591": "Azuleja",
        "6592": "Bacalete antena",
        "6593": "Bacalete del Pacifico oriental",
        "6594": "Bacalete rayado",
        "6595": "Bagre",
        "6596": "Bagre",
        "6597": "Bagre bandera",
        "6598": "Bagre barbón",
        "6599": "Bagre blanco",
        "6600": "Bagre cazón",
        "6601": "Bagre cazón",
        "6602": "Bagre chihuil",
        "6603": "Bagre piedrero",
        "6604": "Bagre pintado",
        "6605": "Bagre tete",
        "6606": "Bagre Tomás",
        "6607": "Banded basslet",
        "6608": "Banded blenny",
        "6609": "Banded jawfish",
        "6610": "Banded puffer",
        "6611": "Barbero rojo",
        "6612": "Barbeta",
        "6613": "Barbudo azul",
        "6614": "Barbudo seis barbas",
        "6615": "Barbul",
        "6616": "Barbul arroyo",
        "6617": "Barbul playero",
        "6618": "Barbul sietebarbas",
        "6619": "Barfin goby",
        "6620": "Barnaclebill blenny",
        "6621": "Barracuda guachanche",
        "6622": "Barracuda mexicana",
        "6623": "Barracuda, picúa",
        "6624": "Barracudina antifaz",
        "6625": "Barracudina traslúcida",
        "6626": "Barred cardinalfish",
        "6627": "Barred clingfish",
        "6628": "Barred flagtail",
        "6629": "Barred hamlet",
        "6630": "Barred sorubim",
        "6631": "Barrendero transparente",
        "6632": "Barsnout goby",
        "6633": "Bartail goby",
        "6634": "Beach flounder",
        "6635": "Bearded goby",
        "6636": "Beardfish",
        "6637": "Beardless codling",
        "6638": "Belted cardinalfish",
        "6639": "Berrugata gurrubata",
        "6640": "Besote",
        "6641": "Besote rayado",
        "6642": "Bicolor damselfish",
        "6643": "Bigeye bass",
        "6644": "Bigeye scad",
        "6645": "Bigeye soldierfish",
        "6646": "Bigeye stargazer",
        "6647": "Bighead sand perch",
        "6648": "Bighead tilefish",
        "6649": "Bigmouth sanddab",
        "6650": "Bigtooth cardinalfish",
        "6651": "Bio del rio",
        "6652": "Black armoured searobin",
        "6653": "Black widow",
        "6654": "Black wing flyingfish",
        "6655": "Blackbar blenny",
        "6656": "Blackcap basslet",
        "6657": "Blackcheek blenny",
        "6658": "Blackear bass",
        "6659": "Blackedge triplefin",
        "6660": "Blackfin blenny",
        "6661": "Blackfin goosefish",
        "6662": "Blackfin soapfish",
        "6663": "Blackmouth bass",
        "6664": "Blacknosed butterflyfish",
        "6665": "Blackspot wrasse",
        "6666": "Blanquillo cabezón",
        "6667": "Blénido",
        "6668": "Blénido anguila",
        "6669": "Blénido anguila",
        "6670": "Blénido pintado",
        "6671": "Blenio de mangle",
        "6672": "Blenio Ostra",
        "6673": "Blennio hocicudo",
        "6674": "Blue and gold snapper",
        "6675": "Blue bobo",
        "6676": "Blue mackerel",
        "6677": "Bluebanded goby",
        "6678": "Bluegold goby",
        "6679": "Bluethroat pikeblenny",
        "6680": "Blunthead puffer",
        "6681": "Bobito",
        "6682": "Bocachico",
        "6683": "Bocón cola larga",
        "6684": "Bocón prieto",
        "6685": "Bocón rayado",
        "6686": "Boga",
        "6687": "Bombache de roca",
        "6688": "Bombache sabalete",
        "6689": "Bonefish",
        "6690": "Bonito",
        "6691": "Bonito listado, barrilete",
        "6692": "Boquinete",
        "6693": "Boquinete boca de novia",
        "6694": "Boquita de sabalo",
        "6695": "Borracho mono",
        "6696": "Borracho perlado",
        "6697": "Borracho vacilón",
        "6698": "Bostezador rosado",
        "6699": "Botellona",
        "6700": "Botete aletas punteadas",
        "6701": "Botete bonito",
        "6702": "Botete chato",
        "6703": "Botete diana",
        "6704": "Botete oceánico",
        "6705": "Botete verrugoso",
        "6706": "Brama clara",
        "6707": "Bridle cardinalfish",
        "6708": "Bridled burrfish",
        "6709": "Bridled goby",
        "6710": "Bright anchovy",
        "6711": "Bristletooth conger",
        "6712": "Bristly searobin",
        "6713": "Broadband anchovy",
        "6714": "Broadnose worm eel",
        "6715": "Broadsaddle cardinalfish",
        "6716": "Brótula aletiroja",
        "6717": "Brótula aletona",
        "6718": "Brótula amarillenta",
        "6719": "Brótula barbona",
        "6720": "Brótula oscura",
        "6721": "Brótula rojiza",
        "6722": "Brown sole",
        "6723": "Bruja dos espinas",
        "6724": "Brujo con mancha",
        "6725": "Brujo larguiespina",
        "6726": "Brujo morado",
        "6727": "Brujo pintado",
        "6728": "Bullish conger",
        "6729": "Bullseye grenadier",
        "6730": "Bullseye puffer",
        "6731": "Burique, burica",
        "6732": "Burrito",
        "6733": "Burrito rayado",
        "6734": "Burro catalina",
        "6735": "Burro pecoso",
        "6736": "Burro piedra",
        "6737": "Caballa",
        "6738": "Caballeta macarela",
        "6739": "Caballeta punteada",
        "6740": "Caballeta rabicolorada",
        "6741": "Caballeta, Salmonete",
        "6742": "Caballio de mar hocico largo",
        "6743": "Caballito de mar",
        "6744": "Caballito de mar del Pacífico",
        "6745": "Cabezudo",
        "6746": "Cabrilla",
        "6747": "Cabrilla cinta",
        "6748": "Cabrilla de Perú",
        "6749": "Cabrilla del Perú",
        "6750": "Cabrilla doblecola",
        "6751": "Cabrilla doncella",
        "6752": "Cabrilla doncella",
        "6753": "Cabrilla mamita",
        "6754": "Cabrilla menta",
        "6755": "Cabrilla mulata",
        "6756": "Cabrilla piedrera",
        "6757": "Cabrilla pinta",
        "6758": "Cabrilleta cinteada",
        "6759": "Cabrilleta tres rayas",
        "6760": "Cabrilleta violeta",
        "6761": "Cachava",
        "6762": "Cachegua",
        "6763": "Cachorreta",
        "6764": "Cachorreta alacorta",
        "6765": "Cachua",
        "6766": "Cachúa aguja",
        "6767": "Cachúa cosmopolita",
        "6768": "Cachúa isleña",
        "6769": "Cachúa negra",
        "6770": "Cachúa perra",
        "6771": "Cachuo",
        "6772": "Cagua",
        "6773": "Camotillo fino",
        "6774": "Camotillo naranja.",
        "6775": "Candil cardenal",
        "6776": "Candil de lo alto",
        "6777": "Cano toadfish",
        "6778": "Capitanejo",
        "6779": "Carajota de quilla",
        "6780": "Carajota lechero, agujeta",
        "6781": "Carajota manglera",
        "6782": "Carajota ojona",
        "6783": "Carajota sable",
        "6784": "Carajota verde",
        "6785": "Carajuelito",
        "6786": "Carajuelo de arrecife",
        "6787": "Carajuelo gallito",
        "6788": "Carajuelo manchado",
        "6789": "Carajuelo oscuro",
        "6790": "Carajuelo soldado",
        "6791": "Carbonero metálico",
        "6792": "Cardenal",
        "6793": "Cardenal bronceado",
        "6794": "Cardenal cincho",
        "6795": "Cardenal colimanchada",
        "6796": "Cardenal colirrayada",
        "6797": "Cardenal del cobo",
        "6798": "Cardenal dientón",
        "6799": "Cardenal dos puntos",
        "6800": "Cardenal esponjero",
        "6801": "Cardenal estrella blanca",
        "6802": "Cardenal frenado",
        "6803": "Cardenal manchado",
        "6804": "Cardenal mimético",
        "6805": "Cardenal morro listado",
        "6806": "Cardenal pálido",
        "6807": "Cardenal pálido",
        "6808": "Cardenal prieto",
        "6809": "Cardenal punteado",
        "6810": "Cardenal rayado",
        "6811": "Cardenal Rosado o Rojo",
        "6812": "Cardinal soldierfish",
        "6813": "Carduma",
        "6814": "Carecaballo, reloj",
        "6815": "Caribbaean seahorse.",
        "6816": "Caribbaean seahorse.",
        "6817": "Caribbean moonfish",
        "6818": "Caribbean pipefish",
        "6819": "Caribbean sharpnose-puffer",
        "6820": "Caribbean tonguefish",
        "6821": "Caribe, bonito",
        "6822": "Carite rayado",
        "6823": "Carrura",
        "6824": "Casabito",
        "6825": "Castañeta cola de tijera",
        "6826": "Catalina",
        "6827": "Catalufa roquera",
        "6828": "Catalufa roquera",
        "6829": "Catire, pintado, sierra",
        "6830": "Chabanaud's tonguefish",
        "6831": "Chabelita tricolor",
        "6832": "Chachás",
        "6833": "Chalk bass",
        "6834": "Chambo",
        "6835": "Cherna",
        "6836": "Cherna",
        "6837": "Cherna",
        "6838": "Cherna amarilla",
        "6839": "Cherna blanca",
        "6840": "Cherna bonaci",
        "6841": "Cherna criolla",
        "6842": "Cherna de piedra",
        "6843": "Cherna garopa",
        "6844": "Cherna gato",
        "6845": "Cherna negra",
        "6846": "Cherna rabimanchado.",
        "6847": "Chicharo garretón",
        "6848": "Chicharro ojón",
        "6849": "Chicuaca",
        "6850": "Chile arpón",
        "6851": "Chile cadena",
        "6852": "Chile lagarto",
        "6853": "Chivo barbón",
        "6854": "Chivo cabezón",
        "6855": "Chivo de mar",
        "6856": "Chivo mapalé",
        "6857": "Chivo mozo",
        "6858": "Choca pintadita",
        "6859": "Chocolate tonguefish",
        "6860": "Chonga",
        "6861": "Chopa amarilla",
        "6862": "Chopa isleña",
        "6863": "Chopita arrecifal",
        "6864": "Chopita azul",
        "6865": "Chopita manchada",
        "6866": "Chopita parda",
        "6867": "Chopita raboamarillo",
        "6868": "Choque",
        "6869": "Chuhueco colorada",
        "6870": "Chuhueco ojito",
        "6871": "Chupador, ventosa",
        "6872": "Chupapiedra",
        "6873": "Chupapiedra",
        "6874": "Chupapiedra acojinada",
        "6875": "Chupapiedra de barras",
        "6876": "Chupapiedra panámica",
        "6877": "Chupapiedra papilosa",
        "6878": "Chupapiedra roja",
        "6879": "Cintilla",
        "6880": "Cintilla del Atlántico",
        "6881": "Cirujano pardo",
        "6882": "Clark¿s brotula.",
        "6883": "Clown wrasse",
        "6884": "Cobia, bacalao",
        "6885": "Cochino",
        "6886": "Cocoa damselfish",
        "6887": "Cocodrilo",
        "6888": "Cocuyo",
        "6889": "Cofre liso",
        "6890": "Cofre pintado",
        "6891": "Cojinúa amarilla",
        "6892": "Cojinúa azul",
        "6893": "Cojinua negra",
        "6894": "Cola de maguey",
        "6895": "Colombian blenny",
        "6896": "Colombian bocachico",
        "6897": "Colombian goby",
        "6898": "Colon goby",
        "6899": "Comal",
        "6900": "Comepán basto",
        "6901": "Common Atlantic grenadier",
        "6902": "Common halfbeak",
        "6903": "Common rêmora",
        "6904": "Common snook",
        "6905": "Conejo",
        "6906": "Conejo",
        "6907": "Congrillo amarillo",
        "6908": "Congrillo de charco",
        "6909": "Congrio barbudo rosado",
        "6910": "Congrio cabeza corta",
        "6911": "Congrio cabezón",
        "6912": "Congrio cola de bordes",
        "6913": "Congrio con escamas",
        "6914": "Congrio de Cortés",
        "6915": "Congrio del talud",
        "6916": "Congrio dentudo",
        "6917": "Congrio disparatado",
        "6918": "Congrio gigante, zafiro",
        "6919": "Congrio narigón",
        "6920": "Congrio nariz colgada",
        "6921": "Congrio plumilla",
        "6922": "Congriperla auricular",
        "6923": "Congriperla cornuda",
        "6924": "Congriperla cornuda",
        "6925": "Congriperla parda",
        "6926": "Coral hawkfish",
        "6927": "Coral scorpionfish",
        "6928": "Corneta azul",
        "6929": "Corneta roja",
        "6930": "Corocoro congo",
        "6931": "Corocoro gris",
        "6932": "Corocoro perro",
        "6933": "Coroncoro perro",
        "6934": "Corrotucho común",
        "6935": "Corrotucho listado",
        "6936": "Cortez rainbow wrasse",
        "6937": "Corvina de altura",
        "6938": "Corvina de roca",
        "6939": "Corvina enana de altamar",
        "6940": "Corvina granadina",
        "6941": "Corvina ojo chico",
        "6942": "Corvina ojona",
        "6943": "Corvina rayada",
        "6944": "Corvinata cambucú",
        "6945": "Corvinata dorada",
        "6946": "Corvineta alacorta",
        "6947": "Corvineta amarilla",
        "6948": "Corvineta ángel",
        "6949": "Corvineta bizca",
        "6950": "Corvineta blanca",
        "6951": "Corvineta blanca",
        "6952": "Corvineta cambucú",
        "6953": "Corvineta corredora",
        "6954": "Corvineta lisa",
        "6955": "Corvineta marota",
        "6956": "Corvineta ojiamarillo",
        "6957": "Corvineta real",
        "6958": "Corvineta salmón",
        "6959": "Corvineta tonquicha",
        "6960": "Corvinilla amarilla",
        "6961": "Corvinilla baileta",
        "6962": "Corvinilla de fondo",
        "6963": "Corvinilla lisa",
        "6964": "Corvinilla lucía",
        "6965": "Corvinilla punteada",
        "6966": "Corvinilla sureña",
        "6967": "Corvinilla venezolana",
        "6968": "Corvinón rayado",
        "6969": "Crested goby",
        "6970": "Crested pipefish",
        "6971": "Crocodile fish",
        "6972": "Cubera snapper",
        "6973": "Cuchillo de lunar",
        "6974": "Culona amarillenta",
        "6975": "Culona chica",
        "6976": "Cuna lucero",
        "6977": "Curved sweeper",
        "6978": "Daisy midshipman",
        "6979": "Damsel bass",
        "6980": "Dara bandera",
        "6981": "Darwin's slimehead",
        "6982": "Deepreef scorpionfish",
        "6983": "Deepwater scorpionfish",
        "6984": "Deepwater tonguefish",
        "6985": "Derivante fragata portuguesa",
        "6986": "Dientón",
        "6987": "Diphasic goby",
        "6988": "Domingo",
        "6989": "Doncella",
        "6990": "Doncella arcoiris",
        "6991": "Doncella cintaverde",
        "6992": "Doncella de pluma",
        "6993": "Doncella enana",
        "6994": "Doncella payaso",
        "6995": "Doradilla",
        "6996": "Dorado",
        "6997": "Dorado chato.",
        "6998": "Dorado chico",
        "6999": "Dorado común",
        "7000": "Dormilón oscuro",
        "7001": "Dormilona, mojarra peña",
        "7002": "Dove's longfin herring",
        "7003": "Dow's toadfish",
        "7004": "Dragoncillo coralino",
        "7005": "Dragoncillo moteado",
        "7006": "Dusky blenny",
        "7007": "Dusky cardinalfish",
        "7008": "Dusky cusk-eel",
        "7009": "Dusky damselfish",
        "7010": "Dusky jawfish",
        "7011": "Dusky squirrelfish",
        "7012": "Dwarf frogfish",
        "7013": "Dwarf goby",
        "7014": "Dwarf weakfish",
        "7015": "Dwarf wrasse",
        "7016": "Earspot cusk eel",
        "7017": "Eelgrass blenny",
        "7018": "Elongate tonguefish",
        "7019": "Elongated bristlemouth fish",
        "7020": "Emerald clingfish",
        "7021": "Equatorial longfin herring",
        "7022": "Erizo",
        "7023": "Escolar",
        "7024": "Escolar americano",
        "7025": "Escolar rayado",
        "7026": "Escorpión aleta larga",
        "7027": "Escorpión coralino",
        "7028": "Escorpión de arrecife",
        "7029": "Escorpión espinoso",
        "7030": "Escorpión hongo",
        "7031": "Escorpión mejilla lisa",
        "7032": "Escorpión negro",
        "7033": "Escorpión pelón",
        "7034": "Escorpión plumeado",
        "7035": "Escorpión roquero",
        "7036": "Escorpión sapo",
        "7037": "Esmeralda",
        "7038": "Espátula larga",
        "7039": "False papillose blenny",
        "7040": "Farolito cachetiquillada",
        "7041": "Farolito tres espinas",
        "7042": "Feliciano, Boquiparriba",
        "7043": "Flamefish",
        "7044": "Flapnose conger",
        "7045": "Flat needlefish",
        "7046": "Flathead mullet",
        "7047": "Flying halfbeak",
        "7048": "Fourspot flounder",
        "7049": "Freshwater gregory",
        "7050": "Gadela imberbe",
        "7051": "Gafftopsail pompano",
        "7052": "Galapagos blue-banded goby",
        "7053": "Galito-rosado",
        "7054": "Gallina",
        "7055": "Gallina ala corta",
        "7056": "Gallina azul",
        "7057": "Gallina cornúa",
        "7058": "Gallina rosada",
        "7059": "Gallina.",
        "7060": "Gallinazo",
        "7061": "Gallineta",
        "7062": "gallineta",
        "7063": "Gallineta alilarga ojona",
        "7064": "Gallineta alilarga picuda",
        "7065": "Gallineta chica",
        "7066": "Gallineta erizada",
        "7067": "Gallinita",
        "7068": "Gar characin",
        "7069": "Ginsburg's tonguefish",
        "7070": "Glass goby",
        "7071": "Glasseye",
        "7072": "Glasseye",
        "7073": "Glasshead grenadier",
        "7074": "Glassy sweeper",
        "7075": "Góbido crystal",
        "7076": "Góbido de esponja",
        "7077": "Góbido lista amarilla",
        "7078": "Gobio",
        "7079": "Gobio",
        "7080": "Gobio aleta manchada",
        "7081": "Gobio bandeado",
        "7082": "Gobio barreado",
        "7083": "Gobio bonito",
        "7084": "Gobio boquita",
        "7085": "Gobio camaronícola",
        "7086": "Gobio cebra",
        "7087": "Gobio coquetón",
        "7088": "Gobio cristal",
        "7089": "Gobio de riendas",
        "7090": "Gobio dos puntos",
        "7091": "Gobio enano",
        "7092": "Gobio escamoso",
        "7093": "Gobio espes",
        "7094": "Gobio espina alta",
        "7095": "Gobio farol",
        "7096": "Gobio gallo",
        "7097": "Gobio insular",
        "7098": "Gobio jaspeado",
        "7099": "Gobio ligero",
        "7100": "Gobio listado",
        "7101": "Gobio mapache",
        "7102": "Gobio marcado",
        "7103": "Gobio nueve rayas",
        "7104": "Gobio oxidado",
        "7105": "Gobio pálido",
        "7106": "Gobio puntadorada",
        "7107": "Gobio rayado",
        "7108": "Gobio seductor",
        "7109": "Gobio señal",
        "7110": "Gobio Severo",
        "7111": "Gobio tiburoncito",
        "7112": "Gobio violeta",
        "7113": "Goby flathead",
        "7114": "Gold brotula",
        "7115": "Golden mojarra",
        "7116": "Goldentail moray",
        "7117": "Goldentail moray",
        "7118": "Goldline blenny",
        "7119": "Goldspot goby",
        "7120": "Goode croaker",
        "7121": "Gorgeous goby",
        "7122": "Gorgonian tonguefish",
        "7123": "Gourami piel de culebra",
        "7124": "Granadero",
        "7125": "Granadero carapacho",
        "7126": "Granadero caribeño",
        "7127": "Granadero tristón",
        "7128": "Green Jack",
        "7129": "Green razorfish",
        "7130": "Green weakfish",
        "7131": "Greenband wrasse",
        "7132": "Greenbanded goby",
        "7133": "Grey grunt",
        "7134": "Guabina",
        "7135": "Guabina amarilla",
        "7136": "Guanábana",
        "7137": "Guanábana caribeña",
        "7138": "Guaripete",
        "7139": "Guasa bocona, sapo",
        "7140": "Guasa lagunar",
        "7141": "Guasa sureña, sapo",
        "7142": "Guaseta",
        "7143": "Guaseta del Pacifico",
        "7144": "Guato",
        "7145": "Günther's boafish",
        "7146": "Gurrubata",
        "7147": "Hairy blenny",
        "7148": "Hakeling",
        "7149": "Halcón de coral",
        "7150": "Halcón rayadito",
        "7151": "Hancock's blenny",
        "7152": "Harlequin bass",
        "7153": "Highfin goby",
        "7154": "Highfin scorpionfish",
        "7155": "Highspine goby",
        "7156": "Honeycomb cowfish",
        "7157": "Horned whiff",
        "7158": "Horqueta del Pacífico",
        "7159": "Hourglass moray",
        "7160": "Huachinango del Pacífico",
        "7161": "Huavina alilarga.",
        "7162": "Inotted lizardfish",
        "7163": "Inshore sand perch",
        "7164": "Isabelita",
        "7165": "Isabelita",
        "7166": "isabelita",
        "7167": "Isabelita blanca",
        "7168": "Isabelita de piedra",
        "7169": "Isabelita enana",
        "7170": "Isabelita hocicona",
        "7171": "Isabelita mariposa",
        "7172": "Isabelita medioluto",
        "7173": "Isabelita negra",
        "7174": "Isabelita paguara",
        "7175": "Isabelita pintona",
        "7176": "Island frogfish",
        "7177": "Island goby",
        "7178": "Isthmian goby",
        "7179": "Itajara",
        "7180": "Jaboncillo arrecifal",
        "7181": "Jabonero",
        "7182": "Jabonero",
        "7183": "Jabonero doble punteado",
        "7184": "Jabonero punteado",
        "7185": "Jaqueta Acapulco",
        "7186": "Jaqueta bicolor",
        "7187": "Jaqueta castaña",
        "7188": "Jaqueta de tres puntos",
        "7189": "Jaqueta miel",
        "7190": "Jaqueta miel",
        "7191": "Jaqueta prieta",
        "7192": "Jewel moray",
        "7193": "Jewfish",
        "7194": "Jordan's snapper",
        "7195": "Jorobado espejo",
        "7196": "Jorobado lamparosa",
        "7197": "Jorobado luna",
        "7198": "Jorobado papelillo",
        "7199": "Jorobado penacho",
        "7200": "Juancho juancho",
        "7201": "Jurel bonito",
        "7202": "Jurel común",
        "7203": "Jurel fino",
        "7204": "Jurel negro",
        "7205": "Jurel ojón",
        "7206": "Jurel toro",
        "7207": "Jurel volantín",
        "7208": "Keelcheek bass",
        "7209": "Key brotula",
        "7210": "Key worm eel",
        "7211": "King angelfish",
        "7212": "King of herrings",
        "7213": "Lagarto azul",
        "7214": "Lagarto costero",
        "7215": "Lagarto dientón",
        "7216": "Lagarto iguana",
        "7217": "Lagarto manuelito",
        "7218": "Lagarto oceánico",
        "7219": "Lagarto rojo",
        "7220": "Lambe aletón",
        "7221": "Lambe caletero",
        "7222": "Lambe ratón",
        "7223": "Lambe rayado",
        "7224": "Lambe verrugato",
        "7225": "Lancer dragonet",
        "7226": "Lantern bass",
        "7227": "Lapón aleta baja",
        "7228": "Lapón mariposa",
        "7229": "Large-banded blenny",
        "7230": "Large-scaled lantern fish",
        "7231": "Large-scaled spinycheek sleepe",
        "7232": "Largespotted soapfish",
        "7233": "Lebranche",
        "7234": "Lechero",
        "7235": "Lechuza",
        "7236": "Lechuza.",
        "7237": "Lechuza.",
        "7238": "Lee's tonguefish",
        "7239": "Lengua cachete prieto",
        "7240": "Lengua caribeña",
        "7241": "Lengua ceniza",
        "7242": "Lengua chocolate",
        "7243": "Lengua coliblanca",
        "7244": "Lengua colilarga",
        "7245": "Lengua colinegra",
        "7246": "Lengua enana",
        "7247": "Lengua esbelta",
        "7248": "Lengua filo negro",
        "7249": "Lengua perezosa",
        "7250": "Lengua pigmea",
        "7251": "Lengua rayada",
        "7252": "Lenguada",
        "7253": "Lenguado",
        "7254": "Lenguado",
        "7255": "Lenguado",
        "7256": "Lenguado",
        "7257": "Lenguado",
        "7258": "Lenguado amarillo",
        "7259": "Lenguado carbón",
        "7260": "Lenguado colimanchada",
        "7261": "Lenguado cornudo",
        "7262": "Lenguado cuatrojos",
        "7263": "Lenguado de charco",
        "7264": "Lenguado de profundidad",
        "7265": "Lenguado dientón",
        "7266": "Lenguado espinoso",
        "7267": "Lenguado frentón",
        "7268": "Lenguado manchado",
        "7269": "Lenguado ojicornudo",
        "7270": "Lenguado ovalado",
        "7271": "Lenguado pecoso",
        "7272": "Lenguado perro",
        "7273": "Lenguado playero",
        "7274": "Lenguado tapadera",
        "7275": "Lenguado tres manchas",
        "7276": "Leopard goby",
        "7277": "Lichen moray",
        "7278": "Liguija",
        "7279": "Lija amarilla",
        "7280": "Lija áspera",
        "7281": "Lija de clavo",
        "7282": "Lija de hebra",
        "7283": "Lija pintada",
        "7284": "Lija reticulada",
        "7285": "Limpiador azul",
        "7286": "Limpiador lista amarilla",
        "7287": "Lionfish, Ornate butterfly-cod",
        "7288": "Lisa",
        "7289": "Lisa amarilla",
        "7290": "Lisa blanca",
        "7291": "Lisa criolla",
        "7292": "Lisa hospe",
        "7293": "Lisa ojo amarillo",
        "7294": "Lisa rayada",
        "7295": "Lisa saltona",
        "7296": "Little anchovy",
        "7297": "Live sharksucker",
        "7298": "Liza",
        "7299": "Lofty triplefin",
        "7300": "Longfin blenny",
        "7301": "Longfin damselfish",
        "7302": "Longfin damselfish",
        "7303": "Longfin scorpionfish",
        "7304": "Longjaw leatherjack",
        "7305": "Longlure frogfish",
        "7306": "Longnose batfish",
        "7307": "Longnose greeneye",
        "7308": "Longnose puffer",
        "7309": "long-ray searobin",
        "7310": "Longsnout butterflyfish",
        "7311": "Longspine scorpionfish",
        "7312": "Longtail knifefish",
        "7313": "Longtail tonguefish",
        "7314": "Longtailed jawfish",
        "7315": "Lora",
        "7316": "Loreto",
        "7317": "Lorito arcoiris",
        "7318": "Lorito bandazul",
        "7319": "Lorito colombiano",
        "7320": "Lorito cuchilla",
        "7321": "Lorito lomonegro",
        "7322": "Lorito morado",
        "7323": "Lorito orejinegro",
        "7324": "Lorito resbaloso",
        "7325": "Lorito rubio",
        "7326": "Loro azul",
        "7327": "Loro basto",
        "7328": "Loro coliamarilla",
        "7329": "Loro de profundidad",
        "7330": "Loro dientón",
        "7331": "Loro enano",
        "7332": "Loro guacamayo",
        "7333": "Loro jabonero",
        "7334": "Loro listado",
        "7335": "Loro manchado",
        "7336": "Loro negro",
        "7337": "Loro perico",
        "7338": "Loro rabirojo",
        "7339": "Loro rayado",
        "7340": "Loro verde",
        "7341": "Lost pipefish",
        "7342": "Luciérnaga amarilla",
        "7343": "Luciérnaga dentada",
        "7344": "Luminous hake",
        "7345": "Lumptail searobin",
        "7346": "Lyre goby",
        "7347": "Macabí",
        "7348": "Macabí de hebra",
        "7349": "Macarela",
        "7350": "Macarela caballa",
        "7351": "Machete del Atlántico",
        "7352": "Machuelo",
        "7353": "Mackerel scad",
        "7354": "Madrejuile",
        "7355": "Madrejuile flecha",
        "7356": "Mahogany goby",
        "7357": "Malarmados",
        "7358": "Mangrove blenny",
        "7359": "Mangrove molly",
        "7360": "Man-of-war fish",
        "7361": "Mantequero",
        "7362": "Manuelito",
        "7363": "Manytooth conger",
        "7364": "Mapale",
        "7365": "Mapo aguado",
        "7366": "Marbled blenny",
        "7367": "Marbled grouper",
        "7368": "Marbled swamp eel",
        "7369": "Marble-toothed snake-eel",
        "7370": "Margined snake eel",
        "7371": "Margined tonguefish",
        "7372": "Margintail conger",
        "7373": "Mariposa barbero",
        "7374": "Mariposa muñeca",
        "7375": "Mariposa narigona",
        "7376": "Marked goby",
        "7377": "Marlin",
        "7378": "Masked goby",
        "7379": "Masked hamlet",
        "7380": "Matajuelo",
        "7381": "Mazatlan sole",
        "7382": "Medregal amarillo",
        "7383": "Medregal guaimeque",
        "7384": "Medregal limón",
        "7385": "Medregal listado",
        "7386": "Merluza",
        "7387": "Merluza panameña",
        "7388": "Mero",
        "7389": "Mero barril",
        "7390": "Mero cabrilla",
        "7391": "Mero colorado",
        "7392": "Mero criollo",
        "7393": "Mero guasa",
        "7394": "Mero guasa.",
        "7395": "Mero Guato",
        "7396": "Mero listado",
        "7397": "Mero mármol",
        "7398": "Mero naricita",
        "7399": "Mero ojón",
        "7400": "Mero rojo",
        "7401": "Mexican barracuda",
        "7402": "Mexican goatfish",
        "7403": "Mexican sand perch",
        "7404": "Mimic blenny",
        "7405": "Mimic cardinalfish",
        "7406": "Mina, ojón",
        "7407": "Miraestrellas",
        "7408": "Miraestrellas espinoso",
        "7409": "Miraestrellas ojilargo",
        "7410": "Mirón barreado",
        "7411": "Mirón ojicorto",
        "7412": "Mojarra",
        "7413": "Mojarra",
        "7414": "Mojarra alargada",
        "7415": "Mojarra aletas amarillas",
        "7416": "Mojarra azul",
        "7417": "Mojarra blanca",
        "7418": "Mojarra cabucha",
        "7419": "Mojarra caitipa",
        "7420": "Mojarra de ley",
        "7421": "Mojarra de marea",
        "7422": "Mojarra española",
        "7423": "Mojarra ojona",
        "7424": "Mojarra palometa",
        "7425": "Mojarra peña",
        "7426": "Mojarra picona",
        "7427": "Mojarra rayada",
        "7428": "Mojarra rayada",
        "7429": "Mollera luminosa",
        "7430": "Moncholo",
        "7431": "Morena amarilla",
        "7432": "Morena clepsidra",
        "7433": "Morena cola dorada",
        "7434": "Morena cola dorada",
        "7435": "Morena cola pintada",
        "7436": "Morena enana",
        "7437": "Morena liquen",
        "7438": "Morena manglera",
        "7439": "Morena negra",
        "7440": "Morena ondulada",
        "7441": "Morena pinta",
        "7442": "Morena pintada",
        "7443": "Morena polígona",
        "7444": "Morena punteada",
        "7445": "Morena rabirrubia",
        "7446": "Morena rayada",
        "7447": "Morena robusta",
        "7448": "Morena rubicunda",
        "7449": "Morena tigre",
        "7450": "Morena verde",
        "7451": "Morenita",
        "7452": "Moth blenny",
        "7453": "Mottled scorpionfish",
        "7454": "Muchachita",
        "7455": "Murciélago",
        "7456": "Murciélago",
        "7457": "Murciélago",
        "7458": "Murciélago",
        "7459": "Murciélago",
        "7460": "Murciélago ñato",
        "7461": "Murciélago picudo",
        "7462": "Murciélago tres cuernos",
        "7463": "Murico",
        "7464": "Mushroom scorpionfish",
        "7465": "Mutton snapper",
        "7466": "Muzzled blenny",
        "7467": "Naked belly searobin",
        "7468": "Naked-belly searobin",
        "7469": "Nassau grouper",
        "7470": "Navajero azul",
        "7471": "Navajero cirujano",
        "7472": "Navajero común",
        "7473": "Network sole",
        "7474": "Nineline goby",
        "7475": "North American naked sole",
        "7476": "Northern puffer",
        "7477": "Obispo",
        "7478": "Obispo barreado",
        "7479": "Obispo coronado",
        "7480": "Obispo estrellado",
        "7481": "Ocean surgeon",
        "7482": "Oceanic puffer",
        "7483": "Ocellate duckbill",
        "7484": "Ocellated pipefish",
        "7485": "Ochavo redondo",
        "7486": "Offshore hake",
        "7487": "Ojiverde chato",
        "7488": "Ojiverde truculento",
        "7489": "Ojón tira al sol",
        "7490": "Ojona de canto",
        "7491": "Ojona roca",
        "7492": "Ojona toro",
        "7493": "Okapi goby",
        "7494": "Oldwife",
        "7495": "Olomina",
        "7496": "Orange spotted sand perch",
        "7497": "Orangeback bass",
        "7498": "Orangespotted blenny",
        "7499": "Orangespotted goby",
        "7500": "Ornamented flyingfish",
        "7501": "Oval flounder",
        "7502": "Oyster blenny",
        "7503": "Pacific anchoveta",
        "7504": "Pacific bearded brotula",
        "7505": "Pacific bumper",
        "7506": "Pacific crevalle jack",
        "7507": "Pacific deepwater flounder",
        "7508": "Pacific flatiron herring",
        "7509": "Pacific goosefish",
        "7510": "Pacific lined sole",
        "7511": "Pacific moonfish",
        "7512": "Pacific mutton hamlet",
        "7513": "Pacific red snapper",
        "7514": "Pacific seahorse.",
        "7515": "Pacific sierra",
        "7516": "Pacific spadefish",
        "7517": "Pacific spotted scorpionfish",
        "7518": "Pacora",
        "7519": "paguara",
        "7520": "Pale cardinalfish",
        "7521": "Pale green eel",
        "7522": "Paleback goby",
        "7523": "Palehead blenny",
        "7524": "Palito",
        "7525": "Pallid goby",
        "7526": "Palometa",
        "7527": "Palometa ´pintada",
        "7528": "Palometa lucia",
        "7529": "Palometa ojona",
        "7530": "Palometa salema",
        "7531": "Pampano",
        "7532": "Pámpano amarillo",
        "7533": "Pámpano de bandera",
        "7534": "Pámpano de hebra",
        "7535": "Pámpano fino",
        "7536": "Pámpano listado",
        "7537": "Pámpano, pómpano",
        "7538": "Panama anchovy",
        "7539": "Panama hake",
        "7540": "Panama Kingcroaker",
        "7541": "Panamic clingfish",
        "7542": "Panamic sergeant major",
        "7543": "Panamic soldierfish",
        "7544": "Pancake batfish",
        "7545": "Panchito alargado",
        "7546": "Panchito ojón",
        "7547": "Panchito voraz",
        "7548": "Pargo",
        "7549": "Pargo",
        "7550": "Pargo amarillo",
        "7551": "Pargo azul-dorado",
        "7552": "Pargo cachucho",
        "7553": "Pargo camaronero",
        "7554": "Pargo cebal",
        "7555": "Pargo cebao",
        "7556": "Pargo chino",
        "7557": "Pargo colmillón",
        "7558": "Pargo criollo",
        "7559": "Pargo cubera",
        "7560": "Pargo de altura",
        "7561": "Pargo dientón",
        "7562": "Pargo lunarejo",
        "7563": "Pargo manchado",
        "7564": "Pargo mulato",
        "7565": "Pargo ojo amarillo",
        "7566": "Pargo ojón",
        "7567": "Pargo palmero",
        "7568": "Pargo perro",
        "7569": "Pargo pluma",
        "7570": "Pargo prieto",
        "7571": "Pargo rojo",
        "7572": "Pargo sesí",
        "7573": "Pargo tabardillo",
        "7574": "Parguito",
        "7575": "Patudo",
        "7576": "Pearl blenny",
        "7577": "Pearlfish",
        "7578": "Pearlsides",
        "7579": "Peje murciélago",
        "7580": "Peje piloto",
        "7581": "Pejegato bocón",
        "7582": "Pejegato común",
        "7583": "Pejepuerco",
        "7584": "Pejepuerco, peñolero",
        "7585": "Pejesapo",
        "7586": "Pejesapo",
        "7587": "Pejesapo",
        "7588": "Pelada corvina",
        "7589": "Pemferis colinegro",
        "7590": "Peppermint bass",
        "7591": "Peppermint goby",
        "7592": "Pequita esmerilada",
        "7593": "Perla gris",
        "7594": "Perla manchada",
        "7595": "Perla marrón",
        "7596": "Perla oscura",
        "7597": "Perlero del Atlántico",
        "7598": "Peruvian mojarra",
        "7599": "Peruvian scorpionfish",
        "7600": "Petaca banderita",
        "7601": "Peters' banded croaker",
        "7602": "Peto",
        "7603": "Petota",
        "7604": "Pez cocodrilo",
        "7605": "Pez cofre",
        "7606": "Pez conejo",
        "7607": "Pez de pluma",
        "7608": "Pez de San Francisco",
        "7609": "Pez de sargazo",
        "7610": "Pez diablo",
        "7611": "Pez Diablo Narizón",
        "7612": "Pez erizo",
        "7613": "Pez erizo, pez globo",
        "7614": "Pez espada",
        "7615": "Pez gato",
        "7616": "Pez hacha",
        "7617": "Pez hacha aquillado",
        "7618": "Pez joya",
        "7619": "Pez león rojo, Pez león turco",
        "7620": "Pez luna",
        "7621": "Pez murcielago de dos ocelos",
        "7622": "Pez perro",
        "7623": "Pez pina ñato",
        "7624": "Pez pipa",
        "7625": "Pez pipa aletilla",
        "7626": "Pez pipa caribeño",
        "7627": "Pez pipa crestado",
        "7628": "Pez pipa culebra",
        "7629": "Pez pipa gusano",
        "7630": "Pez pipa hocico blanco",
        "7631": "Pez roca",
        "7632": "Pez sapo.",
        "7633": "Pez trompeta",
        "7634": "Pez vela",
        "7635": "Pez volador",
        "7636": "Pico de pato",
        "7637": "Picuda guaguanche",
        "7638": "Picudilla",
        "7639": "Pierna",
        "7640": "Pink brotula",
        "7641": "Pink cardinalfish",
        "7642": "Pink cusk eel",
        "7643": "Pink frogmouth",
        "7644": "Pintona",
        "7645": "Piña bocona",
        "7646": "Pipa",
        "7647": "Pipón",
        "7648": "Pipón",
        "7649": "Piponcita",
        "7650": "Planehead filefish",
        "7651": "Player scorpionfish",
        "7652": "Pluma bajonado",
        "7653": "Pluma cachicato",
        "7654": "Pluma cálamo",
        "7655": "Pluma del Caribe",
        "7656": "Plumed scorpionfish",
        "7657": "Plumilla",
        "7658": "Plumita",
        "7659": "Point-nosed croaker",
        "7660": "Polla",
        "7661": "Polygon moray",
        "7662": "Pompano dolphinfish",
        "7663": "Pond perch",
        "7664": "Prowspine cusk eel",
        "7665": "Prowspine cusk-eel",
        "7666": "Puffcheek blenny",
        "7667": "Pugjaw wormfish",
        "7668": "Pugnose pipefish",
        "7669": "Punctuated snake-eel",
        "7670": "Purplemouth moray",
        "7671": "Purplemouthed conger",
        "7672": "Pygmy goby",
        "7673": "Pygmy sea bass",
        "7674": "Pygmy tonguefish",
        "7675": "Rabilargo",
        "7676": "Rainbow wrasse",
        "7677": "Ranisapo ceboso",
        "7678": "Ranisapo enano",
        "7679": "Ranisapo estriado",
        "7680": "Ranisapo ocelado.",
        "7681": "Ranisapo uniocelado",
        "7682": "Rape de rabo manchado",
        "7683": "Rape hocicón",
        "7684": "Rape pescador",
        "7685": "Rascacio acanalado",
        "7686": "rascacio escorpión",
        "7687": "Rascacio jugador",
        "7688": "Rascacio manchado",
        "7689": "Rascacio negro",
        "7690": "Rascacio orejón",
        "7691": "Rascacio párlamo",
        "7692": "Rascacio richichi",
        "7693": "Rascacio rojo",
        "7694": "Rascacio sapo",
        "7695": "Rascapalo, raspabalsa",
        "7696": "Raton",
        "7697": "Ratón",
        "7698": "Ratona",
        "7699": "Ravijunco",
        "7700": "Ravijunco, rabilargo",
        "7701": "Raya piel de lija",
        "7702": "Red barbier",
        "7703": "Red clingfish",
        "7704": "Red scorpionfish",
        "7705": "reddish scorpionfish",
        "7706": "Redeye triplefin",
        "7707": "Redface moray",
        "7708": "Redfin brotula",
        "7709": "Redfin needlefish",
        "7710": "Redfin parrotfish",
        "7711": "Redhump eartheater",
        "7712": "Redspotted hawkfish",
        "7713": "Reef bass",
        "7714": "Reef croaker",
        "7715": "Reef scorpionfish",
        "7716": "Reef squirrelfish",
        "7717": "Reloj de Darwin",
        "7718": "Rémora",
        "7719": "Rémora",
        "7720": "Rémora alargada",
        "7721": "Rémora aletiblanca",
        "7722": "Rémora blanca",
        "7723": "Rémora común",
        "7724": "Rémora rayada",
        "7725": "Reticulated goosefish",
        "7726": "Reticulated tilefish",
        "7727": "Rey de los arenques",
        "7728": "Ribbon halfbeak",
        "7729": "Rimspine searobin",
        "7730": "Robalo",
        "7731": "Robalo baileta",
        "7732": "Robalo blanco",
        "7733": "Robalo carita blanca",
        "7734": "Robalo chucumite",
        "7735": "Robalo congo",
        "7736": "Robalo mexicano",
        "7737": "Rock beauty",
        "7738": "Ronco",
        "7739": "Ronco amarillo",
        "7740": "Ronco basto",
        "7741": "Ronco blanco",
        "7742": "Ronco boquichica",
        "7743": "Ronco boquilla",
        "7744": "Ronco bravo",
        "7745": "Ronco caco",
        "7746": "Ronco canario",
        "7747": "Ronco carbonero",
        "7748": "Ronco catire",
        "7749": "Ronco coño",
        "7750": "Ronco listado",
        "7751": "Ronco mapurite",
        "7752": "Ronco plateado",
        "7753": "Ronco prieto",
        "7754": "Ronco ruyi",
        "7755": "Roosevelt's goby",
        "7756": "Rosy blenny",
        "7757": "Rosy dory",
        "7758": "Roudi escolar",
        "7759": "Roughbar frogfish.",
        "7760": "Roughhead blenny",
        "7761": "Roughhead triplefin",
        "7762": "Roughlip cardinalfish",
        "7763": "Roughtail goby",
        "7764": "Roughtongue bass",
        "7765": "Round herring",
        "7766": "Royal gramma",
        "7767": "Rubio aleticorta",
        "7768": "rubio cabro",
        "7769": "Rubio gallardete",
        "7770": "rubio jaquita",
        "7771": "Rubio polla",
        "7772": "Rubio rey",
        "7773": "rubio tico",
        "7774": "rubio volador",
        "7775": "Rubio volador.",
        "7776": "Rusty goby",
        "7777": "Sábalo",
        "7778": "Sable",
        "7779": "Saddle stargazer",
        "7780": "Saddled blenny",
        "7781": "Saddled moray",
        "7782": "Safío amarillo",
        "7783": "Saga",
        "7784": "Salema butterfish",
        "7785": "Salmon de lo altso",
        "7786": "Salmón, macarela",
        "7787": "Salmonete amarillo",
        "7788": "Salmonete colorado",
        "7789": "Salmonete manchado",
        "7790": "Salmonete rayado",
        "7791": "Salmonete vanicolense",
        "7792": "Saltona",
        "7793": "San Pedro rojo",
        "7794": "Sand lizardfish",
        "7795": "Sand stargazer",
        "7796": "Sapito",
        "7797": "Sapito",
        "7798": "Sapito común",
        "7799": "Sapito de lalandi",
        "7800": "Sapito punteado",
        "7801": "Sapo",
        "7802": "Sapo cabezón",
        "7803": "Sapo caño",
        "7804": "Sapo de Dow",
        "7805": "Sapo de ojo chico",
        "7806": "Sapo del Pacifico",
        "7807": "Sapo futre",
        "7808": "Sapo lagunero",
        "7809": "Sapo margarita.",
        "7810": "Sapo sureño",
        "7811": "Sardina",
        "7812": "Sardina",
        "7813": "Sardina conchua",
        "7814": "Sardina cuchilla",
        "7815": "Sardina jaguana",
        "7816": "Sardina machete",
        "7817": "Sardina machete chata",
        "7818": "Sardina redonda",
        "7819": "Sardinata",
        "7820": "Sardineta",
        "7821": "Sardineta atlántica",
        "7822": "Sardineta canalera",
        "7823": "Sardineta escamuda",
        "7824": "Sardineta plumilla",
        "7825": "Sardinita plumilla",
        "7826": "Sargassum triggerfish",
        "7827": "Sargento mayor",
        "7828": "Sargo amarillo",
        "7829": "Sargo chopa",
        "7830": "Sargo fino",
        "7831": "Sargo piedra",
        "7832": "Sauro lizardfish",
        "7833": "Sawcheek cardinalfish",
        "7834": "Scaly boy",
        "7835": "Scissortail damselfish",
        "7836": "Scrawled sole",
        "7837": "Seaweed blenny",
        "7838": "Sechura lizardfish",
        "7839": "Secretary blenny",
        "7840": "Seminole goby",
        "7841": "Semiscaled goby",
        "7842": "Serrano arenero",
        "7843": "Serrano arlequín",
        "7844": "Serrano cabaicucho",
        "7845": "Serrano cabaicucho.",
        "7846": "Serrano carabonita",
        "7847": "Serrano de altura",
        "7848": "Serrano de arena",
        "7849": "Serrano extranjero",
        "7850": "Serrano guaseta",
        "7851": "Serrano lengua rasposa",
        "7852": "Serrano linterna",
        "7853": "Serrano mexicano",
        "7854": "Serrano naranja",
        "7855": "Serrano Nevado",
        "7856": "Serrano ojón",
        "7857": "Serrano ojón",
        "7858": "Serrano oreja negra",
        "7859": "Serrano pálido",
        "7860": "Serrano pigmeo",
        "7861": "Sharknose goby",
        "7862": "Shelf goby",
        "7863": "Shining drum",
        "7864": "Shortchin stargazer",
        "7865": "Shortfin molly",
        "7866": "Shortfin pipefish",
        "7867": "Shorthead lizardfish",
        "7868": "Shortnose greeneye",
        "7869": "Short-tailed pipefish",
        "7870": "Short-tube scorpionfish",
        "7871": "Shrimp eel",
        "7872": "Shy hamlet",
        "7873": "Sierra del Pacífico",
        "7874": "Sierra, carite",
        "7875": "Sietecueros caspín",
        "7876": "Sietecueros meona",
        "7877": "Sietecueros palometa",
        "7878": "Silio",
        "7879": "Silver hatchetfish",
        "7880": "Silver weakfish",
        "7881": "Singlespot frogfish",
        "7882": "Slender filefish",
        "7883": "Slender searobin",
        "7884": "Slender snipe eel",
        "7885": "Slendertail cardinalfish",
        "7886": "Slipper sole",
        "7887": "Sloane's viperfish",
        "7888": "Slope bass",
        "7889": "Slope conger",
        "7890": "Small sandab",
        "7891": "Smalleye midshipman",
        "7892": "Smallmouth grunt",
        "7893": "Smallscale weakfish",
        "7894": "Smooth stardrum",
        "7895": "Smooth-cheek scorpionfish",
        "7896": "Smooth-eye blenny",
        "7897": "Smooth-head scorpionfish",
        "7898": "Snow bass",
        "7899": "Soldado panámico",
        "7900": "Spaghetti eel",
        "7901": "Speckled blenny",
        "7902": "Speckled puffer",
        "7903": "Speckled sanddab",
        "7904": "Speckled worm eel",
        "7905": "Speckled-tail flounder",
        "7906": "Spicule anchovy",
        "7907": "Spined whiff",
        "7908": "Spinycheek sleeper",
        "7909": "Spinythroat scorpionfish",
        "7910": "Splitnose searobin",
        "7911": "Splittail bass",
        "7912": "Sponge cardinalfish",
        "7913": "Spotcheek blenny",
        "7914": "Spotcheek goby",
        "7915": "Spotfin dragonet",
        "7916": "Spotfin tonguefish",
        "7917": "Spotjaw blenny",
        "7918": "Spotlight goby",
        "7919": "Spotted batfish",
        "7920": "Spotted dragonet",
        "7921": "Spotted rose snapper",
        "7922": "Spotted scorpionfish",
        "7923": "Spotted sharpnosed puffer",
        "7924": "Spotted soapfish",
        "7925": "Spotted whiff",
        "7926": "Spottedfin sole",
        "7927": "Spottedtail anger",
        "7928": "Spotted-tail moray",
        "7929": "Squint-eyed croaker",
        "7930": "Stardrum",
        "7931": "Stareye lightfish",
        "7932": "Starry grouper",
        "7933": "Steep lined drum",
        "7934": "Steeplined drum",
        "7935": "Stellate codlet",
        "7936": "Stout beardfish",
        "7937": "Stout moray",
        "7938": "Streamer bass",
        "7939": "Striated frogfish",
        "7940": "Striped drum",
        "7941": "Striped escolar",
        "7942": "Striped mojarra",
        "7943": "Striped weakfish",
        "7944": "Stripefin brotula",
        "7945": "Suela carioca",
        "7946": "Suela chancleta",
        "7947": "Suela desnuda",
        "7948": "Suela desnuda",
        "7949": "Suela pintada",
        "7950": "Suela plomiza",
        "7951": "Suela reticulada",
        "7952": "Surf blenny",
        "7953": "Surf eel",
        "7954": "Suspicious lizardfish",
        "7955": "Swordtail jawfish",
        "7956": "Tailspot cardinalfish",
        "7957": "Tamboril norteño",
        "7958": "Tamborín narizón",
        "7959": "Tarpón",
        "7960": "Tessellated blenny",
        "7961": "Threadtail conger",
        "7962": "Threebanded butterflyfish",
        "7963": "Three-eye flounder",
        "7964": "Threeline basslet",
        "7965": "Threespine bass",
        "7966": "Threespot damselfish",
        "7967": "Tieso amarillo",
        "7968": "Tieso antillano",
        "7969": "Tieso camaronero",
        "7970": "Tieso chato",
        "7971": "Tieso de cayo",
        "7972": "Tieso dentón",
        "7973": "Tieso dientes romos",
        "7974": "Tieso dos rayas",
        "7975": "Tieso gusano",
        "7976": "Tieso manchado",
        "7977": "Tieso pintado",
        "7978": "Tieso punteado",
        "7979": "Tieso rabiagudo",
        "7980": "Tijera esbelta",
        "7981": "Tinícalo cabezón",
        "7982": "Tinícalo común",
        "7983": "Tinícalo de piedra",
        "7984": "Tinícalo lagunar",
        "7985": "Tinícalo playón",
        "7986": "Tira al sol",
        "7987": "Tismiche",
        "7988": "Tobaccofish",
        "7989": "Tolomba",
        "7990": "Toothed flounder",
        "7991": "Topote de manglar",
        "7992": "Topote del Atlántico",
        "7993": "Torito azul",
        "7994": "Torito hexagonal",
        "7995": "Torito panal",
        "7996": "Torpedo sand perch",
        "7997": "Traganavi",
        "7998": "Traganavi dorado",
        "7999": "Trahira",
        "8000": "Trambollito aletinegra",
        "8001": "Trambollito barra oscura",
        "8002": "Trambollito carinegro",
        "8003": "Trambollito de la resaca",
        "8004": "Trambollito ocelado",
        "8005": "Trambollito ojiliso",
        "8006": "Trambollo caripálida",
        "8007": "Trambollo ensillado",
        "8008": "Trambollo fumador",
        "8009": "Trambollo lineado",
        "8010": "Trambollo lunado",
        "8011": "Trambollo mímico",
        "8012": "Trambollo pardo",
        "8013": "Trambollo peludo",
        "8014": "Trambollo príncipe",
        "8015": "Trambollo rosado",
        "8016": "Trancanil, siete cueros",
        "8017": "Tras-tras",
        "8018": "Tres aletas aletón",
        "8019": "Tres aletas de barras",
        "8020": "Tres aletas orlado",
        "8021": "Tres aletas rugoso",
        "8022": "Tricorn batfish",
        "8023": "Tropical hatchetfish",
        "8024": "Tubícola afilado",
        "8025": "Tubícola aletón",
        "8026": "Tubícola anguila",
        "8027": "Tumba",
        "8028": "Turkey fish",
        "8029": "Tusked goby",
        "8030": "Twinhorn blenny",
        "8031": "Twinspot bass",
        "8032": "Twospot brotula",
        "8033": "Twospot cardinalfish",
        "8034": "Undulated moray",
        "8035": "Vaca bicolor",
        "8036": "Vaca de vientre amarillo",
        "8037": "Vaquita blindada flaca",
        "8038": "Venezuelan grouper",
        "8039": "Verdulón",
        "8040": "Vermiculated goby",
        "8041": "Verraco alto",
        "8042": "Verrugato corocoro",
        "8043": "Verrugato rayado",
        "8044": "Viejita",
        "8045": "Viejita manchada",
        "8046": "Violet goby",
        "8047": "Vípora marina",
        "8048": "Viscaina",
        "8049": "Vizcana",
        "8050": "Volador aleta negra",
        "8051": "Volador aletón",
        "8052": "Volador alinegra",
        "8053": "Volador bonito",
        "8054": "Volador bordiblanco",
        "8055": "Volador común",
        "8056": "Volador de fondo",
        "8057": "Volador espejo",
        "8058": "Volador golondrina",
        "8059": "Volador jorobado",
        "8060": "Volador manchado",
        "8061": "Volador ñato",
        "8062": "Volador verdadero",
        "8063": "Walker's anchovy",
        "8064": "Warteye stargazer",
        "8065": "Web burrfish",
        "8066": "Western softhead grenadier",
        "8067": "White mullet",
        "8068": "White suckerfish",
        "8069": "White-eye goby",
        "8070": "Whitenose pipefish",
        "8071": "whitesnout searobin",
        "8072": "Whitestar cardinalfish",
        "8073": "Wisner's lantern fish",
        "8074": "Yellow snake-eel",
        "8075": "Yellow snapper",
        "8076": "Yellowbelly hamlet",
        "8077": "Yelloweye croaker",
        "8078": "Yellowface pikeblenny",
        "8079": "Yellowfin goatfish",
        "8080": "Yellowhead jawfish",
        "8081": "Yellowline goby",
        "8082": "Zafiro",
        "8083": "Zanahoria.",
        "8084": "Zapatero sierrita",
        "8085": "Zebra fish, Fire fish",
        "8086": "Zebra goby",
        "8087": "Otro"    
        }
    df["vernacularName"] = df["NOMB_COMUN"].map(mapeo_nombre).fillna("")


    # Verificar si "TAMANO_cm" existe y tiene al menos un valor no vacío
    if "TAMANO_cm" in df.columns and df["TAMANO_cm"].dropna().empty is False:
        df["measurementValue"] = df["TAMANO_cm"]
        df["measurementType"] = "Tamaño promedio aproximado (de cola a pico en centímetros) de los individuos"
        df["measurementUnit"] = "cm"
    else:
        df["measurementValue"] = None
        df["measurementType"] = None
        df["measurementUnit"] = None
    
    # Mapear FECHA_MFL a verbatimEventDate
    if "FECHA" in df.columns:
        df["verbatimEventDate"] = df["FECHA"]
    else:

        df["verbatimEventDate"] = None


    # Calcular `eventDate`, `year`, `month`, `day` y conservamos `verbatimEventDate`
    if "verbatimEventDate" in df.columns:

        df["eventDate"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.strftime('%Y-%m-%d')
        df["year"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.year
        df["month"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.month
        df["day"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.day

    else:

        df["eventDate"] = None
        df["year"] = None
        df["month"] = None
        df["day"] = None

    
    if "DIRE_GRUPO" in df.columns:
        df["DIRE_GRUPO"] = df["DIRE_GRUPO"].fillna(0).astype(float).astype(int).astype(str).replace({"0": ""})

        mapa_dire = {
        "171": "Towards (T)",
        "172": "Away (A)",
        "173": "Same direction (S)",
        "174": "Opposite direction (O)",
        "175": "Crossing perpendicular (C)",
        "176": "Variable (V)"
    }
    df["temp_measurementValue_1"] = df["DIRE_GRUPO"].map(mapa_dire).fillna("")

    if "temp_measurementValue_1" in df.columns and df["temp_measurementValue_1"].dropna().empty is False:
        df["measurementValue"] = df["temp_measurementValue_1"]
        df["measurementType"] = " Comportamiento de viaje de los individuos respecto a la plataforma de observación"
    else:
        df["measurementValue"] = None
        df["measurementType"] = None


    return df



# Función principal para procesar peces_OFM
def procesar_peces_OFM(ruta_gdb, capa_peces_OFM, tabla_peces_OFM, enlace_peces_OFM, ruta_excel_peces_OFM, archivo_entrada_peces_OFM, archivo_salida_peces_OFM, epsg_origen):
    try:
        # Abrir la Geodatabase
        gdb = gdal.OpenEx(ruta_gdb, gdal.OF_VECTOR)
        if not gdb:
            raise RuntimeError(f"❌ No se pudo abrir la GDB en {ruta_gdb}")

        # Extraer coordenadas de la capa de peces_OFM
        datos_capa = extraer_coordenadas(gdb.GetLayerByName(capa_peces_OFM), epsg_origen)

        # Extraer atributos de la tabla de peces_OFM
        datos_tabla = []
        layer = gdb.GetLayerByName(tabla_peces_OFM)
        for feature in layer:
            datos_tabla.append(feature.items()) 

        resultado = realizar_join(datos_capa, datos_tabla, enlace_peces_OFM)

        if resultado is None or resultado.empty:

            return

        # Primero mapear especie
        resultado = mapear_espec_nomb(resultado)

        # Exportar el resultado del join a un archivo Excel intermedio
        exportar_excel(resultado, ruta_excel_peces_OFM)

        # Leer el archivo Excel intermedio y agregar taxonRank
        agregar_datos_api_a_excel(ruta_excel_peces_OFM)

        # Leer el archivo Excel con taxonRank agregado
        df_intermedio = pd.read_excel(ruta_excel_peces_OFM)

        # Procesar campos específicos
        df_intermedio = procesar_campos_especificos(df_intermedio)

        # Crear DataFrame final con todos los atributos de lista_atributos
        df_final = pd.DataFrame(columns=lista_atributos)

        # Mapear los datos del DataFrame intermedio al DataFrame final
        for columna_intermedia, columna_final in mapeo.items():
            if columna_intermedia in df_intermedio.columns:
                df_final[columna_final] = df_intermedio[columna_intermedia]

        crs = pyproj.CRS.from_user_input(epsg_origen)
        if crs.is_geographic:
            coord_system = "Coordenadas geográficas"
        else:
            coord_system = "Coordenadas proyectadas"
        # Valores constantes para peces_OFM
        valores_constantes = {
            "occurrenceStatus": "present",
            "language": "es",
            "continent": "América del Sur",
            "country": "Colombia",
            "countryCode": "CO",
            "geodeticDatum": "WGS84",
            "verbatimSRS": epsg_origen,
            "verbatimCoordinateSystem": coord_system
        }
        # Agregar valores constantes
        for clave, valor in valores_constantes.items():
            df_final[clave] = valor

        # Agregar los campos calculados
        
        df_final["verbatimCoordinates"] = df_intermedio["verbatimCoordinates"]
        #df_final["verbatimIdentification"] = df_intermedio["verbatimIdentification"]
        df_final["occurrenceID"] = df_intermedio["occurrenceID"]
        df_final["type"] = df_intermedio["type"]
        df_final["basisOfRecord"] = df_intermedio["basisOfRecord"]
        df_final["occurrenceRemarks"] = df_intermedio["occurrenceRemarks"]
        df_final["eventTime"] = df_intermedio["eventTime"]
        df_final["order"] = df_intermedio["order"]
        #df_final["temp_suborden"] = df_intermedio["temp_suborden"]
        df_final["family"] = df_intermedio["family"]
        df_final["genus"] = df_intermedio["genus"]
        df_final["higherClassification"] = df_intermedio["higherClassification"]
        df_final["scientificName"] = df_intermedio["scientificName"]
        df_final["measurementValue"] = df_intermedio["measurementValue"]
        df_final["measurementType"] = df_intermedio["measurementType"]
        df_final["eventDate"] = df_intermedio["eventDate"]
        df_final["year"] = df_intermedio["year"]
        df_final["month"] = df_intermedio["month"]
        df_final["day"] = df_intermedio["day"]
        df_final["verbatimEventDate"] = df_intermedio["verbatimEventDate"]
       


        # Exportar el DataFrame final a un archivo Excel
        exportar_excel(df_final, archivo_salida_peces_OFM)

    except Exception as e:
        return None