from shapely.geometry import shape
import pyproj
import pandas as pd
import requests
import re
from osgeo import gdal, ogr
from concurrent.futures import ThreadPoolExecutor
import time

from ..utils.codigos import departamentos, municipios
from ..utils.config import EPSG_DESTINO

# Diccionario de mapeo de atributos para muestreo hidrobiotico
mapeo = {
    #"ID_PUNTO_M": "id",
    "OPERADOR": "institutionCode",
    "LABORAT": "recordedBy",
    "DENS_CANTI": "organismQuantity",
    "ABUND_ABS": "individualCount",
    "ID_MUESTRA": "materialSampleID",
    #"ID_MUES_PT": "eventID",
    "NOMBRE": "fieldNumber",
    #"PROF_MUES": "verbatimDepth",
    "FEC_ANALIS": "dateIdentified",
    #"NOMBRE": "locationRemarks",
    #"TIP_FU_SUP": "habitat",
    "CATE_MONIT": "eventRemarks",
    #"DESCRIP": "samplingProtocol",
    "NOM_C_AG": "waterBody",
    "VEREDA": "locality",
    "latitud": "decimalLatitude",
    "longitud": "decimalLongitude",
    "COOR_NORTE": "verbatimLatitude",
    "COOR_ESTE": "verbatimLongitude",
    "ESPECIE": "specificEpithet", 
    "canonicalName": "acceptedNameUsage",
    "scientificNameAuthorship": "scientificNameAuthorship",
    "taxonomicStatus": "taxonomicStatus",
    "DIVISION": "phylum",
    "CLASE": "class",
    "ORDEN": "order",
    "FAMILIA": "family",
    "GENERO": "genus",
    "taxonRank": "taxonRank",
    #"HIDROBIOTA": "vernacularName",
    "PROYECTO": "datasetName"
}



# Lista completa de atributos en el archivo final
lista_atributos = [
    "id", "type", "language", "institutionID", "institutionCode", "datasetName",
    "basisOfRecord", "occurrenceID", "recordNumber", "recordedBy", "organismQuantity", "individualCount",
    "organismQuantityType", "occurrenceStatus", "materialSampleID", "occurrenceRemarks", "organismRemarks", "eventID",
    "parentEventID", "fieldNumber", "eventDate", "year", "month", "day",
    "verbatimEventDate", "dateIdentified", "habitat", "samplingProtocol", "sampleSizeValue", "sampleSizeUnit",
    "eventRemarks", "continent", "waterBody", "country",
    "countryCode", "stateProvince", "county", "municipality", "locality",
    "minimumElevationInMeters", "maximumElevationInMeters", "verbatimElevation",
    "locationRemarks", "verbatimDepth", "minimumDepthInMeters", "decimalLatitude", "decimalLongitude", "geodeticDatum",
    "verbatimCoordinates", "verbatimLatitude", "verbatimLongitude",
    "verbatimCoordinateSystem", "verbatimSRS", "footprintWKT", "footprintSRS",
    "verbatimIdentification", "identificationQualifier", "scientificName",
    "acceptedNameUsage", "higherClassification", "kingdom", "phylum", "class",
    "order", "family", "genus", "specificEpithet", "infraspecificEpithet",
    "taxonRank", "verbatimTaxonRank", "scientificNameAuthorship", "vernacularName",
    "taxonomicStatus", "measurementValue_1", "measurementType_1", "measurementUnit_1",
    "measurementValue_2", "measurementType_2", "measurementUnit_2",
    "measurementValue_3", "measurementType_3", "measurementUnit_3"
]


def consultar_taxon_rank(especie):
    url = "https://api.gbif.org/v1/species/match"
    params = {"name": especie}
    
    try:
        time.sleep(1) 
        response = requests.get(url, params=params)
        if response.status_code == 200:
            data = response.json()
            scientific_name = data.get("scientificName", "")
            canonical_name = data.get("canonicalName", "")
            
            # Extraer la autoría eliminando el nombre científico del nombre canónico
            if canonical_name and scientific_name and scientific_name != canonical_name:
                scientific_name_authorship = scientific_name.replace(canonical_name, "").strip()
            else:
                scientific_name_authorship = None

            return {
                "taxonRank": data.get("rank", None),
                "canonicalName": canonical_name,
                "scientificNameAuthorship": scientific_name_authorship,
                "taxonomicStatus": data.get("status", None)
            }
        else:

            return None
    except Exception as e:

        return None

# 🔹 Función para agregar los datos de la API al archivo Excel existente
def agregar_datos_api_a_excel(ruta_excel_hidrobiotico):
    try:

        df = pd.read_excel(ruta_excel_hidrobiotico)

        if "ESPECIE" not in df.columns:
            raise ValueError("❌ ERROR: La columna 'ESPECIE' no está en el archivo Excel.")

        df["nombre_cientifico"] = df.apply(
            lambda row: (
                f"{row['GENERO']} {row['ESPECIE']}" if pd.notna(row["GENERO"]) and pd.notna(row["ESPECIE"]) else
                row["GENERO"] if pd.notna(row["GENERO"]) else
                row["ESPECIE"] if pd.notna(row["ESPECIE"]) else
                None
            ),
            axis=1
        )


        especies = df["nombre_cientifico"].tolist()

        # Ejecutar consultas a la API en paralelo 
        with ThreadPoolExecutor(max_workers=30) as executor:
            resultados = list(executor.map(consultar_taxon_rank, especies))

        # Convertir a DataFrame
        datos_api = pd.DataFrame([r if isinstance(r, dict) else {} for r in resultados])

        # Agregar columnas nuevas al dataframe original
        df["taxonRank"] = datos_api["taxonRank"]
        df["canonicalName"] = datos_api["canonicalName"]
        df["scientificNameAuthorship"] = datos_api["scientificNameAuthorship"]
        df["taxonomicStatus"] = datos_api["taxonomicStatus"]


        df.to_excel(ruta_excel_hidrobiotico, index=False)


    except Exception as e:
        return None

# Función para extraer coordenadas
def extraer_coordenadas(layer, epsg_origen):
    resultados = []
    transformer = pyproj.Transformer.from_crs(epsg_origen, EPSG_DESTINO, always_xy=True)

    for feature in layer:
        geom = feature.GetGeometryRef()
        if geom and geom.GetGeometryType() == ogr.wkbPoint:
            lon, lat = transformer.transform(geom.GetX(), geom.GetY())

            atributos = feature.items()
            atributos["latitud"] = lat
            atributos["longitud"] = lon
            resultados.append(atributos)

    return resultados


# Función para realizar el join
def realizar_join(capa, tabla, enlace_hidrobiotico):
    try:


        capa_df = pd.DataFrame(capa)
        tabla_df = pd.DataFrame(tabla)

        if capa_df.empty or tabla_df.empty:

            return None

        resultado_df = pd.merge(capa_df, tabla_df, on=enlace_hidrobiotico, how="inner")

        if resultado_df.empty:

            return None

        return resultado_df

    except Exception as e:

        return None

# Función para exportar a Excel
def exportar_excel(dataframe, ruta_salida):
    try:

        dataframe.to_excel(ruta_salida, index=False)

    except Exception as e:
        return None


# Función para extraer los nombres de los departamentos y los municipios
def convertir_codigos_nombres(df):
    """Convierte los códigos de departamentos y municipios a nombres."""
    if "DEPTO" in df.columns:

        df["stateProvince"] = df["DEPTO"].apply(lambda x: departamentos.get(x, "Desconocido"))
    else:
        df["stateProvince"] = None


    if "MUNICIPIO" in df.columns:

        df["county"] = df["MUNICIPIO"].apply(lambda x: municipios.get(x, "Desconocido"))
    else:
        df["county"] = None

    return df

# Función para procesar campos específicos
def procesar_campos_especificos(df):
    """Calcula los campos en orden secuencial asegurando que cada campo esté disponible antes de ser usado."""

    if df is None or df.empty:

        return df  

    # Mapear ID_PUNTO_M a id y eventID
    if "ID_PUNTO_M" in df.columns:
        df["id"] = df["ID_PUNTO_M"]
        df["eventID"] = df["ID_PUNTO_M"]
    else:

        df["id"] = None
        df["eventID"] = None

    # Mapear FEC_MUEST a verbatimEventDate
    if "FEC_TOM" in df.columns:
        df["verbatimEventDate"] = df["FEC_TOM"]
    else:

        df["verbatimEventDate"] = None

    # Mapear COTA a verbatimElevation
    if "COTA" in df.columns:
        df["verbatimElevation"] = df["COTA"]
    else:

        df["verbatimElevation"] = None

     # Mapear COOR_NORTE a verbatimLatitude
    if "COOR_NORTE" in df.columns:
        df["verbatimLatitude"] = df["COOR_NORTE"]
    else:

        df["verbatimLatitude"] = None
        

    # Mapear COOR_ESTE a verbatimLongitude
    if "COOR_ESTE" in df.columns:
        df["verbatimLongitude"] = df["COOR_ESTE"]
    else:

        df["verbatimLongitude"] = None

    # Mapear verbatimDepth
    if "PROF_MUES" in df.columns:
        df["verbatimDepth"] = df["PROF_MUES"]
    else:

        df["verbatimDepth"] = None

    # Se calcula  `recordNumber`
    if "OBJECTID" in df.columns:
        df["recordNumber"] = df["OBJECTID"]
    else:

        df["recordNumber"] = None

    # Calcular "organismQuantityType" 
    if "UNIDAD_DEN" in df.columns:
        mapeo_unidad = {
            120301: "Ind/mL",
            120302: "Ind/m2",
            120303: "Ind/cm2",
            120304: "Unidades"
        }
        df["organismQuantityType"] = df["UNIDAD_DEN"].map(mapeo_unidad).fillna("")
    else:

        df["organismQuantityType"] = None

    # Calcular habitat
    if "TIP_FU_SUP" in df.columns:
        mapeo_habitat = {
            2210: "Río",
            2211: "Quebrada",
            2212: "Arroyo",
            2213: "Caño",
            2214: "Canal",
            2215: "Lago",
            2216: "Laguna",
            2217: "Ciénaga",
            2218: "Pantano",
            2219: "Embalse",
            2220: "Estero",
            2221: "Jagüey",
            2222: "Estuario",
            2223: "Manantial",
            2224: "Área del fondo del pit-minería",
            2225: "Planta de Tratamiento"
        }
        df["habitat"] = df["TIP_FU_SUP"].map(mapeo_habitat).fillna("")
    else:

        df["habitat"] = None

    # Calcular vernacularName
    if "HIDROBIOTA" in df.columns:
        mapeo_name = {
            120201: "Macroinvertebrados (Bentos)",
            120202: "Perifiton",
            120203: "Zooplancton",
            120204: "Fitoplancton",
            120205: "Macrófitas",
            120206: "Meiofauna",
            120207: "Ictioplancton",
            120208: "Macrofauna",
            120209: "Ictiofauna"
        }
        df["vernacularName"] = df["HIDROBIOTA"].map(mapeo_name).fillna("")
    else:

        df["vernacularName"] = None

    # Calcular `occurrenceID`
    if "OPERADOR" in df.columns and "recordNumber" in df.columns:
        df["occurrenceID"] = df.apply(
            lambda row: re.sub(r'[^A-Za-z0-9]', '', row["OPERADOR"]) + ":hidrobiotico:" + str(row["recordNumber"]),
            axis=1
        )
    else:

        df["occurrenceID"] = None


    # Calcular `eventDate`, `year`, `month`, `day` y conservamos `verbatimEventDate`
    if "verbatimEventDate" in df.columns:
        df["eventDate"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.strftime('%Y-%m-%d')
        df["year"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.year
        df["month"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.month
        df["day"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.day

    else:

        df["eventDate"] = None
        df["year"] = None
        df["month"] = None
        df["day"] = None


    # Asignar `verbatimElevation` a `minimumElevationInMeters` y `maximumElevationInMeters`
    if "verbatimElevation" in df.columns:
        df["minimumElevationInMeters"] = df["verbatimElevation"]
        df["maximumElevationInMeters"] = df["verbatimElevation"]
    else:

        df["minimumElevationInMeters"] = None
        df["maximumElevationInMeters"] = None

    # Asignar verbatimDepth a minimumDepthInMeters
    if "verbatimDepth" in df.columns:
        df["minimumDepthInMeters"] = df["verbatimDepth"]
    else:

        df["minimumDepthInMeters"] = None

    # Concatenar `COOR_ESTE` y `COOR_NORTE` para `verbatimCoordinates`
    if "verbatimLongitude" in df.columns and "verbatimLatitude" in df.columns:
        df["verbatimCoordinates"] = df.apply(
            lambda row: str(row["verbatimLongitude"]).replace(",", ".") + ", " + str(row["verbatimLatitude"]).replace(",", "."),
            axis=1
        )

    else:

        df["verbatimCoordinates"] = None


    # Asignar `ESPECIE` a `verbatimIdentification`
    if "ESPECIE" in df.columns:
        df["verbatimIdentification"] = df["ESPECIE"]
    else:

        df["verbatimIdentification"] = None

    # Asignar cada campo de medición individualmente

    # Verificar si "AREA_EST" existe y tiene al menos un valor no vacío
    if "AREA_EST" in df.columns and df["AREA_EST"].dropna().empty is False:
        df["measurementValue_1"] = df["AREA_EST"]
        df["measurementType_1"] = df["AREA_EST"].apply(lambda x: "Área estimada del cuerpo de agua" if pd.notna(x) else None)
        df["measurementUnit_1"] = df["AREA_EST"].apply(lambda x: "m2" if pd.notna(x) else None)
    else:
        df["measurementValue_1"] = None
        df["measurementType_1"] = None
        df["measurementUnit_1"] = None

    # Verificar si "NIV_AGUA" existe y tiene al menos un valor no vacío
    if "NIV_AGUA" in df.columns and not df["NIV_AGUA"].dropna().empty:
        df["measurementValue_2"] = df["NIV_AGUA"]
        df["measurementType_2"] = df["NIV_AGUA"].apply(lambda x: "Nivel de la corriente o cuerpo de agua" if pd.notna(x) else None)
        df["measurementUnit_2"] = df["NIV_AGUA"].apply(lambda x: "m" if pd.notna(x) else None)
    else:
        df["measurementValue_2"] = None
        df["measurementType_2"] = None
        df["measurementUnit_2"] = None

    # Verificar si "PROF_SECCH" existe y tiene al menos un valor no vacío
    if "PROF_SECCH" in df.columns and not df["PROF_SECCH"].dropna().empty:
        df["measurementValue_3"] = df["PROF_SECCH"]
        df["measurementType_3"] = df["PROF_SECCH"].apply(lambda x: "Profundidad de disco secchi" if pd.notna(x) else None)
        df["measurementUnit_3"] = df["PROF_SECCH"].apply(lambda x: "m" if pd.notna(x) else None)
    else:
        df["measurementValue_3"] = None
        df["measurementType_3"] = None
        df["measurementUnit_3"] = None

    # Mapear higherClassification 
    campos_clasificacion = [
        "DIVISION", "CLASE", "ORDEN", "FAMILIA", "GENERO"
    ]

    for col in campos_clasificacion:
        if col not in df.columns:
            df[col] = None

    df["higherClassification"] = df[campos_clasificacion]\
        .applymap(lambda x: x if pd.notna(x) and str(x).strip().lower() not in ["", "none", "nan", "<null>"] else pd.NA)\
        .apply(lambda row: " | ".join(row.dropna().astype(str)), axis=1)

    
    
     # concatenar GENERO y ESPECIE para scientificName
    if "ESPECIE" in df.columns and "GENERO" in df.columns:
        df["scientificName"] = df.apply(
            lambda row: f"{row['GENERO']}" if pd.notna(row["GENERO"]) and pd.isna(row["ESPECIE"]) else
                        f"{row['ESPECIE']}" if pd.notna(row["ESPECIE"]) and pd.isna(row["GENERO"]) else
                        f"{row['GENERO']} {row['ESPECIE']}" if pd.notna(row["GENERO"]) and pd.notna(row["ESPECIE"]) else None,
            axis=1
        )
    
    # Mapear a scientificName con validación
    df["scientificName"] = df.apply(
        lambda row: row["scientificName"]
        if pd.notna(row["scientificName"]) and str(row["scientificName"]).strip().lower() not in ["", "none", "nan", "<null>"]
        else (
            row["higherClassification"].split(" ")[-1]
            if pd.notna(row["higherClassification"]) and " " in row["higherClassification"]
            else row["higherClassification"]
        ),
        axis=1
    )

   
    return df



# Función principal para procesar muestreio hidrobiotico
def procesar_hidrobiotico(ruta_gdb, capa_hidrobiotico, tabla_hidrobiotico, enlace_hidrobiotico, ruta_excel_hidrobiotico, archivo_entrada_hidrobiotico, archivo_salida_hidrobiotico, epsg_origen):
    try:
        # Abrir la Geodatabase
        gdb = gdal.OpenEx(ruta_gdb, gdal.OF_VECTOR)
        if not gdb:
            raise RuntimeError(f"No se pudo abrir la GDB en {ruta_gdb}")

        # Extraer coordenadas de la capa de muestreo hidrobiotico
        datos_capa = extraer_coordenadas(gdb.GetLayerByName(capa_hidrobiotico), epsg_origen)

        # Extraer atributos de la tabla de muestreo hidrobiotico
        datos_tabla = []
        layer = gdb.GetLayerByName(tabla_hidrobiotico)
        for feature in layer:
            datos_tabla.append(feature.items()) 


        resultado = realizar_join(datos_capa, datos_tabla, enlace_hidrobiotico)

        if resultado is None or resultado.empty:

            return

        # Convertir códigos a nombres
        resultado = convertir_codigos_nombres(resultado)

        # Exportar el resultado del join a un archivo Excel intermedio
        exportar_excel(resultado, ruta_excel_hidrobiotico)

        # Leer el archivo Excel intermedio y agregar taxonRank
        agregar_datos_api_a_excel(ruta_excel_hidrobiotico)

        # Leer el archivo Excel con taxonRank agregado
        df_intermedio = pd.read_excel(ruta_excel_hidrobiotico)

        # Procesar campos específicos
        df_intermedio = procesar_campos_especificos(df_intermedio)

        # Crear DataFrame final con todos los atributos de lista_atributos
        df_final = pd.DataFrame(columns=lista_atributos)

        # Mapear los datos del DataFrame intermedio al DataFrame final
        for columna_intermedia, columna_final in mapeo.items():
            if columna_intermedia in df_intermedio.columns:
                df_final[columna_final] = df_intermedio[columna_intermedia]

        crs = pyproj.CRS.from_user_input(epsg_origen)
        if crs.is_geographic:
            coord_system = "Coordenadas geográficas"
        else:
            coord_system = "Coordenadas proyectadas"
        # Valores constantes para muestreo hidrobiotico
        valores_constantes = {
            "type": "Event",
            "occurrenceStatus": "present",
            "language": "es",
            "continent": "América del Sur",
            "country": "Colombia",
            "countryCode": "CO",
            "geodeticDatum": "WGS84",
            "verbatimSRS": epsg_origen,
            "verbatimCoordinateSystem": coord_system,
            "basisOfRecord": "HumanObservation",
        }
        
        # Agregar valores constantes
        for clave, valor in valores_constantes.items():
            df_final[clave] = valor

        # Agregar los campos calculados
        df_final["recordNumber"] = df_intermedio["recordNumber"]
        df_final["eventDate"] = df_intermedio["eventDate"]
        df_final["year"] = df_intermedio["year"]
        df_final["month"] = df_intermedio["month"]
        df_final["day"] = df_intermedio["day"]
        df_final["verbatimEventDate"] = df_intermedio["verbatimEventDate"]
        df_final["verbatimElevation"] = df_intermedio["verbatimElevation"]
        df_final["minimumElevationInMeters"] = df_intermedio["minimumElevationInMeters"]
        df_final["maximumElevationInMeters"] = df_intermedio["maximumElevationInMeters"]
        df_final["verbatimCoordinates"] = df_intermedio["verbatimCoordinates"]
        df_final["verbatimIdentification"] = df_intermedio["verbatimIdentification"]
        df_final["occurrenceID"] = df_intermedio["occurrenceID"]
        df_final["stateProvince"] = df_intermedio["stateProvince"]
        df_final["county"] = df_intermedio["county"]
        df_final["organismQuantityType"] = df_intermedio["organismQuantityType"]
        df_final["habitat"] = df_intermedio["habitat"]
        df_final["verbatimDepth"] = df_intermedio["verbatimDepth"]
        df_final["minimumDepthInMeters"] = df_intermedio["minimumDepthInMeters"]
        df_final["vernacularName"] = df_intermedio["vernacularName"]
        df_final["id"] = df_intermedio["id"]
        df_final["eventID"] = df_intermedio["eventID"]
        df_final["higherClassification"] = df_intermedio["higherClassification"]
        df_final["scientificName"] = df_intermedio["scientificName"]

     

        df_final["measurementValue_1"] = df_intermedio["measurementValue_1"]
        df_final["measurementType_1"] = df_intermedio["measurementType_1"]
        df_final["measurementUnit_1"] = df_intermedio["measurementUnit_1"]

        df_final["measurementValue_2"] = df_intermedio["measurementValue_2"]
        df_final["measurementType_2"] = df_intermedio["measurementType_2"]
        df_final["measurementUnit_2"] = df_intermedio["measurementUnit_2"]

        df_final["measurementValue_3"] = df_intermedio["measurementValue_3"]
        df_final["measurementType_3"] = df_intermedio["measurementType_3"]
        df_final["measurementUnit_3"] = df_intermedio["measurementUnit_3"]

        measurement_columns_to_drop  = []

        for col in df_final.columns:
            if col.startswith("measurementValue_"):
                suf = col.split("_")[-1] 
                v = f"measurementValue_{suf}"
                t = f"measurementType_{suf}"
                u = f"measurementUnit_{suf}"

                serie = df_final[v].astype("string")

                vacio = (
                    serie.isna() |
                    (serie.str.strip() == "")
                ).all()

                if vacio:
                    measurement_columns_to_drop .extend([v, t, u])

        
        measurement_columns_to_drop  = [c for c in measurement_columns_to_drop  if c in df_final.columns]
        if measurement_columns_to_drop :
            df_final.drop(columns=measurement_columns_to_drop , inplace=True)

        # Exportar el DataFrame final a un archivo Excel
        exportar_excel(df_final, archivo_salida_hidrobiotico)

    except Exception as e:
        return None