import pyproj
import pandas as pd
import requests
import re
from osgeo import gdal, ogr
from ..utils.codigos import departamentos, municipios
from ..utils.config import EPSG_ORIGEN, EPSG_DESTINO

# Diccionario de mapeo de atributos para Muestreo Agua Marino
mapeo = {
    "OPERADOR": "institutionCode",
    "ABUND_ABS": "individualCount",
    "ID_PUNTO_M": "eventID",
    "NOMBRE": "fieldNumber",
    "HABITAT": "habitat",
    "DESCRIP": "samplingProtocol",
    "T_ESF_MUEST": "samplingEffort",
    "PROFUND": "verbatimDepth",
    "OBSERV": "eventRemarks",
    #"LABORAT": "recordedBy",
    #"ID_MUESTRA": "materialSampleID",
    #"HORA": "eventTime",
    #"FEC_ANALIS": "dateIdentified",
    "CUERPO_AGU": "waterBody",
    "VEREDA": "locality",
    "latitud": "decimalLatitude",
    "longitud": "decimalLongitude",
    "PROYECTO": "datasetName"
   
}

# Valores constantes para Muestreo Agua Marino
valores_constantes = {
    "occurrenceStatus": "present",
    "language": "es",
    "continent": "América del Sur",
    "country": "Colombia",
    "countryCode": "CO",
    "geodeticDatum": "WGS84",
    "verbatimSRS": EPSG_ORIGEN,
    "verbatimCoordinateSystem": "Coordenadas proyectadas"
}

# Lista completa de atributos en el archivo final
lista_atributos = [
    "id", "type", "language", "institutionID", "institutionCode", "datasetName",
    "basisOfRecord", "occurrenceID", "recordNumber", "individualCount",
    "occurrenceStatus", "occurrenceRemarks", "organismRemarks", "eventID",
    "parentEventID", "fieldNumber", "eventDate", "year", "month", "day",
    "verbatimEventDate", "habitat", "samplingProtocol", "samplingEffort", "verbatimDepth" "sampleSizeValue", "sampleSizeUnit",
    "eventRemarks", "continent", "waterBody", "country",
    "countryCode", "stateProvince", "county", "municipality", "locality",
    "minimumElevationInMeters", "maximumElevationInMeters", "verbatimElevation",
    "locationRemarks", "decimalLatitude", "decimalLongitude", "geodeticDatum",
    "verbatimCoordinates", "verbatimLatitude", "verbatimLongitude",
    "verbatimCoordinateSystem", "verbatimSRS", "footprintWKT", "footprintSRS",
    "verbatimIdentification", "identificationQualifier", "scientificName",
    "acceptedNameUsage", "higherClassification", "kingdom", "phylum", "class",
    "order", "family", "genus", "specificEpithet", "infraspecificEpithet",
    "taxonRank", "verbatimTaxonRank", "scientificNameAuthorship", "vernacularName",
    "taxonomicStatus"
]



# Función para extraer coordenadas
def extraer_coordenadas(layer):
    resultados = []
    transformer = pyproj.Transformer.from_crs(EPSG_ORIGEN, EPSG_DESTINO, always_xy=True)

    for feature in layer:
        geom = feature.GetGeometryRef()
        if geom and geom.GetGeometryType() == ogr.wkbPoint:
            lon, lat = transformer.transform(geom.GetX(), geom.GetY())

            atributos = feature.items()
            atributos["latitud"] = lat
            atributos["longitud"] = lon
            resultados.append(atributos)

    return resultados

def realizar_join(capa, tabla, enlace_hidrobiotico):
    try:

        capa_df = pd.DataFrame(capa)
        tabla_df = pd.DataFrame(tabla)

        if capa_df.empty or tabla_df.empty:
            return None

        # Solo se deja el campo de enlace y FEC_TOM
        tabla_df = tabla_df[[enlace_hidrobiotico, "FEC_TOM"]].drop_duplicates(subset=[enlace_hidrobiotico])

        # se realiza un left join, manteniendo solo los datos de capa + FEC_TOM de tabla
        resultado_df = pd.merge(
            capa_df,
            tabla_df,
            on=enlace_hidrobiotico,
            how="left"
        )

        if resultado_df.empty:
            
            return None

        return resultado_df

    except Exception as e:

        return None


# Función para exportar a Excel
def exportar_excel(dataframe, ruta_salida):
    try:

        dataframe.to_excel(ruta_salida, index=False)

    except Exception as e:
        return None




# Función para procesar campos específicos
def procesar_campos_especificos(df):
    """Calcula los campos en orden secuencial asegurando que cada campo esté disponible antes de ser usado."""

    if df is None or df.empty:

        return df  

    # Mapear FEC_TOM a verbatimEventDate
    if "FEC_TOM" in df.columns:
        df["verbatimEventDate"] = df["FEC_TOM"]
    else:

        df["verbatimEventDate"] = None

    # Mapear COOR_NORTE a verbatimLatitude
    if "COOR_NORTE" in df.columns:
        df["verbatimLatitude"] = df["COOR_NORTE"]
    else:

        df["verbatimLatitude"] = None

    # Mapear COOR_ESTE a verbatimLongitude
    if "COOR_ESTE" in df.columns:
        df["verbatimLongitude"] = df["COOR_ESTE"]
    else:

        df["verbatimLongitude"] = None

    # Se calcula  `recordNumber`
    if "OBJECTID" in df.columns:
        df["recordNumber"] = df["OBJECTID"]
    else:

        df["recordNumber"] = None

    # Se calcula `occurrenceID`
    if "OPERADOR" in df.columns and "recordNumber" in df.columns:
        df["occurrenceID"] = df.apply(
            lambda row: re.sub(r'[^A-Za-z0-9]', '', row["OPERADOR"]) + ":agua-marino:" + str(row["recordNumber"]),
            axis=1
        )
    else:

        df["occurrenceID"] = None

     # Se asigna `occurrenceID` a `id`
    if "occurrenceID" in df.columns:
        df["id"] = df["occurrenceID"]
    else:

        df["id"] = None

    # Se calcula `eventDate`, `year`, `month`, `day` y conservamos `verbatimEventDate`
    if "verbatimEventDate" in df.columns:
        df["eventDate"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.strftime('%Y-%m-%d')
        df["year"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.year
        df["month"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.month
        df["day"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.day
        
    else:

        df["eventDate"] = None
        df["year"] = None
        df["month"] = None
        df["day"] = None

    # Concatenar `COOR_ESTE` y `COOR_NORTE` para `verbatimCoordinates`
    if "verbatimLongitude" in df.columns and "verbatimLatitude" in df.columns:
        df["verbatimCoordinates"] = df.apply(
            lambda row: str(row["verbatimLongitude"]).replace(",", ".") + ", " + str(row["verbatimLatitude"]).replace(",", "."),
            axis=1
        )

    else:

        df["verbatimCoordinates"] = None


    # Asignar `ESPECIE` a `verbatimIdentification`
    if "ESPECIE" in df.columns:
        df["verbatimIdentification"] = df["ESPECIE"]
    else:

        df["verbatimIdentification"] = None

    return df



# 🔹 Función principal para procesar Muestreo Agua Marino
def procesar_marino(ruta_gdb, capa_marino, tabla_marino, enlace_marino, ruta_excel_marino, archivo_entrada_marino, archivo_salida_marino):
    try:
       # Abrir la Geodatabase
        gdb = gdal.OpenEx(ruta_gdb, gdal.OF_VECTOR)
        if not gdb:
            raise RuntimeError(f"❌ No se pudo abrir la GDB en {ruta_gdb}")

        # Extraer coordenadas de la capa de muestreo hidrobiotico
        datos_capa = extraer_coordenadas(gdb.GetLayerByName(capa_marino))

        # Extraer atributos de la tabla de muestreo hidrobiotico
        datos_tabla = []
        layer = gdb.GetLayerByName(tabla_marino)
        for feature in layer:
            datos_tabla.append(feature.items()) 


        resultado = realizar_join(datos_capa, datos_tabla, enlace_marino)

        if resultado is None or resultado.empty:

            return


        # Verificar los datos después del join



        # Exportar el resultado del join a un archivo Excel intermedio
        exportar_excel(resultado, ruta_excel_marino)

        # Leer el archivo Excel con taxonRank agregado
        df_intermedio = pd.read_excel(ruta_excel_marino)

        # Procesar campos específicos
        df_intermedio = procesar_campos_especificos(df_intermedio)

        # Crear DataFrame final con todos los atributos de lista_atributos
        df_final = pd.DataFrame(columns=lista_atributos)

        # Mapear los datos del DataFrame intermedio al DataFrame final
        for columna_intermedia, columna_final in mapeo.items():
            if columna_intermedia in df_intermedio.columns:
                df_final[columna_final] = df_intermedio[columna_intermedia]

        # Agregar valores constantes
        for clave, valor in valores_constantes.items():
            df_final[clave] = valor

        # Agregar los campos calculados
        df_final["recordNumber"] = df_intermedio["recordNumber"]
        df_final["verbatimCoordinates"] = df_intermedio["verbatimCoordinates"]
        df_final["occurrenceID"] = df_intermedio["occurrenceID"]
        df_final["id"] = df_intermedio["id"]
        df_final["verbatimEventDate"] = df_intermedio["verbatimEventDate"]
        df_final["eventDate"] = df_intermedio["eventDate"]
        df_final["year"] = df_intermedio["year"]
        df_final["month"] = df_intermedio["month"]
        df_final["day"] = df_intermedio["day"]


        # Verificar los datos antes de la exportación final



        # Exportar el DataFrame final a un archivo Excel
        exportar_excel(df_final, archivo_salida_marino)

    except Exception as e:
        return None