import pyproj
import pandas as pd
import requests
import re
from osgeo import gdal, ogr
from concurrent.futures import ThreadPoolExecutor
import time

from ..utils.codigos import departamentos, municipios
from ..utils.config import EPSG_ORIGEN, EPSG_DESTINO

# Diccionario de mapeo de atributos para mamiferos_OFM
mapeo = {
    "OPERADOR": "institutionCode",
    #"NOMBRE": "fieldNumber",
    "latitud": "decimalLatitude",
    "longitud": "decimalLongitude",
    "ID_INGRESO": "eventID",
    "ID_AVISTAM": "fieldNumber",
    "FECHA": "verbatimEventDate",
    "PROFUND_m": "verbatimDepth",
    #"ORDEN": "order",
    #"FAMILIA": "family",
    "genus": "genus",   
    "especie": "specificEpithet", 
    #"NOMB_COMUN": "vernacularName",
    "taxonRank": "taxonRank",
    "canonicalName": "acceptedNameUsage",
    "scientificNameAuthorship": "scientificNameAuthorship",
    "taxonomicStatus": "taxonomicStatus",
    "scientificNameID": "scientificNameID",
    "nameAccordingTo": "nameAccordingTo",
    "nameAccordingToID": "nameAccordingToID",
    "NUM_ANIMAL": "individualCount",
    "PROYECTO": "datasetName"
    
}

# Valores constantes para mamiferos_OFM
valores_constantes = {
    "occurrenceStatus": "present",
    "language": "es",
    "continent": "América del Sur",
    "country": "Colombia",
    "countryCode": "CO",
    "geodeticDatum": "WGS84",
    "verbatimSRS": EPSG_ORIGEN,
    "verbatimCoordinateSystem": "Coordenadas proyectadas"
}

# Lista completa de atributos en el archivo final
lista_atributos = [
    "id", "type", "language", "institutionID", "institutionCode", "datasetName", "basisOfRecord", "occurrenceID", "recordedBy", 
    "individualCount", "lifeStage", "organismQuantity", "organismQuantityType", "behavior", "occurrenceStatus", "occurrenceRemarks", 
    "eventID", "parentEventID", "fieldNumber", "eventDate", "eventTime", "year", "month", "day", "verbatimEventDate", "samplingProtocol", 
    "samplingEffort", "eventRemarks", "locationID", "continent", "waterBody", "country", "countryCode", "stateProvince", "minimumDepthInMeters", "maximumDepthInMeters",
    "verbatimDepth", "decimalLatitude", "decimalLongitude", "geodeticDatum", "verbatimCoordinates", "verbatimLatitude", "verbatimLongitude", "verbatimCoordinateSystem",
    "verbatimSRS", "verbatimIdentification", "identificationQualifier", "dateIdentified", "scientificNameID", "nameAccordingToID", "scientificName", "acceptedNameUsage", 
    "nameAccordingTo", "higherClassification", "kingdom", "phylum", "class", "order", "family", "genus", "specificEpithet", "taxonRank", "verbatimTaxonRank", 
    "scientificNameAuthorship", "vernacularName", "taxonomicStatus", "identificationVerificationStatus"
]


def consultar_taxon_rank(especie):
    url = f"http://www.marinespecies.org/rest/AphiaRecordsByMatchNames?scientificnames[]={especie}&marine_only=false"

    try:
        time.sleep(1)  
        response = requests.get(url)

        if response.status_code == 200:
            data = response.json()
            especie_data = data[0][0] if data and data[0] else {}

            return {
                "taxonRank": especie_data.get("rank", None),
                "canonicalName": especie_data.get("valid_name", None),
                "scientificNameAuthorship": especie_data.get("authority", None),
                "taxonomicStatus": especie_data.get("status", None),
                "scientificNameID": f"https://www.marinespecies.org/aphia.php?p=taxdetails&id={especie_data.get('AphiaID')}" if especie_data.get("AphiaID") else None,
                "nameAccordingTo": "World Register of Marine Species",
                "nameAccordingToID": "https://www.marinespecies.org"
            }

        else:

            return None

    except Exception as e:

        return None


def agregar_datos_api_a_excel(ruta_excel_peces_OFM):
    try:

        df = pd.read_excel(ruta_excel_peces_OFM)

        if "especie" not in df.columns:
            raise ValueError("❌ ERROR: La columna 'ESPEC_NOMB' no está en el archivo Excel.")

        df["nombre_cientifico"] = df.apply(
            lambda row: (
                f"{row['genus']} {row['especie']}" if pd.notna(row["genus"]) and pd.notna(row["especie"]) else
                row["genus"] if pd.notna(row["genus"]) else
                row["especie"] if pd.notna(row["especie"]) else
                None
            ),
            axis=1
        )


        especies = df["nombre_cientifico"].tolist()

        # Ejecutar en paralelo las consultas a la API
        with ThreadPoolExecutor(max_workers=1) as executor:
            resultados = list(executor.map(consultar_taxon_rank, especies))

        # Convertir a DataFrame
        datos_api = pd.DataFrame([r if isinstance(r, dict) else {} for r in resultados])

        # Asegurar que todas las columnas necesarias existan aunque estén vacías
        columnas_requeridas = [
            "taxonRank", "canonicalName", "scientificNameAuthorship",
            "taxonomicStatus", "scientificNameID", "nameAccordingTo", "nameAccordingToID"
        ]
        for col in columnas_requeridas:
            if col not in datos_api.columns:
                datos_api[col] = None

        # Agregar columnas nuevas al DataFrame original
        df["taxonRank"] = datos_api["taxonRank"]
        df["canonicalName"] = datos_api["canonicalName"]
        df["scientificNameAuthorship"] = datos_api["scientificNameAuthorship"]
        df["taxonomicStatus"] = datos_api["taxonomicStatus"]
        df["scientificNameID"] = datos_api.get("scientificNameID")
        df["nameAccordingTo"] = datos_api.get("nameAccordingTo")
        df["nameAccordingToID"] = datos_api.get("nameAccordingToID")


        df.to_excel(ruta_excel_peces_OFM, index=False)


    except Exception as e:
        return None


# Función para extraer coordenadas
def extraer_coordenadas(layer):
    resultados = []
    transformer = pyproj.Transformer.from_crs(EPSG_ORIGEN, EPSG_DESTINO, always_xy=True)

    for feature in layer:
        geom = feature.GetGeometryRef()
        if geom and geom.GetGeometryType() == ogr.wkbPoint:
            lon, lat = transformer.transform(geom.GetX(), geom.GetY())

            atributos = feature.items()
            atributos["latitud"] = lat
            atributos["longitud"] = lon
            resultados.append(atributos)

    return resultados


# Función para realizar el join
def realizar_join(capa, tabla, enlace_mamiferos_OFM):
    try:


        capa_df = pd.DataFrame(capa)
        tabla_df = pd.DataFrame(tabla)

        if capa_df.empty or tabla_df.empty:

            return None

        resultado_df = pd.merge(capa_df, tabla_df, on=enlace_mamiferos_OFM, how="inner")

        if resultado_df.empty:

            return None

        return resultado_df

    except Exception as e:

        return None


# Función para mapear especie
def mapear_espec_nomb(df):
    if "ESPEC_NOMB" in df.columns:
        df["ESPEC_NOMB"] = df["ESPEC_NOMB"].fillna(0).astype(float).astype(int).astype(str).replace({"0", ""})

    mapa_espec_nomb = {
        "1701": "attenuata",
        "1702": "Balaenoptera borealis",
        "1703": "Balaenoptera edeni",
        "1704": "borealis",
        "1705": "bredanensis",
        "1706": "capensis",
        "1707": "clymene",
        "1708": "coeruleoalba",
        "1709": "crassidens",
        "1710": "Delphinus capensis",
        "1711": "edeni",
        "1712": "frontalis",
        "1713": "Globicephala macrorhynchus",
        "1714": "Grampus griseus",
        "1715": "griseus",
        "1716": "guianensis",
        "1717": "longirostris",
        "1718": "macrocephalus",
        "1719": "macrorhynchus",
        "1720": "Megaptera novaeangliae",
        "1721": "novaeangliae",
        "1722": "orca",
        "1723": "Orcinus orca",
        "1724": "Physeter macrocephalus",
        "1725": "Pseudorca crassidens",
        "1726": "Sotalia guianensis",
        "1727": "Stenella attenuata",
        "1728": "Stenella clymene",
        "1729": "Stenella coeruleoalba",
        "1730": "Stenella frontalis",
        "1731": "Stenella longirostris",
        "1732": "Steno bredanensis",
        "1733": "truncatus",
        "1734": "Tursiops truncatus",
        "1735": "Otros"    
    }
    if "ESPEC_NOMB" in df.columns:
        df["especie"] = df["ESPEC_NOMB"].map(mapa_espec_nomb).fillna(df["ESPEC_NOMB"])
    else:

        df["especie"] = None

    
    return df

# Función para mapear genero
def mapear_genero(df):
    if "GENERO" in df.columns:
        df["GENERO"] = df["GENERO"].fillna(0).astype(float).astype(int).astype(str).replace({"0": ""})

    mapa_genero = {
        "1601": "Balaenoptera",
        "1602": "Delphinus",
        "1603": "Globicephala",
        "1604": "Grampus",
        "1605": "Megaptera",
        "1606": "Orcinus",
        "1607": "Physeter",
        "1608": "Pseudorca",
        "1609": "Sotalia",
        "1610": "Stenella",
        "1611": "Steno",
        "1612": "Tursiops",
        "1613": "Otro"
    }

    if "GENERO" in df.columns:
        df["genus"] = df["GENERO"].map(mapa_genero).fillna(df["GENERO"])
    else:

        df["genus"] = None

    return df




# Función para exportar a Excel
def exportar_excel(dataframe, ruta_salida):
    try:

        dataframe.to_excel(ruta_salida, index=False)

    except Exception as e:
        return None


# Función para procesar campos específicos
def procesar_campos_especificos(df):
    """Calcula los campos en orden secuencial asegurando que cada campo esté disponible antes de ser usado."""

    if df is None or df.empty:

        return df  

     # Mapear COOR_NORTE a verbatimLatitude
    if "COOR_NORTE" in df.columns:
        df["verbatimLatitude"] = df["COOR_NORTE"]
    else:

        df["verbatimLatitude"] = None

    # Mapear COOR_ESTE a verbatimLongitude
    if "COOR_ESTE" in df.columns:
        df["verbatimLongitude"] = df["COOR_ESTE"]
    else:

        df["verbatimLongitude"] = None

    # Se calcula  `recordNumber`
    if "OBJECTID" in df.columns:
        df["recordNumber"] = df["OBJECTID"]
    else:

        df["recordNumber"] = None

    # Calcular `occurrenceID`
    if "OPERADOR" in df.columns and "recordNumber" in df.columns:
        df["occurrenceID"] = df.apply(
            lambda row: re.sub(r'[^A-Za-z0-9]', '', row["OPERADOR"]) + ":MamiferosOFM:" + str(row["recordNumber"]),
            axis=1
        )
    else:

        df["occurrenceID"] = None

     # Asignar `occurrenceID` a `id`
    if "occurrenceID" in df.columns:
        df["id"] = df["occurrenceID"]
    else:

        df["id"] = None


    # Concatenar `COOR_ESTE` y `COOR_NORTE` para `verbatimCoordinates`
    if "verbatimLongitude" in df.columns and "verbatimLatitude" in df.columns:
        df["verbatimCoordinates"] = df.apply(
            lambda row: str(row["verbatimLongitude"]).replace(",", ".") + ", " + str(row["verbatimLatitude"]).replace(",", "."),
            axis=1
        )

    else:

        df["verbatimCoordinates"] = None


     # se mapea identificationVerificationStatus
    if "CERTE_DATO" in df.columns:
        mapeo_certe = {
            1201: "Sure(1)",
            1202: "Probable(2)",
            1203: "Doubtful(3)"
                      
        }
        df["identificationVerificationStatus"] = df["CERTE_DATO"].map(mapeo_certe).fillna("")
    else:

        df["identificationVerificationStatus"] = None

    #  se mapea a variables temporales para despues sacar el type
    if "EVENTTYPE" in df.columns:
        mapeo_type = {
            1101: "Event",
            1102: "Event",
            1103: "Event",
            1104: "Event",
            1105: "PhysicalObject",
            1106: "Event"            
        }
        df["temp_type"] = df["EVENTTYPE"].map(mapeo_type).fillna("")
    else:

        df["temp_type"] = None

    if "EVIDENCIA" in df.columns:
        mapeo_evidencia = {
            181: "StillImage",
            182: "MovingImage",
            183: "MovingImage",
            184: "Event"           
        }
        df["temp_type_2"] = df["EVIDENCIA"].map(mapeo_evidencia).fillna("")
    else:

        df["temp_type_2"] = None

    if "temp_type" in df.columns and "temp_type_2" in df.columns:
        df["type"] = df.apply(
            lambda row: (
                str(row["temp_type"]) if pd.notna(row["temp_type"]) and pd.isna(row["temp_type_2"]) else
                str(row["temp_type_2"]) if pd.notna(row["temp_type_2"]) and pd.isna(row["temp_type"]) else
                f"{row['temp_type']} | {row['temp_type_2']}" if pd.notna(row["temp_type"]) and pd.notna(row["temp_type_2"]) else
                None
            ),
            axis=1
        )
        

    else:

        df["type"] = None



    # Calcular `basisOfRecord`, que depende de `type`
    if "temp_type" in df.columns:
        mapeo_basis = {
            "Event": "HumanObservation",
            "PhysicalObject": "MaterialSample"
        }
        df["temp_basisOfRecord"] = df["temp_type"].map(mapeo_basis).fillna("")
    else:

        df["temp_basisOfRecord"] = None

    if "temp_type_2" in df.columns:
        mapeo_basis_2 = {
            "StillImage": "MachineObservation",
            "MovingImage": "MachineObservation",
            "Event": "HumanObservation"
        }
        df["temp_basisOfRecord_2"] = df["temp_type_2"].map(mapeo_basis_2).fillna("")
    else:

        df["temp_basisOfRecord_2"] = None
    
    
    if "temp_basisOfRecord" in df.columns and "temp_basisOfRecord_2" in df.columns:
        df["basisOfRecord"] = df.apply(
            lambda row: (
                str(row["temp_basisOfRecord"]) if pd.notna(row["temp_basisOfRecord"]) and pd.isna(row["temp_basisOfRecord_2"]) else
                str(row["temp_basisOfRecord_2"]) if pd.notna(row["temp_basisOfRecord_2"]) and pd.isna(row["temp_basisOfRecord"]) else
                f"{row['temp_basisOfRecord']} | {row['temp_basisOfRecord_2']}" if pd.notna(row["temp_basisOfRecord"]) and pd.notna(row["temp_basisOfRecord_2"]) else
                None
            ),
            axis=1
        )
        
    else:  

        df["basisOfRecord"] = None
    
    
    
    # Calcular `occurrenceRemarks`
    if "EVENTTYPE" in df.columns:
        mapeo_eventtype = {
            1101: "Afloat",
            1102: "Boat strike",
            1103: "Entanglement",
            1104: "Sighting",
            1105: "Skeletal Remains",
            1106: "Stranding"
        }
        df["temp_occurrenceRemarks"] = df["EVENTTYPE"].map(mapeo_eventtype).fillna("")
    else:

        df["temp_occurrenceRemarks"] = None

    if "EVIDENCIA" in df.columns:
        mapeo_evidencia = {
            181: "Fotografía",
            182: "video",
            183: "Video y fotografía",
            184: "Solo observación" 
        }
        df["temp_occurrenceRemarks_2"] = df["EVIDENCIA"].map(mapeo_evidencia).fillna("")
    else:

        df["temp_occurrenceRemarks_2"] = None
    
    
    # Mapear occurrenceRemarks 
    campos_occurrenceRemarks = [
        "temp_occurrenceRemarks", "temp_occurrenceRemarks_2", "OBSERVACIO_y"
    ]
    

    for col in campos_occurrenceRemarks:
        if col not in df.columns:
            df[col] = None

  
    df["occurrenceRemarks"] = df[campos_occurrenceRemarks]\
        .applymap(lambda x: x if pd.notna(x) and str(x).strip().lower() not in ["", "none", "nan", "<null>"] else pd.NA)\
        .apply(lambda row: " | ".join(row.dropna().astype(str)), axis=1)
    


    
    
    # Mapeamos a lifeStage    
    if "NUM_ADULT" in df.columns and "NUM_JUVE" in df.columns:
        df["lifeStage"] = df.apply(
            lambda row: (
                f"{int(row['NUM_ADULT'])} adultos" if pd.notna(row["NUM_ADULT"]) and pd.isna(row["NUM_JUVE"]) else
                f"{int(row['NUM_JUVE'])} juveniles" if pd.notna(row["NUM_JUVE"]) and pd.isna(row["NUM_ADULT"]) else
                f"{int(row['NUM_ADULT'])} adultos | {int(row['NUM_JUVE'])} juveniles" if pd.notna(row["NUM_ADULT"]) and pd.notna(row["NUM_JUVE"]) else
                None
            ),
            axis=1
        )
        
    else:

        df["lifeStage"] = None

    #  Mapear TIEMPO_I y TIEMPO_F a eventTime con formato HH:MM:SS
    if "TIEMPO_I" in df.columns or "TIEMPO_F" in df.columns:
        def formatear_hora(horas):
            tiempo = pd.to_timedelta(horas, unit="h")
            return f"{int(tiempo.components.hours):02}:{int(tiempo.components.minutes):02}:{int(tiempo.components.seconds):02}"
        

        df["eventTime"] = df.apply(
            lambda row: (
                formatear_hora(row.get("TIEMPO_I")) if pd.notna(row.get("TIEMPO_I")) and pd.isna(row.get("TIEMPO_F")) else
                formatear_hora(row.get("TIEMPO_F")) if pd.notna(row.get("TIEMPO_F")) and pd.isna(row.get("TIEMPO_I")) else
                f"{formatear_hora(row.get('TIEMPO_I'))} / {formatear_hora(row.get('TIEMPO_F'))}"
                if pd.notna(row.get("TIEMPO_I")) and pd.notna(row.get("TIEMPO_F")) else None
            ),
            axis=1
        )

        
    


    # Mapeamos a order
    if "ORDEN" in df.columns:
        mapeo_orden = {
            1301: "Cetacea",
            1302: "Otro"        
        }
        df["order"] = df["ORDEN"].map(mapeo_orden).fillna("")
    else:

        df["order"] = None


    # Mapeamos SUBORDEN a una variable temporal
    if "SUBORDEN" in df.columns:
        mapeo_suborden = {
            1401: "Misticeti",
            1402: "Odontoceti",
            1403: "Otro"
        }
        df["temp_suborden"] = df["SUBORDEN"].map(mapeo_suborden).fillna("")
    else:

        df["temp_suborden"] = None

    # Mapeamos a family
    if "FAMILIA" in df.columns:
        mapeo_familia = {
            1501: "Balaenopteridae",
            1502: "Delphinidae",
            1503: "Physeteridae",
            1504: "otro"
        }
        df["family"] = df["FAMILIA"].map(mapeo_familia).fillna("")
    else:

        df["family"] = None


    
    # Mapear higherClassification 
    campos_clasificacion = [
        "order", "temp_suborden", "family", "genus"
    ]

    for col in campos_clasificacion:
        if col not in df.columns:
            df[col] = None

    df["higherClassification"] = df[campos_clasificacion]\
        .applymap(lambda x: x if pd.notna(x) and str(x).strip().lower() not in ["", "none", "nan", "<null>"] else pd.NA)\
        .apply(lambda row: " | ".join(row.dropna().astype(str)), axis=1)

    # Mapear a scientificName con validación
    df["scientificName"] = df.apply(
        lambda row: (
            row["especie"]
            if pd.notna(row["especie"]) and str(row["especie"]).strip().lower() not in ["", "none", "nan", "<null>", "Otros", "Otro", "otro","Otra","Otras"]
            else (
                row["higherClassification"].split(" | ")[-1]
                if pd.notna(row["higherClassification"]) and " | " in row["higherClassification"]
                else row["higherClassification"]
            )
        ),
        axis=1
    )

    # Mapeamos NOMB_COMUN
    if "NOMB_COMUN" in df.columns:
        df["NOMB_COMUN"] = df["NOMB_COMUN"].fillna(0).astype(float).astype(int).astype(str).replace({"0": ""})

        mapeo_nombre = {
            "1801": "Ballena de Bryde",
            "1802": "Ballena Jorobada",
            "1803": "Ballena Sei",
            "1804": "Cachalote",
            "1805": "Calderón negro de aletas cortas",
            "1806": "Delfín clymene",
            "1807": "Delfín común de hocico largo",
            "1808": "Delfín de dientes rugosos",
            "1809": "Delfín de Risso",
            "1810": "Delfín gris/ Tucuxi",
            "1811": "Delfín hocico de botella",
            "1812": "Delfín listado",
            "1813": "Delfín moteado del Atlántico",
            "1814": "Delfín moteado pantropical",
            "1815": "Delfín tornillo",
            "1816": "Falsa Orca",
            "1817": "Orca",
            "1818": "Otro"
     }
        df["vernacularName"] = df["NOMB_COMUN"].map(mapeo_nombre).fillna("")
    else:
        df["vernacularName"] = None
    

    # Se calcula verbatimIdentification 
    if "nombre_cientifico" in df.columns:
        df["scientificName"] = df["nombre_cientifico"]
        df["verbatimIdentification"] = df["nombre_cientifico"]
    else:

        df["scientificName "] = None
        df["verbatimIdentification"] = None   

    # Mapear FECHA_ a verbatimEventDate
    if "FECHA" in df.columns:
        df["verbatimEventDate"] = df["FECHA"]
    else:

        df["verbatimEventDate"] = None


    # Calcular `eventDate`, `year`, `month`, `day` y conservamos `verbatimEventDate`
    if "verbatimEventDate" in df.columns:

        df["eventDate"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.strftime('%Y-%m-%d')
        df["year"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.year
        df["month"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.month
        df["day"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.day

    else:

        df["eventDate"] = None
        df["year"] = None
        df["month"] = None
        df["day"] = None

    return df



# Función principal para procesar mamiferos_OFM
def procesar_mamiferos_OFM(ruta_gdb, capa_mamiferos_OFM, tabla_mamiferos_OFM, enlace_mamiferos_OFM, ruta_excel_mamiferos_OFM, archivo_entrada_mamiferos_OFM, archivo_salida_mamiferos_OFM):
    try:
        # Abrir la Geodatabase
        gdb = gdal.OpenEx(ruta_gdb, gdal.OF_VECTOR)
        if not gdb:
            raise RuntimeError(f"❌ No se pudo abrir la GDB en {ruta_gdb}")

        # Extraer coordenadas de la capa de mamiferos_OFM
        datos_capa = extraer_coordenadas(gdb.GetLayerByName(capa_mamiferos_OFM))

        # Extraer atributos de la tabla de mamiferos_OFM
        datos_tabla = []
        layer = gdb.GetLayerByName(tabla_mamiferos_OFM)
        for feature in layer:
            datos_tabla.append(feature.items())  # Obtiene los atributos correctamente


        resultado = realizar_join(datos_capa, datos_tabla, enlace_mamiferos_OFM)

        if resultado is None or resultado.empty:

            return

        # Primero mapear especie
        resultado = mapear_espec_nomb(resultado)
        resultado = mapear_genero(resultado)

        # Exportar el resultado del join a un archivo Excel intermedio
        exportar_excel(resultado, ruta_excel_mamiferos_OFM)

        # Leer el archivo Excel intermedio y agregar taxonRank
        agregar_datos_api_a_excel(ruta_excel_mamiferos_OFM)

        # Leer el archivo Excel con taxonRank agregado
        df_intermedio = pd.read_excel(ruta_excel_mamiferos_OFM)
    
        # Procesar campos específicos
        df_intermedio = procesar_campos_especificos(df_intermedio)

        # Crear DataFrame final con todos los atributos de lista_atributos
        df_final = pd.DataFrame(columns=lista_atributos)

        # Mapear los datos del DataFrame intermedio al DataFrame final
        for columna_intermedia, columna_final in mapeo.items():
            if columna_intermedia in df_intermedio.columns:
                df_final[columna_final] = df_intermedio[columna_intermedia]

        # Agregar valores constantes
        for clave, valor in valores_constantes.items():
            df_final[clave] = valor

        # Agregar los campos calculados
        
        df_final["verbatimCoordinates"] = df_intermedio["verbatimCoordinates"]
        df_final["verbatimIdentification"] = df_intermedio["verbatimIdentification"]
        df_final["scientificName"] = df_intermedio["scientificName"]
        df_final["occurrenceID"] = df_intermedio["occurrenceID"]
        df_final["type"] = df_intermedio["type"]
        df_final["basisOfRecord"] = df_intermedio["basisOfRecord"]
        df_final["occurrenceRemarks"] = df_intermedio["occurrenceRemarks"]
        df_final["lifeStage"] = df_intermedio["lifeStage"]
        df_final["eventTime"] = df_intermedio["eventTime"]
        df_final["order"] = df_intermedio["order"]
        #df_final["temp_suborden"] = df_intermedio["temp_suborden"]
        df_final["family"] = df_intermedio["family"]
        df_final["genus"] = df_intermedio["genus"]
        df_final["higherClassification"] = df_intermedio["higherClassification"]
        df_final["scientificName"] = df_intermedio["scientificName"]
        df_final["identificationVerificationStatus"] = df_intermedio["identificationVerificationStatus"]
        df_final["vernacularName"] = df_intermedio["vernacularName"]
        df_final["eventDate"] = df_intermedio["eventDate"]
        df_final["year"] = df_intermedio["year"]
        df_final["month"] = df_intermedio["month"]
        df_final["day"] = df_intermedio["day"]
        df_final["verbatimEventDate"] = df_intermedio["verbatimEventDate"]

      


        # Exportar el DataFrame final a un archivo Excel
        exportar_excel(df_final, archivo_salida_mamiferos_OFM)

    except Exception as e:
        return None