from shapely.geometry import shape
import pyproj
import pandas as pd
import requests
import re
from osgeo import gdal, ogr
from concurrent.futures import ThreadPoolExecutor
import time

from ..utils.codigos import departamentos, municipios
from ..utils.config import EPSG_ORIGEN, EPSG_DESTINO

# Diccionario de mapeo de atributos para Flora Fustal
mapeo = {
    "OPERADOR": "institutionCode",
    "ABUND_ABS": "individualCount",
    "ID_MUEST": "eventID",
    #"ID_INDV_MU": "parentEventID",
    "ID_INDV_MU": "organismID",
    "NOMBRE": "fieldNumber",
    "N_COBERT": "habitat",
    "AREA_UM_ha": "sampleSizeValue",
    "DESCRIP": "samplingProtocol",
    "CUERPO_AGU": "waterBody",
    #"DEPTO": "stateProvince",
    #"MUNICIPIO": "county",
    "VEREDA": "locality",
    "latitud": "decimalLatitude",
    "longitud": "decimalLongitude",
    "COOR_NORTE": "verbatimLatitude",
    "COOR_ESTE": "verbatimLongitude",
    "LOCALIDAD": "locationRemarks",
    "ESPECIE": "scientificName",  
    "canonicalName": "acceptedNameUsage",
    "scientificNameAuthorship": "scientificNameAuthorship",
    "taxonomicStatus": "taxonomicStatus",
    "DIVISION": "phylum",
    "CLASE": "class",
    "ORDEN": "order",
    "FAMILIA": "family",
    "GENERO": "genus",
    "taxonRank": "taxonRank",
    "N_COMUN": "vernacularName",
    "OBSERV": "occurrenceRemarks",
    "PROYECTO": "datasetName"
}

# Valores constantes para Flora Fustal
valores_constantes = {
    "type": "Event",
    "basisOfRecord": "HumanObservation",
    "occurrenceStatus": "present",
    "language": "es",
    "sampleSizeUnit": "Hectáreas",
    "continent": "América del Sur",
    "country": "Colombia",
    "countryCode": "CO",
    "geodeticDatum": "WGS84",
    "verbatimSRS": EPSG_ORIGEN,
    "verbatimCoordinateSystem": "Coordenadas proyectadas",
    "kingdom": "Plantae" 
}

# Lista completa de atributos en el archivo final
lista_atributos = [
    "id", "type", "language", "institutionID", "institutionCode", "datasetName",
    "basisOfRecord", "occurrenceID", "recordNumber", "individualCount",
    "occurrenceStatus", "occurrenceRemarks", "organismRemarks", "eventID",
    "parentEventID", "organismID", "fieldNumber", "eventDate", "year", "month", "day",
    "verbatimEventDate", "habitat", "samplingProtocol", "sampleSizeValue", "sampleSizeUnit",
    "eventRemarks", "continent", "waterBody", "country",
    "countryCode", "stateProvince", "county", "municipality", "locality",
    "minimumElevationInMeters", "maximumElevationInMeters", "verbatimElevation",
    "locationRemarks", "decimalLatitude", "decimalLongitude", "geodeticDatum",
    "verbatimCoordinates", "verbatimLatitude", "verbatimLongitude",
    "verbatimCoordinateSystem", "verbatimSRS", "footprintWKT", "footprintSRS",
    "verbatimIdentification", "identificationQualifier", "scientificName",
    "acceptedNameUsage", "higherClassification", "kingdom", "phylum", "class",
    "order", "family", "genus", "specificEpithet", "infraspecificEpithet",
    "taxonRank", "verbatimTaxonRank", "scientificNameAuthorship", "vernacularName",
    "taxonomicStatus", "measurementValue_1", "measurementType_1", "measurementUnit_1",
    "measurementValue_2", "measurementType_2", "measurementUnit_2",
    "measurementValue_3", "measurementType_3", "measurementUnit_3",
    "measurementValue_4", "measurementType_4", "measurementUnit_4",
    "measurementValue_5", "measurementType_5", "measurementUnit_5",
    "measurementValue_6", "measurementType_6", "measurementUnit_6",
    "measurementValue_7", "measurementType_7", "measurementUnit_7",
    "measurementValue_8", "measurementType_8", "measurementUnit_8",
    "permitText"
]


# Función para consultar la API de GBIF

def consultar_taxon_rank(especie):
    url = "https://api.gbif.org/v1/species/match"
    params = {"name": especie}
    
    try:
        time.sleep(1) 
        response = requests.get(url, params=params)
        if response.status_code == 200:
            data = response.json()
            scientific_name = data.get("scientificName", "")
            canonical_name = data.get("canonicalName", "")
            
            # Extraer la autoría eliminando el nombre científico del nombre canónico
            if canonical_name and scientific_name and scientific_name != canonical_name:
                scientific_name_authorship = scientific_name.replace(canonical_name, "").strip()
            else:
                scientific_name_authorship = None

            return {
                "taxonRank": data.get("rank", None),
                "canonicalName": canonical_name,
                "scientificNameAuthorship": scientific_name_authorship,
                "taxonomicStatus": data.get("status", None)
            }
        else:

            return None
    except Exception as e:

        return None

# 🔹 Función para agregar los datos de la API al archivo Excel existente
def agregar_datos_api_a_excel(ruta_excel_flora_fustal):
    try:

        df = pd.read_excel(ruta_excel_flora_fustal)

        if "ESPECIE" not in df.columns:
            raise ValueError("❌ ERROR: La columna 'ESPECIE' no está en el archivo Excel.")


        especies = df["ESPECIE"].tolist()

        # Ejecutar consultas a la API en paralelo
        with ThreadPoolExecutor(max_workers=50) as executor:
            resultados = list(executor.map(consultar_taxon_rank, especies))

        # Convertir a DataFrame
        datos_api = pd.DataFrame([r if isinstance(r, dict) else {} for r in resultados])

        # Agregar columnas nuevas al dataframe original
        df["taxonRank"] = datos_api["taxonRank"]
        df["canonicalName"] = datos_api["canonicalName"]
        df["scientificNameAuthorship"] = datos_api["scientificNameAuthorship"]
        df["taxonomicStatus"] = datos_api["taxonomicStatus"]


        df.to_excel(ruta_excel_flora_fustal, index=False)


    except Exception as e:
        return None

# Función para extraer coordenadas
def extraer_coordenadas(layer):
    resultados = []
    transformer = pyproj.Transformer.from_crs(EPSG_ORIGEN, EPSG_DESTINO, always_xy=True)

    for feature in layer:
        geom = feature.GetGeometryRef()
        if geom and geom.GetGeometryType() == ogr.wkbPoint:
            lon, lat = transformer.transform(geom.GetX(), geom.GetY())

            atributos = feature.items()
            atributos["latitud"] = lat
            atributos["longitud"] = lon
            resultados.append(atributos)

    return resultados


# Función para realizar el join
def realizar_join(capa, tabla, enlace_flora_fustal):
    try:


        capa_df = pd.DataFrame(capa)
        tabla_df = pd.DataFrame(tabla)

        if capa_df.empty or tabla_df.empty:

            return None

        resultado_df = pd.merge(capa_df, tabla_df, on=enlace_flora_fustal, how="inner")

        if resultado_df.empty:

            return None

        return resultado_df

    except Exception as e:

        return None

# Función para exportar a Excel
def exportar_excel(dataframe, ruta_salida):
    try:

        dataframe.to_excel(ruta_salida, index=False)

    except Exception as e:
        return None


# Función para extraer los nombres de los departamentos y los municipios
def convertir_codigos_nombres(df):
    """Convierte los códigos de departamentos y municipios a nombres."""
    if "DEPTO" in df.columns:

        df["stateProvince"] = df["DEPTO"].apply(lambda x: departamentos.get(x, "Desconocido"))
    else:
        df["stateProvince"] = None


    if "MUNICIPIO" in df.columns:

        df["county"] = df["MUNICIPIO"].apply(lambda x: municipios.get(x, "Desconocido"))
    else:
        df["county"] = None


    return df

# Función para procesar campos específicos
def procesar_campos_especificos(df):
    """Calcula los campos en orden secuencial asegurando que cada campo esté disponible antes de ser usado."""

    if df is None or df.empty:

        return df  

    #Mapear a parentEventID
    if "ID_MUEST" in df.columns:
        df["parentEventID"] = df["ID_MUEST"]
    else:

        df["parentEventID"] = None

    # Mapear FEC_MUEST a verbatimEventDate
    if "FEC_MUEST" in df.columns:
        df["verbatimEventDate"] = df["FEC_MUEST"]
    else:

        df["verbatimEventDate"] = None

    # Mapear COTA a verbatimElevation
    if "COTA" in df.columns:
        df["verbatimElevation"] = df["COTA"]
    else:

        df["verbatimElevation"] = None

     # Mapear COOR_NORTE a verbatimLatitude
    if "COOR_NORTE" in df.columns:
        df["verbatimLatitude"] = df["COOR_NORTE"]
    else:

        df["verbatimLatitude"] = None

    # Mapear COOR_ESTE a verbatimLongitude
    if "COOR_ESTE" in df.columns:
        df["verbatimLongitude"] = df["COOR_ESTE"]
    else:

        df["verbatimLongitude"] = None

    # Calcular  `recordNumber`
    if "OBJECTID" in df.columns:
        df["recordNumber"] = df["OBJECTID"]
    else:

        df["recordNumber"] = None

   
    # Calcular `occurrenceID`
    if "OPERADOR" in df.columns and "recordNumber" in df.columns:
        df["occurrenceID"] = df.apply(
            lambda row: re.sub(r'[^A-Za-z0-9]', '', row["OPERADOR"]) + ":Flora-Fustal:" + str(row["recordNumber"]),
            axis=1
        )
    else:

        df["occurrenceID"] = None

     # Asignar `occurrenceID` a `id`
    if "occurrenceID" in df.columns:
        df["id"] = df["occurrenceID"]
    else:

        df["id"] = None


    # Calcular `eventDate`, `year`, `month`, `day` y conservamos `verbatimEventDate`
    if "verbatimEventDate" in df.columns:

        df["eventDate"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.strftime('%Y-%m-%d')
        df["year"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.year
        df["month"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.month
        df["day"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.day
       
    else:

        df["eventDate"] = None
        df["year"] = None
        df["month"] = None
        df["day"] = None

    # Asignar `verbatimElevation` a `minimumElevationInMeters` y `maximumElevationInMeters`
    if "verbatimElevation" in df.columns:
        df["minimumElevationInMeters"] = df["verbatimElevation"]
        df["maximumElevationInMeters"] = df["verbatimElevation"]
    else:

        df["minimumElevationInMeters"] = None
        df["maximumElevationInMeters"] = None

    # Concatenar `COOR_ESTE` y `COOR_NORTE` para `verbatimCoordinates`
    if "verbatimLongitude" in df.columns and "verbatimLatitude" in df.columns:
        df["verbatimCoordinates"] = df.apply(
            lambda row: str(row["verbatimLongitude"]).replace(",", ".") + ", " + str(row["verbatimLatitude"]).replace(",", "."),
            axis=1
        )

    else:

        df["verbatimCoordinates"] = None


    # Asignar `ESPECIE` a `verbatimIdentification`
    if "ESPECIE" in df.columns:
        df["verbatimIdentification"] = df["ESPECIE"]
    else:

        df["verbatimIdentification"] = None


    # Agregar cada campo de medición individualmente
    df["measurementValue_1"] = df["DAP_INDIV"] if "DAP_INDIV" in df.columns else None
    df["measurementType_1"] = "Diámetro a la altura del pecho (DAP)" if "DAP_INDIV" in df.columns else None
    df["measurementUnit_1"] = "m" if "DAP_INDIV" in df.columns else None

    df["measurementValue_2"] = df["AB_INDIV"] if "AB_INDIV" in df.columns else None
    df["measurementType_2"] = "Área basal" if "AB_INDIV" in df.columns else None
    df["measurementUnit_2"] = "m2" if "AB_INDIV" in df.columns else None

    df["measurementValue_3"] = df["H_TOTAL"] if "H_TOTAL" in df.columns else None
    df["measurementType_3"] = "Altura total" if "H_TOTAL" in df.columns else None
    df["measurementUnit_3"] = "m" if "H_TOTAL" in df.columns else None

    df["measurementValue_4"] = df["H_FUSTE"] if "H_FUSTE" in df.columns else None
    df["measurementType_4"] = "Altura comercial" if "H_FUSTE" in df.columns else None
    df["measurementUnit_4"] = "m" if "H_FUSTE" in df.columns else None

    df["measurementValue_5"] = df["VOL_TOTAL"] if "VOL_TOTAL" in df.columns else None
    df["measurementType_5"] = "Volumen Total" if "VOL_TOTAL" in df.columns else None
    df["measurementUnit_5"] = "m3" if "VOL_TOTAL" in df.columns else None

    df["measurementValue_6"] = df["VOL_COM"] if "VOL_COM" in df.columns else None
    df["measurementType_6"] = "Volumen Comercial" if "VOL_COM" in df.columns else None
    df["measurementUnit_6"] = "m3" if "VOL_COM" in df.columns else None

    df["measurementValue_7"] = df["BIOM_INDIV"] if "BIOM_INDIV" in df.columns else None
    df["measurementType_7"] = "Biomasa aérea calculada" if "BIOM_INDIV" in df.columns else None
    df["measurementUnit_7"] = "kg" if "BIOM_INDIV" in df.columns else None

    df["measurementValue_8"] = df["CARB_INDIV"] if "CARB_INDIV" in df.columns else None
    df["measurementType_8"] = "Carbono calculado" if "CARB_INDIV" in df.columns else None
    df["measurementUnit_8"] = "kg" if "CARB_INDIV" in df.columns else None

     # Mapear higherClassification 
    campos_clasificacion = [
        "DIVISION", "CLASE", "ORDEN", "FAMILIA", "GENERO"
    ]

    for col in campos_clasificacion:
        if col not in df.columns:
            df[col] = None

    df["higherClassification"] = df[campos_clasificacion]\
        .applymap(lambda x: x if pd.notna(x) and str(x).strip().lower() not in ["", "none", "nan", "<null>"] else pd.NA)\
        .apply(lambda row: " | ".join(row.dropna().astype(str)), axis=1)

    # Mapear a scientificName con validación
    df["scientificName"] = df.apply(
        lambda row: (
            row["ESPECIE"]
            if pd.notna(row["ESPECIE"]) and str(row["ESPECIE"]).strip().lower() not in ["", "none", "nan", "<null>"]
            else (
                row["higherClassification"].split(" | ")[-1]
                if pd.notna(row["higherClassification"]) and " | " in row["higherClassification"]
                else row["higherClassification"]
            )
        ),
        axis=1
    )

    # Mapear permitText 
    df["permitText"] = df.apply(
        lambda row: "ANLA:{0}:{1}:{2}".format(
            row.get("NUM_ACT_AD") if pd.notna(row.get("NUM_ACT_AD")) and str(row.get("NUM_ACT_AD")).strip() != "" else "xxx",
            row.get("FEC_ACT_AD") if pd.notna(row.get("FEC_ACT_AD")) and str(row.get("FEC_ACT_AD")).strip() != "" else "xxx",
            row.get("OPERADOR")   if pd.notna(row.get("OPERADOR"))   and str(row.get("OPERADOR")).strip()   != "" else "xxx"
        ),
        axis=1
    )

    # Mapear ESTACIONAL a eventRemarks
    if "ESTACIONAL" in df.columns:
        mapeo_eventRemarks = {
            301: "Seco",
            302: "Húmedo",
            303: "Medio",
            304: "Todo el año"
        }
        df["eventRemarks"] = df["ESTACIONAL"].map(mapeo_eventRemarks).fillna("")
    else:

        df["eventRemarks"] = None

    return df





# Función principal para procesar Flora Fustal
def procesar_flora_fustal(ruta_gdb, capa_flora_fustal, tabla_flora_fustal, enlace_flora_fustal, ruta_excel_flora_fustal, archivo_entrada_flora_fustal, archivo_salida_flora_fustal):
    try:
        # Abrir la Geodatabase
        gdb = gdal.OpenEx(ruta_gdb, gdal.OF_VECTOR)
        if not gdb:
            raise RuntimeError(f"❌ No se pudo abrir la GDB en {ruta_gdb}")

        # Extraer coordenadas de la capa de fauna
        datos_capa = extraer_coordenadas(gdb.GetLayerByName(capa_flora_fustal))

        # Extraer atributos de la tabla de fauna
        datos_tabla = []
        layer = gdb.GetLayerByName(tabla_flora_fustal)
        for feature in layer:
            datos_tabla.append(feature.items())  # Obtiene los atributos correctamente


        resultado = realizar_join(datos_capa, datos_tabla, enlace_flora_fustal)

        if resultado is None or resultado.empty:

            return

        # Convertir códigos a nombres
        resultado = convertir_codigos_nombres(resultado)

        # Exportar el resultado del join a un archivo Excel intermedio
        exportar_excel(resultado, ruta_excel_flora_fustal)

        # Leer el archivo Excel intermedio y agregar taxonRank
        agregar_datos_api_a_excel(ruta_excel_flora_fustal)

        # Leer el archivo Excel con taxonRank agregado
        df_intermedio = pd.read_excel(ruta_excel_flora_fustal)

        # Procesar campos específicos
        df_intermedio = procesar_campos_especificos(df_intermedio)

        # Crear DataFrame final con todos los atributos de lista_atributos
        df_final = pd.DataFrame(columns=lista_atributos)

        # Mapear los datos del DataFrame intermedio al DataFrame final
        for columna_intermedia, columna_final in mapeo.items():
            if columna_intermedia in df_intermedio.columns:
                df_final[columna_final] = df_intermedio[columna_intermedia]

        # Agregar valores constantes
        for clave, valor in valores_constantes.items():
            df_final[clave] = valor

        # Agregar los campos calculados
       
        df_final["recordNumber"] = df_intermedio["recordNumber"]        
        df_final["eventDate"] = df_intermedio["eventDate"]
        df_final["year"] = df_intermedio["year"]
        df_final["month"] = df_intermedio["month"]
        df_final["day"] = df_intermedio["day"]
        df_final["verbatimEventDate"] = df_intermedio["verbatimEventDate"]
        df_final["verbatimElevation"] = df_intermedio["verbatimElevation"]
        df_final["minimumElevationInMeters"] = df_intermedio["minimumElevationInMeters"]
        df_final["maximumElevationInMeters"] = df_intermedio["maximumElevationInMeters"]
        df_final["verbatimCoordinates"] = df_intermedio["verbatimCoordinates"]
        df_final["verbatimIdentification"] = df_intermedio["verbatimIdentification"]
        df_final["occurrenceID"] = df_intermedio["occurrenceID"]
        df_final["id"] = df_intermedio["id"]
        df_final["stateProvince"] = df_intermedio["stateProvince"]
        df_final["county"] = df_intermedio["county"]

        df_final["measurementValue_1"] = df_intermedio["measurementValue_1"]
        df_final["measurementType_1"] = df_intermedio["measurementType_1"]
        df_final["measurementUnit_1"] = df_intermedio["measurementUnit_1"]

        df_final["measurementValue_2"] = df_intermedio["measurementValue_2"]
        df_final["measurementType_2"] = df_intermedio["measurementType_2"]
        df_final["measurementUnit_2"] = df_intermedio["measurementUnit_2"]

        df_final["measurementValue_3"] = df_intermedio["measurementValue_3"]
        df_final["measurementType_3"] = df_intermedio["measurementType_3"]
        df_final["measurementUnit_3"] = df_intermedio["measurementUnit_3"]

        df_final["measurementValue_4"] = df_intermedio["measurementValue_4"]
        df_final["measurementType_4"] = df_intermedio["measurementType_4"]
        df_final["measurementUnit_4"] = df_intermedio["measurementUnit_4"]

        df_final["measurementValue_5"] = df_intermedio["measurementValue_5"]
        df_final["measurementType_5"] = df_intermedio["measurementType_5"]
        df_final["measurementUnit_5"] = df_intermedio["measurementUnit_5"]

        df_final["measurementValue_6"] = df_intermedio["measurementValue_6"]
        df_final["measurementType_6"] = df_intermedio["measurementType_6"]
        df_final["measurementUnit_6"] = df_intermedio["measurementUnit_6"]

        df_final["measurementValue_7"] = df_intermedio["measurementValue_7"]
        df_final["measurementType_7"] = df_intermedio["measurementType_7"]
        df_final["measurementUnit_7"] = df_intermedio["measurementUnit_7"]

        df_final["measurementValue_8"] = df_intermedio["measurementValue_8"]
        df_final["measurementType_8"] = df_intermedio["measurementType_8"]
        df_final["measurementUnit_8"] = df_intermedio["measurementUnit_8"]

        df_final["permitText"] = df_intermedio["permitText"]
        df_final["higherClassification"] = df_intermedio["higherClassification"]
        df_final["scientificName"] = df_intermedio["scientificName"]
        df_final["eventRemarks"] = df_intermedio["eventRemarks"]


        # Exportar el DataFrame final a un archivo Excel
        exportar_excel(df_final, archivo_salida_flora_fustal)

    except Exception as e:
        return None