from shapely.geometry import shape, Point, MultiLineString
from shapely.ops import transform
from shapely.wkt import loads as wkt_loads
import pyproj
import pandas as pd
import requests
import re
from osgeo import ogr, osr
from concurrent.futures import ThreadPoolExecutor

from ..utils.codigos import departamentos, municipios
from ..utils.config import EPSG_ORIGEN, EPSG_DESTINO


# Diccionario de mapeo de atributos para Fauna Transecto
mapeo = {
    "OPERADOR": "institutionCode",
    "ABUND_ABS": "individualCount",
    "ID_MUES_PT": "eventID",
    "NOMBRE": "fieldNumber",
    "HABITAT": "habitat",
    "DESCRIP": "samplingProtocol",
    "LONGITUD_M":"sampleSizeValue",
    "CUERPO_AGU": "waterBody",
    #"DEPTO": "stateProvince",
    #"MUNICIPIO": "county",
    "VEREDA": "locality",
    "COTA_MIN": "minimumElevationInMeters",
    "COTA_MAX": "maximumElevationInMeters",
    "latitud": "decimalLatitude",
    "longitud": "decimalLongitude",
    "COOR_NORTE": "verbatimLatitude",
    "COOR_ESTE": "verbatimLongitude",
    "ESPECIE": "scientificName",  
    "canonicalName": "acceptedNameUsage",
    "scientificNameAuthorship": "scientificNameAuthorship",
    "taxonomicStatus": "taxonomicStatus",
    "DIVISION": "phylum",
    "CLASE": "class",
    "ORDEN": "order",
    "FAMILIA": "family",
    "GENERO": "genus",
    "taxonRank": "taxonRank",
    "N_COMUN": "vernacularName",
    "PROYECTO": "datasetName"
}

# Valores constantes para Fauna Transecto
valores_constantes = {
    "occurrenceStatus": "present",
    "language": "es",
    "continent": "América del Sur",
    "country": "Colombia",
    "countryCode": "CO",
    "geodeticDatum": "WGS84",
    "footprintSRS": EPSG_DESTINO,
    "verbatimCoordinateSystem": "Coordenadas proyectadas",
    "sampleSizeUnit": "Metros",
}

# Lista completa de atributos en el archivo final
lista_atributos = [
    "id", "type", "language", "institutionID", "institutionCode", "datasetName",
    "basisOfRecord", "occurrenceID", "recordNumber", "individualCount",
    "occurrenceStatus", "occurrenceRemarks", "organismRemarks", "eventID",
    "parentEventID", "fieldNumber", "eventDate", "year", "month", "day",
    "verbatimEventDate", "habitat", "samplingProtocol", "sampleSizeValue", "sampleSizeUnit",
    "eventRemarks", "continent", "waterBody", "country",
    "countryCode", "stateProvince", "county", "municipality", "locality",
    "minimumElevationInMeters", "maximumElevationInMeters", "verbatimElevation",
    "locationRemarks", "decimalLatitude", "decimalLongitude", "geodeticDatum",
    "verbatimCoordinates", "verbatimLatitude", "verbatimLongitude",
    "verbatimCoordinateSystem", "verbatimSRS", "footprintWKT", "footprintSRS",
    "verbatimIdentification", "identificationQualifier", "scientificName",
    "acceptedNameUsage", "higherClassification", "kingdom", "phylum", "class",
    "order", "family", "genus", "specificEpithet", "infraspecificEpithet",
    "taxonRank", "verbatimTaxonRank", "scientificNameAuthorship", "vernacularName",
    "taxonomicStatus", "permitText"
]



def consultar_taxon_rank(especie):
    url = "https://api.gbif.org/v1/species/match"
    params = {"name": especie}
    
    try:
        response = requests.get(url, params=params)
        if response.status_code == 200:
            data = response.json()
            scientific_name = data.get("scientificName", "")
            canonical_name = data.get("canonicalName", "")
            
            # Extrae la autoría eliminando el nombre científico del nombre canónico
            if canonical_name and scientific_name and scientific_name != canonical_name:
                scientific_name_authorship = scientific_name.replace(canonical_name, "").strip()
            else:
                scientific_name_authorship = None

            return {
                "taxonRank": data.get("rank", None),
                "canonicalName": canonical_name,
                "scientificNameAuthorship": scientific_name_authorship,
                "taxonomicStatus": data.get("status", None)
            }
        else:

            return None
    except Exception as e:

        return None

# Función para agregar los datos de la API al archivo Excel existente
def agregar_datos_api_a_excel(ruta_excel_fauna_transecto):
    try:

        df = pd.read_excel(ruta_excel_fauna_transecto)

        if "ESPECIE" not in df.columns:
            raise ValueError("❌ ERROR: La columna 'ESPECIE' no está en el archivo Excel.")


        especies = df["ESPECIE"].tolist()

        # Ejecutar consultas a la API en paralelo
        with ThreadPoolExecutor(max_workers=100) as executor:
            resultados = list(executor.map(consultar_taxon_rank, especies))

        # Convertir a DataFrame
        datos_api = pd.DataFrame([r if isinstance(r, dict) else {} for r in resultados])

        # Agregar columnas nuevas al dataframe original
        df["taxonRank"] = datos_api["taxonRank"]
        df["canonicalName"] = datos_api["canonicalName"]
        df["scientificNameAuthorship"] = datos_api["scientificNameAuthorship"]
        df["taxonomicStatus"] = datos_api["taxonomicStatus"]


        df.to_excel(ruta_excel_fauna_transecto, index=False)


    except Exception as e:
        return None


# Función para convertir geometría de OGR a Shapely
def ogr_to_shapely(feature):
    geom_ogr = feature.GetGeometryRef()  # Obtener geometría OGR
    if geom_ogr is None:
        return None
    return shape(wkt_loads(geom_ogr.ExportToWkt()))  # Convertir OGR → Shapely directamente

# Función para extraer coordenadas y calcular WKT, centroide, y vertice
def extraer_coordenadas(capa):
    resultados = []
    try:
        transformer = pyproj.Transformer.from_crs(EPSG_ORIGEN, EPSG_DESTINO, always_xy=True)

        for feature in capa:
            geom = ogr_to_shapely(feature)  # Convertir OGR → Shapely

            if geom is None:

                continue

            # Transformar la geometría a EPSG:4326
            geom_4326 = transform(transformer.transform, geom)

            # Convertir la geometría transformada a WKT
            wkt = geom_4326.wkt

            centroid = geom.centroid

            # Transformar el centroide a EPSG:4326
            centroid_x, centroid_y = transformer.transform(centroid.x, centroid.y)

            # Manejar geometrías múltiples
            coords = []
            if isinstance(geom, MultiLineString):
                for line in geom.geoms:  # Iterar sobre las sub-geometrías
                    coords.extend(line.coords)
            else:
                coords = geom.coords

            # Encontrar el vértice más cercano al centroide
            nearest_vertex = None
            min_distance = float('inf')
            for coord in coords:
                point = Point(coord)
                distance = centroid.distance(point)
                if distance < min_distance:
                    min_distance = distance
                    nearest_vertex = coord

            # Transformar el vértice más cercano a EPSG:4326
            nearest_vertex_x, nearest_vertex_y = transformer.transform(nearest_vertex[0], nearest_vertex[1])

            atributos = {field: feature.GetField(field) for field in feature.keys()}
            atributos["footprintWKT"] = wkt
            atributos["decimalLatitude"] = nearest_vertex_y
            atributos["decimalLongitude"] = nearest_vertex_x
            resultados.append(atributos)


    except Exception as e:
        return None


    return resultados

# Función para realizar el join
def realizar_join(capa, tabla, enlace_fauna_transecto):
    try:


        capa_df = pd.DataFrame(capa)
        tabla_df = pd.DataFrame(tabla)

        if capa_df.empty or tabla_df.empty:

            return None

        resultado_df = pd.merge(capa_df, tabla_df, on=enlace_fauna_transecto, how="inner")

        if resultado_df.empty:

            return None

        return resultado_df

    except Exception as e:

        return None

# Función para exportar a Excel
def exportar_excel(dataframe, ruta_salida):
    try:

        dataframe.to_excel(ruta_salida, index=False)

    except Exception as e:
        return None

# Función para extraer los nombres de los departamentos y los municipios
def convertir_codigos_nombres(df):
    """Convierte los códigos de departamentos y municipios a nombres."""
    if "DEPTO" in df.columns:

        df["stateProvince"] = df["DEPTO"].apply(lambda x: departamentos.get(x, "Desconocido"))
    else:
        df["stateProvince"] = None


    if "MUNICIPIO" in df.columns:

        df["county"] = df["MUNICIPIO"].apply(lambda x: municipios.get(x, "Desconocido"))
    else:
        df["county"] = None


    return df

# Función para procesar campos específicos
def procesar_campos_especificos(df):
    """Calcula los campos en orden secuencial asegurando que cada campo esté disponible antes de ser usado."""

    if df is None or df.empty:

        return df  

    # Mapear FEC_MUEST a verbatimEventDate
    if "FEC_MUEST" in df.columns:
        df["verbatimEventDate"] = df["FEC_MUEST"]
    else:

        df["verbatimEventDate"] = None


    #  Calcula `recordNumber`
    if "OBJECTID" in df.columns:
        df["recordNumber"] = df["OBJECTID"]
    else:

        df["recordNumber"] = None
    

    # Calcula `type`, ya que otros campos dependerán de él
    if "DETERM" in df.columns:
        mapeo_type = {
            411: "Event",
            413: "Event",
            414: "PhysicalObject",
            415: "Sound",
            416: "PhysicalObject",
            417: "PhysicalObject",
            418: "PhysicalObject",
            419: ""
        }
        df["type"] = df["DETERM"].map(mapeo_type).fillna("")
    else:

        df["type"] = None

    # Calcula `occurrenceID`
    if "OPERADOR" in df.columns and "recordNumber" in df.columns:
        df["occurrenceID"] = df.apply(
            lambda row: re.sub(r'[^A-Za-z0-9]', '', row["OPERADOR"]) + ":Fauna-Transecto:" + str(row["recordNumber"]),
            axis=1
        )
    else:

        df["occurrenceID"] = None

     # Asigna `occurrenceID` a `id`
    if "occurrenceID" in df.columns:
        df["id"] = df["occurrenceID"]
    else:

        df["id"] = None

    # Calcula `basisOfRecord`, que depende de `type`
    if "type" in df.columns:
        mapeo_basis = {
            "Event": "HumanObservation",
            "StillImage": "MachineObservation",
            "PhysicalObject": "MaterialSample",
            "Sound": "HumanObservation"
        }
        df["basisOfRecord"] = df["type"].map(mapeo_basis).fillna("Desconocido")
    else:

        df["basisOfRecord"] = None

    # Calcula `occurrenceRemarks`, que depende de `basisOfRecord`
    if "basisOfRecord" in df.columns and "DETERM" in df.columns:
        mapeo_determ = {
            411: "Captura de individuos",
            413: "Observación",
            414: "Marcas de Individuos",
            415: "Detección auditiva",
            416: "Huellas",
            417: "Heces",
            418: "Pelos",
            419: "Otro"
        }

        usar_ot_det = "OT_DETERM" in df.columns

        def obtener_occurrence(row):
            valor = mapeo_determ.get(row["DETERM"], "Desconocido")
            if valor == "Otro":
                if usar_ot_det:
                    return row["OT_DETERM"]
                else:
                    return "Otro" 
            return valor

        df["occurrenceRemarks"] = df.apply(obtener_occurrence, axis=1)

    else:

        df["occurrenceRemarks"] = None

    # Calcula `eventDate`, `year`, `month`, `day` y se conserva `verbatimEventDate`
    if "verbatimEventDate" in df.columns:

        df["eventDate"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.strftime('%Y-%m-%d')
        df["year"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.year
        df["month"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.month
        df["day"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.day

    else:

        df["eventDate"] = None
        df["year"] = None
        df["month"] = None
        df["day"] = None

    # Asignar `ESPECIE` a `verbatimIdentification`
    if "ESPECIE" in df.columns:
        df["verbatimIdentification"] = df["ESPECIE"]
    else:

        df["verbatimIdentification"] = None

    # Mapear higherClassification 
    campos_clasificacion = [
        "DIVISION", "CLASE", "ORDEN", "FAMILIA", "GENERO"
    ]

    for col in campos_clasificacion:
        if col not in df.columns:
            df[col] = None

    df["higherClassification"] = df[campos_clasificacion]\
        .applymap(lambda x: x if pd.notna(x) and str(x).strip().lower() not in ["", "none", "nan", "<null>"] else pd.NA)\
        .apply(lambda row: " | ".join(row.dropna().astype(str)), axis=1)

    # Mapear a scientificName con validación
    df["scientificName"] = df.apply(
        lambda row: (
            row["ESPECIE"]
            if pd.notna(row["ESPECIE"]) and str(row["ESPECIE"]).strip().lower() not in ["", "none", "nan", "<null>"]
            else (
                row["higherClassification"].split(" | ")[-1]
                if pd.notna(row["higherClassification"]) and " | " in row["higherClassification"]
                else row["higherClassification"]
            )
        ),
        axis=1
    )

    # Mapear permitText 
    df["permitText"] = df.apply(
        lambda row: "ANLA:{0}:{1}:{2}".format(
            row.get("NUM_ACT_AD") if pd.notna(row.get("NUM_ACT_AD")) and str(row.get("NUM_ACT_AD")).strip() != "" else "xxx",
            row.get("FEC_ACT_AD") if pd.notna(row.get("FEC_ACT_AD")) and str(row.get("FEC_ACT_AD")).strip() != "" else "xxx",
            row.get("OPERADOR")   if pd.notna(row.get("OPERADOR"))   and str(row.get("OPERADOR")).strip()   != "" else "xxx"
        ),
        axis=1
    )

    # Mapear ESTACIONAL a eventRemarks
    if "ESTACIONAL" in df.columns:
        mapeo_eventRemarks = {
            301: "Seco",
            302: "Húmedo",
            303: "Medio",
            304: "Todo el año"
        }
        df["eventRemarks"] = df["ESTACIONAL"].map(mapeo_eventRemarks).fillna("")
    else:

        df["eventRemarks"] = None

    return df



# Función principal para procesar fauna transecto
def procesar_fauna_transecto(ruta_gdb, capa_fauna_transecto, tabla_fauna_transecto, enlace_fauna_transecto, ruta_excel_fauna_transecto, archivo_entrada_fauna_transecto, archivo_salida_fauna_transecto):
    try:
         # Abrir la geodatabase con GDAL
        driver = ogr.GetDriverByName("OpenFileGDB")
        gdb = driver.Open(ruta_gdb, 0)
        if gdb is None:

            return

        # Cargar la capa de fauna transecto con GDAL
        capa = gdb.GetLayerByName(capa_fauna_transecto)
        if capa is None:

            return

        datos_capa = extraer_coordenadas(capa) 

        # Cargar la tabla de fauna con GDAL
        tabla = gdb.GetLayerByName(tabla_fauna_transecto)
        if tabla is None:

            return

        datos_tabla = [registro.items() for registro in tabla]

        resultado = realizar_join(datos_capa, datos_tabla, enlace_fauna_transecto)

        if resultado is None or resultado.empty:

            return

        # Convertir códigos a nombres
        resultado = convertir_codigos_nombres(resultado)

        
        # Exportar el resultado del join a un archivo Excel intermedio
        exportar_excel(resultado, ruta_excel_fauna_transecto)

        # Leer el archivo Excel intermedio y agregar taxonRank
        agregar_datos_api_a_excel(ruta_excel_fauna_transecto)

        # Leer el archivo Excel con taxonRank agregado
        df_intermedio = pd.read_excel(ruta_excel_fauna_transecto)

        # Procesar campos específicos
        df_intermedio = procesar_campos_especificos(df_intermedio)

        # Crear DataFrame final con todos los atributos de lista_atributos
        df_final = pd.DataFrame(columns=lista_atributos)

        # Mapear los datos del DataFrame intermedio al DataFrame final
        for columna_intermedia, columna_final in mapeo.items():
            if columna_intermedia in df_intermedio.columns:
                df_final[columna_final] = df_intermedio[columna_intermedia]

        # Agregar valores constantes
        for clave, valor in valores_constantes.items():
            df_final[clave] = valor

        # Agregar los campos calculados
        df_final["type"] = df_intermedio["type"]
        df_final["recordNumber"] = df_intermedio["recordNumber"]
        df_final["basisOfRecord"] = df_intermedio["basisOfRecord"]
        df_final["occurrenceRemarks"] = df_intermedio["occurrenceRemarks"]
        df_final["eventDate"] = df_intermedio["eventDate"]
        df_final["year"] = df_intermedio["year"]
        df_final["month"] = df_intermedio["month"]
        df_final["day"] = df_intermedio["day"]
        df_final["verbatimEventDate"] = df_intermedio["verbatimEventDate"]       
        df_final["verbatimIdentification"] = df_intermedio["verbatimIdentification"]
        df_final["occurrenceID"] = df_intermedio["occurrenceID"]
        df_final["id"] = df_intermedio["id"]
        df_final["footprintWKT"] = df_intermedio["footprintWKT"]
        df_final["decimalLatitude"] = df_intermedio["decimalLatitude"]
        df_final["decimalLongitude"] = df_intermedio["decimalLongitude"]
        df_final["stateProvince"] = df_intermedio["stateProvince"]
        df_final["county"] = df_intermedio["county"]
        df_final["higherClassification"] = df_intermedio["higherClassification"]
        df_final["scientificName"] = df_intermedio["scientificName"]
        df_final["permitText"] = df_intermedio["permitText"]
        df_final["eventRemarks"] = df_intermedio["eventRemarks"]



        # Exportar el DataFrame final a un archivo Excel
        exportar_excel(df_final, archivo_salida_fauna_transecto)

    except Exception as e:
        return None