from pathlib import Path
from osgeo import gdal
import sys
from PyQt5.QtCore import QCoreApplication
import time
import os



from .utils.fauna import procesar_fauna
from .utils.flora_fustal import procesar_flora_fustal
from .utils.fauna_transecto import procesar_fauna_transecto
from .utils.flora_regeneracion import procesar_flora_regeneracion
from .utils.aprovechamiento_forestal import procesar_aprovechamiento
from .utils.muestreo_hidrobiotico import procesar_hidrobiotico
from .utils.muestreo_fisicoquimico import procesar_fisicoquimico
from .utils.agua_marino import procesar_marino
from .utils.hidrobio_marino import procesar_hidrobio
from .utils.fisicoquim_marino import procesar_fisicoquim
from .utils.mamiferos_OFM import procesar_mamiferos_OFM
from .utils.aves_OFM import procesar_aves_OFM
from .utils.peces_OFM import procesar_peces_OFM
from .utils.tortugas_OFM import procesar_tortugas_OFM
from .utils.otrafauna_OFM import procesar_otrafauna_OFM
from .utils.veda import procesar_veda
from .utils.fauna_marina import procesar_fauna_marina
from .utils.transecto_fauna_marina import procesar_transecto_fm
from .utils.flora_marina import procesar_flora_marina

def main(gdb_path=None, output_dir=None, progress=None):
    BASE_DIR = Path(__file__).resolve().parent
    ruta_gdb = Path(gdb_path)
    output_dir = Path(output_dir) if output_dir and output_dir.strip() else BASE_DIR / "output"
    output_dir.mkdir(parents=True, exist_ok=True)  # <-- crea si no existe
    





    # Obtener capas-tablas disponibles
    gdb = gdal.OpenEx(str(ruta_gdb), gdal.OF_VECTOR)
    capas_disponibles = set()
    if gdb:
        for i in range(gdb.GetLayerCount()):
            layer = gdb.GetLayerByIndex(i)
            capas_disponibles.add(layer.GetName())

    # Registro de módulos a ejecutar
    modulos_disponibles = {
        "fauna": {
            "func": procesar_fauna,
            "capas": ["PuntoMuestreoFauna"],
            "tablas": ["MuestreoFaunaTB"],
            "args": lambda: [
                str(ruta_gdb), "PuntoMuestreoFauna", "MuestreoFaunaTB", "ID_MUES_PT",
                str(output_dir / "cruce_fauna.xlsx"),
                str(output_dir / "cruce_fauna.xlsx"),
                str(output_dir / "fauna_final.xlsx")
            ]
        },
        "flora_fustal": {
            "func": procesar_flora_fustal,
            "capas": ["PuntoMuestreoFlora"],
            "tablas": ["MuestreoFloraFustalTB"],
            "args": lambda: [
                str(ruta_gdb), "PuntoMuestreoFlora", "MuestreoFloraFustalTB", "ID_MUEST",
                str(output_dir / "cruce_flora_fustal.xlsx"),
                str(output_dir / "cruce_flora_fustal.xlsx"),
                str(output_dir / "flora_fustal_final.xlsx")
            ]
        },
        "fauna_transecto": {
            "func": procesar_fauna_transecto, 
            "capas": ["TransectoMuestreoFauna"],
            "tablas": ["MuestreoFaunaTB"],
            "args": lambda: [
                str(ruta_gdb), "TransectoMuestreoFauna", "MuestreoFaunaTB", "ID_MUES_TR",
                str(output_dir / "cruce_fauna_transecto.xlsx"),
                str(output_dir / "cruce_fauna_transecto.xlsx"),
                str(output_dir / "fauna_transecto_final.xlsx")
            ]
        },
        "flora_regeneracion": {
            "func": procesar_flora_regeneracion,
            "capas": ["PuntoMuestreoFlora"],
            "tablas": ["MuestreoFloraRegeneracionTB"],
            "args": lambda: [
                str(ruta_gdb), "PuntoMuestreoFlora", "MuestreoFloraRegeneracionTB", "ID_MUEST",
                str(output_dir / "cruce_flora_regeneracion.xlsx"),
                str(output_dir / "cruce_flora_regeneracion.xlsx"),
                str(output_dir / "flora_regeneracion_final.xlsx")
            ]
        },
        "aprovechamiento": {
            "func": procesar_aprovechamiento,
            "capas": ["AprovechaForestalPT"],
            "tablas": [],
            "args": lambda: [
                str(ruta_gdb), "AprovechaForestalPT",
                str(output_dir / "cruce_aprovechamiento.xlsx"),
                str(output_dir / "cruce_aprovechamiento.xlsx"),
                str(output_dir / "aprovechamiento_final.xlsx")
            ]
        },
        "hidrobiotico": {
            "func": procesar_hidrobiotico,
            "capas": ["PuntoMuestreoAguaSuper"],
            "tablas": ["MuestreoHidrobioTB"],
            "args": lambda: [
                str(ruta_gdb), "PuntoMuestreoAguaSuper", "MuestreoHidrobioTB", "ID_PUNTO_M",
                str(output_dir / "cruce_hidrobiotico.xlsx"),
                str(output_dir / "cruce_hidrobiotico.xlsx"),
                str(output_dir / "hidrobiotico_final.xlsx")
            ]
        },
        "fisicoquimico": {
            "func": procesar_fisicoquimico,
            "capas": [],
            "tablas": ["MuestreoFisicoquimSuperTB", "ParamFisicoquimSuperTB"],
            "args": lambda: [
                str(ruta_gdb), "MuestreoFisicoquimSuperTB", "ParamFisicoquimSuperTB", "ID_MUESTRA",
                str(output_dir / "cruce_fisicoquimico.xlsx"),
                str(output_dir / "cruce_fisicoquimico.xlsx"),
                str(output_dir / "fisicoquimico_final.xlsx")
            ]
        },
        "agua_marino": {
            "func": procesar_marino,
            "capas": ["PuntoMuestreoAguaMarino"],
            "tablas": ["MuestreoHidrobioMarinoTB"],
            "args": lambda: [
                str(ruta_gdb), "PuntoMuestreoAguaMarino", "MuestreoHidrobioMarinoTB", "ID_PUNTO_M",
                str(output_dir / "cruce_marino.xlsx"),
                str(output_dir / "cruce_marino.xlsx"),
                str(output_dir / "marino_evento_final.xlsx")
            ]
        },
        "hidrobio_marino": {
            "func": procesar_hidrobio,
            "capas": [],
            "tablas": ["MuestreoHidrobioMarinoTB"],
            "args": lambda: [
                str(ruta_gdb), "MuestreoHidrobioMarinoTB",
                str(output_dir / "cruce_hidrobio_marino.xlsx"),
                str(output_dir / "cruce_hidrobio_marino.xlsx"),
                str(output_dir / "hidrobio_marino_final.xlsx")
            ]
        },
        "fisicoquim_marino": {
            "func": procesar_fisicoquim,
            "capas": [],
            "tablas": ["MuestreoFisicoquimMarinoTB"],
            "args": lambda: [
                str(ruta_gdb), "MuestreoFisicoquimMarinoTB",
                str(output_dir / "cruce_fisicoquim_marino.xlsx"),
                str(output_dir / "cruce_fisicoquim_marino.xlsx"),
                str(output_dir / "fisicoquim_marino_final.xlsx")
            ]
        },
        "mamiferos_OFM": {
            "func": procesar_mamiferos_OFM,
            "capas": ["PuntoObservacionFaunaMarina"],
            "tablas": ["MamiferosOFM_TB"],
            "args": lambda: [
                str(ruta_gdb), "PuntoObservacionFaunaMarina", "MamiferosOFM_TB", "ID_INGRESO",
                str(output_dir / "cruce_mamiferos_OFM.xlsx"),
                str(output_dir / "cruce_fauna.xlsx"),
                str(output_dir / "mamiferos_OFM_final.xlsx")
            ]
        },
        "aves_OFM": {
            "func": procesar_aves_OFM,
            "capas": ["PuntoObservacionFaunaMarina"],
            "tablas": ["AvesOFM_TB"],
            "args": lambda: [
                str(ruta_gdb), "PuntoObservacionFaunaMarina", "AvesOFM_TB", "ID_INGRESO",
                str(output_dir / "cruce_aves_OFM.xlsx"),
                str(output_dir / "cruce_fauna.xlsx"),
                str(output_dir / "aves_OFM_final.xlsx")
            ]
        },
        "peces_OFM": {
            "func": procesar_peces_OFM,
            "capas": ["PuntoObservacionFaunaMarina"],
            "tablas": ["PecesOFM_TB"],
            "args": lambda: [
                str(ruta_gdb), "PuntoObservacionFaunaMarina", "PecesOFM_TB", "ID_INGRESO",
                str(output_dir / "cruce_peces_OFM.xlsx"),
                str(output_dir / "cruce_fauna.xlsx"),
                str(output_dir / "peces_OFM_final.xlsx")
            ]
        },
        "tortugas_OFM": {
            "func": procesar_tortugas_OFM,
            "capas": ["PuntoObservacionFaunaMarina"],
            "tablas": ["TortugasOFM_TB"],
            "args": lambda: [
                str(ruta_gdb), "PuntoObservacionFaunaMarina", "TortugasOFM_TB", "ID_INGRESO",
                str(output_dir / "cruce_tortugas_OFM.xlsx"),
                str(output_dir / "cruce_fauna.xlsx"),
                str(output_dir / "tortugas_OFM_final.xlsx")
            ]
        },
        "otrafauna_OFM": {
            "func": procesar_otrafauna_OFM,
            "capas": ["PuntoObservacionFaunaMarina"],
            "tablas": ["OtraFaunaOFM_TB"],
            "args": lambda: [
                str(ruta_gdb), "PuntoObservacionFaunaMarina", "OtraFaunaOFM_TB", "ID_INGRESO",
                str(output_dir / "cruce_otrafauna_OFM.xlsx"),
                str(output_dir / "cruce_fauna.xlsx"),
                str(output_dir / "otrafauna_OFM_final.xlsx")
            ]
        },
        "veda": {
            "func": procesar_veda,
            "capas": ["PuntoMuestreoVeda"],
            "tablas": ["MuestreoVedaTB"],
            "args": lambda: [
                str(ruta_gdb), "PuntoMuestreoVeda", "MuestreoVedaTB", "ID_VEDA",
                str(output_dir / "cruce_veda.xlsx"),
                str(output_dir / "cruce_veda.xlsx"),
                str(output_dir / "veda_final.xlsx")
            ]
        },
        "fauna_marina": {
            "func": procesar_fauna_marina,
            "capas": ["MuestreoFaunaMarina"],
            "tablas": ["MuestreoFaunaMarinaTB"],
            "args": lambda: [
                str(ruta_gdb), "MuestreoFaunaMarina", "MuestreoFaunaMarinaTB", "ID_MUEST_P",
                str(output_dir / "cruce_fauna_marina.xlsx"),
                str(output_dir / "cruce_fauna_marina.xlsx"),
                str(output_dir / "fauna_marina_final.xlsx")
            ]
        },
        "transecto_fm": {
            "func": procesar_transecto_fm,
            "capas": ["TansectoFaunaMarina"],
            "tablas": ["MuestreoFaunaMarinaTB"],
            "args": lambda: [
                str(ruta_gdb), "TansectoFaunaMarina", "MuestreoFaunaMarinaTB", "ID_MUEST_T",
                str(output_dir / "cruce_transecto_fm.xlsx"),
                str(output_dir / "cruce_transecto_fm.xlsx"),
                str(output_dir / "transecto_fm_final.xlsx")
            ]
        },
        "flora_marina": {
            "func": procesar_flora_marina,
            "capas": ["MuestreoFloraMarina"],
            "tablas": ["MuestreoFloraMarinaTB"],
            "args": lambda: [
                str(ruta_gdb), "MuestreoFloraMarina", "MuestreoFloraMarinaTB", "ID_MUEST",
                str(output_dir / "cruce_flora_marina.xlsx"),
                str(output_dir / "cruce_flora_marina.xlsx"),
                str(output_dir / "flora_marina_final.xlsx")
            ]
        },

        
    }

    # Se ejecutan los módulos disponibles
    archivos_eliminados = set()
    for nombre, config in modulos_disponibles.items():
        if all(c in capas_disponibles for c in config["capas"]) and all(t in capas_disponibles for t in config["tablas"]):
            try:
                if progress:
                    progress.setLabelText(f"Ejecutando módulo: {nombre}")
                    progress.repaint()
                    QCoreApplication.processEvents()


                config["func"](*config["args"]())
                time.sleep(5)  

                args = config["args"]()
                for arg in args:
                    if isinstance(arg, str) and "cruce_" in arg and arg.endswith(".xlsx"):
                        if arg not in archivos_eliminados:
                            try:
                                os.remove(arg)
                                archivos_eliminados.add(arg)

                            except FileNotFoundError:
                                pass  # Ya se eliminó o no existe
                            except Exception as e:
                                pass


            except Exception as e:
                pass
        


if __name__ == "__main__":
    import sys
    gdb_path_arg = sys.argv[1] if len(sys.argv) > 1 else None
    main(gdb_path_arg)

