from shapely.geometry import shape
import pyproj
import pandas as pd
import requests
import re
from osgeo import gdal, ogr
from concurrent.futures import ThreadPoolExecutor
import time
from ..utils.codigos import departamentos, municipios
from ..utils.config import EPSG_ORIGEN, EPSG_DESTINO


# Diccionario de mapeo de atributos para veda 
mapeo = {
    "OPERADOR": "institutionCode",
    "INDIVIDUOS": "individualCount",
    "ID_VEDA": "eventID",
    "ID_INDV_MU": "parentEventID",
    "NOMBRE": "fieldNumber",
    "N_COBERT": "habitat",
    "AREA_UM_ha": "sampleSizeValue",
    "DESCRIPCION": "samplingProtocol",
    "CUERPO_AGU": "waterBody",
    #"DEPTO": "stateProvince",
    #"MUNICIPIO": "county",
    "VEREDA": "locality",
    "latitud": "decimalLatitude",
    "longitud": "decimalLongitude",
    "COOR_NORTE": "verbatimLatitude",
    "COOR_ESTE": "verbatimLongitude",
    "LOCALIDAD": "locationRemarks",
    "ESPECIE": "scientificName", 
    "canonicalName": "acceptedNameUsage",
    "scientificNameAuthorship": "scientificNameAuthorship",
    "taxonomicStatus": "taxonomicStatus",
    "DIVISION": "phylum",
    "CLASE": "class",
    "ORDEN": "order",
    "FAMILIA": "family",
    "GENERO": "genus",
    "taxonRank": "taxonRank",
    "N_COMUN": "vernacularName",
    "OBSERV": "occurrenceRemarks",
    "OBSERVACIONES": "eventRemarks",
    "PROYECTO": "datasetName"
}

# Valores constantes para veda 
valores_constantes = {
    "type": "Event",
    "basisOfRecord": "HumanObservation",
    "occurrenceStatus": "present",
    "language": "es",
    #"sampleSizeUnit": "Hectáreas",
    "continent": "América del Sur",
    "country": "Colombia",
    "countryCode": "CO",
    "geodeticDatum": "WGS84",
    "verbatimSRS": EPSG_ORIGEN,
    "verbatimCoordinateSystem": "Coordenadas proyectadas",
    "kingdom": "Plantae"
}

# Lista completa de atributos en el archivo final
lista_atributos = [
    "id", "type", "language", "institutionID", "institutionCode", "datasetName",
    "basisOfRecord", "occurrenceID", "recordNumber", "individualCount",
    "occurrenceStatus", "occurrenceRemarks", "organismRemarks", "eventID",
    "parentEventID", "fieldNumber", "eventDate", "year", "month", "day",
    "verbatimEventDate", "habitat", "samplingProtocol", "sampleSizeValue", "sampleSizeUnit",
    "eventRemarks", "continent", "waterBody", "country",
    "countryCode", "stateProvince", "county", "municipality", "locality",
    "minimumElevationInMeters", "maximumElevationInMeters", "verbatimElevation",
    "locationRemarks", "decimalLatitude", "decimalLongitude", "geodeticDatum",
    "verbatimCoordinates", "verbatimLatitude", "verbatimLongitude",
    "verbatimCoordinateSystem", "verbatimSRS", "footprintWKT", "footprintSRS",
    "verbatimIdentification", "identificationQualifier", "scientificName",
    "acceptedNameUsage", "higherClassification", "kingdom", "phylum", "class",
    "order", "family", "genus", "specificEpithet", "infraspecificEpithet",
    "taxonRank", "verbatimTaxonRank", "scientificNameAuthorship", "vernacularName",
    "taxonomicStatus", "measurementValue", "measurementType",
    "measurementValue_1", "measurementType_1"
]


def consultar_taxon_rank(especie):
    url = "https://api.gbif.org/v1/species/match"
    params = {"name": especie}
    
    try:
        time.sleep(1)  
        response = requests.get(url, params=params)
        if response.status_code == 200:
            data = response.json()
            scientific_name = data.get("scientificName", "")
            canonical_name = data.get("canonicalName", "")
            
            # Extraer la autoría eliminando el nombre científico del nombre canónico
            if canonical_name and scientific_name and scientific_name != canonical_name:
                scientific_name_authorship = scientific_name.replace(canonical_name, "").strip()
            else:
                scientific_name_authorship = None

            return {
                "taxonRank": data.get("rank", None),
                "canonicalName": canonical_name,
                "scientificNameAuthorship": scientific_name_authorship,
                "taxonomicStatus": data.get("status", None)
            }
        else:

            return None
    except Exception as e:

        return None

# 🔹 Función para agregar los datos de la API al archivo Excel existente
def agregar_datos_api_a_excel(ruta_excel_veda):
    try:

        df = pd.read_excel(ruta_excel_veda)

        if "ESPECIE" not in df.columns:
            raise ValueError("❌ ERROR: La columna 'ESPECIE' no está en el archivo Excel.")



        especies = df["ESPECIE"].tolist()

        # Ejecutar consultas a la API en paralelo 
        with ThreadPoolExecutor(max_workers=100) as executor:
            resultados = list(executor.map(consultar_taxon_rank, especies))

        # Convertir a DataFrame
        datos_api = pd.DataFrame(resultados)

        # Agregar columnas nuevas al dataframe original
        df["taxonRank"] = datos_api["taxonRank"]
        df["canonicalName"] = datos_api["canonicalName"]
        df["scientificNameAuthorship"] = datos_api["scientificNameAuthorship"]
        df["taxonomicStatus"] = datos_api["taxonomicStatus"]

        # Guardar de nuevo el Excel

        df.to_excel(ruta_excel_veda, index=False)


    except Exception as e:
        return None
    
# 🔹 Función para extraer coordenadas
def extraer_coordenadas(layer):
    resultados = []
    transformer = pyproj.Transformer.from_crs(EPSG_ORIGEN, EPSG_DESTINO, always_xy=True)

    for feature in layer:
        geom = feature.GetGeometryRef()
        if geom and geom.GetGeometryType() == ogr.wkbPoint:
            lon, lat = transformer.transform(geom.GetX(), geom.GetY())

            atributos = feature.items()
            atributos["latitud"] = lat
            atributos["longitud"] = lon
            resultados.append(atributos)

    return resultados


# Función para realizar el join
def realizar_join(capa, tabla, enlace_veda):
    try:


        capa_df = pd.DataFrame(capa)
        tabla_df = pd.DataFrame(tabla)

        if capa_df.empty or tabla_df.empty:

            return None

        resultado_df = pd.merge(capa_df, tabla_df, on=enlace_veda, how="inner")

        if resultado_df.empty:

            return None

        return resultado_df

    except Exception as e:

        return None

# Función para exportar a Excel
def exportar_excel(dataframe, ruta_salida):
    try:

        dataframe.to_excel(ruta_salida, index=False)

    except Exception as e:
        return None


# Función para extraer los nombres de los departamentos y los municipios
def convertir_codigos_nombres(df):
    """Convierte los códigos de departamentos y municipios a nombres."""
    if "DEPARTAMENTO" in df.columns:

        df["stateProvince"] = df["DEPARTAMENTO"].apply(lambda x: departamentos.get(x, "Desconocido"))
    else:

        df["stateProvince"] = None

    if "MUNICIPIO" in df.columns:

        df["county"] = df["MUNICIPIO"].apply(lambda x: municipios.get(x, "Desconocido"))
    else:

        df["county"] = None

    return df

# Función para procesar campos específicos
def procesar_campos_especificos(df):
    """Calcula los campos en orden secuencial asegurando que cada campo esté disponible antes de ser usado."""

    if df is None or df.empty:

        return df  


    # Mapear FEC_MUEST a verbatimEventDate
    if "FEC_MUESTRA" in df.columns:
        df["verbatimEventDate"] = df["FEC_MUESTRA"]
    else:

        df["verbatimEventDate"] = None

    # Calcular `eventDate`, `year`, `month`, `day` y conservamos `verbatimEventDate`
    if "verbatimEventDate" in df.columns:
        df["eventDate"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.strftime('%Y-%m-%d')
        df["year"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.year
        df["month"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.month
        df["day"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.day
    else:

        df["eventDate"] = None
        df["year"] = None
        df["month"] = None
        df["day"] = None
    
     # Mapear NORTE a verbatimLatitude
    if "NORTE" in df.columns:
        df["verbatimLatitude"] = df["NORTE"]
    else:

        df["verbatimLatitude"] = None

    # Mapear ESTE a verbatimLongitude
    if "ESTE" in df.columns:
        df["verbatimLongitude"] = df["ESTE"]
    else:

        df["verbatimLongitude"] = None

    # Calcular  `recordNumber`
    if "OBJECTID" in df.columns:
        df["recordNumber"] = df["OBJECTID"]
    else:

        df["recordNumber"] = None

   
    # Calcular `occurrenceID`
    if "OPERADOR" in df.columns and "recordNumber" in df.columns:
        df["occurrenceID"] = df.apply(
            lambda row: re.sub(r'[^A-Za-z0-9]', '', row["OPERADOR"]) + ":veda:" + str(row["recordNumber"]),
            axis=1
        )
    else:

        df["occurrenceID"] = None

     # Asignar `occurrenceID` a `id`
    if "occurrenceID" in df.columns:
        df["id"] = df["occurrenceID"]
    else:

        df["id"] = None


    # Concatenar `COOR_ESTE` y `COOR_NORTE` para `verbatimCoordinates`
    if "verbatimLongitude" in df.columns and "verbatimLatitude" in df.columns:
        df["verbatimCoordinates"] = df.apply(
            lambda row: str(row["verbatimLongitude"]).replace(",", ".") + ", " + str(row["verbatimLatitude"]).replace(",", "."),
            axis=1
        )
    else:

        df["verbatimCoordinates"] = None


    # Asignar `ESPECIE` a `verbatimIdentification`
    if "ESPECIE" in df.columns:
        df["verbatimIdentification"] = df["ESPECIE"]
    else:

        df["verbatimIdentification"] = None


    # Mapear organismRemarks 
    campos_clasificacion = [
        "FORM_CRECIM", "TIPO_PLANTA", "NO_VASCULAR"
    ]

    for col in campos_clasificacion:
        if col not in df.columns:
            df[col] = None

    df["organismRemarks"] = df[campos_clasificacion]\
        .applymap(lambda x: x if pd.notna(x) and str(x).strip().lower() not in ["", "none", "nan", "<null>"] else pd.NA)\
        .apply(lambda row: " | ".join(row.dropna().astype(str)), axis=1)



    # Verificar si "FITOSANITARIO" existe y tiene al menos un valor no vacío
    if "FITOSANITARIO" in df.columns and df["FITOSANITARIO"].dropna().empty is False:
        df["measurementValue"] = df["FITOSANITARIO"]
        df["measurementType"] = "Estado fitosanitario general del individuo"
    else:

        df["measurementValue"] = None
        df["measurementType"] = None

    # Verificar si "MEDIDA_MANEJO" existe y tiene al menos un valor no vacío
    if "MEDIDA_MANEJO" in df.columns and df["MEDIDA_MANEJO"].dropna().empty is False:
        df["measurementValue_1"] = df["MEDIDA_MANEJO"]
        df["measurementType_1"] = "Tipo de medida de manejo"
    else:

        df["measurementValue_1"] = None
        df["measurementType_1"] = None


    # Mapear higherClassification 
    campos_clasificacion = [
        "DIVISION", "CLASE", "ORDEN", "FAMILIA", "GENERO"
    ]

    for col in campos_clasificacion:
        if col not in df.columns:
            df[col] = None

    df["higherClassification"] = df[campos_clasificacion]\
        .applymap(lambda x: x if pd.notna(x) and str(x).strip().lower() not in ["", "none", "nan", "<null>"] else pd.NA)\
        .apply(lambda row: " | ".join(row.dropna().astype(str)), axis=1)

    # Mapear a scientificName con validación
    df["scientificName"] = df.apply(
        lambda row: (
            row["ESPECIE"]
            if pd.notna(row["ESPECIE"]) and str(row["ESPECIE"]).strip().lower() not in ["", "none", "nan", "<null>"]
            else (
                row["higherClassification"].split(" | ")[-1]
                if pd.notna(row["higherClassification"]) and " | " in row["higherClassification"]
                else row["higherClassification"]
            )
        ),
        axis=1
    )


    
    return df





# Función principal para procesar veda 
def procesar_veda(ruta_gdb, capa_veda, tabla_veda, enlace_veda, ruta_excel_veda, archivo_entrada_veda, archivo_salida_veda):
    try:
        # Abrir la Geodatabase
        gdb = gdal.OpenEx(ruta_gdb, gdal.OF_VECTOR)
        if not gdb:
            raise RuntimeError(f"❌ No se pudo abrir la GDB en {ruta_gdb}")

        # Extraer coordenadas de la capa de veda
        datos_capa = extraer_coordenadas(gdb.GetLayerByName(capa_veda))

        # Extraer atributos de la tabla de veda
        datos_tabla = []
        layer = gdb.GetLayerByName(tabla_veda)
        for feature in layer:
            datos_tabla.append(feature.items())  

        resultado = realizar_join(datos_capa, datos_tabla, enlace_veda)

        if resultado is None or resultado.empty:

            return

        # Convertir códigos a nombres
        resultado = convertir_codigos_nombres(resultado)

        # Exportar el resultado del join a un archivo Excel intermedio
        exportar_excel(resultado, ruta_excel_veda)

        # Leer el archivo Excel intermedio y agregar taxonRank
        agregar_datos_api_a_excel(ruta_excel_veda)

        # Leer el archivo Excel con taxonRank agregado
        df_intermedio = pd.read_excel(ruta_excel_veda)

        # Procesar campos específicos
        df_intermedio = procesar_campos_especificos(df_intermedio)

        # Crear DataFrame final con todos los atributos de lista_atributos
        df_final = pd.DataFrame(columns=lista_atributos)

        # Mapear los datos del DataFrame intermedio al DataFrame final
        for columna_intermedia, columna_final in mapeo.items():
            if columna_intermedia in df_intermedio.columns:
                df_final[columna_final] = df_intermedio[columna_intermedia]

        # Agregar valores constantes
        for clave, valor in valores_constantes.items():
            df_final[clave] = valor

        # Agregar los campos calculados
       
        df_final["recordNumber"] = df_intermedio["recordNumber"]    
        df_final["eventDate"] = df_intermedio["eventDate"]
        df_final["year"] = df_intermedio["year"]
        df_final["month"] = df_intermedio["month"]
        df_final["day"] = df_intermedio["day"]
        df_final["verbatimEventDate"] = df_intermedio["verbatimEventDate"]    
        df_final["verbatimCoordinates"] = df_intermedio["verbatimCoordinates"]
        df_final["verbatimIdentification"] = df_intermedio["verbatimIdentification"]
        df_final["occurrenceID"] = df_intermedio["occurrenceID"]
        df_final["id"] = df_intermedio["id"]
        df_final["stateProvince"] = df_intermedio["stateProvince"]
        df_final["county"] = df_intermedio["county"]
        df_final["measurementValue"] = df_intermedio["measurementValue"]
        df_final["measurementType"] = df_intermedio["measurementType"]
        df_final["measurementValue_1"] = df_intermedio["measurementValue_1"]
        df_final["measurementType_1"] = df_intermedio["measurementType_1"]
        df_final["higherClassification"] = df_intermedio["higherClassification"]
        df_final["scientificName"] = df_intermedio["scientificName"]


        # Exportar el DataFrame final a un archivo Excel
        exportar_excel(df_final, archivo_salida_veda)

    except Exception as e:
        return None