from shapely.geometry import shape
import pyproj
import pandas as pd
import requests
import re
from osgeo import gdal, ogr
from concurrent.futures import ThreadPoolExecutor
import time

from ..utils.codigos import departamentos, municipios
from ..utils.config import EPSG_ORIGEN, EPSG_DESTINO


# Diccionario de mapeo de atributos para Flora Regeneracion
mapeo = {
    "OPERADOR": "institutionCode",
    "INDIVIDUOS": "individualCount",
    "ID_MUEST": "parentEventID",
    #"ID_INDV_MU": "parentEventID",
    "NOMBRE": "fieldNumber",
    "N_COBERT": "habitat",
    "AREA_UM_ha": "sampleSizeValue",
    "DESCRIP": "samplingProtocol",
    "CUERPO_AGU": "waterBody",
    #"DEPTO": "stateProvince",
    #"MUNICIPIO": "county",
    "VEREDA": "locality",
    "latitud": "decimalLatitude",
    "longitud": "decimalLongitude",
    "COOR_NORTE": "verbatimLatitude",
    "COOR_ESTE": "verbatimLongitude",
    "LOCALIDAD": "locationRemarks",
    "ESPECIE": "scientificName", 
    "canonicalName": "acceptedNameUsage",
    "scientificNameAuthorship": "scientificNameAuthorship",
    "taxonomicStatus": "taxonomicStatus",
    "DIVISION": "phylum",
    "CLASE": "class",
    "ORDEN": "order",
    "FAMILIA": "family",
    "GENERO": "genus",
    "taxonRank": "taxonRank",
    "N_COMUN": "vernacularName",
    "OBSERV": "occurrenceRemarks",
    "PROYECTO": "datasetName"
}

# Valores constantes para Flora Regeneracion
valores_constantes = {
    "type": "Event",
    "basisOfRecord": "HumanObservation",
    "occurrenceStatus": "present",
    "language": "es",
    "sampleSizeUnit": "Hectáreas",
    "continent": "América del Sur",
    "country": "Colombia",
    "countryCode": "CO",
    "geodeticDatum": "WGS84",
    "verbatimSRS": EPSG_ORIGEN,
    "verbatimCoordinateSystem": "Coordenadas proyectadas",
    "kingdom": "Plantae"
    
}

# Lista completa de atributos en el archivo final
lista_atributos = [
    "id", "type", "language", "institutionID", "institutionCode", "datasetName",
    "basisOfRecord", "occurrenceID", "recordNumber", "individualCount",
    "occurrenceStatus", "occurrenceRemarks", "organismRemarks", "eventID",
    "parentEventID", "fieldNumber", "eventDate", "year", "month", "day",
    "verbatimEventDate", "habitat", "samplingProtocol", "sampleSizeValue", "sampleSizeUnit",
    "eventRemarks", "continent", "waterBody", "country",
    "countryCode", "stateProvince", "county", "municipality", "locality",
    "minimumElevationInMeters", "maximumElevationInMeters", "verbatimElevation",
    "locationRemarks", "decimalLatitude", "decimalLongitude", "geodeticDatum",
    "verbatimCoordinates", "verbatimLatitude", "verbatimLongitude",
    "verbatimCoordinateSystem", "verbatimSRS", "footprintWKT", "footprintSRS",
    "verbatimIdentification", "identificationQualifier", "scientificName",
    "acceptedNameUsage", "higherClassification", "kingdom", "phylum", "class",
    "order", "family", "genus", "specificEpithet", "infraspecificEpithet",
    "taxonRank", "verbatimTaxonRank", "scientificNameAuthorship", "vernacularName",
    "taxonomicStatus", "permitText",
]


def consultar_taxon_rank(especie):
    url = "https://api.gbif.org/v1/species/match"
    params = {"name": especie}
    
    try:
        time.sleep(1)  
        response = requests.get(url, params=params)
        if response.status_code == 200:
            data = response.json()
            scientific_name = data.get("scientificName", "")
            canonical_name = data.get("canonicalName", "")
            
            # Extraer la autoría eliminando el nombre científico del nombre canónico
            if canonical_name and scientific_name and scientific_name != canonical_name:
                scientific_name_authorship = scientific_name.replace(canonical_name, "").strip()
            else:
                scientific_name_authorship = None

            return {
                "taxonRank": data.get("rank", None),
                "canonicalName": canonical_name,
                "scientificNameAuthorship": scientific_name_authorship,
                "taxonomicStatus": data.get("status", None)
            }
        else:

            return None
    except Exception as e:

        return None

# Función para agregar los datos de la API al archivo Excel existente
def agregar_datos_api_a_excel(ruta_excel_flora_regeneracion):
    try:

        df = pd.read_excel(ruta_excel_flora_regeneracion)

        if "ESPECIE" not in df.columns:
            raise ValueError("❌ ERROR: La columna 'ESPECIE' no está en el archivo Excel.")


        especies = df["ESPECIE"].tolist()

        # Ejecutar consultas a la API en paralelo 
        with ThreadPoolExecutor(max_workers=50) as executor:
            resultados = list(executor.map(consultar_taxon_rank, especies))

        # Convertir a DataFrame
        datos_api = pd.DataFrame([r if isinstance(r, dict) else {} for r in resultados])

        # Agregar columnas nuevas al dataframe original
        df["taxonRank"] = datos_api["taxonRank"]
        df["canonicalName"] = datos_api["canonicalName"]
        df["scientificNameAuthorship"] = datos_api["scientificNameAuthorship"]
        df["taxonomicStatus"] = datos_api["taxonomicStatus"]


        df.to_excel(ruta_excel_flora_regeneracion, index=False)


    except Exception as e:
        return None

# Función para extraer coordenadas
def extraer_coordenadas(layer):
    resultados = []
    transformer = pyproj.Transformer.from_crs(EPSG_ORIGEN, EPSG_DESTINO, always_xy=True)

    for feature in layer:
        geom = feature.GetGeometryRef()
        if geom and geom.GetGeometryType() == ogr.wkbPoint:
            lon, lat = transformer.transform(geom.GetX(), geom.GetY())

            atributos = feature.items()
            atributos["latitud"] = lat
            atributos["longitud"] = lon
            resultados.append(atributos)

    return resultados


# Función para realizar el join
def realizar_join(capa, tabla, enlace_flora_regeneracion):
    try:


        capa_df = pd.DataFrame(capa)
        tabla_df = pd.DataFrame(tabla)

        if capa_df.empty or tabla_df.empty:

            return None

        resultado_df = pd.merge(capa_df, tabla_df, on=enlace_flora_regeneracion, how="inner")

        if resultado_df.empty:

            return None

        return resultado_df

    except Exception as e:

        return None

# Función para exportar a Excel
def exportar_excel(dataframe, ruta_salida):
    try:

        dataframe.to_excel(ruta_salida, index=False)

    except Exception as e:
        return None



# Función para extraer los nombres de los departamentos y los municipios
def convertir_codigos_nombres(df):
    """Convierte los códigos de departamentos y municipios a nombres."""
    if "DEPTO" in df.columns:

        df["stateProvince"] = df["DEPTO"].apply(lambda x: departamentos.get(x, "Desconocido"))
    else:
        df["stateProvince"] = None


    if "MUNICIPIO" in df.columns:

        df["county"] = df["MUNICIPIO"].apply(lambda x: municipios.get(x, "Desconocido"))
    else:
        df["county"] = None

    return df

# Función para procesar campos específicos
def procesar_campos_especificos(df):
    """Calcula los campos en orden secuencial asegurando que cada campo esté disponible antes de ser usado."""

    if df is None or df.empty:

        return df  

    # Mapear FEC_MUEST a verbatimEventDate
    if "FEC_MUEST" in df.columns:
        df["verbatimEventDate"] = df["FEC_MUEST"]
    else:

        df["verbatimEventDate"] = None

    # Mapear COTA a verbatimElevation
    if "COTA" in df.columns:
        df["verbatimElevation"] = df["COTA"]
    else:

        df["verbatimElevation"] = None

     # Mapear COOR_NORTE a verbatimLatitude
    if "COOR_NORTE" in df.columns:
        df["verbatimLatitude"] = df["COOR_NORTE"]
    else:

        df["verbatimLatitude"] = None

    # Mapear COOR_ESTE a verbatimLongitude
    if "COOR_ESTE" in df.columns:
        df["verbatimLongitude"] = df["COOR_ESTE"]
    else:

        df["verbatimLongitude"] = None

    # Calcular primero `recordNumber`, ya que no depende de ningún otro campo
    if "OBJECTID" in df.columns:
        df["recordNumber"] = df["OBJECTID"]
    else:

        df["recordNumber"] = None

   
    # Calcular `occurrenceID`
    if "OPERADOR" in df.columns and "recordNumber" in df.columns:
        df["occurrenceID"] = df.apply(
            lambda row: re.sub(r'[^A-Za-z0-9]', '', row["OPERADOR"]) + ":Flora-regeneracion:" + str(row["recordNumber"]),
            axis=1
        )
    else:

        df["occurrenceID"] = None

     # Asignar `occurrenceID` a `id`
    if "occurrenceID" in df.columns:
        df["id"] = df["occurrenceID"]
    else:

        df["id"] = None


    # Calcular `eventDate`, `year`, `month`, `day` y conservamos `verbatimEventDate`
    if "verbatimEventDate" in df.columns:

        df["eventDate"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.strftime('%Y-%m-%d')
        df["year"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.year
        df["month"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.month
        df["day"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.day

    else:

        df["eventDate"] = None
        df["year"] = None
        df["month"] = None
        df["day"] = None

    # Asignar `verbatimElevation` a `minimumElevationInMeters` y `maximumElevationInMeters`
    if "verbatimElevation" in df.columns:
        df["minimumElevationInMeters"] = df["verbatimElevation"]
        df["maximumElevationInMeters"] = df["verbatimElevation"]
    else:

        df["minimumElevationInMeters"] = None
        df["maximumElevationInMeters"] = None

    # Concatenar `COOR_ESTE` y `COOR_NORTE` para `verbatimCoordinates`
    if "verbatimLongitude" in df.columns and "verbatimLatitude" in df.columns:
        df["verbatimCoordinates"] = df.apply(
            lambda row: str(row["verbatimLongitude"]).replace(",", ".") + ", " + str(row["verbatimLatitude"]).replace(",", "."),
            axis=1
        )

    else:

        df["verbatimCoordinates"] = None


    # Asignar `ESPECIE` a `verbatimIdentification`
    if "ESPECIE" in df.columns:
        df["verbatimIdentification"] = df["ESPECIE"]
    else:

        df["verbatimIdentification"] = None

     # Mapear higherClassification 
    campos_clasificacion = [
        "DIVISION", "CLASE", "ORDEN", "FAMILIA", "GENERO"
    ]

    for col in campos_clasificacion:
        if col not in df.columns:
            df[col] = None

    df["higherClassification"] = df[campos_clasificacion]\
        .applymap(lambda x: x if pd.notna(x) and str(x).strip().lower() not in ["", "none", "nan", "<null>"] else pd.NA)\
        .apply(lambda row: " | ".join(row.dropna().astype(str)), axis=1)

    # Mapear a scientificName con validación
    df["scientificName"] = df.apply(
        lambda row: (
            row["ESPECIE"]
            if pd.notna(row["ESPECIE"]) and str(row["ESPECIE"]).strip().lower() not in ["", "none", "nan", "<null>"]
            else (
                row["higherClassification"].split(" | ")[-1]
                if pd.notna(row["higherClassification"]) and " | " in row["higherClassification"]
                else row["higherClassification"]
            )
        ),
        axis=1
    )

    # Mapear permitText 
    df["permitText"] = df.apply(
        lambda row: "ANLA:{0}:{1}:{2}".format(
            row.get("NUM_ACT_AD") if pd.notna(row.get("NUM_ACT_AD")) and str(row.get("NUM_ACT_AD")).strip() != "" else "xxx",
            row.get("FEC_ACT_AD") if pd.notna(row.get("FEC_ACT_AD")) and str(row.get("FEC_ACT_AD")).strip() != "" else "xxx",
            row.get("OPERADOR")   if pd.notna(row.get("OPERADOR"))   and str(row.get("OPERADOR")).strip()   != "" else "xxx"
        ),
        axis=1
    )

    # Concatenamos los dos ID
    if "ID_MUEST" in df.columns and "ID_S_MUEST" in df.columns:
        df["eventID"] = df.apply(
            lambda row: f"{row['ID_MUEST']}" if pd.notna(row["ID_MUEST"]) and pd.isna(row["ID_S_MUEST"]) else
                        f"{row['ID_S_MUEST']}" if pd.notna(row["ID_S_MUEST"]) and pd.isna(row["ID_MUEST"]) else
                        f"{row['ID_MUEST']}_{row['ID_S_MUEST']}" if pd.notna(row["ID_MUEST"]) and pd.notna(row["ID_S_MUEST"]) else None,
            axis=1
        )
    

    # Mapeamos los dominios de T_REGEN y TIPO_HAB 
    if "T_REGEN" in df.columns:
        df["T_REGEN"] = df["T_REGEN"].fillna(0).astype(float).astype(int).astype(str).replace({"0": ""})

        mapeo_regen = {
            "1": "Renuevo o plántula",
            "2": "Brinzal",
            "3": "Latizal"                    
        }
        df["regen_temp"] = df["T_REGEN"].map(mapeo_regen).fillna("")
    else:

        df["regen_temp"] = None

    if "TIPO_HAB" in df.columns:
        df["TIPO_HAB"] = df["TIPO_HAB"].fillna(0).astype(float).astype(int).astype(str).replace({"0": ""})

        mapeo_hab = {
            "371": "Arbol",
            "372": "Arbusto",
            "373": "Hierba",     
            "374": "Sufrútice", 
            "375": "Enredadera", 
            "376": "Liana", 
            "377": "Epífita", 
            "378": "Hemiparásita", 
            "379": "Suculentas", 
            "380": "Otro",              
        }
        df["hab_temp"] = df["TIPO_HAB"].map(mapeo_hab).fillna("")
    else:

        df["hab_temp"] = None


    # Mapear a organismRemarks
    if "regen_temp" in df.columns and "hab_temp" in df.columns:
        df["organismRemarks"] = df.apply(
            lambda row: f"{row['regen_temp']}" if pd.notna(row["regen_temp"]) and pd.isna(row["hab_temp"]) else
                        f"{row['hab_temp']}" if pd.notna(row["hab_temp"]) and pd.isna(row["regen_temp"]) else
                        f"{row['regen_temp']} | {row['hab_temp']}" if pd.notna(row["regen_temp"]) and pd.notna(row["hab_temp"]) else None,
            axis=1
        )

    # Mapear ESTACIONAL a eventRemarks
    if "ESTACIONAL" in df.columns:
        mapeo_eventRemarks = {
            301: "Seco",
            302: "Húmedo",
            303: "Medio",
            304: "Todo el año"
        }
        df["eventRemarks"] = df["ESTACIONAL"].map(mapeo_eventRemarks).fillna("")
    else:

        df["eventRemarks"] = None


    return df





# Función principal para procesar Flora regeneracion
def procesar_flora_regeneracion(ruta_gdb, capa_flora_regeneracion, tabla_flora_regeneracion, enlace_flora_regeneracion, ruta_excel_flora_regeneracion, archivo_entrada_flora_regeneracion, archivo_salida_flora_regeneracion):
    try:
        # Abrir la Geodatabase
        gdb = gdal.OpenEx(ruta_gdb, gdal.OF_VECTOR)
        if not gdb:
            raise RuntimeError(f"❌ No se pudo abrir la GDB en {ruta_gdb}")

        # Extraer coordenadas de la capa de flora regeneracion
        datos_capa = extraer_coordenadas(gdb.GetLayerByName(capa_flora_regeneracion))

        # Extraer atributos de la tabla de flora regeneracion
        datos_tabla = []
        layer = gdb.GetLayerByName(tabla_flora_regeneracion)
        for feature in layer:
            datos_tabla.append(feature.items())  # Obtiene los atributos correctamente

        resultado = realizar_join(datos_capa, datos_tabla, enlace_flora_regeneracion)

        if resultado is None or resultado.empty:

            return

        # Convertir códigos a nombres
        resultado = convertir_codigos_nombres(resultado)


        # Exportar el resultado del join a un archivo Excel intermedio
        exportar_excel(resultado, ruta_excel_flora_regeneracion)

        # Leer el archivo Excel intermedio y agregar taxonRank
        agregar_datos_api_a_excel(ruta_excel_flora_regeneracion)

        # Leer el archivo Excel con taxonRank agregado
        df_intermedio = pd.read_excel(ruta_excel_flora_regeneracion)

        # Procesar campos específicos
        df_intermedio = procesar_campos_especificos(df_intermedio)

        # Crear DataFrame final con todos los atributos de lista_atributos
        df_final = pd.DataFrame(columns=lista_atributos)

        # Mapear los datos del DataFrame intermedio al DataFrame final
        for columna_intermedia, columna_final in mapeo.items():
            if columna_intermedia in df_intermedio.columns:
                df_final[columna_final] = df_intermedio[columna_intermedia]

        # Agregar valores constantes
        for clave, valor in valores_constantes.items():
            df_final[clave] = valor

        # Agregar los campos calculados
       
        df_final["recordNumber"] = df_intermedio["recordNumber"]        
        df_final["eventDate"] = df_intermedio["eventDate"]
        df_final["year"] = df_intermedio["year"]
        df_final["month"] = df_intermedio["month"]
        df_final["day"] = df_intermedio["day"]
        df_final["verbatimEventDate"] = df_intermedio["verbatimEventDate"]
        df_final["verbatimElevation"] = df_intermedio["verbatimElevation"]
        df_final["minimumElevationInMeters"] = df_intermedio["minimumElevationInMeters"]
        df_final["maximumElevationInMeters"] = df_intermedio["maximumElevationInMeters"]
        df_final["verbatimCoordinates"] = df_intermedio["verbatimCoordinates"]
        df_final["verbatimIdentification"] = df_intermedio["verbatimIdentification"]
        df_final["occurrenceID"] = df_intermedio["occurrenceID"]
        df_final["id"] = df_intermedio["id"]
        df_final["stateProvince"] = df_intermedio["stateProvince"]
        df_final["county"] = df_intermedio["county"]
        df_final["higherClassification"] = df_intermedio["higherClassification"]
        df_final["scientificName"] = df_intermedio["scientificName"]
        df_final["permitText"] = df_intermedio["permitText"]
        df_final["organismRemarks"] = df_intermedio["organismRemarks"]
        df_final["eventID"] = df_intermedio["eventID"]
        df_final["eventRemarks"] = df_intermedio["eventRemarks"]


        # Exportar el DataFrame final a un archivo Excel
        exportar_excel(df_final, archivo_salida_flora_regeneracion)

    except Exception as e:
        return None