import pyproj
import pandas as pd
import requests
import re
from osgeo import gdal, ogr
from concurrent.futures import ThreadPoolExecutor
import time

from ..utils.codigos import departamentos, municipios
from ..utils.config import EPSG_ORIGEN, EPSG_DESTINO

# Diccionario de mapeo de atributos para aves_OFM
mapeo = {
    "OPERADOR": "institutionCode",
    #"NOMBRE": "fieldNumber",
    "latitud": "decimalLatitude",
    "longitud": "decimalLongitude",
    "ID_INGRESO": "eventID",
    "CERTE_DATO": "identificationVerificationStatus",
    "ID_AVISTAM": "fieldNumber",
    "FECHA": "verbatimEventDate",
    "PROFUND_m": "verbatimDepth",
    #"ORDEN": "order",
    #"FAMILIA": "family",
    #"GENERO": "genus",   
    "especie": "scientificName", 
    "NOMB_COMUN": "vernacularName",
    "taxonRank": "taxonRank",
    "canonicalName": "acceptedNameUsage",
    "scientificNameAuthorship": "scientificNameAuthorship",
    "taxonomicStatus": "taxonomicStatus",
    "scientificNameID": "scientificNameID",
    "nameAccordingTo": "nameAccordingTo",
    "nameAccordingToID": "nameAccordingToID",
    "NUM_ANIMAL": "individualCount",
    "COMPORTAMI": "behavior",
    "PROYECTO": "datasetName"
    
}

# Valores constantes para aves_OFM
valores_constantes = {
    "occurrenceStatus": "present",
    "language": "es",
    "continent": "América del Sur",
    "country": "Colombia",
    "countryCode": "CO",
    "geodeticDatum": "WGS84",
    "verbatimSRS": EPSG_ORIGEN,
    "verbatimCoordinateSystem": "Coordenadas proyectadas"
}

# Lista completa de atributos en el archivo final
lista_atributos = [
    "id", "type", "language", "institutionID", "institutionCode", "datasetName", "basisOfRecord", "occurrenceID", "recordedBy", 
    "individualCount", "lifeStage", "organismQuantity", "organismQuantityType", "behavior", "occurrenceStatus", "occurrenceRemarks", 
    "eventID", "parentEventID", "fieldNumber", "eventDate", "eventTime", "year", "month", "day", "verbatimEventDate", "samplingProtocol", 
    "samplingEffort", "eventRemarks", "locationID", "continent", "waterBody", "country", "countryCode", "stateProvince", "minimumDepthInMeters", "maximumDepthInMeters",
    "verbatimDepth", "decimalLatitude", "decimalLongitude", "geodeticDatum", "verbatimCoordinates", "verbatimLatitude", "verbatimLongitude", "verbatimCoordinateSystem",
    "verbatimSRS", "verbatimIdentification", "identificationQualifier", "dateIdentified", "scientificNameID", "nameAccordingToID", "scientificName", "acceptedNameUsage", 
    "nameAccordingTo", "higherClassification", "kingdom", "phylum", "class", "order", "family", "genus", "specificEpithet", "taxonRank", "verbatimTaxonRank", 
    "scientificNameAuthorship", "vernacularName", "taxonomicStatus", "identificationVerificationStatus", "measurementValue", "measurementType", "measurementUnit", 
    "measurementValue_1", "measurementType_1"
]


def consultar_taxon_rank(especie):
    url = f"http://www.marinespecies.org/rest/AphiaRecordsByMatchNames?scientificnames[]={especie}&marine_only=false"

    try:
        time.sleep(1) 
        response = requests.get(url)

        if response.status_code == 200:
            data = response.json()
            especie_data = data[0][0] if data and data[0] else {}

            return {
                "taxonRank": especie_data.get("rank", None),
                "canonicalName": especie_data.get("valid_name", None),
                "scientificNameAuthorship": especie_data.get("authority", None),
                "taxonomicStatus": especie_data.get("status", None),
                "scientificNameID": f"https://www.marinespecies.org/aphia.php?p=taxdetails&id={especie_data.get('AphiaID')}" if especie_data.get("AphiaID") else None,
                "nameAccordingTo": "World Register of Marine Species",
                "nameAccordingToID": "https://www.marinespecies.org"
            }

        else:

            return None

    except Exception as e:

        return None


def agregar_datos_api_a_excel(ruta_excel_peces_OFM):
    try:

        df = pd.read_excel(ruta_excel_peces_OFM)

        if "especie" not in df.columns:
            raise ValueError("❌ ERROR: La columna 'ESPEC_NOMB' no está en el archivo Excel.")


        especies = df["especie"].tolist()

        # Ejecutar en paralelo las consultas a la API
        with ThreadPoolExecutor(max_workers=5) as executor:
            resultados = list(executor.map(consultar_taxon_rank, especies))

        # Convertir a DataFrame
        datos_api = pd.DataFrame([r if isinstance(r, dict) else {} for r in resultados])

        # Asegurar que todas las columnas necesarias existan aunque estén vacías
        columnas_requeridas = [
            "taxonRank", "canonicalName", "scientificNameAuthorship",
            "taxonomicStatus", "scientificNameID", "nameAccordingTo", "nameAccordingToID"
        ]
        for col in columnas_requeridas:
            if col not in datos_api.columns:
                datos_api[col] = None

        # Agregar columnas nuevas al DataFrame original
        df["taxonRank"] = datos_api["taxonRank"]
        df["canonicalName"] = datos_api["canonicalName"]
        df["scientificNameAuthorship"] = datos_api["scientificNameAuthorship"]
        df["taxonomicStatus"] = datos_api["taxonomicStatus"]
        df["scientificNameID"] = datos_api.get("scientificNameID")
        df["nameAccordingTo"] = datos_api.get("nameAccordingTo")
        df["nameAccordingToID"] = datos_api.get("nameAccordingToID")


        df.to_excel(ruta_excel_peces_OFM, index=False)


    except Exception as e:
        return None

# 🔹 Función para extraer coordenadas
def extraer_coordenadas(layer):
    resultados = []
    transformer = pyproj.Transformer.from_crs(EPSG_ORIGEN, EPSG_DESTINO, always_xy=True)

    for feature in layer:
        geom = feature.GetGeometryRef()
        if geom and geom.GetGeometryType() == ogr.wkbPoint:
            lon, lat = transformer.transform(geom.GetX(), geom.GetY())

            atributos = feature.items()
            atributos["latitud"] = lat
            atributos["longitud"] = lon
            resultados.append(atributos)

    return resultados


# Función para realizar el join
def realizar_join(capa, tabla, enlace_aves_OFM):
    try:


        capa_df = pd.DataFrame(capa)
        tabla_df = pd.DataFrame(tabla)

        if capa_df.empty or tabla_df.empty:

            return None

        resultado_df = pd.merge(capa_df, tabla_df, on=enlace_aves_OFM, how="inner")

        if resultado_df.empty:

            return None

        return resultado_df

    except Exception as e:

        return None


# Función para mapear especie
def mapear_espec_nomb(df):
    if "ESPEC_NOMB" in df.columns:
        df["ESPEC_NOMB"] = df["ESPEC_NOMB"].fillna(0).astype(float).astype(int).astype(str).replace({"0": ""})

    mapa_espec_nomb = {
        "2701": "Actitis macularia",
        "2702": "Actitis macularius",
        "2703": "Amaurolimnas concolor",
        "2704": "Amazilia franciae",
        "2705": "Amazilia tzacatl",
        "2706": "Amazona autumnalis",
        "2707": "Amazona farinosa",
        "2708": "Amazona ochrocephala",
        "2709": "Anas clypeata",
        "2710": "Anas cyanoptera",
        "2711": "Anas discors",
        "2712": "Anthracothorax prevostii",
        "2713": "Aphriza virgata",
        "2714": "Aramides wolfi",
        "2715": "Aratinga pertinax",
        "2716": "Ardea alba",
        "2717": "Ardea herodias",
        "2718": "Arenaria interpres",
        "2719": "Bubulcus ibis",
        "2720": "Burhinus bistriatus",
        "2721": "Buteogallus anthracinus",
        "2722": "Buteogallus subtilis",
        "2723": "Butorides striata",
        "2724": "Butorides striatus",
        "2725": "Butorides virescens",
        "2726": "Cairina moschata",
        "2727": "Calidris alba",
        "2728": "Calidris bairdii",
        "2729": "Calidris fuscicollis",
        "2730": "Calidris mauri",
        "2731": "Calidris melanotos",
        "2732": "Calidris minutilla",
        "2733": "Calidris pusilla",
        "2734": "Calyptura cristata",
        "2735": "Camarhynchus heliobates",
        "2736": "Camarhynchus pallidus",
        "2737": "Camarhynchus parvulus",
        "2738": "Camarhynchus pauper",
        "2739": "Camarhynchus psittacula",
        "2740": "Campylorhynchus griseus",
        "2741": "Caracara cheriway",
        "2742": "Cardinalis phoeniceus",
        "2743": "Cassidix  mexicanus",
        "2744": "Cathartes aura",
        "2745": "Catharus ustulatus",
        "2746": "Catoptrophorus semipalmatus",
        "2747": "Certhidea crassirostris",
        "2748": "Certhidea fusca",
        "2749": "Charadrius semipalmatus",
        "2750": "Charadrius vociferus",
        "2751": "Charadrius wilsonia",
        "2752": "Chlidonias niger",
        "2753": "Chloroceryle aenea",
        "2754": "Chloroceryle amazona",
        "2755": "Chloroceryle americana",
        "2756": "Claravis pretiosa",
        "2757": "Coereba flaveola",
        "2758": "Coereba flaveolaxxx",
        "2759": "Conirostrum bicolor",
        "2760": "Coragyps atratus",
        "2761": "Crotophaga ani",
        "2762": "Cyanerpes cyaneus",
        "2763": "Dendrocygna autumnalis",
        "2764": "Dendroica striata",
        "2765": "Egretta caerulea",
        "2766": "Egretta rufescens",
        "2767": "Egretta thula",
        "2768": "Egretta tricolor",
        "2769": "Elaenia martinica",
        "2770": "Elanoides forficatus",
        "2771": "Eudocimus albus",
        "2772": "Eudocimus ruber",
        "2773": "Falco peregrinus",
        "2774": "Falco sparverius",
        "2775": "Fregata magnificens",
        "2776": "Fulica caribaea",
        "2777": "Galbula ruficauda",
        "2778": "Gallinula galeata",
        "2779": "Gelochelidon nilotica",
        "2780": "Geospiza conirostris",
        "2781": "Geospiza difficilis",
        "2782": "Geospiza fortis",
        "2783": "Geospiza fuliginosa",
        "2784": "Geospiza magnirostris",
        "2785": "Geospiza scandens",
        "2786": "Geotrygon montana",
        "2787": "Haematopus palliatus",
        "2788": "Hemitriccus margaritaceiventer",
        "2789": "Himantopus mexicanus",
        "2790": "Hirundo rustica",
        "2791": "Hydranassa tricolor",
        "2792": "Hydroprogne caspia",
        "2793": "Hylocharis humboldtii",
        "2794": "Hypnelus ruficollis",
        "2795": "Icterus nigrogularis",
        "2796": "Larus atricilla",
        "2797": "Larus marinus",
        "2798": "Larus minutus",
        "2799": "Larus pipixcan",
        "2800": "Leucophaeus atricilla",
        "2801": "Leucophaeus pipixcan",
        "2802": "Leucopternis princeps",
        "2803": "Limosa fedoa",
        "2804": "Megaceryle alcyon",
        "2805": "Megaceryle torcuata",
        "2806": "Megaceryle torquata",
        "2807": "Melanerpes rubricapillus",
        "2808": "Mimus gilvus",
        "2809": "Mitrospingus cassinii",
        "2810": "Mitrospingus oleagineus",
        "2811": "Mniotilta varia",
        "2812": "Mycteria americana",
        "2813": "Numenius phaeopus",
        "2814": "Nyctanassa violacea",
        "2815": "Nycticorax nycticorax",
        "2816": "Pandion haliaetus",
        "2817": "Parkerthraustes humeralis",
        "2818": "Parkesia noveboracensis",
        "2819": "Patagioenas cayennensis",
        "2820": "Patagioenas leucocephala",
        "2821": "Patagioenas speciosa",
        "2822": "Pelecanus occidentalis",
        "2823": "Phaethon aethereus",
        "2824": "Phalacrocorax bouganvillii",
        "2825": "Phalacrocorax brasilianus",
        "2826": "Phibalura flavirostris",
        "2827": "Phoebastria irrorata",
        "2828": "Phoenicopterus ruber",
        "2829": "Piaya cayana",
        "2830": "Pionopsitta pulchra",
        "2831": "Pionus menstruus",
        "2832": "Piprites chloris",
        "2833": "Piprites pileata",
        "2834": "Piranga rubra",
        "2835": "Platalea ajaja",
        "2836": "Plegadis falcinellus",
        "2837": "Pluvialis squatarola",
        "2838": "Polioptila plumbea",
        "2839": "Progne chalybea",
        "2840": "Protonotaria citrea",
        "2841": "Pterodroma hasitata",
        "2842": "Pteroglossus sanguineus",
        "2843": "Puffinus griseus",
        "2844": "Pyrocephalus rubinus",
        "2845": "Quiscalus mexicanus",
        "2846": "Ramphastos ambiguus",
        "2847": "Ramphocelus flammigerus",
        "2848": "Rhodinocichla rosea",
        "2849": "Rynchops niger",
        "2850": "Saltator atricollis",
        "2851": "Saltator atripennis",
        "2852": "Saltator aurantiirostris",
        "2853": "Saltator cinctus",
        "2854": "Saltator coerulescens",
        "2855": "Saltator fuliginosus",
        "2856": "Saltator grossus",
        "2857": "Saltator maxillosus",
        "2858": "Saltator maximus",
        "2859": "Saltator nigriceps",
        "2860": "Saltator orenocensis",
        "2861": "Saltator similis",
        "2862": "Saltator striatipectus",
        "2863": "Saltatricula multicolor",
        "2864": "Setophaga castanea",
        "2865": "Setophaga magnolia",
        "2866": "Setophaga palmarum",
        "2867": "Setophaga petechia",
        "2868": "Setophaga ruticilla",
        "2869": "Setophoga ruticilla",
        "2870": "Stercorarius pomarinus",
        "2871": "Sterna elegans",
        "2872": "Sterna hirundo",
        "2873": "Sterna maxima",
        "2874": "Sterna nilotica",
        "2875": "Sterna sandvicensis",
        "2876": "Sula dactrylatra",
        "2877": "Sula leucogaster",
        "2878": "Sula nebouxii",
        "2879": "Sula variegata",
        "2880": "Thalasseus maximus",
        "2881": "Thalasseus sandvicensis",
        "2882": "Thectocercus acuticaudatus",
        "2883": "Theristicus caudatus",
        "2884": "Thraupis episcopus",
        "2885": "Thraupis palmarum",
        "2886": "Tiaris bicolor",
        "2887": "Tiaris fuliginosus",
        "2888": "Tiaris obscurus",
        "2889": "Tiaris olivacea",
        "2890": "Tiaris olivaceus",
        "2891": "Tigrisoma lineatum",
        "2892": "Tigrisoma mexicanum",
        "2893": "Tinamus major",
        "2894": "Tringa flavipes",
        "2895": "Tringa melanoleuca",
        "2896": "Tringa semipalmata",
        "2897": "Tringa solitaria",
        "2898": "Trogon massena",
        "2899": "Tyrannus dominicensis",
        "2900": "Tyrannus melancholicus",
        "2901": "Tyrannus tyrannus",
        "2902": "Tyto alba",
        "2903": "Vireo altiloquus",
        "2904": "Vireo crassirostris",
        "2905": "Vireo olivaceus",
        "2906": "Zenaida asiatica",
        "2907": "Zenaida auriculata",
        "2908": "Otro"
    
    }
    if "ESPEC_NOMB" in df.columns:
        df["especie"] = df["ESPEC_NOMB"].map(mapa_espec_nomb).fillna(df["ESPEC_NOMB"])
    else:

        df["especie"] = None
    
    return df



# Función para exportar a Excel
def exportar_excel(dataframe, ruta_salida):
    try:

        dataframe.to_excel(ruta_salida, index=False)

    except Exception as e:
        return None


#  Función para procesar campos específicos
def procesar_campos_especificos(df):
    """Calcula los campos en orden secuencial asegurando que cada campo esté disponible antes de ser usado."""

    if df is None or df.empty:

        return df  

     # Mapear COOR_NORTE a verbatimLatitude
    if "COOR_NORTE" in df.columns:
        df["verbatimLatitude"] = df["COOR_NORTE"]
    else:

        df["verbatimLatitude"] = None

    # Mapear COOR_ESTE a verbatimLongitude
    if "COOR_ESTE" in df.columns:
        df["verbatimLongitude"] = df["COOR_ESTE"]
    else:

        df["verbatimLongitude"] = None

    # Se calcula  `recordNumber`
    if "OBJECTID" in df.columns:
        df["recordNumber"] = df["OBJECTID"]
    else:

        df["recordNumber"] = None

    # Calcular `occurrenceID`
    if "OPERADOR" in df.columns and "recordNumber" in df.columns:
        df["occurrenceID"] = df.apply(
            lambda row: re.sub(r'[^A-Za-z0-9]', '', row["OPERADOR"]) + ":avesOFM:" + str(row["recordNumber"]),
            axis=1
        )
    else:

        df["occurrenceID"] = None

     # Asignar `occurrenceID` a `id`
    if "occurrenceID" in df.columns:
        df["id"] = df["occurrenceID"]
    else:

        df["id"] = None


    # Concatenar `COOR_ESTE` y `COOR_NORTE` para `verbatimCoordinates`
    if "verbatimLongitude" in df.columns and "verbatimLatitude" in df.columns:
        df["verbatimCoordinates"] = df.apply(
            lambda row: str(row["verbatimLongitude"]).replace(",", ".") + ", " + str(row["verbatimLatitude"]).replace(",", "."),
            axis=1
        )

    else:

        df["verbatimCoordinates"] = None


    # se mapea a variables temporales para despues sacar el type

    if "EVIDENCIA" in df.columns:
        mapeo_evidencia = {
            181: "StillImage",
            182: "MovingImage",
            183: "MovingImage",
            184: "Event"           
        }
        df["type"] = df["EVIDENCIA"].map(mapeo_evidencia).fillna("")
    else:

        df["type"] = None

    # Calcular `basisOfRecord`, que depende de `type`

    if "type" in df.columns:
        mapeo_basis_2 = {
            "StillImage": "MachineObservation",
            "MovingImage": "MachineObservation",
            "Event": "HumanObservation"
        }
        df["basisOfRecord"] = df["type"].map(mapeo_basis_2).fillna("")
    else:

        df["basisOfRecord"] = None
      
    
    # Calcular `occurrenceRemarks`
    if "EVIDENCIA" in df.columns:
        mapeo_evidencia = {
            181: "Fotografía",
            182: "video",
            183: "Video y fotografía",
            184: "Solo observación" 
        }
        df["temp_occurrenceRemarks"] = df["EVIDENCIA"].map(mapeo_evidencia).fillna("")
    else:

        df["temp_occurrenceRemarks"] = None

    if "temp_occurrenceRemarks" in df.columns and "OBSERVACIO_y" in df.columns:
        df["occurrenceRemarks"] = df.apply(
            lambda row: (
                str(row["temp_occurrenceRemarks"]) if pd.notna(row["temp_occurrenceRemarks"]) and pd.isna(row["OBSERVACIO_y"]) else
                str(row["OBSERVACIO_y"]) if pd.notna(row["OBSERVACIO_y"]) and pd.isna(row["temp_occurrenceRemarks"]) else
                f"{row['temp_occurrenceRemarks']} | {row['OBSERVACIO_y']}" if pd.notna(row["temp_occurrenceRemarks"]) and pd.notna(row["OBSERVACIO_y"]) else
                None
            ),
            axis=1
        )
    
    
    # Mapeamos a lifeStage    
    if "NUM_ADULT" in df.columns and "NUM_JUVE" in df.columns:
        df["lifeStage"] = df.apply(
            lambda row: (
                f"{int(row['NUM_ADULT'])} adultos" if pd.notna(row["NUM_ADULT"]) and pd.isna(row["NUM_JUVE"]) else
                f"{int(row['NUM_JUVE'])} juveniles" if pd.notna(row["NUM_JUVE"]) and pd.isna(row["NUM_ADULT"]) else
                f"{int(row['NUM_ADULT'])} adultos | {int(row['NUM_JUVE'])} juveniles" if pd.notna(row["NUM_ADULT"]) and pd.notna(row["NUM_JUVE"]) else
                None
            ),
            axis=1
        )

    # Mapear HORA a eventTime (formato HH:MM:SS)
    if "HORA" in df.columns:
        df["eventTime"] = pd.to_timedelta(df["HORA"], unit="h").dt.components.apply(
            lambda x: f"{int(x.hours):02}:{int(x.minutes):02}:{int(x.seconds):02}", axis=1
        )
    else:

        df["eventTime"] = None




    # Mapeamos a order
    if "ORDEN" in df.columns:
        df["ORDEN"] = df["ORDEN"].fillna(0).astype(float).astype(int).astype(str).replace({"0": ""})

        mapeo_orden = {
            "2101": "Accipitriformes",
            "2102": "Anseriformes",
            "2103": "Apodiformes",
            "2104": "Cathartiformes",
            "2105": "Charadriiformes",
            "2106": "Ciconiiformes",
            "2107": "Columbiformes",
            "2108": "Coraciiformes",
            "2109": "Cuculiformes",
            "2110": "Eurypygiformes",
            "2111": "Falconiformes",
            "2112": "Galbuliformes",
            "2113": "Galliformes",
            "2114": "Gruiformes",
            "2115": "Opisthocomiformes",
            "2116": "Passeriformes",
            "2117": "Pelecaniformes",
            "2118": "Phaethontiformes",
            "2119": "Phoenicopteriformes",
            "2120": "Piciformes",
            "2121": "Podicipediformes",
            "2122": "Procellariiformes",
            "2123": "Psittaciformes",
            "2124": "Sphenisciformes",
            "2125": "Strigiformes",
            "2126": "Tinamiformes",
            "2127": "Trogoniformes",
            "2128": "Otro"
        
        }
        df["order"] = df["ORDEN"].map(mapeo_orden).fillna("")
    else:

        df["order"] = None


    # Mapeamos SUBORDEN a una variable temporal
    if "SUBORDEN" in df.columns:
        df["SUBORDEN"] = df["SUBORDEN"].fillna(0).astype(float).astype(int).astype(str).replace({"0": ""})
        mapeo_suborden = {
            "2201": "Charadrii",
            "2202": "Lari",
            "2203": "Oscines",
            "2204": "Scolopaci",
            "2205": "Tyranni",
            "2206": "Otro"
        }
        df["temp_suborden"] = df["SUBORDEN"].map(mapeo_suborden).fillna("")
    else:

        df["temp_suborden"] = None

    # Mapeamos a family
    if "FAMILIA" in df.columns:
        df["FAMILIA"] = df["FAMILIA"].fillna(0).astype(float).astype(int).astype(str).replace({"0": ""})
        mapeo_familia = {
            "2301": "Accipitridae",
            "2302": "Accipitridae",
            "2303": "Alaudidae",
            "2304": "Alcedinidae",
            "2305": "Anatidae",
            "2306": "Ardeidae",
            "2307": "Ardeidae",
            "2308": "Bombycillidae",
            "2309": "Bucconidae",
            "2310": "Burhinidae",
            "2311": "Capitonidae",
            "2312": "Caprimulgidae",
            "2313": "Cardinalidae",
            "2314": "Cathartidae",
            "2315": "Charadriidae",
            "2316": "Charadriidae",
            "2317": "Ciconiidae",
            "2318": "Cinclidae",
            "2319": "Columbidae",
            "2320": "Conopophagidae",
            "2321": "Corvidae",
            "2322": "Cotingidae",
            "2323": "Cracidae",
            "2324": "Cuculidae",
            "2325": "Cuculidae",
            "2326": "Diomedeidae",
            "2327": "Donacobiidae",
            "2328": "Emberizidae",
            "2329": "Eurypygidae",
            "2330": "Falconidae",
            "2331": "Formicariidae",
            "2332": "Fregatidae",
            "2333": "Fringillidae",
            "2334": "Furnariidae",
            "2335": "Galbulidae",
            "2336": "Grallariidae",
            "2337": "Haematopodidae",
            "2338": "Hirundinidae",
            "2339": "Hydrobatidae",
            "2340": "Icteridae",
            "2341": "Incertaesedis",
            "2342": "IncertaeSedis",
            "2343": "Jacanidae",
            "2344": "Laridae",
            "2345": "Mimidae",
            "2346": "Motacillidae",
            "2347": "Odontophoridae",
            "2348": "Opisthocomidae",
            "2349": "Pandionidae",
            "2350": "Parulidae",
            "2351": "Pelecanidae",
            "2352": "Pelecanidae",
            "2353": "Phaethontidae",
            "2354": "Phaethontidae",
            "2355": "Phalacrocoracidae",
            "2356": "Phoenicopteridae",
            "2357": "Picidae",
            "2358": "Pipridae",
            "2359": "Podicipedidae",
            "2360": "Polioptilidae",
            "2361": "Procellariidae",
            "2362": "Procellariidae",
            "2363": "Procellariidae",
            "2364": "Psittacidae",
            "2365": "Rallidae",
            "2366": "Ramphastidae",
            "2367": "Recurvirostridae",
            "2368": "Rhinocryptidae",
            "2369": "Rynchopidae",
            "2370": "Sapayoidae",
            "2371": "Scolopacidae",
            "2372": "Scolopacidae",
            "2373": "Semnornithidae",
            "2374": "Spheniscidae",
            "2375": "Stercorariidae",
            "2376": "Sulidae",
            "2377": "Thamnophilidae",
            "2378": "Thraupidae",
            "2379": "Threskiornithidae",
            "2380": "Tinamidae",
            "2381": "Tityridae",
            "2382": "Trochilidae",
            "2383": "Troglodytidae",
            "2384": "Trogonidae",
            "2385": "Turdidae",
            "2386": "Tyrannidae",
            "2387": "Tytonidae",
            "2388": "Vireonidae",
            "2389": "Otro"

        }
        df["family"] = df["FAMILIA"].map(mapeo_familia).fillna("")
    else:

        df["family"] = None


    # Mapeamos SUBFAMILIA a una variable temporal
    if "SUBFAMILIA" in df.columns:
        df["SUBFAMILIA"] = df["SUBFAMILIA"].fillna(0).astype(float).astype(int).astype(str).replace({"0": ""})
        mapeo_subfamilia = {
            "2401": "Cerylinae",
            "2402": "Columbinae",
            "2403": "Crotophaginae",
            "2404": "Crotophaginae",
            "2405": "Cuculinae",
            "2406": "Dendrocolaptinae",
            "2407": "Furnariinae",
            "2408": "Hydrobatinae",
            "2409": "Neomorphinae",
            "2410": "Oceanitinae",
            "2411": "Psittacinae",
            "2412": "Sclerurinae",
            "2413": "Tinaminae",
            "2414": "Trochilinae",
            "2415": "Otro"
        }
        df["temp_subfamily"] = df["SUBFAMILIA"].map(mapeo_subfamilia).fillna("")
    else:

        df["temp_subfamily"] = None


    # Mapeamos a genus
    if "GENERO" in df.columns:
        df["GENERO"] = df["GENERO"].fillna(0).astype(float).astype(int).astype(str).replace({"0": ""})
        mapeo_genero = {
            "2501": "Actitis sp.",
            "2502": "Amaurolimnas sp.",
            "2503": "Amazilia sp.",
            "2504": "Amazona sp.",
            "2505": "Anas sp.",
            "2506": "Anthracothorax sp.",
            "2507": "Aphriza sp.",
            "2508": "Aramides sp.",
            "2509": "Aratinga sp.",
            "2510": "Ardea sp.",
            "2511": "Arenaria sp.",
            "2512": "Bubulcus sp.",
            "2513": "Burhinus sp.",
            "2514": "Buteogallus sp.",
            "2515": "Butorides sp.",
            "2516": "Cairina sp.",
            "2517": "Calidris sp.",
            "2518": "Calyptura sp.",
            "2519": "Camarhynchus sp.",
            "2520": "Campylorhynchus sp.",
            "2521": "Caracara sp.",
            "2522": "Cardinalis sp.",
            "2523": "Cassidix  sp.",
            "2524": "Cathartes sp.",
            "2525": "Catharus sp.",
            "2526": "Catoptrophorus sp.",
            "2527": "Certhidea sp.",
            "2528": "Charadrius sp.",
            "2529": "Chlidonias sp.",
            "2530": "Chloroceryle sp.",
            "2531": "Claravis sp.",
            "2532": "Coereba sp.",
            "2533": "Coereba sp.",
            "2534": "Conirostrum sp.",
            "2535": "Coragyps sp.",
            "2536": "Crotophaga sp.",
            "2537": "Cyanerpes sp.",
            "2538": "Dendrocygna sp.",
            "2539": "Dendroica sp.",
            "2540": "Egretta sp.",
            "2541": "Elaenia sp.",
            "2542": "Elanoides sp.",
            "2543": "Eudocimus sp.",
            "2544": "Falco sp.",
            "2545": "Fregata sp.",
            "2546": "Fulica sp.",
            "2547": "Galbula sp.",
            "2548": "Gallinula sp.",
            "2549": "Gelochelidon sp.",
            "2550": "Geospiza sp.",
            "2551": "Geotrygon sp.",
            "2552": "Haematopus sp.",
            "2553": "Hemitriccus sp.",
            "2554": "Himantopus sp.",
            "2555": "Hirundo sp.",
            "2556": "Hydranassa sp.",
            "2557": "Hydroprogne sp.",
            "2558": "Hylocharis sp.",
            "2559": "Hypnelus sp.",
            "2560": "Icterus sp.",
            "2561": "Larus sp.",
            "2562": "Leucophaeus sp.",
            "2563": "Leucopternis sp.",
            "2564": "Limosa sp.",
            "2565": "Megaceryle sp.",
            "2566": "Melanerpes sp.",
            "2567": "Mimus sp.",
            "2568": "Mitrospingus sp.",
            "2569": "Mniotilta sp.",
            "2570": "Mycteria sp.",
            "2571": "Myiarchus sp.",
            "2572": "Numenius sp.",
            "2573": "Nyctanassa sp.",
            "2574": "Nycticorax sp.",
            "2575": "Pandion sp.",
            "2576": "Parkerthraustes sp.",
            "2577": "Parkesia sp.",
            "2578": "Patagioenas sp.",
            "2579": "Pelecanus sp.",
            "2580": "Phaethon sp.",
            "2581": "Phalacrocorax sp.",
            "2582": "Phibalura sp.",
            "2583": "Phoebastria sp.",
            "2584": "Phoenicopterus sp.",
            "2585": "Piaya sp.",
            "2586": "Pionopsitta sp.",
            "2587": "Pionus sp.",
            "2588": "Piprites sp.",
            "2589": "Piranga sp.",
            "2590": "Platalea sp.",
            "2591": "Platyspiza sp.",
            "2592": "Plegadis sp.",
            "2593": "Pluvialis sp.",
            "2594": "Polioptila sp.",
            "2595": "Progne sp.",
            "2596": "Protonotaria sp.",
            "2597": "Pterodroma sp.",
            "2598": "Pteroglossus sp.",
            "2599": "Puffinus sp.",
            "2600": "Pyrocephalus sp.",
            "2601": "Quiscalus sp.",
            "2602": "Ramphastos sp.",
            "2603": "Ramphocelus sp.",
            "2604": "Rhodinocichla sp.",
            "2605": "Rynchops sp.",
            "2606": "Saltator sp.",
            "2607": "Saltatricula sp.",
            "2608": "Setophaga sp.",
            "2609": "Setophoga sp.",
            "2610": "Stercorarius sp.",
            "2611": "Sterna sp.",
            "2612": "Sula sp.",
            "2613": "Thalasseus sp.",
            "2614": "Thectocercus sp.",
            "2615": "Theristicus sp.",
            "2616": "Thraupis sp.",
            "2617": "Tiaris sp.",
            "2618": "Tigrisoma sp.",
            "2619": "Tinamus sp.",
            "2620": "Tringa sp.",
            "2621": "Trogon sp.",
            "2622": "Tyrannus sp.",
            "2623": "Tyto sp.",
            "2624": "Vireo sp.",
            "2625": "Zenaida sp.",
            "2626": "Otro"

        }
        df["genus"] = df["GENERO"].map(mapeo_genero).fillna("")
    else:

        df["genus"] = None
    
    # Mapear higherClassification 
    campos_clasificacion = [
        "order", "temp_suborden", "family", "temp_subfamily", "genus"
    ]

    for col in campos_clasificacion:
        if col not in df.columns:
            df[col] = None

    df["higherClassification"] = df[campos_clasificacion]\
        .applymap(lambda x: x if pd.notna(x) and str(x).strip().lower() not in ["", "none", "nan", "<null>"] else pd.NA)\
        .apply(lambda row: " | ".join(row.dropna().astype(str)), axis=1)

    # Mapear a scientificName con validación
    df["scientificName"] = df.apply(
        lambda row: (
            row["especie"]
            if pd.notna(row["especie"]) and str(row["especie"]).strip().lower() not in ["", "none", "nan", "<null>", "Otro","otro","Otros","Otra"]
            else (
                row["higherClassification"].split(" | ")[-1]
                if pd.notna(row["higherClassification"]) and " | " in row["higherClassification"]
                else row["higherClassification"]
            )
        ),
        axis=1
    )


    # Verificar si "TAMANO_cm" existe y tiene al menos un valor no vacío
    if "TAMANO_cm" in df.columns and df["TAMANO_cm"].dropna().empty is False:
        df["measurementValue"] = df["TAMANO_cm"]
        df["measurementType"] = "Tamaño promedio aproximado (de cola a pico en centímetros) de los individuos"
        df["measurementUnit"] = "cm"
    else:

        df["measurementValue"] = None
        df["measurementType"] = None
        df["measurementUnit"] = None
    
   
    if "DIRE_GRUPO" in df.columns:
        df["DIRE_GRUPO"] = df["DIRE_GRUPO"].fillna(0).astype(float).astype(int).astype(str).replace({"0": ""})
        mapa_dire = {
        "151": "N (Norte)",
        "152": "S (Sur)",
        "153": "E (Este)",
        "154": "O (Oeste)",
        "155": "NE (Nor-Este)",
        "156": "SE (Sur-Este)",
        "157": "SO (Sur-Oeste)",
        "158": "NO (Nor-Oeste)"
        }
        df["temp_measurementValue_1"] = df["DIRE_GRUPO"].map(mapa_dire).fillna("")
    else:

        df["temp_measurementValue_1"] = None

    if "temp_measurementValue_1" in df.columns and df["temp_measurementValue_1"].dropna().empty is False:
        df["measurementValue_1"] = df["temp_measurementValue_1"]
        df["measurementType_1"] = " Comportamiento de viaje de los individuos respecto a la plataforma de observación"
    else:

        df["measurementValue_1"] = None
        df["measurementType_1"] = None

    # Mapear FECHA_MFL a verbatimEventDate
    if "FECHA" in df.columns:
        df["verbatimEventDate"] = df["FECHA"]
    else:

        df["verbatimEventDate"] = None


    # Calcular `eventDate`, `year`, `month`, `day` y conservamos `verbatimEventDate`
    if "verbatimEventDate" in df.columns:

        df["eventDate"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.strftime('%Y-%m-%d')
        df["year"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.year
        df["month"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.month
        df["day"] = pd.to_datetime(df["verbatimEventDate"], errors='coerce').dt.day

    else:

        df["eventDate"] = None
        df["year"] = None
        df["month"] = None
        df["day"] = None

    return df



# 🔹 Función principal para procesar aves_OFM
def procesar_aves_OFM(ruta_gdb, capa_aves_OFM, tabla_aves_OFM, enlace_aves_OFM, ruta_excel_aves_OFM, archivo_entrada_aves_OFM, archivo_salida_aves_OFM):
    try:
        # Abrir la Geodatabase
        gdb = gdal.OpenEx(ruta_gdb, gdal.OF_VECTOR)
        if not gdb:
            raise RuntimeError(f"❌ No se pudo abrir la GDB en {ruta_gdb}")

        # Extraer coordenadas de la capa de aves_OFM
        datos_capa = extraer_coordenadas(gdb.GetLayerByName(capa_aves_OFM))

        # Extraer atributos de la tabla de aves_OFM
        datos_tabla = []
        layer = gdb.GetLayerByName(tabla_aves_OFM)
        for feature in layer:
            datos_tabla.append(feature.items())  # Obtiene los atributos correctamente


        resultado = realizar_join(datos_capa, datos_tabla, enlace_aves_OFM)

        if resultado is None or resultado.empty:
            return




        # Primero mapear especie
        resultado = mapear_espec_nomb(resultado)

        # Exportar el resultado del join a un archivo Excel intermedio
        exportar_excel(resultado, ruta_excel_aves_OFM)

        # Leer el archivo Excel intermedio y agregar taxonRank
        agregar_datos_api_a_excel(ruta_excel_aves_OFM)

        # Leer el archivo Excel con taxonRank agregado
        df_intermedio = pd.read_excel(ruta_excel_aves_OFM)

        # Procesar campos específicos
        df_intermedio = procesar_campos_especificos(df_intermedio)

        # Crear DataFrame final con todos los atributos de lista_atributos
        df_final = pd.DataFrame(columns=lista_atributos)

        # Mapear los datos del DataFrame intermedio al DataFrame final
        for columna_intermedia, columna_final in mapeo.items():
            if columna_intermedia in df_intermedio.columns:
                df_final[columna_final] = df_intermedio[columna_intermedia]

        # Agregar valores constantes
        for clave, valor in valores_constantes.items():
            df_final[clave] = valor

        # Agregar los campos calculados
        
        df_final["verbatimCoordinates"] = df_intermedio["verbatimCoordinates"]
        #df_final["verbatimIdentification"] = df_intermedio["verbatimIdentification"]
        df_final["occurrenceID"] = df_intermedio["occurrenceID"]
        df_final["type"] = df_intermedio["type"]
        df_final["basisOfRecord"] = df_intermedio["basisOfRecord"]
        df_final["occurrenceRemarks"] = df_intermedio["occurrenceRemarks"]
        df_final["lifeStage"] = df_intermedio["lifeStage"]
        df_final["eventTime"] = df_intermedio["eventTime"]
        df_final["order"] = df_intermedio["order"]
        #df_final["temp_suborden"] = df_intermedio["temp_suborden"]
        df_final["family"] = df_intermedio["family"]
        df_final["genus"] = df_intermedio["genus"]
        df_final["higherClassification"] = df_intermedio["higherClassification"]
        df_final["scientificName"] = df_intermedio["scientificName"]
        df_final["measurementValue"] = df_intermedio["measurementValue"]
        df_final["measurementType"] = df_intermedio["measurementType"]
        df_final["measurementUnit"] = df_intermedio["measurementUnit"]
        df_final["measurementValue_1"] = df_intermedio["measurementValue_1"]
        df_final["measurementType_1"] = df_intermedio["measurementType_1"]
        df_final["eventDate"] = df_intermedio["eventDate"]
        df_final["year"] = df_intermedio["year"]
        df_final["month"] = df_intermedio["month"]
        df_final["day"] = df_intermedio["day"]
        df_final["verbatimEventDate"] = df_intermedio["verbatimEventDate"]
      





        # Exportar el DataFrame final a un archivo Excel
        exportar_excel(df_final, archivo_salida_aves_OFM)

    except Exception as e:
        return None