# -*- coding: utf-8 -*-
from qgis.PyQt.QtWidgets import QAction, QMessageBox
from qgis.PyQt.QtGui import QIcon
from qgis.core import QgsProcessingProvider, QgsApplication

from .geobiotool_shannon_algorithm      import GeoBioToolShannonAlgorithm
from .geobiotool_simpson_algorithm      import GeoBioToolSimpsonAlgorithm
from .geobiotool_fhd_algorithm          import GeoBioToolFHDAlgorithm
from .geobiotool_rumple_algorithm       import GeoBioToolRumpleAlgorithm
from .geobiotool_rugosity_algorithm     import GeoBioToolRugosityAlgorithm
from .geobiotool_lai_vci_algorithm      import GeoBioToolLAIVCIAlgorithm
from .geobiotool_beta_local_algorithm   import GeoBioToolBetaLocalAlgorithm

from . import resources  # load qrc


def _plugin_icon() -> QIcon:
    ico = QIcon(":/icons/icon.png")
    if not ico.isNull():
        return ico
    ico = QIcon(":/plugins/GeoBioTool/icons/icon.png")
    if not ico.isNull():
        return ico
    from os.path import dirname, join
    return QIcon(join(dirname(__file__), "icons", "icon.png"))


# ---- Ordered wrappers (Toolbox는 displayName으로 정렬되므로 접두 번호로 고정) ----
# Raster 그룹
class _OrderedShannon(GeoBioToolShannonAlgorithm):
    def displayName(self) -> str:
        return "1) Compute Shannon–Wiener (1948)"
    def group(self) -> str:
        return "1 Raster"

class _OrderedSimpson(GeoBioToolSimpsonAlgorithm):
    def displayName(self) -> str:
        return "2) Compute Simpson (1949)"
    def group(self) -> str:
        return "1 Raster"

class _OrderedBetaLocal(GeoBioToolBetaLocalAlgorithm):
    def displayName(self) -> str:
        return "3) Compute Local Beta (Jaccard / Sorensen)"
    def group(self) -> str:
        return "1 Raster"

# ASCII 그룹
class _OrderedFHD(GeoBioToolFHDAlgorithm):
    def displayName(self) -> str:
        return "1) Compute FHD Index (MacArthur & MacArthur, 1961)"
    def group(self) -> str:
        return "2 ASCII"

class _OrderedLAIVCI(GeoBioToolLAIVCIAlgorithm):
    def displayName(self) -> str:
        return "2) Compute LAI & VCI"
    def group(self) -> str:
        return "2 ASCII"

class _OrderedRugosity(GeoBioToolRugosityAlgorithm):
    def displayName(self) -> str:
        return "3) Compute Rugosity Index (Dustan et al. 2013)"
    def group(self) -> str:
        return "2 ASCII"

class _OrderedRumple(GeoBioToolRumpleAlgorithm):
    def displayName(self) -> str:
        return "4) Compute Rumple Index (Parker et al. 2004)"
    def group(self) -> str:
        return "2 ASCII"


class GeoBioToolProvider(QgsProcessingProvider):
    def loadAlgorithms(self):
        # Raster (고정 순서)
        self.addAlgorithm(_OrderedShannon())
        self.addAlgorithm(_OrderedSimpson())
        self.addAlgorithm(_OrderedBetaLocal())
        # ASCII (고정 순서)
        self.addAlgorithm(_OrderedFHD())
        self.addAlgorithm(_OrderedLAIVCI())
        self.addAlgorithm(_OrderedRugosity())
        self.addAlgorithm(_OrderedRumple())

    def id(self):
        return "geobiotool"

    def name(self):
        return "GeoBioTool"

    def longName(self):
        return "GeoBioTool: Biodiversity and Canopy Metrics"

    def icon(self):
        return _plugin_icon()


class GeoBioToolPlugin:
    def __init__(self, iface):
        self.iface = iface
        self.provider = None
        self.actions_toolbar = []  # keep only two here
        self.actions_menu = []     # menu can list all (incl. separators)

    def initGui(self):
        # register provider
        self.provider = GeoBioToolProvider()
        QgsApplication.processingRegistry().addProvider(self.provider)

        icon = _plugin_icon()

        # --- Toolbar: only Shannon + Simpson ---
        act_shannon = QAction(icon, "GeoBioTool — Run Shannon–Wiener", self.iface.mainWindow())
        act_shannon.triggered.connect(lambda: self._open_alg("geobiotool:compute_shannon_1948"))
        self.iface.addToolBarIcon(act_shannon)
        self.actions_toolbar.append(act_shannon)

        act_simpson = QAction(icon, "GeoBioTool — Run Simpson", self.iface.mainWindow())
        act_simpson.triggered.connect(lambda: self._open_alg("geobiotool:compute_simpson_1949"))
        self.iface.addToolBarIcon(act_simpson)
        self.actions_toolbar.append(act_simpson)

        # --- Helpers for plugin menu ---
        def add_menu_action(text, alg_id):
            a = QAction(icon, text, self.iface.mainWindow())
            a.triggered.connect(lambda: self._open_alg(alg_id))
            self.iface.addPluginToMenu("&GeoBioTool", a)
            self.actions_menu.append(a)

        def add_menu_separator():
            sep = QAction(self.iface.mainWindow())
            sep.setSeparator(True)
            self.iface.addPluginToMenu("&GeoBioTool", sep)
            self.actions_menu.append(sep)

        # --- Plugin menu: ASCII 보고서 vs Raster 출력 구분선 ---
        # ASCII-type outputs (TXT) — 메뉴 텍스트는 기존 그대로 유지(원하면 숫자 접두도 가능)
        add_menu_action("Run Shannon–Wiener",                    "geobiotool:compute_shannon_1948")
        add_menu_action("Run Simpson",                           "geobiotool:compute_simpson_1949")
        add_menu_action("Run Local Beta (Jaccard / Sorensen)",   "geobiotool:compute_beta_local")

        # Separator between ASCII and Raster sections
        add_menu_separator()

        # Raster outputs
        add_menu_action("Run FHD (MacArthur & MacArthur, 1961)", "geobiotool:compute_fhd_index_macarthur_1961")
        add_menu_action("Run LAI & VCI",                         "geobiotool:compute_lai_vci")
        add_menu_action("Run Rugosity (Dustan et al. 2013)",     "geobiotool:compute_rugosity_index_dustan_2013")
        add_menu_action("Run Rumple (Parker et al. 2004)",       "geobiotool:compute_rumple_index_parker_2004")

    def unload(self):
        for a in self.actions_toolbar:
            try:
                self.iface.removeToolBarIcon(a)
            except Exception:
                pass
        self.actions_toolbar.clear()

        for a in self.actions_menu:
            try:
                self.iface.removePluginMenu("&GeoBioTool", a)
            except Exception:
                pass
        self.actions_menu.clear()

        if self.provider:
            try:
                QgsApplication.processingRegistry().removeProvider(self.provider)
            except Exception:
                pass
            self.provider = None

    def _open_alg(self, alg_id: str):
        try:
            from qgis import processing
            processing.execAlgorithmDialog(alg_id)
        except Exception as e:
            QMessageBox.critical(self.iface.mainWindow(), "GeoBioTool", f"Failed to open dialog:\n{e}")
