# -*- coding: utf-8 -*-
"""
/***************************************************************************
 GeoQGIS
                                 A QGIS plugin
 GeoQGIS
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2019-01-21
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Geo
        email                : hmd@geo.dk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from PyQt5.QtCore import *
from PyQt5.QtGui import *
from PyQt5.QtWidgets import *
# Initialize Qt resources from file resources.py

from qgis.gui import *
from qgis.core import *
import os
import re

from .utils import *
from .virtualBoring import *
from .ApiKeyGetter import *
from .SliceTool import *
from .layer_manager import LayerManager
from .model_manager import ModelManager
from .resources import *
from .Crosssection import *
from .report import *


class GeoQGIS:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        # Used for finding files in the directory.
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        # Currently unused, but useful if we start translating.
        # locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'GeoQGIS_{}.qm'.format("en"))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)

            if qVersion() > '4.3.3':
                QCoreApplication.installTranslator(self.translator)

        self.settings = Settings()
        try:
            self.settings.settings_updated.disconnect(self.reloadMenu)
        except TypeError:
            pass
        self.settings.settings_updated.connect(self.reloadMenu)
        self.options_factory = OptionsFactory(self.settings)
        self.options_factory.setTitle(self.tr('GeoAtlas'))
        iface.registerOptionsWidgetFactory(self.options_factory)
        iface.mapCanvas().setPreviewJobsEnabled(False)
       

        # Declare instance attributes
        # Actions for action bar
        self.actions = []
        self.menu = self.tr(u'&GeoQGIS')
        self.layersMenu = None
        self.hydromodelsMenu = None
        self.currentModels = None
        self.myToolBar = None
        
        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        for i in range(10):
            debugMsg("")
        debugMsg("Loaded plugin")
        self.first_start = None
        self.svgWidget = None
        self.modelid = 0 
        self.apiKeyGetter = ApiKeyGetter(self.iface, self.settings)
        self.apiKey = self.apiKeyGetter.getApiKey()
        self.model_manager = ModelManager(self)
        if self.apiKey is not None:
            self.model_manager.ensureElemDict()
        self.virtualBoring = VirtualBoringTool(self.iface, self.model_manager.elemdict, self.apiKeyGetter, self.settings)
        self.sliceTool = SliceTool(self.iface, self.model_manager.elemdict, self.apiKeyGetter, self.settings)
        self.crosssectionTool = Crosssection(self.iface, self.model_manager.elemdict, self.apiKeyGetter, self.settings)
        self.report = ReportTool(self.iface, self.apiKeyGetter)
        self.layer_manager = LayerManager(self)
        # Timer is used for regularly updating tokens and keeping access to 
        # wms layers as the tokens only last for 22 hours.
        self.register_timer_for_token_updater()
        self.update_GAL_layers_with_tokens()
        

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('GeoQGIS', message)

    def register_timer_for_token_updater(self):
        # When we load a new project, update the tokens in it.
        self.iface.projectRead.connect(self.update_token_on_project_load)
        self.timer = QTimer()
        
        self.timer.timeout.connect(self.update_GAL_layers_with_tokens)
        self.timer.start(1000 * 60 * 30) #Is in miliseconds. So runs every half hour
        # Should be okay as it last for 22 hours.

    def update_token_on_project_load(self):
        # Wait for 10 seconds before updating tokens, as it seems doing it while loading 
        # makes QGIS crash.
        self.timer.singleShot(10000, self.update_GAL_layers_with_tokens)

    def makeMenu(self):
        if self.menu and hasattr(self.menu, 'menuAction'):
            self.iface.mainWindow().menuBar().removeAction(self.menu.menuAction())
            self.menu.deleteLater()
        self.menu = None
        # Tool bar menu.
        self.menu = QMenu( "GeoAtlas", self.iface.mainWindow().menuBar() )
        actions = self.iface.mainWindow().menuBar().actions()
        lastAction = actions[-1]
        self.iface.mainWindow().menuBar().insertMenu( lastAction, self.menu )
        self.menu.addAction( 'Add models to map', self.model_manager.addModelsToMap)
        #self.menu.addAction( 'Print Api Key', self.apiKeyGetter.printApiKey)
        self.menu.addAction( 'Add Boreholes to map', lambda: self.layer_manager.addLayer("GAL - Boreholes", "borehole", "borehole-labels"))
        self.hydromodelsMenu = QMenu('Add hydromodels to map', self.menu)
        self.menu.addMenu(self.hydromodelsMenu)
        self.layer_manager.populateHydromodelsMenu(self.hydromodelsMenu)
        # Create nested submenu for adding layers
        # We'll set the title with count after populating
        self.layersMenu = QMenu('Add layers to map', self.menu)
        self.menu.addMenu(self.layersMenu)
        
        # Populate the layers menu
        self.layer_manager.populateLayersMenu(self.layersMenu)
        self.menu.addAction('Refresh hydromodel catalogue', self.layer_manager.refreshHydromodelsMenu)
        self.menu.addAction('Refresh layer catalogue', self.layer_manager.refreshLayersMenu)

        self.menu.addAction( 'Update Tokens', self.update_GAL_layers_with_tokens)
        self.menu.addAction( 'Help', self.helpmessagebox)
        self.menu.addAction( 'About', self.aboutmessagebox)

        self.myToolBar = self.iface.mainWindow().findChild( QToolBar, u'GeoAtlasToolBar' )
        if not self.myToolBar:
            self.myToolBar = self.iface.addToolBar( u'GeoAtlasToolBar' )
            self.myToolBar.setObjectName( u'GeoAtlasToolBar' )

        self.myToolBar.clear()
        self.addActionsToActionBar()
        # add toolbar button and menu item


    def update_GAL_layers_with_tokens(self):
        debugMsg("Updating Tokens.")
        self.apiKey = self.apiKeyGetter.getApiKey() # update key
        self.model_manager.ensureElemDict()
        
        token_regex = r'(&|%26)?token([=:]|%3A|%3D)(?P<Token>[\d\w\.=+-_\/]*)'
        #Find all layers with tokens in them, which are updatable and created by us.
        for layer in self.iface.mapCanvas().layers():
            if not layer.name().startswith("GAL"):
                continue
            if not type(layer) is QgsRasterLayer:
                continue
            if not callable(getattr(layer, "dataProvider", None)):
                continue    
            if not callable(getattr(layer.dataProvider(), "dataSourceUri", None)):
                continue
            uri = layer.dataProvider().dataSourceUri()
            token =  re.search(token_regex, uri)
            if not token:
                continue
            token = token.group('Token')
            # make sure the function exists, else we crash
            if callable(getattr(layer, "setDataSource", None)):
                debugMsg("  Updated Token for layer: " + layer.name())
                uri = uri.replace(token, self.apiKeyGetter.getApiKeyNoBearer())
                layer.setDataSource(uri, layer.name(), 'wms', QgsDataProvider.ProviderOptions()) 



    def addActionsToActionBar(self):
        # The action menu bar. 
        crosstool = QAction(QIcon( self.plugin_dir + "/images/cross.png"), 'Get profile of existing line', self.iface.mainWindow())
        crosstool.triggered.connect(self.crosssectionTool.crossectionExistingLine)
        self.myToolBar.addAction(crosstool)
        crosstool2 = QAction(QIcon( self.plugin_dir + "/images/crossNew.png"), 'Get profile of new line', self.iface.mainWindow())
        crosstool2.triggered.connect(self.crosssectionTool.createNewLineAndCrossSection)
        self.myToolBar.addAction(crosstool2)
        slicetool = QAction(QIcon( self.plugin_dir + "/images/slice.png"), 'Open Slice view', self.iface.mainWindow())
        slicetool.triggered.connect(self.sliceTool.startSliceTool)
        self.myToolBar.addAction(slicetool)
        boretool = QAction(QIcon( self.plugin_dir + "/images/bore.png"), 'Make virtual borehole', self.iface.mainWindow())
        boretool.triggered.connect(self.virtualBoring.changeToBoringTool)
        self.myToolBar.addAction(boretool)

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        self.makeMenu()
        # will be set False in run()
        self.first_start = True

    def unload(self):
        self.clearMenu()
        if self.myToolBar:
            self.iface.mainWindow().removeToolBar(self.myToolBar)
            self.myToolBar = None
        if hasattr(self, 'options_factory'):
            self.iface.unregisterOptionsWidgetFactory(self.options_factory)
        
    def helpmessagebox(self):
        msgBox = QMessageBox()
        msgBox.setWindowTitle( "Help" )
        msgBox.setTextFormat( Qt.RichText )
        msgBox.setText( "<br>We have two manuals to help you along<br>" 
            + "GeoAtlasLive Manual: <a href='{0}'>{0}</a><br><br>".format("https://wgn.geo.dk/geodata/GeoAtlasLive_Manual.pdf")
            + "Plugin Manual: <a href='{0}'>{0}</a><br><br>".format("https://wgn.geo.dk/geodata/GeoAtlasPlugin_Manual.pdf"))

        msgBox.setStandardButtons( QMessageBox.Ok )
        msgBox.exec_() 

    def aboutmessagebox(self):
        title = "About"
        message = "QGIS implementation of GeoAtlasLive\n"
        message += "Version 1.3\n"
        message += "Copyright (c) 2022 GEO\n"
        message += "data@geo.dk"
        QMessageBox.information(self.iface.mainWindow(), title, message)

    def reloadMenu(self):
        self.clearMenu()
        self.makeMenu()
    
    def clearMenu(self):
        if getattr(self, 'myToolBar', None):
            self.myToolBar.clear()
        if self.menu:
            self.iface.mainWindow().menuBar().removeAction(self.menu.menuAction())
            self.menu.deleteLater()
            self.menu = None
        self.layersMenu = None
        self.hydromodelsMenu = None


