"""
/***************************************************************************
 GarminCustomMap
						         A QGIS plugin
 Export map canvas to a Garmin Custom Map
						      -------------------
        begin                : 2012-01-17
        copyright            : (C) 2012 by Stefan Blumentrath - Norwegian Institute for Nature Research (NINA)
        email                : stefan.blumentrath@nina.no
 ***************************************************************************/

/***************************************************************************
 *																		 *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.						           *
 *																		 *
 ***************************************************************************/
"""
# To dos
# Skip empty tiles flag in GUI
# nodata-value after reprojection should be 255 for all 3 bands
# Setting hints should be included into plugin GUI (no extra message)
# Leave plugin GUI open and reapply user-settings
# Further memory optimisation for the QImage process
# 
# Multiprocess tiling
#
# Import the PyQt and QGIS libraries
from PyQt4.QtCore import *
from PyQt4.QtGui import *
from qgis.core import *
from qgis.gui import *
from qgis.utils import *

from osgeo import gdal
from osgeo import gdalnumeric
from osgeo import gdalconst
from osgeo import osr

import sys, itertools, os, subprocess, zipfile, zlib, tempfile

from math import *

# Initialize Qt resources from file resources.py
import resources
# Import the code for the dialog
from garmincustommapdialog import GarminCustomMapDialog

class GarminCustomMap:

    def __init__(self, iface):
        # Save reference to the QGIS interface
        self.iface = iface

    def initGui(self):
        # Create action that will start plugin configuration
        self.action = QAction(QIcon(":/plugins/garmincustommap/gcm_icon.png"), \
            "GarminCustomMap", self.iface.mainWindow())
        # connect the action to the run method
        QObject.connect(self.action, SIGNAL("triggered()"), self.run)

        # Add toolbar button and menu item
        self.iface.addToolBarIcon(self.action)
        self.iface.addPluginToMenu("&GarminCustomMap", self.action)

    def unload(self):
        # Remove the plugin menu item and icon
        self.iface.removePluginMenu("&GarminCustomMap",self.action)
        self.iface.removeToolBarIcon(self.action)

    def CleanUp(self):
        # Reset mapRenderer
        mapRenderer.setOutputSize(QSize(old_width, old_height), old_dpi)
        
        if os.path.exists(out_put + ".png") :
            os.remove(out_put + ".png")
        
        if os.path.exists(out_put + ".png.aux.xml") :
            os.remove(out_put + ".png.aux.xml")
		
        if os.path.exists(output_geofile) :
            os.remove(output_geofile)
		
        if os.path.exists(os.path.join(out_folder, input_geofile)) :
            os.remove(os.path.join(out_folder, input_geofile))
		
        if os.path.exists(os.path.join(out_folder, 'doc.kml')) :
            os.remove(os.path.join(out_folder, 'doc.kml'))
		
        if os.path.exists(os.path.join(out_folder, t_name)) :
            os.remove(os.path.join(out_folder, t_name))
		
        if os.path.exists(out_folder) :
            os.rmdir(out_folder)
		

    # run method that performs all the real work
    def run(self):

		#Set output file name
		# prepare dialog parameters
		settings = QSettings()
		lastDir = settings.value( "/UI/lastShapefileDir" )
		filter = "GarminCustomMap-files (*.kmz)"
		out_putFile = QgsEncodingFileDialog(None, "Select output file", 'My_CustomMap.kmz', "GarminCustomMap (*.kmz)" )
		out_putFile.setDefaultSuffix( "kmz" )
		out_putFile.setFileMode( QFileDialog.AnyFile )
		out_putFile.setAcceptMode( QFileDialog.AcceptSave )
		out_putFile.setConfirmOverwrite( True )
		if out_putFile.exec_() == QDialog.Accepted:
			kmz_file = str(out_putFile.selectedFiles()[0])	
			#Get mapCanvas and mapRenderer variables
			canvas = self.iface.mapCanvas()
			mapRenderer = canvas.mapRenderer()
			mapRect = mapRenderer.extent()
			width = mapRenderer.width()
			height = mapRenderer.height()
			srs = mapRenderer.destinationCrs()
			SourceCRS = str(srs.authid())
			#Save settings for resetting the mapRenderer after GCM production
			old_width = mapRenderer.width()
			old_height = mapRenderer.height()
			old_dpi = mapRenderer.outputDpi()
			#Give information about project projection, mapCanvas size and Custom map settings
			if (height * width ) % ( 1024.0 * 1024.0 ) >= 1 : expected_tile_n_unzoomed = int( ( height * width ) / ( 1024.0 * 1024.0 ) ) + 1
			else : expected_tile_n_unzoomed = int( ( height * width ) / ( 1024.0 * 1024.0 ) )
			
			if len(str(int(sqrt( 100 / ( ( height * width ) / ( 1024.0 * 1024.0 ) ) ) ) ) ) == 1 : max_zoom = str(sqrt( 100 / ( ( height * width ) / ( 1024.0 * 1024.0 ) ) ) )[0:3]
			else : max_zoom = str(sqrt( 100 / ( ( height * width ) / ( 1024.0 * 1024.0 ) ) ) )[0:4]
			
			canvas_msg = QMessageBox()
			canvas_msg.setWindowTitle("Info about map input")
			
			if SourceCRS != 'EPSG:4326':
				iface.messageBar().pushMessage("WARNING", "The coordinate system of your project differs from WGS 1984 (EPSG: 4326).\n"
	"It is likely, that you will get a better Custom Map\n"
	"when your project and data has CRS\n"
	"WGS 1984!\n"
	"\n"
	"The number of rows and columns in the map\n"
	"will be affected by reprojecting to WGS 1984!", level=QgsMessageBar.WARNING, duration=10)
			
			
			canvas_msg.setText("The following information should help you\n"
	"to adjust the settings for your Garmin Custom Map\n"
	"in the next window.\n"
	"\n"
	"Your current map canvas contains\n"
	+ str(height) + " rows and\n"
	+ str(width) + " colums.\n"
	"\n"
	"Zooming level 1.0 will result in " + str(expected_tile_n_unzoomed) + " tile(s),\n"
	"which is " + str(expected_tile_n_unzoomed) + "% of the maximum possible number of tiles\n"
	"across all Custom Maps on your GPS unit.\n"
	"\n"
	"To comply with Garmins limit for the number of tiles,\n"
	"you should use a zoom factor <= " + max_zoom + "\n"
	"\n"
	"For more information on size limits\n"
	"in Garmin Custom Maps see Info tab\n"
	"in the next window.\n")
			
			canvas_msg.exec_()
			# create and show the dialog
			dlg = GarminCustomMapDialog()
			# show the dialog
			#dlg.show()
			result = dlg.exec_()
			# See if OK was pressed
			if result == 1:
				#Set variables
				optimize =  int(dlg.ui.flag_optimize.isChecked())
				skip_empty = int(dlg.ui.flag_skip_empty.isChecked())
				#skip_empty =  1
				max_y_ext_general = int(dlg.ui.nrows.value())
				max_x_ext_general = int(dlg.ui.ncols.value())
				qual = int(dlg.ui.jpg_quality.value())
				# Set options for jpg-production
				options = []
				options.append("QUALITY=" + str(qual))
				draworder = dlg.ui.draworder.value()
				zoom = float(dlg.ui.zoom.value())
				in_file = os.path.basename(kmz_file[0:(len(kmz_file)-4)])
				max_pix = ( 1024 * 1024 )
				#Create tmp-folder
				out_folder = tempfile.mkdtemp('_tmp', 'gcm_')
				out_put = os.path.join(str(out_folder), str(in_file))
				input_file = out_put + ".png"
				#input_file="/home/stefan/testfile2.png"
				
				tname = in_file
				#Set QGIS objects
				target_dpi = int(round(zoom * mapRenderer.outputDpi()))
				# Initialise temporary output image
				x, y = 0, 0
				width = int(round(mapRenderer.width() * zoom))
				height = int(round(mapRenderer.height() * zoom))
				
				# create output image and initialize it
				image = QImage(QSize(width, height), QImage.Format_RGB555)
				image.fill(qRgb(255,255,255))
				
				#adjust map canvas (renderer) to the image size and render
				imagePainter = QPainter(image)
				mapRenderer.setOutputSize(QSize(width, height), target_dpi)
				mapRenderer.render(imagePainter)
				imagePainter.end()
				
				#Save the image
				image.save(input_file, "png")
				
				#Errorhandler
				#if os.path_exists(input_file) :
					#writesuccess == False:
					#Give error message
				#	iface.messageBar().pushMessage("ERROR", "Could not write temporary outputfile to " + str(input_file) + " with " + str(width) + " x " + str(height) + " pixels. This is likely due to memory (RAM) limitations. Please check your zoom-factor settings", level=QgsMessageBar.CRITICAL)
					#Reset mapRenderer to old size (monitor)
				#	CleanUp()
					#mapRenderer.setOutputSize(QSize(old_width, old_height), old_dpi)
					#Save user defined values to UI settings
					#UI_settings = QSettings()
				
				###Set Geotransform and NoData values 
				input_dataset = gdal.Open(input_file)
				
				#Set Geotransform values
				ULx = mapRect.xMinimum()
				ULy = mapRect.yMaximum()
				LRx = mapRect.xMaximum()
				LRy = mapRect.yMinimum()
				xScale = ( LRx - ULx ) /  image.width()
				yScale = ( ULy - LRy ) /  image.height()
				
				input_dataset.SetGeoTransform([ULx,xScale,0,ULy,0,yScale])
				
				#Set NoData values
				#input_dataset.GetRasterBand(1).SetNoDataValue( 255 )
				#input_dataset.GetRasterBand(2).SetNoDataValue( 255 )
				#input_dataset.GetRasterBand(3).SetNoDataValue( 255 )
				
				#Close dataset
				input_dataset = None

				##Export tfw-file
								
				#f = open(out_put + ".pgw", 'w')
				#f.write(str(xScale) + '\n')
				#f.write(str(0) + '\n')
				#f.write(str(0) + '\n')
				#f.write('-' + str(yScale) + '\n')
				#f.write(str(ULx) + '\n')
				#f.write(str(ULy))
				#f.close()
				
				#Reset mapRenderer to old size (monitor)
				mapRenderer.setOutputSize(QSize(old_width, old_height), old_dpi)
				
				#Warp the exported image to WGS84 if necessary
				if SourceCRS != 'EPSG:4326':
					#Define input and output file
					input_geofile = out_put + "wgs84.tif"
					output_geofile = os.path.join(out_folder, input_geofile)
					#Register tif-driver
					driver = gdal.GetDriverByName("GTiff")
					driver.Register()
					#Define input CRS
					in_CRS = str(srs.toWkt())
					#Define output CRS
					out_CRS = str(QgsCoordinateReferenceSystem(4326, QgsCoordinateReferenceSystem.EpsgCrsId).toWkt())
					#Open input dataset
					input_dataset = gdal.Open(input_file)
					#Reproject					
					reproj_file = gdal.AutoCreateWarpedVRT(input_dataset, in_CRS, out_CRS)
					#reproj_file.GetRasterBand(1).SetNoDataValue( 255 )
					#reproj_file.GetRasterBand(2).SetNoDataValue( 255 )
					#reproj_file.GetRasterBand(3).SetNoDataValue( 255 )
					reproj_file.GetRasterBand(1).Fill(255)
					reproj_file.GetRasterBand(2).Fill(255)
					reproj_file.GetRasterBand(3).Fill(255)
					
					#, gdal.padfDstNoDataReal=255, gdal.padfDstNoDataImag=255
					
					gdal.ReprojectImage(input_dataset, reproj_file, in_CRS, out_CRS)
					reproj_attributes = reproj_file.GetGeoTransform()
					
					#Update relevant georef variables
					ULx = reproj_attributes[0]
					ULy = reproj_attributes[3]
					xScale = reproj_attributes[1]
					yScale = reproj_attributes[5]
					
					driver = gdal.GetDriverByName("GTiff")
					warped_input = driver.CreateCopy(output_geofile, reproj_file, 0)
					
					input_dataset = None
					reproj_file = None
					warped_input = None
					input_file = output_geofile
								
				#Calculate tile size and number of tiles
				indataset = gdal.Open(input_file)
				
				x_extent = indataset.RasterXSize
				y_extent = indataset.RasterYSize
				
				if optimize == 1 :
					#Identify length of the short and long side of the map canvas and their relation 
					short_ext = min(x_extent, y_extent)
					long_ext = max(x_extent, y_extent)
					s_l_side_relation = 0
					
					#Estimate number of tiles in the result
					if float( x_extent * y_extent ) % (1024 * 1024) >= 1 : expected_tile_n = int( float( x_extent * y_extent ) / (1024 * 1024) ) + 1
					else : expected_tile_n = int(float( x_extent * y_extent ) / (1024 * 1024))

					#Find settings for tiling with:
					# 1 minimum number of tiles,
					# 2 a short / long size relation close to 1,
					# 3 and a minimum numer of pixels in each tile
					for tc in range(1, expected_tile_n + 1, 1):
						if expected_tile_n % tc >= 1 : continue
						else :
							
							if short_ext % tc >= 1 : s_pix = int( short_ext / tc ) + 1
							else : s_pix = int( short_ext / tc )
							
							if long_ext % tc >= 1 : l_pix = int( long_ext / ( expected_tile_n / tc ) ) + 1
							else : l_pix = int( long_ext / ( expected_tile_n / tc ) )
							
							if ( s_pix * l_pix ) <= ( 1024 * 1024 ):
								#print("min s/l-relation is " +str(min( ( float(s_pix) / float(l_pix) ), ( float(l_pix) / float(s_pix) ) ) ) )
								if min( ( float(s_pix) / float(l_pix) ), ( float(l_pix) / float(s_pix) ) ) >= s_l_side_relation : 
									s_l_side_relation = min( ( float(s_pix) / float(l_pix) ), ( float(l_pix) / float(s_pix) ) )
									s_pix_opt = s_pix
									l_pix_opt = l_pix 

					#
					#Set tile size variable according to optimal setings
					if short_ext == x_extent :
						max_x_ext_general = s_pix_opt
						max_y_ext_general = l_pix_opt
					else : 
						max_y_ext_general = s_pix_opt
						max_x_ext_general = l_pix_opt

				#
				#Identify number of rows and columns
				n_cols_rest = x_extent % max_x_ext_general
				n_rows_rest = y_extent % max_y_ext_general
				#
				if n_cols_rest >= 1 : n_cols=( x_extent / max_x_ext_general ) + 1
				else : n_cols=( x_extent / max_x_ext_general )
				
				#
				if n_rows_rest >= 1 : n_rows=( y_extent / max_y_ext_general ) + 1
				else : n_rows=( y_extent / max_y_ext_general )
				
				#
				#Check if number of tiles is below Garmins limit of 100 tiles (across all custom maps)
				n_tiles = ( n_rows * n_cols )
				if n_tiles > 100 :
					iface.messageBar().pushMessage("WARNING", "The number of tiles is likely to exceed Garmins limit of 100 tiles! Not all tiles will be displayed on your GPS unit. Consider reducing your map size (extend or zoom-factor).", level=QgsMessageBar.WARNING, duration=5)
				
				progressMessageBar = iface.messageBar().createMessage("Producing tiles...")
				progress = QProgressBar()
				progress.setMaximum(n_tiles)
				progress.setAlignment(Qt.AlignLeft|Qt.AlignVCenter)
				progressMessageBar.layout().addWidget(progress)
				iface.messageBar().pushWidget(progressMessageBar, iface.messageBar().INFO)

				#
				#Check if size of tiles is below Garmins limit of 1 megapixel (for each tile)
				n_pix = ( max_x_ext_general * max_y_ext_general )
				
				if n_pix > max_pix : 
					iface.messageBar().pushMessage("WARNING", "The number of pixels in a tile exceeds Garmins limit of 1 megapixel per tile! Images will not be displayed properly.", level=QgsMessageBar.WARNING, duration=5)
				
				kmz = zipfile.ZipFile(kmz_file, 'w')
				kml = open(os.path.join(out_folder, 'doc.kml'), 'w')
				#
				#Write kml header
				kml.write('<?xml version="1.0" encoding="UTF-8"?>\n')
				kml.write('<kml xmlns="http://www.opengis.net/kml/2.2">\n')
				kml.write('  <Document>\n')
				kml.write('    <name>' + tname + '</name>\n')
				#
				#Produce .jpg tiles using gdal_translate looping through the complete rows and columns (1024x1024 pixel)
				y_offset = 0
				x_offset = 0
				r = 1
				c = 1
				n_tiles = 0
				empty_tiles = 0
				#Loop through rows
				for r in range(1, n_rows + 1, 1):
					#Define maximum Y-extend of tiles
					if r == ( n_rows ) and n_rows_rest > 0 : max_y_ext = n_rows_rest
					else: max_y_ext = max_y_ext_general
					
					#(Within row-loop) Loop through columns
					for c in range(1, n_cols + 1, 1):
						#Define maximum X-extend of tiles
						if c == ( n_cols ) and n_cols_rest > 0 : max_x_ext = n_cols_rest
						else: max_x_ext = max_x_ext_general
						#Define name for tile-jpg
						t_name = tname + '_%(r)d_%(c)d.jpg' %{"r":r, "c":c}
						#Set parameters for "gdal_translate" (JPEG-driver has no Create() (but CreateCopy()) method so first a VRT has to be created band by band
						#Create VRT dataset for tile
						mem_driver = gdal.GetDriverByName("MEM")
						mem_driver.Register()
						t_file = mem_driver.Create('',max_x_ext,max_y_ext,3,gdalconst.GDT_Byte)
						t_band_1 = indataset.GetRasterBand(1).ReadAsArray(x_offset,y_offset,max_x_ext,max_y_ext)
						t_band_2 = indataset.GetRasterBand(2).ReadAsArray(x_offset,y_offset,max_x_ext,max_y_ext)
						t_band_3 = indataset.GetRasterBand(3).ReadAsArray(x_offset,y_offset,max_x_ext,max_y_ext)
						if skip_empty == 1 :
							if t_band_1.min() == 255 and t_band_2.min() == 255 and t_band_3.min() == 255 :
								empty_tiles = empty_tiles + 1
						
						t_file.GetRasterBand(1).WriteArray(t_band_1)
						t_file.GetRasterBand(2).WriteArray(t_band_2)
						t_file.GetRasterBand(3).WriteArray(t_band_3)
						t_band_1 = None
						t_band_2 = None
						t_band_3 = None
						
						#Translate MEM dataset to JPG
						jpg_driver = gdal.GetDriverByName("JPEG")
						jpg_driver.Register()
						jpg_driver.CreateCopy( os.path.join(out_folder, t_name), t_file, options=options )
						
						#Close GDAL-datasets
						t_file = None
						t_jpg = None
						#Get bounding box for tile
						n = ULy - ( y_offset * yScale )
						s = ULy - ( ( y_offset + max_y_ext ) * yScale )
						e = ULx + ( ( x_offset + max_x_ext ) * xScale )
						w = ULx + ( x_offset * xScale )
						#Add .jpg to .kmz-file and remove it together with its meta-data afterwards
						kmz.write(os.path.join(out_folder, t_name), t_name)
						os.remove(os.path.join(out_folder, t_name))
						#
						#Write kml-tags for each tile (Name, DrawOrder, JPEG-Reference, GroundOverlay)
						kml.write('')    
						kml.write('    <GroundOverlay>\n')
						kml.write('        <name>' + tname + ' Tile ' + str(r) + '_' + str(c) + '</name>\n') #%{"r":r, "c":c}
						kml.write('        <drawOrder>' + str(draworder) + '</drawOrder>\n') #%{"draworder":draworder}
						kml.write('        <Icon>\n')
						kml.write('          <href>' + tname + '_' + str(r) + '_' + str(c) + '.jpg</href>\n') #%{"r":r, "c":c}
						kml.write('        </Icon>\n')
						kml.write('        <LatLonBox>\n')
						kml.write('          <north>' + str(n) + '</north>\n')
						kml.write('          <south>' + str(s) + '</south>\n')
						kml.write('          <east>' + str(e) + '</east>\n')
						kml.write('          <west>' + str(w) + '</west>\n')
						kml.write('        </LatLonBox>\n')
						kml.write('    </GroundOverlay>\n')
						
						#Calculate new X-offset
						x_offset=( x_offset + max_x_ext )
						n_tiles=( n_tiles + 1 )
						#Update progress bar
						progress.setValue(n_tiles)
					#Calculate new Y-offset
					y_offset=( y_offset + max_y_ext )
					#Reset X-offset
					x_offset=0
								
				#
				#Write kml footer
				kml.write('  </Document>\n')
				kml.write('</kml>\n')
				#Close kml file
				kml.close()
				
				#Close GDAL dataset
				indataset = None
				
				#Remove temporary geo-tif file
				os.remove(out_put + ".png")
				os.remove(out_put + ".png.aux.xml")
				
				
				#Remove reprojected temporary geo-tif file if necessary
				if SourceCRS != 'EPSG:4326':
					os.remove(output_geofile)
				
				#Add .kml to .kmz-file and remove it together with the rest of the temporary files
				kmz.write(os.path.join(out_folder, 'doc.kml'), 'doc.kml')
				os.remove(os.path.join(out_folder, 'doc.kml'))
				kmz.close()
				os.rmdir(out_folder)
				#Clear progressbar
				iface.messageBar().clearWidgets()
				
				tiles_total = n_tiles - empty_tiles
				#Give success message
				iface.messageBar().pushMessage("Done", "Produced " + str(tiles_total) + " tiles, with " +str(n_rows) + " rows and " + str(n_cols) + " colums.", level=QgsMessageBar.INFO, duration=5)
