"""
/***************************************************************************
 GarminCustomMap
						         A QGIS plugin
 Export map canvas to a Garmin Custom Map
						      -------------------
        begin                : 2012-01-17
        copyright            : (C) 2012 by Stefan Blumentrath - Norwegian Institute for Nature Research (NINA)
        email                : stefan.blumentrath@nina.no
 ***************************************************************************/

/***************************************************************************
 *																		 *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.						           *
 *																		 *
 ***************************************************************************/
"""
# Import the PyQt and QGIS libraries
from PyQt4.QtCore import *
from PyQt4.QtGui import *
from qgis.core import *
from qgis.gui import *

from osgeo import gdal
from osgeo import gdalnumeric
from osgeo import gdalconst

import sys, itertools, os, subprocess, zipfile, zlib, tempfile

from math import *

# Initialize Qt resources from file resources.py
#import resources
# Import the code for the dialog
from garmincustommapdialog import GarminCustomMapDialog

class GarminCustomMap:

    def __init__(self, iface):
        # Save reference to the QGIS interface
        self.iface = iface

    def initGui(self):
        # Create action that will start plugin configuration
        self.action = QAction(QIcon(":/plugins/garmincustommap/icon.png"), \
            "GarminCustomMap", self.iface.mainWindow())
        # connect the action to the run method
        QObject.connect(self.action, SIGNAL("triggered()"), self.run)

        # Add toolbar button and menu item
        self.iface.addToolBarIcon(self.action)
        self.iface.addPluginToMenu("&GarminCustomMap", self.action)

    def unload(self):
        # Remove the plugin menu item and icon
        self.iface.removePluginMenu("&GarminCustomMap",self.action)
        self.iface.removeToolBarIcon(self.action)

    # run method that performs all the real work
    def run(self):

		#Set output file name
		# prepare dialog parameters
		settings = QSettings()
		lastDir = settings.value( "/UI/lastShapefileDir" ).toString()
		filter = QString( "GarminCustomMap-files (*.kmz)" )
		out_putFile = QgsEncodingFileDialog(None, "Select output file", 'My_CustomMap.kmz', "GarminCustomMap (*.kmz)" )
		out_putFile.setDefaultSuffix( QString( "kmz" ) )
		out_putFile.setFileMode( QFileDialog.AnyFile )
		out_putFile.setAcceptMode( QFileDialog.AcceptSave )
		out_putFile.setConfirmOverwrite( True )
		if out_putFile.exec_() == QDialog.Accepted:
			kmz_file = str(out_putFile.selectedFiles()[0])	
			#Get mapCanvas and mapRenderer variables
			canvas = self.iface.mapCanvas()
			mapRenderer = canvas.mapRenderer()
			mapRect = mapRenderer.extent()
			width = mapRenderer.width()
			height = mapRenderer.height()
			srs=mapRenderer.destinationCrs()
			
			#Give information about project projection, mapCanvas size and Custom map settings
			canvas_msg = QMessageBox()
			canvas_msg.setWindowTitle("Info about map input");
			if srs.epsg() != 4326:
				#Give warning message
				canvas_msg.setText("The following information should help you\n"
	"to adjust the settings for your Garmin Custom Map\n"
	"in the next window.\n"
	"\n"
	"Your current map canvas contains\n"
	+ str(height) + " rows and\n"
	+ str(width) + " colums.\n"
	"\n"
	"This is " + str(float(height * width * 100) / (1024 * 1024 *100))[0:4] + " of the maximum possible pixels\n"
	"across all Custom Maps on your GPS unit.\n"
	"\n"
	"For more information on size limits\n"
	"in Garmin Custom Maps see Info tab\n"
	"in the next window.\n"
	"\n"
	"WARNING: The coordinate system of your project\n"
	"differs from WGS 1984 (EPSG: 4326).\n"
	"It is likely, that you will get a better Custom Map\n"
	"when your project and data has CRS\n"
	"WGS 1984!\n"
	"\n"
	"The number of rows and columns in the map\n"
	"will be affected by reprojecting to WGS 1984!")
			
			else : 
				canvas_msg.setText("The following information should help you\n"
	"to adjust the settings for your Garmin Custom Map\n"
	"in the next window.\n"
	"\n"
	"Your current map canvas contains\n"
	+ str(height) + " rows and\n"
	+ str(width) + " colums.\n"
	"\n"
	"This is " + str(float(height * width * 100) / (1024 * 1024 *100))[0:4] + " of the maximum possible pixels\n"
	"across all Custom Maps on your GPS unit.\n"
	"\n"
	"For more information on size limits\n"
	"in Garmin Custom Maps see Info tab\n"
	"in the next window.\n")
			
			canvas_msg.exec_()
			# create and show the dialog
			dlg = GarminCustomMapDialog()
			# show the dialog
			dlg.show()
			result = dlg.exec_()
			# See if OK was pressed
			if result == 1:
				#Set variables
				max_y_ext_general = int(dlg.ui.nrows.value())
				max_x_ext_general = int(dlg.ui.ncols.value())
				qual = int(dlg.ui.jpg_quality.value())
				draworder = dlg.ui.draworder.value()
				zoom = float(dlg.ui.zoom.value())
				in_file=os.path.basename(kmz_file[0:(len(kmz_file)-4)])
				#Create tmp-folder
				out_folder = tempfile.mkdtemp('_tmp', 'gcm_')
				out_put = os.path.join(str(out_folder), str(in_file))
				input_file=out_put + ".tif"
				tname=in_file
				#Set QGIS objects
				target_dpi = int(round(zoom * mapRenderer.outputDpi()))
				# set up composition
				c = QgsComposition(mapRenderer)
				c.setPlotStyle(QgsComposition.Print)
				c.setPrintResolution(target_dpi)
				c.setPaperSize(mapRenderer.width()/(mapRenderer.outputDpi()/25.4), mapRenderer.height()/(mapRenderer.outputDpi()/25.4))
				dpi = c.printResolution()
				dpmm = dpi / 25.4
				# add a map to the composition
				x, y = 0, 0
				w, h = c.paperWidth(), c.paperHeight()
				width = int(round(dpmm * c.paperWidth()))
				height = int(round(dpmm * c.paperHeight()))
				composerMap = QgsComposerMap(c, x,y,w,h)
				c.addItem(composerMap)
				
				# create output image and initialize it
				image = QImage(QSize(width, height), QImage.Format_ARGB32)
				image.setDotsPerMeterX(dpmm * 1000)
				image.setDotsPerMeterY(dpmm * 1000)
				image.fill(0)
				
				#render the composition
				imagePainter = QPainter(image)
				sourceArea = QRectF(0, 0, c.paperWidth(), c.paperHeight())
				targetArea = QRectF(0, 0, width, height)
				c.render(imagePainter, targetArea, sourceArea)
				imagePainter.end()
				
				#Count empty lines at bottom and right border

				#Set initial looping parameters
				lastPixelY = image.height() - 1
				yPixelValue = '(255, 255, 255, 255)'
				previousYrgbasum = 1020 #refering to a value of 255 in all 4 bands
				Yrgbasum = 1020 #refering to a value of 255 in all 4 bands

				while lastPixelY > 0 and previousYrgbasum >= Yrgbasum and yPixelValue != '(0, 0, 0, 255)':
					previousYrgba = yPixelValue.strip("("")").split(",")
					previousYrgbasum = int(previousYrgba[0]) + int(previousYrgba[1]) + int(previousYrgba[2]) + int(previousYrgba[3])
					lastPixelY = lastPixelY - 1
					yPixelValue = str(QColor(image.pixel(int(image.width()/2),lastPixelY)).getRgb())
					Yrgba = yPixelValue.strip("("")").split(",")
					Yrgbasum = int(Yrgba[0]) + int(Yrgba[1]) + int(Yrgba[2]) + int(Yrgba[3])

				#Set initial looping parameters
				lastPixelX = image.width() - 1
				xPixelValue = '(255, 255, 255, 255)'
				previousXrgbasum = 1020 #refering to a value of 255 in all 4 bands
				Xrgbasum = 1020 #refering to a value of 255 in all 4 bands
					
				while lastPixelX > 0 and previousXrgbasum >= Xrgbasum and xPixelValue != '(0, 0, 0, 255)':
					previousXrgba = xPixelValue.strip("("")").split(",")
					previousXrgbasum = int(previousXrgba[0]) + int(previousXrgba[1]) + int(previousXrgba[2]) + int(previousXrgba[3])
					lastPixelX = lastPixelX - 1
					xPixelValue = str(QColor(image.pixel(lastPixelX,int(image.height()/2))).getRgb())
					Xrgba = xPixelValue.strip("("")").split(",")
					Xrgbasum = int(Xrgba[0]) + int(Xrgba[1]) + int(Xrgba[2]) + int(Xrgba[3])

				if xPixelValue != '(0, 0, 0, 255)': lastPixelX = lastPixelX + 1
				if yPixelValue != '(0, 0, 0, 255)': lastPixelY = lastPixelY + 1
				
				
				#Make a copy of the image containing only the map
				exportImage = image.copy(0, 0, lastPixelX, lastPixelY)	
				exportImage.save(input_file, "tif")
				
				#Export tfw-file
				xScale = (mapRect.xMaximum() - mapRect.xMinimum()) /  exportImage.width()
				yScale = (mapRect.yMaximum() - mapRect.yMinimum()) /  exportImage.height()
				
				yDif=(( mapRenderer.height() / mapRenderer.outputDpi()) * target_dpi)-exportImage.height()
				xDif=(( mapRenderer.width() / mapRenderer.outputDpi()) * target_dpi)-exportImage.width()
				
				f = open(out_put + ".tfw", 'w')
				f.write(str(xScale) + '\n')
				f.write(str(0) + '\n')
				f.write(str(0) + '\n')
				f.write('-' + str(yScale) + '\n')
				f.write(str(mapRect.xMinimum()) + '\n')
				f.write(str(mapRect.yMaximum()))
				f.close()
				
				if srs.epsg() != 4326:
					input_geofile=out_put + "wgs84.tif"
					params = [
					'-s_srs',
					'EPSG:' + str(srs.epsg()),
					'-t_srs',
					'EPSG:4326',
					input_file,
					os.path.join(out_folder, input_geofile),
					'-q'
					]
					program = r'gdalwarp'
					subprocess.call([program] + params)
					input_file = os.path.join(out_folder, input_geofile)			
								
				#Calculate tile size and number of tiles
				indataset=gdal.Open(input_file)
				x_extent=indataset.RasterXSize
				y_extent=indataset.RasterYSize
				#Close GDAL dataset
				indataset=''
				#
				n_cols_rest=x_extent % max_x_ext_general
				n_rows_rest=y_extent % max_y_ext_general
				#
				if n_cols_rest >= 1 : n_cols=( x_extent / max_x_ext_general ) + 1
				else : n_cols=( x_extent / max_x_ext_general )
				
				#
				if n_rows_rest >= 1 : n_rows=( y_extent / max_y_ext_general ) + 1
				else : n_rows=( y_extent / max_y_ext_general )
				
				#
				#Check if number of tiles is below Garmins limit of 100 tiles (across all custom maps)
				n_tiles=( n_rows * n_cols )
				if n_tiles >= 100 : QMessageBox(2, "Warning:", 'The number of tiles is likely to exceed Garmins limit of 100 tiles! Not all tiles will be displayed on your GPS unit. Consider reducing your map size (extend or zoom-factor).').exec_()
				
				#
				#Check if size of tiles is below Garmins limit of 1 megapixel (for each tile)
				max_pix=( 1024 * 1024 )
				n_pix=( max_x_ext_general * max_y_ext_general )
				#
				if n_pix > max_pix : QMessageBox(2, "Warning:", 'The number of pixels in a tile exceeds Garmins limit of 1 megapixel per tile! Images will not be displayed properly.').exec_(); sys.exit()
				kmz=zipfile.ZipFile(kmz_file, 'w')
				kml=open(os.path.join(out_folder, 'doc.kml'), 'w')
				#
				#Write kml header
				kml.write('<?xml version="1.0" encoding="UTF-8"?>\n')
				kml.write('<kml xmlns="http://www.opengis.net/kml/2.2">\n')
				kml.write('  <Document>\n')
				kml.write('    <name>' + tname + '</name>\n')
				#
				#Produce .jpg tiles using gdal_translate looping through the complete rows and columns (1024x1024 pixel)
				y_offset=0
				x_offset=0
				r=1
				c=1
				n_tiles=0
				#Loop through rows
				for r in range(1, n_rows + 1, 1):
					#Define maximum Y-extend of tiles
					if r == ( n_rows ): max_y_ext=n_rows_rest
					else: max_y_ext=max_y_ext_general
					
					#(Within row-loop) Loop through columns
					for c in range(1, n_cols + 1, 1):
						#Define maximum X-extend of tiles
						if (c == ( n_cols ) ): max_x_ext=n_cols_rest
						else: max_x_ext=max_x_ext_general
						#Define name for tile-jpg
						t_name=tname + '_%(r)d_%(c)d.jpg' %{"r":r, "c":c}
						#Set parameters for gdal_translate
						srcwin='-srcwin %(x_offset)d %(y_offset)d %(max_x_ext)d %(max_y_ext)d ' %{"x_offset":x_offset, "y_offset":y_offset, "max_x_ext":max_x_ext, "max_y_ext":max_y_ext}
						params = [
						'-of',
						'JPEG',
						'-co',
						'QUALITY=' + str(qual),
						'-srcwin',
						str(x_offset),
						str(y_offset),
						str(max_x_ext),
						str(max_y_ext),
						input_file,
						os.path.join(out_folder, t_name),
						'-q'
						]
						program = r'gdal_translate'
						#Execute gdal_translate using subprocess
						subprocess.call([program] + params)
						#Extract bounding box for tile using GDAL
						tdataset=gdal.Open(os.path.join(out_folder, t_name))
						n=tdataset.GetGeoTransform(can_return_null = True)[3]
						s=tdataset.GetGeoTransform(can_return_null = True)[3] + ( tdataset.GetGeoTransform(can_return_null = True)[5] * tdataset.RasterYSize )
						e=tdataset.GetGeoTransform(can_return_null = True)[0] + ( tdataset.GetGeoTransform(can_return_null = True)[1] * tdataset.RasterXSize )
						w=tdataset.GetGeoTransform(can_return_null = True)[0]
						#Close GDAL dataset
						tdataset=''
						#Add .jpg to .kmz-file and remove it together with its meta-data afterwards
						kmz.write(os.path.join(out_folder, t_name), t_name)
						os.remove(os.path.join(out_folder, t_name))
						os.remove(os.path.join(out_folder, t_name) + '.aux.xml')
						#
						#Write kml-tags for each tile (Name, DrawOrder, JPEG-Reference, GroundOverlay)
						kml.write('')    
						kml.write('    <GroundOverlay>\n')
						kml.write('        <name>' + tname + ' Tile ' + str(r) + '_' + str(c) + '</name>\n') #%{"r":r, "c":c}
						kml.write('        <drawOrder>' + str(draworder) + '</drawOrder>\n') #%{"draworder":draworder}
						kml.write('        <Icon>\n')
						kml.write('          <href>' + tname + '_' + str(r) + '_' + str(c) + '.jpg</href>\n') #%{"r":r, "c":c}
						kml.write('        </Icon>\n')
						kml.write('        <LatLonBox>\n')
						kml.write('          <north>' + str(n) + '</north>\n')
						kml.write('          <south>' + str(s) + '</south>\n')
						kml.write('          <east>' + str(e) + '</east>\n')
						kml.write('          <west>' + str(w) + '</west>\n')
						kml.write('        </LatLonBox>\n')
						kml.write('    </GroundOverlay>\n')
						
						#Calculate new X-offset
						x_offset=( x_offset + max_x_ext )
						n_tiles=( n_tiles + 1 )
					#Calculate new Y-offset
					y_offset=( y_offset + max_y_ext )
					#Reset X-offset
					x_offset=0
								
				#
				#Write kml footer
				kml.write('  </Document>\n')
				kml.write('</kml>\n')
				#Close kml file
				kml.close()
				
				#Remove temporary geo-tif file
				os.remove(out_put + ".tif")
				os.remove(out_put + ".tfw")
				
				#Remove reprojected temporary geo-tif file if necessary
				if srs.epsg() != 4326:
					os.remove(os.path.join(out_folder, input_geofile))
				
				#Add .kml to .kmz-file and remove it together with the rest of the temporary files
				kmz.write(os.path.join(out_folder, 'doc.kml'), 'doc.kml')
				os.remove(os.path.join(out_folder, 'doc.kml'))
				kmz.close()
				os.rmdir(out_folder)
				#Give success message
				success_msg = QMessageBox()
				success_msg.setWindowTitle("Done!");
				success_msg.setText("Produced " + str(n_tiles) + " tiles,\n" "with " +str(n_rows) + " rows\n" "and " + str(n_cols) + " colums.")
				success_msg.exec_()
