# -*- coding: utf-8 -*-
"""
/***************************************************************************
 FotovolCat
                                 A QGIS plugin
 Spatial analysis automation for solar power station sitting in Catalonia
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-08-20
        git sha              : $Format:%H$
        copyright            : (C) 2023 by Lluc Rehues
        email                : fotovolcat@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QPointF, Qt
from qgis.PyQt.QtGui import QIcon, QColor, QFont, QPolygonF
from functools import reduce
from random import randrange
from qgis.PyQt.QtWidgets import QAction, QFileDialog
from qgis.core import *
from qgis import processing

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .photovol_analysis_dialog import FotovolCatDialog
import os.path


class FotovolCat:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'FotovolCat_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&FotovolCAT')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('FotovolCat', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/photovol_analysis/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'FotovolCAT'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&FotovolCAT'),
                action)
            self.iface.removeToolBarIcon(action)

    def select_input_file(self):
        filename, _filter = QFileDialog.getOpenFileName(self.dlg, "Select input file", "", '*.shp')
        self.dlg.leInput.setText(filename)

    def select_output_directory(self):
        directory = QFileDialog.getExistingDirectory(self.dlg, "Select output directory", "/home/",
                                                     QFileDialog.ShowDirsOnly)
        self.dlg.leOutput.setText(directory)
        
    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = FotovolCatDialog()
            self.dlg.pbInput.clicked.connect(self.select_input_file)
            self.dlg.pbOutput.clicked.connect(self.select_output_directory)
        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:

            filename = self.dlg.leInput.text()

            pfv = QgsVectorLayer(filename, "Parque fotovoltaico", "ogr")

            pfv_buffer = processing.run("native:buffer",
                                        {'INPUT': pfv, 'DISTANCE': 2000,
                                         'SEGMENTS': 100, 'END_CAP_STYLE': 0,
                                         'JOIN_STYLE': 0, 'MITER_LIMIT': 2,
                                         'DISSOLVE': False, 'OUTPUT': 'memory:'})['OUTPUT']

            area_estudio = processing.run("native:polygonfromlayerextent",
                                          {'INPUT': pfv_buffer,
                                           'OUTPUT': 'memory:'})['OUTPUT']

            ####cargamos wms

            uri1 = "crs=EPSG:25831&dpiMode=7&format=image/png&layers=capacitat-agrologica&styles&url=https://geoserveis.icgc.cat/servei/catalunya/capacitat-agrologica/wms"
            agro = QgsRasterLayer(uri1, "Capacidad agrologica", "wms")

            uri2 = "crs=EPSG:25831&dpiMode=7&format=image/png&layers=HY.Network&styles&url=https://servicios.idee.es/wms-inspire/hidrografia"
            hidro = QgsRasterLayer(uri2, "Cursos fluviales", "wms")

            uri3 = "crs=EPSG:25831&dpiMode=7&format=image/png&layers=OMB2m&styles&url=https://geoserveis.icgc.cat/icgc_mdt2m/wms/service"
            sombras = QgsRasterLayer(uri3, "Sombras", "wms")

            uri4 = "crs=EPSG:25831&dpiMode=7&format=image/png&layers=MP20P5M_PA&styles&url=https://geoserveis.icgc.cat/icgc_mp20p5m/wms/service"
            pendiente = QgsRasterLayer(uri4, "Pendiente > 20%", "wms")

            uri5 = "crs=EPSG:25831&dpiMode=7&format=image/png&layers=orto25m&styles&url=https://geoserveis.icgc.cat/icc_mapesbase/wms/service"
            orto = QgsRasterLayer(uri5, "Ortofotomapa", "wms")

            uri6 = "crs=EPSG:25831&dpiMode=7&format=image/jpeg&layers=mtc50msg&styles&url=https://geoserveis.icgc.cat/icc_fonstopografic/wms/service"
            topo_gris = QgsRasterLayer(uri6, "Topográfico gris", "wms")

            rlayers = [topo_gris, orto, sombras, agro, hidro, pendiente]

            # cargamos wfs
            uri7 = "https://sig.gencat.cat/ows/XARXES_TRANSPORT/wfs?srsname=EPSG:25831&typename=XARXES_TRANSPORT:XT_GRAF_CATALEG_CARRETERES&version=1.1.0&request=GetFeature&service=WFS"
            carreteras = QgsVectorLayer(uri7, "Carreteras", "wfs")

            uri8 = "https://sig.gencat.cat/ows/XARXES_TRANSPORT/wfs?srsname=EPSG:25831&typename=XARXES_TRANSPORT:XT_GRAF_CATALEG_FERRO&version=1.1.0&request=GetFeature&service=WFS"
            tren = QgsVectorLayer(uri8, "Ferrocarril", "wfs")

            uri9 = "https://sig.gencat.cat/ows/ESPAIS_NATURALS/wfs?srsname=EPSG:25831&typename=ESPAIS_NATURALS:ESPAISNATURALS_XARNAT_2000&version=1.1.0&request=GetFeature&service=WFS"
            rn2000 = QgsVectorLayer(uri9, "Red Natura 2000", "wfs")

            uri10 = "https://sig.gencat.cat/ows/PLANEJAMENT/wfs?srsname=EPSG:25831&typename=PLANEJAMENT:PLANSTP_ESPAIS_OBERTS&version=1.1.0&request=GetFeature&service=WFS"
            plan_ter = QgsVectorLayer(uri10, "Planeamiento territorial", "wfs")

            uri11 = "https://sig.gencat.cat/ows/PAISATGE/wfs?srsname=EPSG:25831&typename=PAISATGE:PAISATGE_UNITATS&version=1.1.0&request=GetFeature&service=WFS"
            paisaje = QgsVectorLayer(uri11, "Unidades de paisaje", "wfs")

            uri12 = "pagingEnabled='true' preferCoordinatesForWfsT11='false' restrictToRequestBBOX='1' srsname='EPSG:25831' typename='PATRIMONI_CULTURAL:PATRIMONI_JACIMENT_02' url='http://sig.gencat.cat/ows/PATRIMONI_CULTURAL/wfs' version='auto'"
            patrimonio = QgsVectorLayer(uri12, "Patrimonio", "wfs")

            layers = [carreteras, tren, rn2000, plan_ter, paisaje, patrimonio]

            # Cortamos todos los wfs con nuestra area de estudio
            vlayers = []  # creamos lista vacía donde guardar las capas cortadas
            for layer in layers:
                i = processing.run("native:clip",
                                   {'INPUT': layer, 'OVERLAY': area_estudio,
                                    'OUTPUT': 'memory:'})['OUTPUT']
                i.setName(layer.name())
                vlayers.append(i)

            ########Procesamos capas que lo requieran

            # Buffer de 100 metros alrededor de yacimientos arqueológicos
            patrimonio_buffer = processing.run("native:buffer",
                                               {'INPUT': vlayers[5], 'DISTANCE': 100,
                                                'SEGMENTS': 100, 'END_CAP_STYLE': 0,
                                                'JOIN_STYLE': 0, 'MITER_LIMIT': 2,
                                                'DISSOLVE': True, 'OUTPUT': 'memory:'})['OUTPUT']
            patrimonio_buffer.setName("Protección patrimonio")
            vlayers.append(patrimonio_buffer)

            # Buffer de 50 metros de protección a las vías de tren
            tren_buffer = processing.run("native:buffer",
                                         {'INPUT': vlayers[1], 'DISTANCE': 50,
                                          'SEGMENTS': 100, 'END_CAP_STYLE': 0,
                                          'JOIN_STYLE': 0, 'MITER_LIMIT': 2,
                                          'DISSOLVE': True, 'OUTPUT': 'memory:'})['OUTPUT']
            tren_buffer.setName("Protección ferrocarril")
            vlayers.append(tren_buffer)

            # Buffers de diferentes magnitudes segun tipo carretera

            dist_dict = {'BasicaPrimaria': 20, 'BasicaSecund': 15, 'Comarcal': 15, 'Local': 10, 'Transeuropea': 30}
            qgis_expr = f'map_get(map{reduce(lambda x, y: x + y, dist_dict.items())}, "FUNCIONAL")'

            param = {'INPUT': vlayers[0], 'DISTANCE': QgsProperty.fromExpression(qgis_expr),
                     'SEGMENTS': 100, 'END_CAP_STYLE': 0,
                     'JOIN_STYLE': 0, 'MITER_LIMIT': 2,
                     'DISSOLVE': False,
                     'OUTPUT': 'memory:'}

            carreteras_buffer = processing.run("native:buffer", param)['OUTPUT']
            carreteras_buffer.setName("Protección carreteras")
            vlayers.append(carreteras_buffer)

            # reordenamos lista para que los buffers nos salgan por debajo de las capas principales
            vlayers_order = [8, 7, 6, 0, 1, 2, 3, 4, 5]
            vlayers[:] = [vlayers[i] for i in vlayers_order]

            ########Simbolizamos capas

            # Creamos simbologia para cada capa
            carreteras_symbol = QgsLineSymbol.createSimple(
                {'line_style': 'solid', 'line_width': '0.5', 'line_color': '0,0,255,255'})
            tren_symbol = QgsLineSymbol.createSimple(
                {'line_style': 'dash', 'line_width': '1.0', 'line_color': '88,88,88,255'})
            rn2000_symbol = QgsFillSymbol.createSimple(
                {'outline_style': 'solid', 'outline_width': '0.5', 'outline_color': '0,0,0,255',
                 'color': '38,205,110,170'})
            plan_ter_symbol = QgsFillSymbol.createSimple(
                {'outline_style': 'solid', 'outline_width': '0.5', 'outline_color': '0,0,0,255',
                 'color': '070,164,0,170'})
            paisaje_symbol = QgsFillSymbol.createSimple(
                {'outline_style': 'solid', 'outline_width': '0.5', 'outline_color': '0,0,0,255',
                 'color': '231,166,0,170'})
            patrimonio_symbol = QgsFillSymbol.createSimple(
                {'outline_style': 'solid', 'outline_width': '0.26', 'outline_color': '0,0,0,255',
                 'color': '143,50,55,255'})
            patrimonio_buffer_symbol = QgsFillSymbol.createSimple(
                {'outline_style': 'solid', 'outline_width': '0.26', 'outline_color': '0,0,0,255',
                 'color': '143,50,55,255', 'style': 'b_diagonal'})
            tren_buffer_symbol = QgsFillSymbol.createSimple(
                {'outline_style': 'solid', 'outline_width': '0.26', 'outline_color': '0,0,0,255',
                 'color': '164,0,123,170'})
            carreteras_buffer_symbol = QgsFillSymbol.createSimple(
                {'outline_style': 'solid', 'outline_width': '0.26', 'outline_color': '0,0,0,255',
                 'color': '0,234,230,170'})

            symbol_list = [carreteras_buffer_symbol, tren_buffer_symbol, patrimonio_buffer_symbol, carreteras_symbol,
                           tren_symbol, rn2000_symbol, plan_ter_symbol, paisaje_symbol, patrimonio_symbol]

            # Aplicamos simbologia a las capas
            for i in range(0, len(vlayers)):
                vlayers[i].renderer().setSymbol(symbol_list[i])
                vlayers[i].triggerRepaint()

            # Simbolizamos "Planeamiento Territorial" por categorias

            symbol1 = QgsFillSymbol.createSimple(
                {'outline_style': 'solid', 'outline_width': '0.5', 'outline_color': '0,0,0,255',
                 'color': '166,184,107,170'})
            symbol2 = QgsFillSymbol.createSimple(
                {'outline_style': 'solid', 'outline_width': '0.5', 'outline_color': '0,0,0,255',
                 'color': '206,212,144,170'})
            symbol3 = QgsFillSymbol.createSimple(
                {'outline_style': 'solid', 'outline_width': '0.5', 'outline_color': '0,0,0,255',
                 'color': '254,254,213,170'})

            cat1 = QgsRendererCategory('Sòl de protecció especial', symbol1, 'Protección especial')
            cat2 = QgsRendererCategory('Sòl de protecció territorial', symbol2, 'Protección territorial')
            cat3 = QgsRendererCategory('Sòl de protecció preventiva', symbol3, 'Protección preventiva')

            categories = [cat1, cat2, cat3]
            renderer_pt = QgsCategorizedSymbolRenderer("CATEGORIA", categories)

            vlayers[6].setRenderer(renderer_pt)
            vlayers[6].triggerRepaint()

            # Añadimos etiquetas para Unidades de Paisaje

            vlayers[7].setLabelsEnabled(True)
            vlayers[7].triggerRepaint()

            # Simbolizamos Unidad de Paisajes para que cada Unidad tenga un color diferente
            fni = vlayers[7].fields().indexFromName('NOM_UP')
            unique_values = vlayers[7].uniqueValues(fni)

            categories = []
            for unique_value in unique_values:
                psymbol = QgsSymbol.defaultSymbol(vlayers[7].geometryType())
                layer_style = {}
                layer_style['color'] = '%d, %d, %d, 170' % (randrange(0, 256), randrange(0, 256), randrange(0, 256))
                layer_style['outline'] = '#000000'
                symbol_layer = QgsSimpleFillSymbolLayer.create(layer_style)
                if symbol_layer is not None:
                    psymbol.changeSymbolLayer(0, symbol_layer)
                category = QgsRendererCategory(unique_value, psymbol, str(unique_value))
                categories.append(category)

            prenderer = QgsCategorizedSymbolRenderer('NOM_UP', categories)
            if prenderer is not None:
                vlayers[7].setRenderer(prenderer)

            vlayers[7].triggerRepaint()

            # Añadimos mapa base
            uri_topo = "crs=EPSG:25831&dpiMode=7&format=image/png&layers=mtc50m&styles&url=https://geoserveis.icgc.cat/icc_mapesbase/wms/service"
            base_topo = QgsRasterLayer(uri_topo, "Base Topografica", "wms")
            base_topo.setExtent(QgsRectangle(area_estudio.extent()))
            QgsProject.instance().addMapLayer(base_topo)

            # Cargamos todas las capas raster, y las desactivamos
            for rlayer in rlayers:
                rlayer.setExtent(QgsRectangle(area_estudio.extent()))
                QgsProject.instance().addMapLayer(rlayer)
                QgsProject.instance().layerTreeRoot().findLayer(rlayer).setItemVisibilityChecked(False)

            # Cargamos todas las capas vectoriales, y las desactivamos
            for vlayer in vlayers:
                QgsProject.instance().addMapLayer(vlayer)
                QgsProject.instance().layerTreeRoot().findLayer(vlayer).setItemVisibilityChecked(False)
            # Simbolizamos y cargamos capa parque fotovoltaico
            pfv_symbol = QgsFillSymbol.createSimple(
                {'outline_style': 'solid', 'outline_width': '1', 'outline_color': '255,0,0,255', 'color': '0,0,0,0'})
            pfv.renderer().setSymbol(pfv_symbol)
            pfv.triggerRepaint()
            QgsProject.instance().addMapLayer(pfv)

            def createLayout(layoutName, layer1, layer2 = None, layer3 = None, layer4 = None, legendMode = True, topoMode = True):
                selected_layers = [layer1, layer2, layer3, layer4]
                if topoMode == False:
                    QgsProject.instance().layerTreeRoot().findLayer(base_topo).setItemVisibilityChecked(False)
                else:
                    pass
                for s in range(0, len(selected_layers)):
                    if selected_layers[s] != None:
                        QgsProject.instance().layerTreeRoot().findLayer(selected_layers[s]).setItemVisibilityChecked(
                            True)
                project = QgsProject.instance()
                manager = project.layoutManager()

                # Borramos layouts existentes con el mismo nombre
                layouts_list = manager.printLayouts()
                for layout in layouts_list:
                    if layout.name() == layoutName:
                        manager.removeLayout(layout)

                # Configuramos layout
                layout = QgsPrintLayout(project)
                layout.initializeDefaults()

                layout.setName(layoutName)
                manager.addLayout(layout)

                # Creamos mapa
                map = QgsLayoutItemMap(layout)
                map.setRect(20, 20, 20, 20)

                # Extension del mapa
                ms = QgsMapSettings()

                ms.setLayers(selected_layers)  # elegimos las capas a dibujar

                layer_extent = area_estudio.extent()
                ms.setExtent(layer_extent)
                map.setExtent(layer_extent)
                map.setBackgroundColor(QColor(255, 255, 255, 0))
                map.setFrameEnabled(True)
                layout.addLayoutItem(map)

                # movemos y cambiamos medida mapa
                map.attemptMove(QgsLayoutPoint(25, 8, QgsUnitTypes.LayoutMillimeters))
                map.attemptResize(QgsLayoutSize(248, 178, QgsUnitTypes.LayoutMillimeters))

                # Añadimos marco info
                polygon = QPolygonF()
                polygon.append(QPointF(25, 186))
                polygon.append(QPointF(273, 186))
                polygon.append(QPointF(273, 202))
                polygon.append(QPointF(25, 202))
                polygonItem = QgsLayoutItemPolygon(polygon, layout)
                layout.addLayoutItem(polygonItem)

                props = {}
                props["style"] = "solid"
                props["color"] = "white"
                props["style_border"] = "solid"
                props["color_border"] = "black"
                props["width_border"] = "0.26"
                props["joinstyle"] = "miter"

                polygon_symbol = QgsFillSymbol.createSimple(props)
                polygonItem.setSymbol(polygon_symbol)

                # Añadimos leyenda
                legend = QgsLayoutItemLegend(layout)
                legend.setTitle("Leyenda")
                layerTree = QgsLayerTree()
                layerTree.addLayer(pfv)
                if legendMode == True:
                    for slayer in selected_layers:
                        layerTree.addLayer(slayer)
                legend.model().setRootGroup(layerTree)
                legend.setFrameEnabled(True)
                legend.setBackgroundColor(QColor(255, 255, 255, 180))
                layout.addLayoutItem(legend)
                legend.attemptMove(QgsLayoutPoint(32, 135, QgsUnitTypes.LayoutMillimeters))

                # Añadimos scalebar
                scalebar = QgsLayoutItemScaleBar(layout)
                scalebar.setStyle('Single Box')
                scalebar.setLinkedMap(map)
                scalebar.applyDefaultSize()
                scalebar.update()
                layout.addLayoutItem(scalebar)
                scalebar.attemptMove(QgsLayoutPoint(28, 187.5, QgsUnitTypes.LayoutMillimeters))

                # Añadimos título
                header = QgsLayoutItemLabel(layout)
                header.setText(layoutName)
                header.setFont(QFont("Arial", 16, QFont.Bold))
                header.setFixedSize(QgsLayoutSize(124.75, 13.2, QgsUnitTypes.LayoutMillimeters))
                header.adjustSizeToText()
                header.setHAlign(Qt.AlignCenter)
                layout.addLayoutItem(header)
                header.attemptMove(QgsLayoutPoint(90, 191.5, QgsUnitTypes.LayoutMillimeters))

                #Añadimos etiqueta fotovolCAT
                etiqueta = QgsLayoutItemLabel(layout)
                etiqueta.setText("Creado con FotovolCAT")
                etiqueta.setFont(QFont("Arial", 8))
                etiqueta.setFixedSize(QgsLayoutSize(70, 12, QgsUnitTypes.LayoutMillimeters))
                etiqueta.adjustSizeToText()
                layout.addLayoutItem(etiqueta)
                etiqueta.attemptMove(QgsLayoutPoint(245, 203.5, QgsUnitTypes.LayoutMillimeters))

                # Añadimos flecha norte
                north = QgsLayoutItemPicture(layout)
                north.setMode(QgsLayoutItemPicture.FormatSVG)
                north.setPicturePath(":/images/north_arrows/layout_default_north_arrow.svg")
                north.attemptMove(QgsLayoutPoint(255, 187.5, QgsUnitTypes.LayoutMillimeters))
                north.attemptResize(QgsLayoutSize(*[150, 150], QgsUnitTypes.LayoutPixels))
                layout.addLayoutItem(north)

                directory = self.dlg.leOutput.text()
                exporter = QgsLayoutExporter(layout)
                expSettings = exporter.PdfExportSettings()
                expSettings.dpi = 150
                exporter.exportToPdf(f"{directory}/{layoutName}.pdf", expSettings)

                if topoMode == False:
                    QgsProject.instance().layerTreeRoot().findLayer(base_topo).setItemVisibilityChecked(True)
                else:
                    pass

                # Al acabar layout volvemos a desactivar todas las capas
                for s in range(0, len(selected_layers)):
                    if selected_layers[s] != None:
                        QgsProject.instance().layerTreeRoot().findLayer(selected_layers[s]).setItemVisibilityChecked(
                            False)

            #Exportamos mapas si la casilla está marcada
            if self.dlg.cbExport.isChecked():
                # Orto:
                createLayout("Mapa ortofotográfico", rlayers[1], legendMode=False, topoMode=False)

                # Agro:
                createLayout("Capacidad agrológica", rlayers[3], legendMode=False)

                # Hidro:
                createLayout("Red hidrográfica", rlayers[4], rlayers[0], legendMode=False, topoMode=False)

                # Infraestructuras:
                createLayout("Infraestructuras", vlayers[0], vlayers[1], vlayers[3], vlayers[4])

                # RN2000:
                createLayout("Red Natura 2000", vlayers[5])

                # Plan. territorial:
                createLayout("Planeamiento territorial", vlayers[6])

                # Paisaje:
                createLayout("Unidades de paisaje", vlayers[7])

                # Arqueo:
                createLayout("Yacimientos arqueológicos", vlayers[8], vlayers[2])

                # Topo:
                createLayout("Mapa topográfico", base_topo, legendMode=False)

                # Pendiente:
                createLayout("Pendiente", rlayers[2], rlayers[5], topoMode=False)

            else:
                pass