from qgis.PyQt.QtCore import Qt, pyqtSignal
from qgis.PyQt.QtGui import QColor
from qgis.PyQt.QtWidgets import QColorDialog, QPushButton, QWidget


class IcsColorDialog(QColorDialog):
    """A QColorWidget with colors from the international chronostratigraphic chart
    defined by the International Comission on Stratigraphy (ICS).

    The colors were extracted from https://stratigraphy.org/timescale/
    """

    def __init__(
        self,
        initial: QColor = None,
        alpha: bool = False,
        parent: QWidget = None,
    ):
        """
        Args:
            initial (QColor, optional): Color initially selected when the dialog pops-up. Defaults to None (black).
            alpha (bool, optional): If True, the dialog displays the alpha channel so the user can edit it.
                Otherwise, the alpha channel is hidden and can only be edited. Defaults to False.
            parent (QWidget, optional): The parent of this dialog. Defaults to None.
        """

        super().__init__(initial, parent)
        if initial is None:
            initial = QColor("#ffffff")  # Black
        self.setOption(QColorDialog.ColorDialogOption.ShowAlphaChannel, alpha)
        # Column 1: yellow
        self.setStandardColor(0, QColor(254, 235, 210))  # Holocene
        self.setStandardColor(1, QColor(255, 239, 175))  # Pleistocene
        self.setStandardColor(2, QColor(249, 249, 127))  # Quaternary
        self.setStandardColor(3, QColor(255, 240, 120))  # Langhian
        self.setStandardColor(4, QColor(255, 235, 61))  # Miocene
        self.setStandardColor(5, QColor(255, 219, 75))  # Neogene
        # Column 2: salmon
        self.setStandardColor(6, QColor(255, 198, 148))  # Oligocene
        self.setStandardColor(7, QColor(255, 188, 135))  # Eocene
        self.setStandardColor(8, QColor(255, 167, 111))  # Paleogene
        self.setStandardColor(9, QColor(255, 139, 117))  # Guadalupian
        self.setStandardColor(10, QColor(248, 116, 95))  # Cisuralian
        self.setStandardColor(11, QColor(249, 95, 68))  # Permian
        # Column 3: blue
        self.setStandardColor(12, QColor(212, 243, 253))  # Tithonian
        self.setStandardColor(13, QColor(183, 236, 252))  # Oxfordien
        self.setStandardColor(14, QColor(156, 227, 241))  # Bajocian
        self.setStandardColor(15, QColor(113, 217, 239))  # Middle Jurassic
        self.setStandardColor(16, QColor(141, 216, 244))  # Toarcian
        self.setStandardColor(17, QColor(40, 193, 240))  # Lower Jurassic
        # Column 4: violet
        self.setStandardColor(18, QColor(227, 200, 226))  # Rhaetian
        self.setStandardColor(19, QColor(200, 175, 215))  # Carnian
        self.setStandardColor(20, QColor(201, 156, 205))  # Ladinian
        self.setStandardColor(21, QColor(176, 132, 194))  # Middle Triassic
        self.setStandardColor(22, QColor(153, 91, 174))  # Lower Triassic
        self.setStandardColor(23, QColor(130, 79, 168))  # Triassic
        # Column 5
        self.setStandardColor(24, QColor(244, 237, 211))  # Famenian
        self.setStandardColor(25, QColor(248, 223, 160))  # Givetian
        self.setStandardColor(26, QColor(249, 213, 146))  # Eifelian
        self.setStandardColor(27, QColor(203, 209, 147))  # Emsian
        self.setStandardColor(28, QColor(236, 190, 122))  # Lochkovian
        self.setStandardColor(29, QColor(213, 156, 88))  # Devonian
        # Column 6: System/Period (1/2)
        self.setStandardColor(30, QColor(249, 249, 127))  # Quaternary
        self.setStandardColor(31, QColor(255, 219, 75))  # Neogene
        self.setStandardColor(32, QColor(255, 167, 111))  # Paleogene
        self.setStandardColor(33, QColor(130, 204, 115))  # Cretaceous
        self.setStandardColor(34, QColor(26, 196, 236))  # Jurassic
        self.setStandardColor(35, QColor(130, 79, 168))  # Triassic
        # Column 7: System/Period (2/2)
        self.setStandardColor(36, QColor(249, 95, 68))  # Permian
        self.setStandardColor(37, QColor(95, 182, 181))  # Carboniferous
        self.setStandardColor(38, QColor(213, 156, 88))  # Devonian
        self.setStandardColor(39, QColor(175, 227, 205))  # Silurian
        self.setStandardColor(40, QColor(0, 175, 144))  # Ordovician
        self.setStandardColor(41, QColor(131, 175, 119))  # Cambrian
        # Column 8: gray
        self.setStandardColor(42, QColor(255, 255, 255))  # Black 0 - 100 White
        self.setStandardColor(43, QColor(204, 204, 204))  # Black 20 - 80 White
        self.setStandardColor(44, QColor(153, 153, 153))  # Black 40 - 60 White
        self.setStandardColor(45, QColor(102, 102, 102))  # Black 60 - 40 White
        self.setStandardColor(46, QColor(51, 51, 51))  # Black 80 - 20 White
        self.setStandardColor(47, QColor(0, 0, 0))  # Black 100 - 0 White


class IcsColorButton(QPushButton):
    """A QPushButton that opens a `IcsColorDialog` when clicked and displays the
    selected color

    - Left-click: opens the color selector, while
    - Right-click: resets the color to its default value: None (no-color) by default
    """

    colorChanged = pyqtSignal(object)

    def __init__(
        self,
        parent: QWidget = None,
        color: QColor = None,
        alpha: bool = False,
    ):
        """
        Args:
            initial (QColor, optional): The default color of this selector. Defaults to None.
            alpha (bool, optional): alpha (bool, optional): If True, the dialog displays the alpha channel so the user can edit it.
                Otherwise, the alpha channel is hidden and can only be edited. Defaults to False.
            parent (QWidget, optional): The parent of this dialog. Defaults to None.
        """
        super().__init__(parent)

        self._color = None
        self._default = QColor(color) if color else None

        self.setColor(color)
        self._alpha = alpha
        self.clicked.connect(self.openColorDialog)

    def color(self):
        return self._color

    def setColor(self, value):
        value = QColor(value) if value else None
        if value == self.color():
            return

        self._color = value
        if value:  # avoid
            self.setStyleSheet(f"background-color: {value.name()[:7]};")
        else:
            self.setStyleSheet("")

        current = _qcolor_to_rbga_string(self._color)
        default = _qcolor_to_rbga_string(self._default)
        self.setToolTip(f"current: {current}; default: {default}")
        self.colorChanged.emit(value)

    def openColorDialog(self):
        """Opens a dialog to select a new color"""

        dlg = IcsColorDialog(initial=self._color, alpha=self._alpha)
        if dlg.exec():
            self.setColor(dlg.currentColor())

    def mouseReleaseEvent(self, e):
        if e.button() == Qt.MouseButton.RightButton:
            self.setColor(self._default)

        return super().mouseReleaseEvent(e)


def _qcolor_to_rbga_string(color: QColor):
    if color is None:
        return "None"
    rgb = f"{color.red()}, {color.green()}, {color.blue()}"
    a = color.alpha()
    if a == 255:  # No transparency
        return f"RGB({rgb})"
    return f"RGBA({rgb}, {a})"
