# import rifter
# from forgeo.gmlib.architecture import from_GeoModeller, make_evaluator
# from .algorithm import Algorithm
from pathlib import Path

import numpy as np
from qgis.core import (
    QgsCoordinateTransform,
    QgsProcessingException,
    QgsProcessingParameterExtent,
    QgsProcessingParameterNumber,
)

from .featureextracter import FeatureExtracter


class ZSlicer(
    FeatureExtracter,
    name=Path(__file__).stem,
    display_name="draw model on horizontal slice",
):
    ZSLICE = "ZSLICE"
    EXTENT = "EXTENT"
    NX = "NX"
    NY = "NY"

    def initAlgorithm(self, config=None):  # noqa: ARG002
        self.add_geomodel_parameter()
        self.addParameter(
            QgsProcessingParameterExtent(self.EXTENT, self.tr("Extent"), optional=False)
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                self.ZSLICE,
                self.tr("draw model on horizontal plane"),
                type=QgsProcessingParameterNumber.Double,
            )
        )

        def add_int_parameter(name, description):
            self.addParameter(
                QgsProcessingParameterNumber(
                    name,
                    self.tr(description),
                    defaultValue=10,
                    type=QgsProcessingParameterNumber.Integer,
                )
            )

        add_int_parameter(self.NX, "number of cells along (Ox)")
        add_int_parameter(self.NY, "number of cells along (Oy)")
        self.add_parameter_sinks()

    def processAlgorithm(self, parameters, context, feedback):
        input_layer = self.model_layer(parameters)
        zslice = self.parameterAsDouble(parameters, self.ZSLICE, context)
        extent = self.parameterAsExtent(parameters, self.EXTENT, context)
        extent_crs = self.parameterAsExtentCrs(parameters, self.EXTENT, context)
        # FIXME: should we test crs is the same to skip transform?
        # FIXME: cast extent to model crs
        ct = QgsCoordinateTransform(
            extent_crs, input_layer.crs(), context.transformContext()
        )
        extent = ct.transform(extent)
        nx, ny = (
            self.parameterAsInt(parameters, name, context)
            for name in (self.NX, self.NY)
        )
        if nx <= 0 or ny <= 0:
            msg = "resolution (nx, ny) must be strictly positive"
            raise QgsProcessingException(msg)
        dx = (extent.xMaximum() - extent.xMinimum()) / nx
        dy = (extent.yMaximum() - extent.yMinimum()) / ny
        x = np.linspace(
            extent.xMinimum() + 0.5 * dx, extent.xMaximum() - 0.5 * dx, nx, dtype="d"
        )
        y = np.linspace(
            extent.yMinimum() + 0.5 * dy, extent.yMaximum() - 0.5 * dy, ny, dtype="d"
        )
        pts = np.empty((nx * ny, 3), dtype="d")
        pts[:, :2] = np.column_stack(
            [b.ravel() for b in np.meshgrid(x, y[::-1], indexing="xy")]
        )
        pts[:, 2] = zslice
        cells = np.array(
            [(0, 1, nx + 1), (0, nx + 1, nx)], dtype=int
        )  # single triangle
        cells = np.vstack([cells + k for k in range(nx - 1)])  # strip of triangles
        cells = np.vstack(
            [cells + nx * k for k in range(ny - 1)]
        )  # matrix of triangles

        return self.extract_features_once(
            mesh=(pts, cells),
            parameters=parameters,
            context=context,
            feedback=feedback,
            support_name=f" on z={zslice}",
        )
