import contextlib

from forgeo.interpolation import BSPTreeBuilder
from forgeo.rigs.gmconverter import convert as gmconverter
from qgis.core import (
    QgsPluginLayer,
    QgsProcessingAlgorithm,
    QgsProcessingParameterMapLayer,
    QgsProject,
)
from qgis.PyQt.QtCore import QCoreApplication

from ..layers import FaultNetworkLayer, GmLayer, ModelLayer


class GmProcessingParameterLayer(QgsProcessingParameterMapLayer):
    """ "Helper class to use a GmLayer as Processing parameter"""

    def checkValueIsAcceptable(self, obj, context=None):  # noqa: ARG002
        if isinstance(obj, str):
            with contextlib.suppress(KeyError):
                obj = QgsProject.instance().layerStore().mapLayers()[obj]
        for cls in [GmLayer, ModelLayer, FaultNetworkLayer]:
            # layer type can be return as baseclass QgsPluginLayer...
            if isinstance(obj, cls):
                return True
            if (
                isinstance(obj, QgsPluginLayer)
                and obj.pluginLayerType() == cls.LAYER_TYPE
            ):
                return True
        return False


class GmProcessingAlgorithm(QgsProcessingAlgorithm):
    INPUT = "INPUT"

    def __init_subclass__(cls, /, name, display_name, **kwargs):
        # we must set the identification info before initializing base class
        # (it will used through name and displayName)
        cls.process_name = name
        cls.process_display_name = display_name
        super().__init_subclass__(**kwargs)

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return "Discretization"

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "discretization"

    def tr(self, string):
        """
        Returns a translatable string with the self.tr() function.
        """
        return QCoreApplication.translate("Processing", string)

    def createInstance(self):
        return self.__class__()

    # FIXME: should they be class methods?
    def name(self):
        return self.process_name

    # FIXME: should they be class methods?
    def displayName(self):
        return self.tr(self.process_display_name)

    def shortHelpString(self):
        return self.tr(self.__doc__)

    # We add the input geomodel source. It can have only GmLayer type.
    def add_geomodel_parameter(self):
        self.addParameter(
            GmProcessingParameterLayer(
                name=self.INPUT, description=self.tr("GeoModel layer"), optional=False
            )
        )

    def model_layer(self, parameters):
        # FIXME: This is a hack to avoid self.parameterAsLayer that fails
        return QgsProject.instance().layerStore().mapLayers()[parameters[self.INPUT]]

    def extraction_parameters(self, parameters, **kwargs):
        layer = self.model_layer(parameters)
        assert isinstance(layer, (GmLayer, ModelLayer, FaultNetworkLayer))
        if isinstance(layer, GmLayer):
            return gmconverter(layer.model, **kwargs)
        if isinstance(layer, ModelLayer):
            if (fault_layer_id := layer.faultnetlayer_id) is not None:
                fault_layer = QgsProject.instance().mapLayer(fault_layer_id)
                fault_network = fault_layer.faultnet
            else:
                fault_network = None
            params = BSPTreeBuilder.from_model(layer.model, fault_network)
            assert "is_fault" in params
            return params
        assert isinstance(layer, FaultNetworkLayer)
        params = BSPTreeBuilder.from_fault_network(layer.faultnet)
        assert "is_fault" in params
        return params
