from pathlib import Path

from forgeo.core import ModellingUnit
from qgis.core import (
    QgsMapLayerRenderer,
    QgsPluginLayer,
    QgsPluginLayerType,
    QgsProject,
)
from qgis.PyQt.QtWidgets import QMessageBox
from qgis.utils import iface

import forgeo.io.xml as fxml

from ..utils import get_forgeo_data_dir, qicon
from .utils import _repr_plugin_layer, set_unique_name_for_layer


class PileLayerRenderer(QgsMapLayerRenderer):
    def __init__(self, layerId, rendererContext):
        super().__init__(layerId, rendererContext)

    def render(self):
        return True


class PileLayer(QgsPluginLayer):
    """Custom `qgis.core.QgsMapLayer` to display/interact with geological pile

    Args:
        source: Pile files (`*.xml`).
    """

    LAYER_TYPE = "PileLayer"

    def __init__(self, pile=None):
        super().__init__(PileLayer.LAYER_TYPE)
        self.setValid(True)
        # Set pile layer CRS: not used (at least up to now), but avoids QGIS
        # prompting when opening the project to ask for setting the input CRS
        self.setCrs(QgsProject.instance().crs())
        self.nameChanged.connect(self._on_rename)
        self.pile = pile
        if pile is not None:
            self.setName(pile.name)

    __repr__ = _repr_plugin_layer

    def icon(self):
        return qicon("pile_icon")

    def createMapRenderer(self, rendererContext):
        # TODO: create an default empty Renderer (does it exist?)
        return PileLayerRenderer(self.id(), rendererContext)

    def setTransformContext(self, ct):
        pass

    def readXml(self, node, *_):
        """Called by readLayerXML(), used by children to read state specific to them from project files."""
        super().readXml(node, *_)
        # Load pile from source XML file
        root_dir = Path(QgsProject.instance().absolutePath())
        filepath = node.toElement().attribute("source")
        if not (root_dir / filepath).exists():
            # FIXME To remove "soon", allows reopening old projects
            root_dir = root_dir.parent
            assert (root_dir / filepath).exists()
        self.pile = fxml.load(root_dir / filepath)
        return True

    def writeXml(self, node, *_):
        """Called by writeLayerXML(), used by children to write state specific to them to project files."""
        super().writeXml(node, *_)
        element = node.toElement()
        element.setAttribute("type", "plugin")
        element.setAttribute("name", self.LAYER_TYPE)
        # Dump the pile in XML file
        filepath = get_forgeo_data_dir() / (self.pile.name + ".xml")
        fxml.dump(self.pile, filepath)
        # Set the relative pile XML path as source for the pile layer
        root_dir = Path(QgsProject.instance().absolutePath())
        filepath = filepath.relative_to(root_dir).as_posix()
        element.setAttribute("source", "./" + str(filepath))
        return True

    @classmethod
    def new(cls, name):
        if QgsProject.instance().mapLayersByName(name):
            QMessageBox.warning(
                iface.mainWindow(), "Name already used", "Layer not created"
            )
        else:
            return cls(ModellingUnit(name))
        return None

    @classmethod
    def clone(cls, layer):
        pile = None
        if layer.pile is not None:
            pile = fxml.deep_copy(layer.pile)
        return cls(pile)

    def _on_rename(self):
        name = set_unique_name_for_layer(self)
        if self.pile is not None:
            self.pile.name = name


class PileLayerType(QgsPluginLayerType):
    def __init__(self):
        super().__init__(PileLayer.LAYER_TYPE)

    def createLayer(self):
        return PileLayer()

    def showLayerProperties(self, layer):
        from ..widgets.pile_widget import PileEditionDialog  # noqa: PLC0415

        PileEditionDialog.edit(layer)
        return True
