import numpy as np
from qgis.core import (
    QgsAbstractProfileGenerator,
    QgsAbstractProfileResults,
    QgsAbstractProfileSource,
    QgsDoubleRange,
    QgsFeedback,
    QgsMapLayerElevationProperties,
    QgsProfileRequest,
)

from .utils import ProfileCurve, layer_to_rigs_parameters, renderRigsResults


class FaultNetworkLayerElevationProperties(QgsMapLayerElevationProperties):
    def __init__(self):
        super().__init__(None)

    def hasElevation(self):
        return True

    def isVisibleInZRange(zrange, layer):  # noqa: ARG002
        return True

    def showByDefaultInElevationProfilePlots(self):
        return False

    def writeXml(self, parent, document, context):  # noqa: ARG002
        return document.createElement("FaultNetworkLayerElevation")

    def readXml(self, element, context):  # noqa: ARG002
        return True


class FaultNetworkLayerProfileSource(QgsAbstractProfileSource):
    def __init__(self, layer, cache_profile: bool):
        super().__init__()
        self._layer = layer
        self._cache_profile = cache_profile

    def createProfileGenerator(self, request: QgsProfileRequest):
        # We do not need to keep a reference because the function is tagged as
        # SIP_FACTOR on the C++ side
        return FaultNetworkLayerProfileGenerator(
            self._layer, self._cache_profile, request
        )


class FaultNetworkLayerProfileGenerator(QgsAbstractProfileGenerator):
    def __init__(self, layer, cache_profile, request):
        super().__init__()
        self._layer = layer
        self._request = request
        self._results = None  # ModelLayerProfileResults
        self._feedback = QgsFeedback()
        # Used to cache evaluation results if asked.
        # Cached by the profile generator, as a new ModelLayerProfileResults is
        # stored in self._results at each call of generateProfile()
        self._cached_results = {} if cache_profile else None

    def feedback(self) -> QgsFeedback:
        return self._feedback

    def sourceId(self):
        return self._layer.id()

    def generateProfile(self, context):  # noqa: ARG002
        # Pre-compute rigs parameters to avoid doing it on-the-fly each time the
        # elevation profile updates.
        params = layer_to_rigs_parameters(self._layer, self._request)
        params["fault_network"] = self._layer.faultnet
        # TODO Handle CRS?
        curve = np.array([(P.x(), P.y()) for P in self._request.profileCurve()])
        self._results = FaultNetworkLayerProfileResults(
            curve, params, self._cached_results
        )
        return not self._feedback.isCanceled()

    def takeResults(self):
        return self._results


class FaultNetworkLayerProfileResults(QgsAbstractProfileResults):
    def __init__(self, profile_curve, params, cached_results):
        super().__init__()
        self._profile_curve = ProfileCurve(profile_curve)
        self._fault_network = params.pop("fault_network")
        self._params = params
        self._cached_results = cached_results

    def type(self) -> str:
        return "FaultNetworkLayerProfileResults"

    def asGeometries(self):
        pass

    def asFeatures(self, type, feedback):
        pass

    def renderResults(self, context):
        self._cached_results = renderRigsResults(
            self._profile_curve, self._params, context, self._cached_results
        )

    def zRange(self):
        bbox = self._fault_network.items_bounding_box()
        bbox.dilate(1.5)
        return QgsDoubleRange(bbox.zmin, bbox.zmax)
