from collections.abc import Iterable

from qgis.core import Qgis, QgsMapLayer, QgsPluginLayer
from qgis.gui import QgsMapLayerComboBox
from qgis.PyQt.QtCore import Qt, QVariant
from qgis.PyQt.QtWidgets import (
    QCheckBox,
    QComboBox,
    QDialog,
    QDialogButtonBox,
    QHBoxLayout,
    QLabel,
    QLineEdit,
    QVBoxLayout,
    QWidget,
)
from qgis.utils import iface

from ..utils import cleargridlayoutcell, data_icon_widget, input_data_type


def popup_new_item_collection(title, name):
    # Used for piles and fault networks
    dialog = QDialog(parent=iface.mainWindow())
    dialog.setWindowTitle(title)
    # New name
    name_edit = QLineEdit(name)
    name_hlayout = QHBoxLayout()
    name_hlayout.addWidget(QLabel("Name"))
    name_hlayout.addWidget(name_edit)
    # Option "Create from a layer"
    from_layer_checkbox = QCheckBox(dialog.tr("Create from layer"))
    from_layer_checkbox.stateChanged.connect(
        lambda checkstate: from_layer_widget.setVisible(
            checkstate == Qt.CheckState.Checked
        )
    )
    layer_cbox = qgsMapLayerComboBox(
        Qgis.LayerFilter.VectorLayer, emptyLayerMessage="No layer"
    )
    fields_cbox = QComboBox()

    def update_fields_cbox():
        fields_cbox.clear()
        if (layer := layer_cbox.currentLayer()) is not None:
            fields_cbox.addItems(
                [f.name() for f in layer.fields() if f.type() == QVariant.String]
            )

    update_fields_cbox()  # Call to force the first initialization
    layer_cbox.layerChanged.connect(update_fields_cbox)

    layer_layout = QHBoxLayout()
    layer_layout.addWidget(QLabel(dialog.tr("Layer")))
    layer_layout.addWidget(layer_cbox)
    field_layout = QHBoxLayout()
    field_layout.addWidget(QLabel(dialog.tr("Use field")))
    field_layout.addWidget(fields_cbox)
    from_layer_layout = QVBoxLayout()
    from_layer_layout.addLayout(layer_layout)
    from_layer_layout.addLayout(field_layout)
    from_layer_widget = QWidget()
    from_layer_widget.setLayout(from_layer_layout)
    from_layer_widget.setVisible(False)

    # Buttons
    buttons = QDialogButtonBox(
        QDialogButtonBox.StandardButton.Cancel | QDialogButtonBox.StandardButton.Ok
    )
    buttons.accepted.connect(dialog.accept)
    buttons.rejected.connect(dialog.reject)

    # Layout
    layout = QVBoxLayout()
    layout.addLayout(name_hlayout)
    layout.addWidget(from_layer_checkbox)
    layout.addWidget(from_layer_widget)
    layout.addWidget(buttons)
    dialog.setLayout(layout)

    # Close dialog
    new_name = layer = field = None
    if dialog.exec() == QDialog.DialogCode.Accepted:
        new_name = name_edit.text()
        if from_layer_checkbox.checkState() == Qt.CheckState.Checked:
            layer = layer_cbox.currentLayer()
            field = fields_cbox.currentText()
    return new_name, layer, field


def surface_symbols(color, is_erosion):
    """Create labels for symbols of erosions or contacts"""
    symbols = "~~" if is_erosion else "----"
    symbol_left = QLabel(symbols * 10)
    symbol_left.setAlignment(Qt.AlignmentFlag.AlignLeft | Qt.AlignmentFlag.AlignVCenter)
    symbol_left.setStyleSheet(f"color: {color.name()};")
    symbol_right = QLabel(symbols * 10)
    symbol_right.setAlignment(
        Qt.AlignmentFlag.AlignRight | Qt.AlignmentFlag.AlignVCenter
    )
    symbol_right.setStyleSheet(f"color: {color.name()};")
    return symbol_left, symbol_right


def display_data_icon(item, grid_layout, row_index, column_index):
    cleargridlayoutcell(grid_layout, row_index, column_index)
    if item.item_data is None:
        return
    nb_observations = item.item_data.nb_observation_data()
    nb_orientations = item.item_data.nb_orientation_data()
    widget = data_icon_widget(input_data_type(item), nb_observations, nb_orientations)
    grid_layout.addWidget(widget, row_index, column_index, alignment=Qt.AlignCenter)


def display_data_icon_itemgroup(items, grid_layout, row_index, column_index, row_span):
    nb_observations, nb_orientations = 0, 0
    for item in items:
        if item.item_data is not None:
            nb_observations += item.item_data.nb_observation_data()
            nb_orientations += item.item_data.nb_orientation_data()
    widget = data_icon_widget("vectorData", nb_observations, nb_orientations)
    cleargridlayoutcell(grid_layout, row_index, column_index)
    grid_layout.addWidget(
        widget, row_index, column_index, row_span, 1, alignment=Qt.AlignCenter
    )


def qgsMapLayerComboBox(
    layerFilter: Qgis.LayerFilter,
    defaultLayer: QgsPluginLayer | None = None,
    emptyLayerMessage: str | None = None,
    exceptedLayerList: Iterable[QgsMapLayer] | None = None,
    parent=None,
):
    """Helper function: returns a fully parameterized QgsMapLayerComboBox in a
    single instruction

    Tips:
    - the layer selected in the combobox can be accessed via the `currentLayer` method
    - connect to the `layerChanged` signal to be notified whenever the selection changes
    - full documentation: https://qgis.org/pyqgis/latest/gui/QgsMapLayerComboBox.html

    Parameters
    ----------
    layerFilter: Qgis.LayerFilter
        The type of layer this combobox should display (e.g., Qgis.VectorLayer)
    defaultLayer: QgsMapLayer (optional)
        The layer to select by default when creating the combobox.
        Its type  must match `layerFilter`.
    emptyLayerMessage: str (optional)
        If None, then empty selection is not allowed. Otherwise, empty selection
        is allowed, and the displyed message will be `emptyLayerMessage`
    parent: QWidget
    """
    cbox = QgsMapLayerComboBox(parent=parent)
    # Keep only `PluginLayerType` layers
    cbox.setFilters(layerFilter)
    if exceptedLayerList is not None:
        cbox.setExceptedLayerList(exceptedLayerList)
    if defaultLayer is not None:
        available_layers = [cbox.layer(idx) for idx in range(cbox.count())]
        assert defaultLayer in available_layers
        cbox.setLayer(defaultLayer)
    if emptyLayerMessage is not None:
        cbox.setAllowEmptyLayer(True, cbox.tr(emptyLayerMessage))
    return cbox


class QgsPluginLayerComboBox(QgsMapLayerComboBox):
    """A usual QgsMapLayerComboBox that allows to select a specific QgsPluginLayer
    (sub)type

    Tips:
    - the layer selected in the combobox can be accessed via the `currentLayer` method
    - connect to the `layerChanged` signal to be notified whenever the selection changes
    - full documentation: https://qgis.org/pyqgis/latest/gui/QgsMapLayerComboBox.html
    """

    def __init__(
        self,
        PluginLayerType: type,
        defaultLayer: QgsPluginLayer | None = None,
        emptyLayerMessage: str | None = None,
        parent=None,
    ):
        """
        Parameters
        ----------
        PluginLayerType: type
            A type inheriting from QgsPluginLayer
        defaultLayer: PluginLayerType (optional)
            The layer to select by default when creating the combobox.
            Its type  must match `PluginLayerType`.
        emptyLayerMessage: str (optional)
            If None, then empty selection is not allowed. Otherwise, empty selection
            is allowed, and the displyed message will be `emptyLayerMessage`
        parent: QWidget
        """
        super().__init__(parent=parent)
        assert issubclass(PluginLayerType, QgsPluginLayer)
        # Keep only `PluginLayerType` layers
        self.setFilters(Qgis.LayerFilter.PluginLayer)
        excluded_layers = [
            layer
            for idx in range(self.count())
            if not isinstance(layer := self.layer(idx), PluginLayerType)
        ]
        self.setExceptedLayerList(excluded_layers)
        if defaultLayer is not None:
            assert isinstance(defaultLayer, PluginLayerType)
            self.setLayer(defaultLayer)
        if emptyLayerMessage is not None:
            self.setAllowEmptyLayer(True, self.tr(emptyLayerMessage))
