from pathlib import Path

from forgeo.core import Erosion, ModellingUnit
from qgis.core import QgsProject
from qgis.PyQt.QtCore import QByteArray, QMimeData, QSize, Qt, QVariant
from qgis.PyQt.QtGui import QColor, QDrag, QFont
from qgis.PyQt.QtWidgets import (
    QAction,
    QDialog,
    QDialogButtonBox,
    QFileDialog,
    QGridLayout,
    QHBoxLayout,
    QLabel,
    QLineEdit,
    QMenuBar,
    QMessageBox,
    QPushButton,
    QScrollArea,
    QVBoxLayout,
    QWidget,
)
from qgis.utils import iface

import forgeo.io.xml as fxml

from ..layers import PileLayer
from ..utils import (
    DEFAULT_PILE_NAME,
    get_forgeo_data_dir,
    popup_save_changes,
    save_as_png,
    save_as_xml,
)
from .color_picker import IcsColorDialog as QICSColorDialog
from .model_widget import ModelEditionDialog
from .utils import QgsPluginLayerComboBox, popup_new_item_collection, surface_symbols


class PileEditionDialog(QDialog):
    def __init__(self, pile_layer=None, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Pile edition")
        self.pile = None
        self.pile_layer = None
        self.nb_of_elements = 1

        # Pile name
        self.label_pilename = QLabel("No pile selected")
        self.label_pilename.setAlignment(Qt.AlignmentFlag.AlignCenter)

        # Buttons to add elements
        add_modelling_unit_button = QPushButton("Add modelling unit")
        add_modelling_unit_button.clicked.connect(
            lambda: self.add_modelling_unit()
        )  # FIXME Why is lambda needed?
        add_erosion_button = QPushButton("Add erosion")
        add_erosion_button.clicked.connect(lambda: self.add_erosion())

        # Pile layout and scroll area
        scroll_pile = QScrollArea()
        scroll_pile.setWidgetResizable(True)
        self.pile_widget = QWidget()
        self.pile_layout = QVBoxLayout()
        self.pile_widget.setLayout(self.pile_layout)
        scroll_pile.setWidget(self.pile_widget)

        # Menus
        menu_bar = QMenuBar()
        self.menu_bar = menu_bar
        pile_menu = menu_bar.addMenu(self.tr("Pile"))
        self.pile_menu = pile_menu
        new_action = QAction(self.tr("New"), parent=self)
        new_action.triggered.connect(self.new)
        pile_menu.addAction(new_action)
        open_action = QAction(self.tr("Open"), parent=self)
        open_action.triggered.connect(self.open_layer)
        pile_menu.addAction(open_action)
        load_action = QAction("Load from XML", parent=self)
        load_action.triggered.connect(self.load_from_xml_file)
        pile_menu.addAction(load_action)
        save_as_menu = pile_menu.addMenu(self.tr("Save as"))
        save_as_xml_action = QAction("XML", parent=self)
        save_as_xml_action.triggered.connect(lambda: save_as_xml(self, self.pile))
        save_as_menu.addAction(save_as_xml_action)
        save_as_img_action = QAction("PNG", parent=self)
        save_as_img_action.triggered.connect(
            lambda: save_as_png(self.pile_widget, self, self.pile.name, 100, -50)
        )
        save_as_menu.addAction(save_as_img_action)
        save_as_new_action = QAction(self.tr("New layer"), parent=self)
        save_as_new_action.triggered.connect(self.open_copy)
        save_as_menu.addAction(save_as_new_action)
        pile_menu.addMenu(save_as_menu)

        # Buttons
        model_button = QPushButton("Create a model with this pile")
        model_button.clicked.connect(self.create_new_model_from_this_pile)
        cancel_button = QPushButton("Cancel")
        cancel_button.clicked.connect(self.reject)
        save_button = QPushButton("Save")
        save_button.clicked.connect(lambda: self.save())
        ok_button = QPushButton("OK")
        ok_button.clicked.connect(self.accept)

        # Signals
        self.accepted.connect(lambda: self.save())
        self.finished.connect(self.deleteLater)

        # Layouts
        buttons_add_layout = QHBoxLayout()
        buttons_add_layout.addWidget(add_modelling_unit_button)
        buttons_add_layout.addWidget(add_erosion_button)
        buttons_layout = QGridLayout()
        buttons_layout.addWidget(model_button, 0, 4)
        buttons_layout.addWidget(cancel_button, 1, 2)
        buttons_layout.addWidget(save_button, 1, 3)
        buttons_layout.addWidget(ok_button, 1, 4)
        self.layout = QVBoxLayout()
        self.layout.setMenuBar(menu_bar)
        self.layout.addWidget(self.label_pilename)
        self.layout.addLayout(buttons_add_layout)
        self.layout.addWidget(scroll_pile)
        self.layout.addLayout(buttons_layout)
        self.setLayout(self.layout)
        self.setAcceptDrops(True)
        self.resize(800, 800)

        # Display pile
        self.reset(pile_layer)

    def set_pile_working_copy(self):
        if self.pile_layer is None:
            pile = ModellingUnit(DEFAULT_PILE_NAME)
        else:
            pile = fxml.deep_copy(self.pile_layer.pile)
        self.pile = pile

    def reset(self, pile_layer=None):
        self.pile_layer = pile_layer
        if self.pile_layer is not None:
            assert isinstance(pile_layer, PileLayer)
            # FIXME Cannot connect multiple methods to one nameChanged? (cf. in PileLayer)
            self.pile_layer.nameChanged.connect(self._update_pilename_label)
            self.label_pilename.setText(self.pile_layer.name())
        self.set_pile_working_copy()
        self.refresh()

    def _update_pilename_label(self):
        if self.pile_layer is not None:
            name = self.pile_layer.name()
        else:
            name = "No pile selected"
        self.label_pilename.setText(name)

    def refresh(self):
        # Clear layout except pile title
        for i in range(self.pile_layout.count(), 0, -1):
            self.pile_layout.itemAt(i - 1).widget().deleteLater()
        # Add widgets of the loaded pile
        for element in self.pile.subunits():
            new_widget = ElementWidget(element, self.pile)
            self.pile_layout.insertWidget(0, new_widget)  # Add on top of the pile
            self.nb_of_elements += 1

    def add_modelling_unit(self, name=None):
        if name is None:
            name = f"Unit {self.nb_of_elements}"
        # Force pile elements to have a color, it is reused many times later
        unit = ModellingUnit(name, info={"color": "#55aa00"})
        self.pile.append(unit)
        modelling_unit = ElementWidget(unit, self.pile)
        self.pile_layout.insertWidget(0, modelling_unit)
        self.nb_of_elements += 1

    def add_erosion(self):
        name = f"Erosion {self.nb_of_elements}"
        erosion = Erosion(name, info={"color": "#ff0000"})
        self.pile.append(erosion)
        erosion = ElementWidget(erosion, self.pile)
        self.pile_layout.insertWidget(0, erosion)
        self.nb_of_elements += 1

    def dragEnterEvent(self, event):
        event.acceptProposedAction()

    def open_layer(self):
        dlg = QDialog(parent=iface.mainWindow())
        dlg.setWindowTitle(dlg.tr("Open an existing pile"))
        cbox_layers = QgsPluginLayerComboBox(PileLayer)
        buttons = QDialogButtonBox(
            QDialogButtonBox.StandardButton.Cancel | QDialogButtonBox.StandardButton.Ok
        )
        buttons.accepted.connect(dlg.accept)
        buttons.rejected.connect(dlg.reject)
        layout = QVBoxLayout()
        layout.addWidget(QLabel(dlg.tr("Select a pile")))
        layout.addWidget(cbox_layers)
        layout.addWidget(buttons)
        dlg.setLayout(layout)

        def process_result(result):
            if result == QDialog.DialogCode.Rejected:  # Cancel button
                return
            layer = cbox_layers.currentLayer()
            save_current_pile = popup_save_changes(self.pile.name)
            # Open a new PileEditionDialog
            self.edit(layer)
            # Close the current PileEditionDialog
            if save_current_pile:
                result = QDialog.DialogCode.Accepted
                self.save()
            else:
                result = QDialog.DialogCode.Rejected
            self.done(result)  # Causes self to close(), and emit accepted/rejected

        dlg.finished.connect(process_result)
        dlg.open()

    def dragMoveEvent(self, event):
        mime_data = event.mimeData()
        if mime_data.hasFormat("application/x-pileitem"):
            source_widget = event.source().parent()
            if source_widget:
                drop_position = self.pile_layout.indexOf(source_widget)
                for i in range(self.pile_layout.count()):
                    rect = self.pile_layout.itemAt(i).geometry()
                    position = self.pile_widget.mapFrom(self, event.pos())
                    if rect.contains(position):
                        drop_position = i
                        break
                self.pile_layout.insertWidget(drop_position, source_widget)
                elem = source_widget.elem
                self.pile.removeUnit(elem)
                self.pile.insertAt(elem, len(self.pile.description) - drop_position)
            event.acceptProposedAction()

    def dropEvent(self, event):
        event.acceptProposedAction()

    def load_from_xml_file(self):
        src_dir = get_forgeo_data_dir()
        filename = QFileDialog.getOpenFileName(  # Returns a 2-tuple (filename, filter)
            parent=self,
            caption=self.tr("Load an existing pile"),
            directory=str(src_dir),
            filter=self.tr("XML (*.xml)"),
        )[0]
        if not filename:
            return
        filename = Path(filename)
        pile = fxml.load(filename)
        if pile is None or not isinstance(pile, ModellingUnit):
            return
        # Everything fine, update the widget
        layer = PileLayer(pile)
        QgsProject.instance().addMapLayer(layer)
        self.reset(layer)

    def create_new_model_from_this_pile(self):
        self.accept()
        ModelEditionDialog.new(pile_layer=self.pile_layer)

    def save(self):
        """On save, the pile of the edited layer is overridden with the working copy
        of the widget, and the widget gets a fresh new working copy.

        If no layer exists yet, a new one is created and added to QGIS layer tree
        """

        if self.pile_layer is None or not QgsProject.instance().mapLayersByName(
            self.pile_layer.name()
        ):
            layer = PileLayer(self.pile)
            QgsProject.instance().addMapLayer(layer)
            self.pile_layer = layer
        else:
            # FIXME Do we need to check if layer name changed to update pile name?
            self.pile_layer.pile = self.pile
        self.set_pile_working_copy()

    def open_copy(self):
        self.accept()
        layer = PileLayer.clone(self.pile_layer)
        layer.setName(layer.name() + " copy")
        self.edit(layer)

    @classmethod
    def new(cls):
        title = "New pile"
        name, data_layer, field = popup_new_item_collection(title, DEFAULT_PILE_NAME)
        # If there is another one with this name in que QGIS project, ask for a new name
        while QgsProject.instance().mapLayersByName(name):
            QMessageBox.warning(
                iface.mainWindow(), "Already existing", "Please enter a different name"
            )
            name, data_layer, field = popup_new_item_collection(title, name)
        if name is None:
            return
        layer = PileLayer.new(name)
        dlg = cls(layer, parent=iface.mainWindow())
        if data_layer is not None:
            for unitname in {feature[field] for feature in data_layer.getFeatures()}:
                if not isinstance(unitname, QVariant):
                    dlg.add_modelling_unit(unitname)
        dlg.exec()

    @classmethod
    def edit(cls, layer):
        dlg = cls(layer, parent=iface.mainWindow())
        dlg.exec()


class ElementWidget(QWidget):
    """Widget representing an element of the pile, unit or erosion"""

    def __init__(self, elem, pile, parent=None):
        super().__init__(parent)
        self.elem = elem
        self.pile = pile

        label = ElementLabel(elem, pile)
        delete_button = QPushButton("X")
        delete_button.setFixedSize(QSize(40, 40))
        delete_button.clicked.connect(lambda: self.delete_element())

        # Layout
        layout = QHBoxLayout()
        layout.setContentsMargins(0, 0, 0, 0)
        layout.addWidget(DragIcon())
        layout.addWidget(label)
        layout.addWidget(delete_button)
        self.setLayout(layout)
        self.setFixedHeight(60)

    def delete_element(self):
        self.pile.removeUnit(self.pile[self.elem.name])
        self.deleteLater()


class ElementLabel(QWidget):
    """Name, description, color (and symbols) of an element of the pile"""

    def __init__(self, elem, pile, parent=None):
        super().__init__(parent)
        self.elem = elem
        self.pile = pile
        self.description = elem.info.get("description", None)
        if isinstance(elem, ModellingUnit):
            self.label = ModellingUnitWidget(elem)
        else:  # isinstance(elem, Erosion)
            self.label = ErosionWidget(elem)
        self.set_color(QColor(elem.info["color"]))

        layout = QHBoxLayout()
        layout.setContentsMargins(0, 0, 0, 0)
        layout.addWidget(self.label)
        self.setLayout(layout)

    def set_color(self, color):
        color_name = color.name()
        self.label.label_name.setStyleSheet(f"background-color: {color_name};")
        self.label.label_desc.setStyleSheet(f"background-color: {color_name};")
        if isinstance(self.elem, Erosion):
            self.label.symbolL.setStyleSheet(f"color: {color_name};")
            self.label.symbolR.setStyleSheet(f"color: {color_name};")

    def mouseDoubleClickEvent(self, event):  # noqa: ARG002
        dialog = ElementDialog(self.elem, parent=self)
        ok = dialog.exec()
        if ok:
            # Change color
            old_color = self.elem.info.get("color", None)
            new_color = dialog.color
            if new_color != old_color:
                self.set_color(new_color)
                self.elem.info["color"] = new_color.name()
            # Change name
            old_name = self.elem.name
            new_name = dialog.name
            label_name = self.label.label_name
            if new_name != old_name:
                if self.pile[new_name] is not None:
                    QMessageBox.warning(
                        self, "Renaming impossible", "Name already used"
                    )
                    return
                self.pile[old_name].name = new_name
                label_name.setText(new_name)
            # Change description
            old_desc = self.elem.info.get("description", None)
            new_desc = dialog.desc
            if new_desc == old_desc:
                return
            self.elem.info["description"] = new_desc
            label_desc = self.label.label_desc
            label_desc.setText(new_desc)
            refresh_labels_height(label_name, label_desc)


class ModellingUnitWidget(QWidget):
    def __init__(self, unit, parent=None):
        super().__init__(parent)
        self.label_name = QLabel(unit.name)
        self.label_name.setAlignment(
            Qt.AlignmentFlag.AlignCenter | Qt.AlignmentFlag.AlignVCenter
        )
        self.label_desc = QLabel(unit.info.get("description", None))
        self.label_desc.setAlignment(
            Qt.AlignmentFlag.AlignCenter | Qt.AlignmentFlag.AlignVCenter
        )
        font = QFont()
        font.setPointSize(10)
        self.label_name.setFont(font)
        refresh_labels_height(self.label_name, self.label_desc)

        # Layout
        layout = QVBoxLayout()
        layout.addWidget(self.label_name)
        layout.addWidget(self.label_desc)
        self.setLayout(layout)


class ErosionWidget(QWidget):
    def __init__(self, erosion, parent=None):
        super().__init__(parent)
        self.label_name = QLabel(erosion.name)
        self.label_name.setAlignment(
            Qt.AlignmentFlag.AlignCenter | Qt.AlignmentFlag.AlignVCenter
        )
        self.label_desc = QLabel(erosion.info.get("description", None))
        self.label_desc.setAlignment(
            Qt.AlignmentFlag.AlignCenter | Qt.AlignmentFlag.AlignVCenter
        )
        font = QFont()
        font.setPointSize(10)
        self.label_name.setFont(font)
        color = QColor(erosion.info["color"])
        self.symbolR, self.symbolL = surface_symbols(color, is_erosion=True)
        refresh_labels_height(self.label_name, self.label_desc)

        # Layout
        layout = QHBoxLayout()
        layout_name_desc = QVBoxLayout()
        layout_name_desc.addWidget(self.label_name)
        layout_name_desc.addWidget(self.label_desc)
        layout.addWidget(self.symbolL)
        layout.addLayout(layout_name_desc)
        layout.addWidget(self.symbolR)
        self.setLayout(layout)


class ElementDialog(QDialog):
    """Edition Widget for name, description and color of elements"""

    def __init__(self, elem, parent):
        super().__init__(parent)
        self.color = QColor(elem.info["color"])

        # Name
        name_title = QLabel("Name", alignment=Qt.AlignmentFlag.AlignCenter)
        self.edit_name = QLineEdit(elem.name)
        self.edit_name.setAcceptDrops(True)
        self.edit_name.setClearButtonEnabled(True)
        # Description
        desc_title = QLabel("Description", alignment=Qt.AlignmentFlag.AlignCenter)
        self.edit_desc = QLineEdit(elem.info.get("description", None))
        self.edit_desc.setAcceptDrops(True)
        self.edit_desc.setClearButtonEnabled(True)
        # Color
        self.color_button = QPushButton(self)
        self.color_button.setStyleSheet(f"background-color: {self.color.name()};")
        self.color_button.setText("Color")
        self.color_button.clicked.connect(self.change_color)
        # Buttons
        buttons = QDialogButtonBox(self)
        buttons.setStandardButtons(
            QDialogButtonBox.StandardButton.Cancel | QDialogButtonBox.StandardButton.Ok
        )
        buttons.accepted.connect(self.accept)
        buttons.rejected.connect(self.reject)
        self.finished.connect(self.deleteLater)

        # Layout
        layout = QVBoxLayout()
        layout.addWidget(name_title)
        layout.addWidget(self.edit_name)
        layout.addWidget(desc_title)
        layout.addWidget(self.edit_desc)
        layout.addWidget(self.color_button)
        layout.addWidget(buttons)
        self.setLayout(layout)
        self.setWindowTitle(f"{elem.name} properties")

    @property
    def name(self):
        return self.edit_name.text()

    @property
    def desc(self):
        return self.edit_desc.text()

    def change_color(self):
        """Changes the color of the button.
        The color of the labels are changed when this dialog is accepted"""

        dialog = QICSColorDialog(parent=self, initial=self.color)
        dialog.setWindowTitle(f"Change {self.name} color")
        ok = dialog.exec()
        if ok:
            new_color = dialog.currentColor()
            self.color_button.setStyleSheet(f"background-color: {new_color.name()};")
            self.color = new_color


class DragIcon(QLabel):
    """Widget used to drag and drop ErosionWidgets and ModellingUnitWidgets"""

    def __init__(self, parent=None):
        super().__init__(parent)
        self.color = QColor(Qt.GlobalColor.red)
        self.setText("_\n_\n_")
        self.setAlignment(Qt.AlignmentFlag.AlignCenter | Qt.AlignmentFlag.AlignTop)
        self.setFixedSize(QSize(40, 50))

    def dragEnterEvent(self, event):
        event.acceptProposedAction()

    def mousePressEvent(self, event):
        if event.button() == Qt.MouseButton.LeftButton:
            drag = QDrag(self)
            mime_data = QMimeData()
            mime_data.setData("application/x-pileitem", QByteArray())
            drag.setMimeData(mime_data)
            drag.exec(Qt.DropAction.MoveAction)


def refresh_labels_height(label_name, label_desc):
    if not label_desc.text():
        label_name.setFixedHeight(50)
        label_desc.setFixedHeight(0)
    else:
        label_name.setFixedHeight(25)
        label_desc.setFixedHeight(25)
