from pathlib import Path

from forgeo.core import FaultItem, FaultNetwork
from qgis.core import QgsProject
from qgis.gui import QgsCheckableComboBox
from qgis.PyQt.QtCore import QSize, Qt
from qgis.PyQt.QtGui import QColor, QStandardItem
from qgis.PyQt.QtWidgets import (
    QAction,
    QCheckBox,
    QDialog,
    QDialogButtonBox,
    QFileDialog,
    QHBoxLayout,
    QLabel,
    QLineEdit,
    QMenuBar,
    QMessageBox,
    QPushButton,
    QScrollArea,
    QVBoxLayout,
    QWidget,
)
from qgis.utils import iface

import forgeo.io.xml as fxml

from ..layers import FaultNetworkLayer
from ..layers.fault_network import TemporaryFaultNetworkLayer
from ..utils import (
    DEFAULT_FAULTNET_NAME,
    clearlayout,
    data_icon_widget,
    get_forgeo_data_dir,
    input_data_type,
    popup_save_changes,
    save_as_png,
    save_as_xml,
)
from .color_picker import IcsColorDialog as QICSColorDialog
from .discretization import SurfaceExtractionDialog
from .faults_parameters import FaultPropertiesDialog
from .interpolator_widget import InterpolationParametersDialog
from .utils import QgsPluginLayerComboBox, popup_new_item_collection


class FaultLabel(QLabel):
    """Widget for the name and color of the fault"""

    def __init__(self, fault, network_widget, parent=None):
        assert parent is not None  # FIXME Should not be needed
        super().__init__(fault.name, parent)
        self.fault = fault
        self.network_widget = network_widget
        self.set_color(QColor(Qt.GlobalColor.red))
        if self.fault.info is not None and self.fault.info["color"] is not None:
            self.set_color(QColor(self.fault.info["color"]))
        self.setAlignment(Qt.AlignmentFlag.AlignCenter)
        self.setFixedHeight(25)

    def set_color(self, color):
        self.setStyleSheet(f"background-color: {color.name()};")
        self.color = color

    def mouseDoubleClickEvent(self, event):  # noqa: ARG002
        dialog = FaultDialog(self.fault.name, self.color)
        ok = dialog.exec()
        if ok:
            new_name = dialog.name
            # Prevents from using an already used name
            if self.fault.name != new_name:
                if self.network_widget.faultnet.get_item(new_name) is not None:
                    QMessageBox.warning(
                        self, "Renaming impossible", "Name already used"
                    )
                    return
                # rename
                self.network_widget.rename_fault_in_cboxes(self.fault.name, new_name)
                self.network_widget.faultnet.rename_fault(self.fault, new_name)
                self.setText(new_name)
            # change color
            self.set_color(dialog.color)
            self.fault.info["color"] = str(self.color.name())


class FaultDialog(QDialog):
    """Edition Widget for name and color of a fault"""

    def __init__(self, name, color, parent=None):
        super().__init__(parent)
        self.color = color
        self.name_title = QLabel("Name", alignment=Qt.AlignmentFlag.AlignCenter)
        self.name_title.setFixedHeight(15)
        self.edit_name = QLineEdit()
        self.edit_name.setText(name)
        self.edit_name.setAcceptDrops(True)
        self.edit_name.setClearButtonEnabled(True)
        self.back_color_button = QPushButton(self)
        self.back_color_button.setText("Change background color")
        self.back_color_button.clicked.connect(self.change_color)
        self.set_color(self.color)
        self.buttons = QDialogButtonBox(self)
        self.buttons.setStandardButtons(
            QDialogButtonBox.StandardButton.Cancel | QDialogButtonBox.StandardButton.Ok
        )
        self.buttons.accepted.connect(self.accept)
        self.buttons.rejected.connect(self.reject)
        self.finished.connect(self.deleteLater)

        spacer = QWidget()
        spacer.setFixedSize(10, 25)

        self.layout = QVBoxLayout()
        self.layout.addWidget(self.name_title)
        self.layout.addWidget(self.edit_name)
        self.layout.addWidget(spacer)
        self.layout.addWidget(self.back_color_button)
        self.layout.addWidget(spacer)
        self.layout.addWidget(self.buttons)
        self.setLayout(self.layout)
        self.resize(400, 200)
        self.setWindowTitle(f"{self.name} properties")

    @property
    def name(self):
        return self.edit_name.text()

    def set_color(self, color):
        self.back_color_button.setStyleSheet(f"background-color: {color.name()};")
        self.color = color

    def change_color(self):
        dialog = QICSColorDialog(parent=self)
        dialog.setWindowTitle(f"Change {self.name} color")
        dialog.setCurrentColor(self.color)
        ok = dialog.exec()
        if ok:
            self.set_color(dialog.currentColor())


class FaultEditionWidget(QWidget):
    def __init__(self, fault, filter, is_checked, parent):
        super().__init__(parent)
        self.fault = fault
        self.filter = filter
        self.network_widget = parent
        layout = QHBoxLayout()
        self.layout = layout

        # Active state
        is_active = Qt.CheckState.Checked if is_checked else Qt.CheckState.Unchecked
        active_checkbox = QCheckBox("")
        active_checkbox.setCheckState(is_active)
        active_checkbox.clicked.connect(self.update_active_state)

        # Label
        label = FaultLabel(fault, self.network_widget, parent=self)
        label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        label.setFixedHeight(50)
        color = QColor(fault.info["color"])
        label.setStyleSheet(f"background-color: {color.name()};")
        self.label = label

        # Properties
        self.button_properties = QPushButton("Properties")
        self.button_properties.clicked.connect(self.edit_properties)
        self.data_img = None

        # Stops on
        stops_on_cbox = QgsCheckableComboBox()
        stops_on_cbox.checkedItemsChanged.connect(self.new_fault_checked)
        faultnet = parent.faultnet
        self_index = next(i for i, f in enumerate(faultnet.dataset) if f is fault)
        for other_index, f in enumerate(faultnet.dataset):
            if self_index == other_index:
                continue
            item = QStandardItem(f.name)
            item.setData(f)
            if faultnet.relations[other_index, self_index]:
                item.setEnabled(False)
            else:
                item.setEnabled(True)
                if faultnet.relations[self_index, other_index]:
                    is_active = Qt.CheckState.Checked
                else:
                    is_active = Qt.CheckState.Unchecked
                item.setCheckState(is_active)
            stops_on_cbox.model().appendRow(item)
        self.stops_on_cbox = stops_on_cbox
        self.checked_faultnames = stops_on_cbox.checkedItems()

        # Delete button
        delete_button = QPushButton("X")
        delete_button.setFixedSize(QSize(40, 40))
        delete_button.clicked.connect(lambda: self.network_widget.delete_fault(self))

        # Layouts
        layout.addWidget(active_checkbox, stretch=1)
        layout.addWidget(label, stretch=10)
        self.refresh_data_img(self.fault)
        layout.addWidget(self.button_properties, stretch=1)
        layout.addWidget(stops_on_cbox, stretch=10)
        layout.addWidget(delete_button, stretch=1)
        self.setLayout(layout)

    def update_fault_relation(self, other_name, stops_on):
        # Called when other is modified, so we only consider other stops_on self
        model = self.stops_on_cbox.model()
        other_item = model.findItems(other_name)[0]
        other_item.setEnabled(not stops_on)
        # TODO Re-order with all uncheckable at last

    def new_fault_checked(self):
        new_checked_faultnames = self.stops_on_cbox.checkedItems()
        more_checked = len(self.checked_faultnames) < len(new_checked_faultnames)
        assert len(self.checked_faultnames) - len(new_checked_faultnames) in [-1, 1]
        if more_checked:
            for fname in new_checked_faultnames:
                if fname not in self.checked_faultnames:
                    other_name = fname
        else:
            for fname in self.checked_faultnames:
                if fname not in new_checked_faultnames:
                    other_name = fname
        self_index = self.network_widget.faultnet.get_fault_index(self.fault.name)
        other_index = self.network_widget.faultnet.get_fault_index(other_name)
        if self.network_widget.faultnet.relations[other_index, self_index]:
            # If already checked the other side, uncheck it and do not update relations.
            # setEnabled just makes the item grey, and do not prevent to check it
            # (setCheckable and setSelectable neither)
            item = self.stops_on_cbox.model().findItems(other_name)[0]
            item.setCheckState(Qt.CheckState.Unchecked)
        else:
            self.network_widget.update_fault_relation(self, other_name, more_checked)
            self.checked_faultnames = new_checked_faultnames

    def remove_fault_from_cbox(self, name):
        cbox = self.stops_on_cbox
        index = next(i for i in range(cbox.count()) if cbox.itemText(i) == name)
        cbox.removeItem(index)
        self.checked_faultnames = cbox.checkedItems()

    def update_active_state(self):
        active = self.sender().checkState() == Qt.CheckState.Checked
        self.network_widget.faultnet.set_as_active(self.fault.name, active)

    def edit_properties(self):
        # Warning: do not modify 'parent=self' below! Some subwidgets rely on it
        dlg = FaultPropertiesDialog(self.fault, self.filter, parent=self)

        def _update_fault_params(result):
            if result == QDialog.DialogCode.Rejected:
                return
            # Get results
            updated_item = dlg.fault
            updated_filter = dlg.filter
            interpolator = dlg.interpolator
            # Update self
            self.fault = updated_item
            self.filter = updated_filter
            # Update fault network
            layer = self.network_widget.layer
            fault_network = layer.faultnet
            fault_network.set_item(updated_item)
            fault_network.set_interpolator(interpolator)
            layer.filters[updated_item.name] = updated_filter
            # Refresh display
            self.refresh_data_img(self.fault)

        dlg.finished.connect(_update_fault_params)
        dlg.open()

    def refresh_data_img(self, item):
        if item.item_data is None:
            if self.data_img is not None:
                self.data_img.deleteLater()
            self.layout.removeWidget(self.data_img)
            return
        nb_observations = item.item_data.nb_observation_data()
        nb_orientations = item.item_data.nb_orientation_data()
        widget = data_icon_widget(
            input_data_type(item), nb_observations, nb_orientations
        )
        widget.setFixedHeight(80)
        self.layout.removeWidget(self.data_img)
        self.layout.insertWidget(2, widget)
        self.data_img = widget

    def define_interp(self):
        interpolator = self.network_widget.faultnet.get_interpolator(self.item.name)
        dlg = InterpolationParametersDialog(interpolator)

        def _update_interpolator(result):
            if result == QDialog.DialogCode.Rejected:
                return
            # Update interpolator
            self.network_widget.faultnet.set_interpolator(dlg.interpolator)
            dlg.close()

        dlg.finished.connect(_update_interpolator)
        dlg.show()


class FaultNetworkEditionDialog(QDialog):
    def __init__(self, layer, parent=None):
        super().__init__(parent)
        # Init
        self.setWindowTitle("Fault network edition")
        assert layer is not None
        assert isinstance(layer, FaultNetworkLayer)
        self.layer = layer

        # Menus
        menu_bar = QMenuBar()
        self.menu_bar = menu_bar
        # Faultnet menu
        faultnet_menu = menu_bar.addMenu(self.tr("Fault network"))
        self.faultnet_menu = faultnet_menu
        new_action = QAction(self.tr("New"), parent=self)
        new_action.triggered.connect(self.new)
        faultnet_menu.addAction(new_action)
        open_action = QAction(self.tr("Open"), parent=self)
        open_action.triggered.connect(self.open_layer)
        faultnet_menu.addAction(open_action)
        load_action = QAction("Load from XML", parent=self)
        load_action.triggered.connect(self.load_from_xml_file)
        faultnet_menu.addAction(load_action)
        save_as_menu = faultnet_menu.addMenu(self.tr("Save as"))
        save_as_xml_action = QAction("XML", parent=self)
        save_as_xml_action.triggered.connect(lambda: save_as_xml(self, self.faultnet))
        save_as_menu.addAction(save_as_xml_action)
        save_as_img_action = QAction("PNG", parent=self)
        save_as_img_action.triggered.connect(
            lambda: save_as_png(
                self.scroll_area_widget, self, self.faultnet.name, 70, -10
            )
        )
        save_as_menu.addAction(save_as_img_action)
        save_as_new_action = QAction(self.tr("New layer"), parent=self)
        save_as_new_action.triggered.connect(self.open_copy)
        save_as_menu.addAction(save_as_new_action)
        faultnet_menu.addMenu(save_as_menu)
        # Menu data
        data_menu = menu_bar.addMenu(self.tr("Data"))
        self.data_menu = data_menu
        self.connect_action = QAction(self.tr("Automatic data update"), parent=self)
        self.connect_action.setCheckable(True)
        self.connect_action.setChecked(layer.connected_to_layers)
        self.connect_action.triggered.connect(
            lambda checked: self.automatic_data_update(checked)
        )
        data_menu.addAction(self.connect_action)
        update_action = QAction(self.tr("Update fault network"), parent=self)
        update_action.triggered.connect(self.update_dataset)
        data_menu.addAction(update_action)
        fill_model_action = QAction(
            self.tr("Load data from another fault network"), parent=self
        )
        fill_model_action.triggered.connect(self.load_data_from_faults)
        data_menu.addAction(fill_model_action)
        # Discretization menu
        menu_discretization = menu_bar.addMenu(self.tr("Discretization"))
        self.menu_discretization = menu_discretization
        action_surfaces = QAction(self.tr("Extract surfaces"), parent=self)
        action_surfaces.triggered.connect(self.extract_surfaces)
        menu_discretization.addAction(action_surfaces)

        # Fault network name
        self.label_faultnetname = QLabel(layer.name())
        self.label_faultnetname.setAlignment(Qt.AlignmentFlag.AlignCenter)

        # Add faults
        fault_button = QPushButton("Add fault")
        fault_button.clicked.connect(lambda: self.add_fault())

        # Columns titles
        titles_layout = QHBoxLayout()
        titles_layout.addWidget(
            QLabel("Use", alignment=Qt.AlignmentFlag.AlignCenter), stretch=2
        )
        titles_layout.addWidget(
            QLabel("Fault", alignment=Qt.AlignmentFlag.AlignCenter), stretch=7
        )
        titles_layout.addWidget(
            QLabel("Data and parameters", alignment=Qt.AlignmentFlag.AlignCenter),
            stretch=2,
        )
        titles_layout.addWidget(
            QLabel("Stops on", alignment=Qt.AlignmentFlag.AlignCenter), stretch=7
        )
        titles_layout.addWidget(
            QLabel("Delete", alignment=Qt.AlignmentFlag.AlignCenter), stretch=2
        )

        # Buttons
        cancel_button = QPushButton("Cancel")
        cancel_button.clicked.connect(self.reject)
        save_button = QPushButton("Save")
        save_button.clicked.connect(lambda: self.save())
        ok_button = QPushButton("OK")
        ok_button.clicked.connect(self.accept)

        # Signals
        self.accepted.connect(lambda: self.save())
        self.finished.connect(self.deleteLater)

        # Faults
        scroll_area = QScrollArea()
        scroll_area.setWidgetResizable(True)
        self.scroll_area_widget = QWidget()
        self.faults_layout = QVBoxLayout()
        self.scroll_area_widget.setLayout(self.faults_layout)
        scroll_area.setWidget(self.scroll_area_widget)

        # Layouts
        upper_layout = QVBoxLayout()
        upper_layout.addWidget(self.label_faultnetname)
        upper_layout.addWidget(fault_button)
        upper_layout.addLayout(titles_layout)
        buttons_layout = QHBoxLayout()
        buttons_layout.addWidget(cancel_button, stretch=1)
        buttons_layout.addWidget(save_button, stretch=1)
        buttons_layout.addWidget(ok_button, stretch=1)
        self.layout = QVBoxLayout()
        self.layout.setMenuBar(menu_bar)
        self.layout.addLayout(upper_layout)
        self.layout.addWidget(scroll_area)
        self.layout.addLayout(buttons_layout)
        self.setLayout(self.layout)
        self.resize(800, 500)

        # Display fault network
        self.refresh()

    @property
    def faultnet(self):
        return self.layer.faultnet

    def refresh(self):
        # Delete previous fault network
        clearlayout(self.faults_layout)
        # Display new fault network
        if self.faultnet.nb_faults == 0:
            return
        for fault in self.faultnet.dataset:
            idx = self.faultnet.get_fault_index(fault.name)
            is_checked = self.faultnet.active_faults[idx]
            filter = None
            if (filters := self.layer.filters) is not None:
                filter = filters.get(fault.name)
            # 'parent=self' is mandatory, as used inside FaultEditionWidget.__init__
            self.faults_layout.addWidget(
                FaultEditionWidget(fault, filter, is_checked, parent=self)
            )

    def add_fault(self, name=None):
        if name is None:
            name = f"Fault {self.faultnet.nb_faults}"
        fault = FaultItem(name, info={"color": "#ff0000"})
        self.faultnet.add_fault(fault)
        nb_faults = self.faults_layout.count()
        for i in range(nb_faults):
            f_widget = self.faults_layout.itemAt(i).widget()
            item = QStandardItem(fault.name)
            item.setData(fault)
            if self.faultnet.relations[i, nb_faults]:
                item.setEnabled(True)
                checkstate = Qt.CheckState.Checked
            elif self.faultnet.relations[nb_faults, i]:
                item.setEnabled(False)
                checkstate = Qt.CheckState.Unchecked
            else:
                item.setEnabled(True)
                checkstate = Qt.CheckState.Unchecked
            item.setCheckState(checkstate)
            f_widget.stops_on_cbox.model().appendRow(item)
        # 'parent=self' is mandatory, as used inside FaultEditionWidget.__init__
        self.faults_layout.addWidget(FaultEditionWidget(fault, None, True, parent=self))

    def delete_fault(self, f0_widget):
        # Unlock faults that previously stopped on f0
        f0_name = f0_widget.fault.name
        f0_index = self.faultnet.get_fault_index(f0_name)
        for f_index, stops_on in enumerate(self.faultnet.relations[:, f0_index]):
            if stops_on:
                f_name = self.faultnet.faults[f_index].name
                f_widget = self.get_fault_widget(f_name)
                self.update_fault_relation(f_widget, f0_name, False)
        # Delete from fault network and delete filters
        self.faultnet.delete_fault(f0_widget.fault)
        self.layer.filters.pop(f0_name, None)
        # Remove f0 from the other's comboboxes
        for i in range(self.faults_layout.count()):
            f_widget = self.faults_layout.itemAt(i).widget()
            if f_widget != f0_widget:
                f_widget.remove_fault_from_cbox(f0_widget.fault.name)
        f0_widget.deleteLater()

    def update_fault_relation(self, f1_widget, f2_name, stops_on):
        # Called by f1_widget, who shall already have updated itself
        f1_name = f1_widget.fault.name
        f1_index = self.faultnet.get_fault_index(f1_name)
        f2_index = self.faultnet.get_fault_index(f2_name)
        f2_widget = self.get_fault_widget(f2_name)
        faults = self.faultnet.dataset
        relations = self.faultnet.relations
        relations[f1_index, f2_index] = stops_on
        f2_widget.update_fault_relation(f1_name, stops_on)
        # Prevent faults from stopping on each other in a cycle
        for f_index, stops_on_f in enumerate(relations[:, f1_index]):
            if stops_on_f and f_index != f1_index:
                f_name = faults[f_index].name
                f_widget = self.get_fault_widget(f_name)
                self.update_fault_relation(f_widget, f2_name, stops_on)
        for f_index, stops_on_f in enumerate(relations[f2_index]):
            if stops_on_f and f_index != f2_index:
                f_name = faults[f_index].name
                f_widget = self.get_fault_widget(f_name)
                self.update_fault_relation(f1_widget, f_name, stops_on)
        self.faultnet.relations = relations

    def get_fault_widget(self, name):
        for i in range(self.faults_layout.count()):
            f_widget = self.faults_layout.itemAt(i).widget()
            if f_widget.fault.name == name:
                return f_widget
        msg = f"No fault named {name}"
        raise KeyError(msg)

    def rename_fault_in_cboxes(self, old_name, new_name):
        nb_faults = self.faults_layout.count()
        for i in range(nb_faults):
            f_widget = self.faults_layout.itemAt(i).widget()
            cbox = f_widget.stops_on_cbox
            for j in range(cbox.count()):
                if cbox.itemText(j) == old_name:
                    cbox.setItemText(j, new_name)

    def update_dataset(self):
        self.layer.update_dataset()
        # Update layout
        for i in range(self.faults_layout.count()):
            f_widget = self.faults_layout.itemAt(i).widget()
            f_widget.refresh_data_img(f_widget.fault)

    def automatic_data_update(self, checked):
        if checked:
            dialog = QDialog(parent=iface.mainWindow())
            dialog.setWindowTitle("Automatic data update")
            label = QLabel(
                "The fault network will be automatically updated when a source layer has been modified (Edited, deleted, change of coordinated reference system).\n"
                "Activating automatic data update will save the current fault network."
            )
            buttons = QDialogButtonBox(
                QDialogButtonBox.StandardButton.Cancel
                | QDialogButtonBox.StandardButton.Ok
            )
            buttons.accepted.connect(dialog.accept)
            buttons.rejected.connect(dialog.reject)
            layout = QVBoxLayout()
            layout.addWidget(label)
            layout.addWidget(buttons)
            dialog.setLayout(layout)
            if dialog.exec() == QDialog.DialogCode.Accepted:
                self.save()
                self.model_layer.reconnect_data_layers()
            else:
                self.connect_action.setChecked(Qt.CheckState.Unchecked)
        else:
            self.model_layer.disconnect_data_layers()

    def load_data_from_faults(self):
        dialog = QDialog(parent=iface.mainWindow())
        dialog.setWindowTitle("Load data from another fault network")
        label = QLabel(
            "Replaces data of elements with the same names in both fault networks.\n"
            "Choose source fault network :"
        )
        cbox = QgsPluginLayerComboBox(FaultNetworkLayer)
        buttons = QDialogButtonBox(
            QDialogButtonBox.StandardButton.Cancel | QDialogButtonBox.StandardButton.Ok
        )
        buttons.accepted.connect(dialog.accept)
        buttons.rejected.connect(dialog.reject)
        layout = QVBoxLayout()
        layout.addWidget(label)
        layout.addWidget(cbox)
        layout.addWidget(buttons)
        dialog.setLayout(layout)
        if dialog.exec() == QDialog.DialogCode.Accepted:
            self.layer.load_data_from_model(cbox.currentLayer())
            # Update layout
            for i in range(self.faults_layout.count()):
                f_widget = self.faults_layout.itemAt(i).widget()
                f_widget.refresh_data_img(f_widget.fault)

    def open_layer(self):
        dlg = QDialog(parent=iface.mainWindow())
        dlg.setWindowTitle(dlg.tr("Open an existing fault network"))
        layout = QVBoxLayout()
        dlg.setLayout(layout)
        lbl_description = QLabel(dlg.tr("Select a fault network layer"))
        cbox_layers = QgsPluginLayerComboBox(FaultNetworkLayer)
        buttons = QDialogButtonBox(
            QDialogButtonBox.StandardButton.Cancel | QDialogButtonBox.StandardButton.Ok
        )
        buttons.accepted.connect(dlg.accept)
        buttons.rejected.connect(dlg.reject)
        layout.addWidget(lbl_description)
        layout.addWidget(cbox_layers)
        layout.addWidget(buttons)

        def process_result(result):
            if result == QDialog.DialogCode.Rejected:  # Cancel button
                return
            layer = cbox_layers.currentLayer()
            save_current_faultnet = popup_save_changes(self.faultnet.name)
            # Open a new FaultNetworkEditionDialog
            self.edit(layer)
            # Close the current FaultNetworkEditionDialog
            if save_current_faultnet:
                result = QDialog.DialogCode.Accepted
                self.save()
            else:
                result = QDialog.DialogCode.Rejected
            self.done(result)  # Causes self to close(), and emit accepted/rejected

        dlg.finished.connect(process_result)
        dlg.open()

    def load_from_xml_file(self):
        src_dir = get_forgeo_data_dir()
        filename = QFileDialog.getOpenFileName(  # Returns a 2-tuple (filename, filter)
            parent=self,
            caption=self.tr("Load an existing fault network"),
            directory=str(src_dir),
            filter=self.tr("XML (*.xml)"),
        )[0]
        if not filename:
            return
        fault_network = fxml.load(Path(filename))
        if fault_network is None or not isinstance(fault_network, FaultNetwork):
            return
        # Cancel the changes in the current fault network
        self.done(QDialog.DialogCode.Rejected)
        # Everything fine, update the widget
        layer = FaultNetworkLayer(fault_network)
        QgsProject.instance().addMapLayer(layer)
        # Popup new fault network edition dialog
        self.edit(layer)

    def extract_surfaces(self):
        self.save()
        # FIXME Easier if the widget stores a ref to the permanent layer too?
        permanent_layer = QgsProject.instance().mapLayersByName(self.layer.name())
        permanent_layer = permanent_layer[0]
        assert isinstance(permanent_layer, FaultNetworkLayer)
        # 'parent=self': the dialog will be centered with self
        dlg = SurfaceExtractionDialog(permanent_layer, parent=self)
        dlg.show()

    def save(self):
        """On save, the edited layer captures the changes of the widget temporary layer,
        and the widget gets a fresh new temporary layer.

        If no layer exists yet, a new one is created and added to QGIS layer tree
        """
        # FIXME Easier if the widget stores a ref to the permanent layer too?
        permanent_layer = QgsProject.instance().mapLayersByName(self.layer.name())
        if not permanent_layer:
            permanent_layer = FaultNetworkLayer()
            QgsProject.instance().addMapLayer(permanent_layer)
        else:
            permanent_layer = permanent_layer[0]
            assert isinstance(permanent_layer, FaultNetworkLayer)
        permanent_layer.update_from(self.layer)
        self.layer = TemporaryFaultNetworkLayer.clone(permanent_layer)

    def open_copy(self):
        self.accept()
        layer = FaultNetworkLayer.clone(self.layer)
        layer.setName(layer.name() + " copy")
        self.edit(layer)

    @classmethod
    def new(cls, name=None):
        title = "New fault network"
        name, data_layer, field = popup_new_item_collection(
            title, DEFAULT_FAULTNET_NAME
        )
        # If there is another one with this name in que QGIS project, ask for a new name
        while QgsProject.instance().mapLayersByName(name):
            QMessageBox.warning(
                iface.mainWindow(), "Already existing", "Please enter a different name"
            )
            name, data_layer, field = popup_new_item_collection(title, name)
        if name is None:
            return
        layer = FaultNetworkLayer.new(name)
        # parent=iface.mainWindow() is important, some subwidgets rely on it!
        dlg = cls(layer, parent=iface.mainWindow())
        if data_layer is not None:
            for fault_name in {feature[field] for feature in data_layer.getFeatures()}:
                dlg.add_fault(fault_name)
        dlg.show()

    @classmethod
    def edit(cls, layer):
        layer = TemporaryFaultNetworkLayer.clone(layer)
        # parent=iface.mainWindow() is important, some subwidgets rely on it!
        dlg = cls(layer, parent=iface.mainWindow())
        dlg.exec()
