from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtGui import QColor
from qgis.PyQt.QtWidgets import (
    QDialog,
    QHBoxLayout,
    QLabel,
    QPushButton,
    QRadioButton,
    QSizePolicy,
    QStackedWidget,
    QVBoxLayout,
)

from ..utils import input_data_type
from .pointsfilter_widget import PointsFilterWidget
from .raster_widget import RasterWidget
from .utils import surface_symbols


class AddItemDataDialog(QDialog):
    def __init__(self, item, filter=None, color=None, parent=None):
        """
        Parameters
        ----------
        item: forgeo.Item
            Read-only: only used for accessing "contextual" information
        item_filter: ItemDataSelection | None
        """
        super().__init__(parent)
        self.setWindowTitle(self.tr(f"Data of {item.name}"))
        self.filter = filter
        self.resize(600, 200)

        # Title
        title = QLabel(item.name, alignment=Qt.AlignmentFlag.AlignCenter)
        title.setFixedHeight(30)
        title_layout = QHBoxLayout()
        if color is not None:
            color = QColor(color)
            title.setStyleSheet(f"background-color: {color.name()};")
        else:
            color = QColor("#ffffff")
        self.color = color
        if item.type not in ["Fault", "Unit"]:
            is_erosion = item.type == "Erosion"
            symbol_l, symbol_r = surface_symbols(color, is_erosion)
            title_layout.addWidget(symbol_l)
            title_layout.addWidget(title)
            title_layout.addWidget(symbol_r)
        elif item.type == "Unit":
            title_layout.addWidget(title)

        # Data type
        # Radio buttons that have the same parent are mutually exclusive by default.
        # So checking one will automatically uncheck the other(s)
        points_rbutton = QRadioButton("Points")
        points_rbutton.clicked.connect(lambda: self.data_widget.setCurrentIndex(0))
        raster_rbutton = QRadioButton("Raster")
        raster_rbutton.clicked.connect(lambda: self.data_widget.setCurrentIndex(1))

        points_rbutton.setChecked(True)
        if item.type not in ("Erosion", "Surface only"):
            raster_rbutton.setEnabled(False)
        # FIXME Forbids the use of "Raster data" option, remove next line to
        # authorize again
        raster_rbutton.setEnabled(False)  # To remove after demo

        # Data
        data_widget = QStackedWidget()
        data_widget.setSizePolicy(
            QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Expanding
        )
        data_widget.addWidget(PointsFilterWidget(item, filter, color))  # 0:points
        data_widget.addWidget(RasterWidget(item))  # 1:raster

        # Reload data
        data_type = input_data_type(item)
        if data_type == "rasterData":
            assert item.type in ("Erosion", "Surface only")
            raster_rbutton.setChecked(True)
            data_widget.setCurrentIndex(1)

        # Buttons
        clear_button = QPushButton("Clear")
        clear_button.clicked.connect(self.clear)
        cancel_button = QPushButton("Cancel")
        cancel_button.clicked.connect(self.reject)
        ok_button = QPushButton("OK")
        ok_button.clicked.connect(self.accept)
        self.accepted.connect(lambda: self.save())
        self.finished.connect(self.deleteLater)

        # Layouts
        main_layout = QVBoxLayout()
        main_layout.addLayout(title_layout, stretch=1)

        datatype_layout = QHBoxLayout()
        datatype_layout.addWidget(points_rbutton, alignment=Qt.AlignmentFlag.AlignRight)
        datatype_layout.addWidget(raster_rbutton, alignment=Qt.AlignmentFlag.AlignLeft)
        main_layout.addLayout(datatype_layout, stretch=1)

        main_layout.addWidget(data_widget)
        buttons_layout = QHBoxLayout()
        buttons_layout.addWidget(clear_button)
        buttons_layout.addStretch(1)
        buttons_layout.addWidget(cancel_button)
        buttons_layout.addWidget(ok_button)
        main_layout.addLayout(buttons_layout)
        self.setLayout(main_layout)

        self.data_widget = data_widget

    def clear(self):
        # Clear UI
        for i in range(self.data_widget.count()):
            w = self.data_widget.widget(i)
            if w is not None:
                w.clear()

    def save(self):
        self.filter = self.data_widget.currentWidget().process_selection()
