# import rifter
# from forgeo.gmlib.architecture import from_GeoModeller, make_evaluator
# from .algorithm import Algorithm
import contextlib
from pathlib import Path

from qgis.core import (
    QgsCoordinateTransform,
    QgsException,
    QgsProcessing,
    QgsProcessingException,
    QgsProcessingParameterBoolean,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterNumber,
)

from ._utils import verticalSectionFromPolyLine
from .featureextracter import FeatureExtracter


class GmVerticalSlicer(
    FeatureExtracter,
    name=Path(__file__).stem,
    display_name="vertical section",
):
    """Algorithm that generate vertical sections from geological model as a vector layer with symobology

    1. Create a vertical quads mesh along polylines, with quad side-length based on resolutions.
    2. Refine the mesh tracing model interfaces, cutting facets/edges intersected by unit boundaries.
    3. Evaluated the model at each resulting facet centroid.
    4. Fill output features with a multipolygon for each unit evaluated.

    Args:
        INPUT (GeoModelLayer): The GeoModel to interpolate.
        LINES (qgis.core.QgsVectorLayer): A polylines layer containing the top-view trace of sections to slice.
        RES_U (float): The horizontal resolution [in meters] for the output section (along the trace).
        RES_V (float): The vertical resolution [in meters] for the output section (along the z-dimension).
        CLIP_EXTENT(bool, optional): Whether clip the section's extent to the GeoModel extent.
        CLIP_TOPO(bool, optional): Whether clip the section's top to the GeoModel topography.

    Returns:
        OUTPUT (qgis.core.QgsVectorLayer): Layer with MultiPolygon geometry type and a proper feature renderer.
    """

    LINES = "LINES"
    RES_U = "RES_U"
    RES_V = "RES_V"
    # FIXME: to be factorized
    # CLIP_EXTENT = "CLIP_EXTENT"
    ZMIN = "ZMIN"
    ZMAX = "ZMAX"
    CLIP_TOPO = "CLIP_TOPO"

    def initAlgorithm(self, config=None):  # noqa: ARG002
        self.add_geomodel_parameter()
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                name=self.LINES,
                description=self.tr("Vector layer"),
                types=[QgsProcessing.TypeVectorLine],
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                self.RES_U,
                self.tr("Horizontal resolution [m]"),
                defaultValue=100,
                type=QgsProcessingParameterNumber.Integer,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                self.RES_V,
                self.tr("Vertical resolution [m]"),
                defaultValue=100,
                type=QgsProcessingParameterNumber.Integer,
            )
        )

        def add_double_parameter(name, description, default):
            self.addParameter(
                QgsProcessingParameterNumber(
                    name,
                    self.tr(description),
                    defaultValue=default,
                    type=QgsProcessingParameterNumber.Double,
                )
            )

        add_double_parameter(self.ZMIN, "bottom (zmin)", -1000.0)
        add_double_parameter(self.ZMAX, "top (zmax)", 1000.0)
        # self.addParameter(
        #     QgsProcessingParameterBoolean(
        #         self.CLIP_EXTENT,
        #         self.tr("Clip sections to GeoModel extent"),
        #         defaultValue=True,
        #     )
        # )
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.CLIP_TOPO,
                self.tr("Clip sections to GeoModel topography"),
                defaultValue=True,
            )
        )
        self.add_parameter_sinks()

    def processAlgorithm(self, parameters, context, feedback):
        input_layer = self.model_layer(parameters)
        source = self.parameterAsSource(parameters, self.LINES, context)
        if source is None:
            raise QgsProcessingException(
                self.invalidSourceError(parameters, self.LINES)
            )

        dx, dz, zmin, zmax = (
            self.parameterAsDouble(parameters, name, context)
            for name in (self.RES_U, self.RES_V, self.ZMIN, self.ZMAX)
        )

        # Get clip instructions (and topography evaluator)
        # clip_extent = self.parameterAsBool(parameters, self.CLIP_EXTENT, context)
        clip_topo = self.parameterAsBool(parameters, self.CLIP_TOPO, context)
        # if clip_topo:
        #     topo = lambda x, y: input_layer.model.topography.evaluate_z((x, y))
        # else:
        #     topo = None

        # Extract information about geomodel
        with contextlib.suppress(AttributeError):
            input_layer.box()

        count = source.featureCount()
        step = 100 / count if count > 0 else 1
        ct = QgsCoordinateTransform(
            source.sourceCrs(), input_layer.crs(), context.transformContext()
        )

        self.init_sinks(parameters, context, input_layer.crs())

        model = self.extraction_parameters(
            parameters, with_topography=clip_topo, faults_only=False
        )

        for i, feature in enumerate(source.getFeatures()):
            if feedback.isCanceled():
                break
            feedback.setProgress(i * step)
            if not feature.hasGeometry():
                feedback.reportError(f"No geometry attached to feature {feature.id()}.")
                continue

            # split multipolylines into separate lines
            for geometry in feature.geometry().constParts():
                if feedback.isCanceled():
                    break
                # FIXME: do we want to keep all attributes from source layer
                feature.attributes()
                try:
                    geometry.transform(ct, transformZ=True)
                except QgsException:
                    feedback.reportError(
                        f"Could not reproject feature {feature.id()} to GeoModel CRS."
                    )
                # if clip_extent:
                #     geometry = QgsGeometry.fromPolyline(geometry)
                #     geometry = geometry.clipped(bbox.toRectangle())
                #     if not geometry.isGeosValid():
                #         continue
                mesh = verticalSectionFromPolyLine(geometry, zmin, zmax, dx, dz)
                self.extract_features(model, mesh)

        result = self.finalize(input_layer.name(), context)
        feedback.setProgress(100)
        return result
