import contextlib

import numpy as np
from forgeo.interpolation import BSPTreeBuilder, which_domain
from qgis.core import (
    QgsFeature,
    QgsField,
    QgsPluginLayer,
    QgsProcessing,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterMapLayer,
    QgsProcessingParameterVectorDestination,
    QgsProject,
    QgsWkbTypes,
)
from qgis.PyQt.QtCore import QVariant

from ...layers.model import ModelLayer
from . import SamplingAlgorithm


class ProcessingParameterForgeoModelLayer(QgsProcessingParameterMapLayer):
    """Helper class to use a Forgeo model as Processing parameter"""

    def checkValueIsAcceptable(self, obj, context=None):  # noqa: ARG002
        if isinstance(obj, str):
            with contextlib.suppress(KeyError):
                obj = QgsProject.instance().layerStore().mapLayers()[obj]
        return bool(
            isinstance(obj, QgsPluginLayer)
            and obj.pluginLayerType() == ModelLayer.LAYER_TYPE
        )


class SampleGeoModel(SamplingAlgorithm):
    """This processing creates evaluates a ForGeo model on each points of a layer."""

    # Inputs parameters
    POINTS = "POINTS"
    MODEL = "MODEL"
    # Output parameters
    OUTPUT = "OUTPUT"

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr("Sample model on points")

    def initAlgorithm(self, config=None):  # noqa: ARG002
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        self.addParameter(
            ProcessingParameterForgeoModelLayer(
                self.MODEL,
                self.tr("ForGeo model"),
            )
        )
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.POINTS,
                self.tr("Points layer"),
                [QgsProcessing.TypeVectorPoint],
            )
        )
        self.addParameter(
            QgsProcessingParameterVectorDestination(
                self.OUTPUT,
                self.tr("Points output"),
                optional=True,
                createByDefault=True,
            )
        )

    def processAlgorithm(self, parameters, context, feedback) -> dict:  # noqa: ARG002
        """Here is where the processing itself takes place"""

        # Get input points
        points_layer = self.parameterAsVectorLayer(parameters, self.POINTS, context)
        points = []
        for feature in points_layer.getFeatures():
            for pt in feature.geometry().vertices():
                points.append((pt.x(), pt.y(), pt.z()))

        # Evaluate model on points
        model = (
            QgsProject.instance().layerStore().mapLayers()[parameters[self.MODEL]].model
        )
        params = BSPTreeBuilder.from_params(model=model)
        domains = which_domain(np.array(points), **params)

        # Create output points layer with new field
        fields = points_layer.fields()
        fields.append(QgsField(model.name, QVariant.String))
        output_layer, output_id = self.parameterAsSink(
            parameters,
            self.OUTPUT,
            context,
            fields,
            QgsWkbTypes.PointZ,
            points_layer.sourceCrs(),
        )
        unit_names = {v: k for k, v in params["ids"].items()}
        for feature, unit_id in zip(points_layer.getFeatures(), domains, strict=True):
            attributes = list(feature.attributes())
            attributes.append(unit_names[unit_id])
            new_feature = QgsFeature()
            new_feature.setAttributes(attributes)
            new_feature.setGeometry(feature.geometry())
            output_layer.addFeature(new_feature)

        return {self.OUTPUT: output_id}
