from pandas import read_excel
from qgis.core import (
    Qgis,
    QgsCoordinateReferenceSystem,
    QgsFeature,
    QgsFeatureSink,
    QgsField,
    QgsFields,
    QgsProcessingParameterFeatureSink,
    QgsProcessingParameterFile,
    QgsProcessingParameterNumber,
    QgsProcessingParameterString,
)

from . import UtilsAlgorithm


class PandasReadSpreadsheet(UtilsAlgorithm):
    """Read Excel spreadsheet as attribute table using pandas.

    see: https://pandas.pydata.org/docs/reference/api/pandas.read_excel.html
    """

    INPUT = "INPUT"
    HEADER = "HEADER"
    SHEET_NAME = "SHEET_NAME"
    SKIPROWS = "SKIPROWS"
    DECIMAL = "DECIMAL"
    OUTPUT = "OUTPUT"

    def displayName(self):
        return self.tr("Import spreadsheet")

    def initAlgorithm(self, config=None):  # noqa: ARG002
        self.addParameter(
            QgsProcessingParameterFile(
                self.INPUT,
                self.tr("Input spreadsheet file"),
                fileFilter="Spreadsheet file (*.xls *.xlsx *.xlsm *.xlsb *.odf *.ods *.odt);; All files (*.*)",
            )
        )

        self.addParameter(
            QgsProcessingParameterString(
                self.SHEET_NAME,
                self.tr("Sheet name/id"),
                defaultValue="0",
                optional=True,
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                self.HEADER,
                self.tr("Header row"),
                type=QgsProcessingParameterNumber.Integer,
                minValue=0,
                defaultValue=0,
                optional=True,
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                self.SKIPROWS,
                self.tr("Number of rows to skip (from header)"),
                type=QgsProcessingParameterNumber.Integer,
                minValue=0,
                defaultValue=0,
                optional=True,
            )
        )

        self.addParameter(
            QgsProcessingParameterString(
                self.DECIMAL,
                self.tr("Decimal separator"),
                defaultValue=".",
                optional=True,
            )
        )

        self.addParameter(
            QgsProcessingParameterFeatureSink(self.OUTPUT, self.tr("Output layer"))
        )

    def processAlgorithm(self, parameters, context, feedback):
        file = self.parameterAsFile(parameters, self.INPUT, context)
        sheet = self.parameterAsString(parameters, self.SHEET_NAME, context)
        header = self.parameterAsInt(parameters, self.HEADER, context)
        skiprows = self.parameterAsInt(parameters, self.SKIPROWS, context)
        decimal = self.parameterAsString(parameters, self.DECIMAL, context)

        header = header - 1 if header else None
        sheet = int(sheet) if sheet.isdigit() else sheet

        try:
            content = read_excel(
                file,
                sheet_name=sheet,
                header=header,
                skiprows=skiprows,
                decimal=decimal,
            )
        except Exception as err:
            feedback.reportError(err, fatalError=True)

        feedback.pushInfo(f"'{file}' parsed as pandas.DataFrame")

        fields = QgsFields([QgsField(h) for h in content])
        (sink, dest_id) = self.parameterAsSink(
            parameters,
            self.OUTPUT,
            context,
            fields,
            Qgis.WkbType.NoGeometry,
            QgsCoordinateReferenceSystem(),
        )

        total = len(content)
        for i, row in enumerate(content.itertuples()):
            if feedback.isCanceled():
                feedback.reportError("Algorithm was canceled !", fatalError=True)
            else:
                feedback.setProgress(i / total)

            f = QgsFeature(fields)
            f.setAttributes(list(row))
            sink.addFeature(f, QgsFeatureSink.FastInsert)

        return {self.OUTPUT: dest_id}
