import contextlib

from forgeo.core import ModellingUnit
from qgis.core import (
    QgsFeature,
    QgsField,
    QgsFields,
    QgsGeometry,
    QgsPluginLayer,
    QgsPoint,
    QgsProcessing,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterField,
    QgsProcessingParameterMapLayer,
    QgsProcessingParameterVectorDestination,
    QgsProject,
    QgsWkbTypes,
)
from qgis.PyQt.QtCore import QVariant

from ...layers.pile import PileLayer
from . import BoreholeAlgorithm


class ProcessingParameterForgeoPileLayer(QgsProcessingParameterMapLayer):
    """Helper class to use a Forgeo pile as Processing parameter"""

    def checkValueIsAcceptable(self, obj, context=None):  # noqa: ARG002
        if isinstance(obj, str):
            with contextlib.suppress(KeyError):
                obj = QgsProject.instance().layerStore().mapLayers()[obj]
        return (
            isinstance(obj, QgsPluginLayer)
            and obj.pluginLayerType() == PileLayer.LAYER_TYPE
        )


class WellLogsToMarkers(BoreholeAlgorithm):
    """This processing creates markers from a pile and well logs."""

    # Inputs parameters
    WELLDATA = "WELLDATA"
    WELL_ID = "WELL_ID"
    UNIT = "UNIT"
    ZWELL = "ZWELL"
    DEPTHTOP = "DEPTHTOP"
    PILE = "PILE"
    # Output parameters
    OUTPUT = "OUTPUT"

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr("Well logs to markers of contacts")

    def initAlgorithm(self, config=None):  # noqa: ARG002
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        self.addParameter(
            ProcessingParameterForgeoPileLayer(
                self.PILE,
                self.tr("ForGeo pile"),
            )
        )
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.WELLDATA,
                self.tr("Well data layer"),
                [QgsProcessing.TypeVectorPoint],
            )
        )
        self.addParameter(
            QgsProcessingParameterField(
                self.WELL_ID,
                self.tr("Fields of wells ids"),
                parentLayerParameterName=self.WELLDATA,
                type=QgsProcessingParameterField.String,
            )
        )
        self.addParameter(
            QgsProcessingParameterField(
                self.UNIT,
                self.tr("Field of units"),
                parentLayerParameterName=self.WELLDATA,
                type=QgsProcessingParameterField.String,
            )
        )
        self.addParameter(
            QgsProcessingParameterField(
                self.ZWELL,
                self.tr("Field of Z well head"),
                parentLayerParameterName=self.WELLDATA,
                type=QgsProcessingParameterField.Numeric,
            )
        )
        self.addParameter(
            QgsProcessingParameterField(
                self.DEPTHTOP,
                self.tr("Field of depth of top of units"),
                parentLayerParameterName=self.WELLDATA,
                type=QgsProcessingParameterField.Numeric,
            )
        )
        self.addParameter(
            QgsProcessingParameterVectorDestination(
                self.OUTPUT,
                self.tr("Points output"),
                optional=True,
                createByDefault=True,
            )
        )

    def processAlgorithm(self, parameters, context, feedback) -> dict:  # noqa : ARG002
        """Here is where the processing itself takes place"""

        # Get input data
        pile = (
            QgsProject.instance().layerStore().mapLayers()[parameters[self.PILE]].pile
        )
        points_layer = self.parameterAsVectorLayer(parameters, self.WELLDATA, context)
        well_ids_fieldname = self.parameterAsString(parameters, self.WELL_ID, context)
        unit_fieldname = self.parameterAsString(parameters, self.UNIT, context)
        zwellhead_fieldname = self.parameterAsString(parameters, self.ZWELL, context)
        ztop_fieldname = self.parameterAsString(parameters, self.DEPTHTOP, context)

        # Create markers
        markers = []
        current_well_id = ""
        previous_block = None
        for block in points_layer.getFeatures():
            well_id = block[well_ids_fieldname]
            if well_id != current_well_id:
                current_well_id = well_id
                previous_block = None
            if previous_block is None:
                previous_block = block
                continue
            formation_above = previous_block[unit_fieldname]
            formation_below = block[unit_fieldname]
            previous_block = block
            if (
                formation_above == formation_below
                or not formation_above
                or not formation_below
                or not pile[formation_above]
                or not pile[formation_below]
            ):
                continue
            new_marker = []
            for pt in block.geometry().vertices():
                new_marker.append(pt.x())
                new_marker.append(pt.y())
            new_marker.append(
                block[zwellhead_fieldname] - block[ztop_fieldname]
            )  # Z coordinate
            new_marker.append(block[well_ids_fieldname])
            new_marker.append(block[zwellhead_fieldname])
            new_marker.append(formation_above)
            new_marker.append(formation_below)
            new_marker.append(get_contact(pile, formation_below, formation_above))
            markers.append(new_marker)

        # Create output markers layer
        fields = QgsFields()
        fields.append(QgsField("X", QVariant.Double))
        fields.append(QgsField("Y", QVariant.Double))
        fields.append(QgsField("Z", QVariant.Double))
        fields.append(QgsField(well_ids_fieldname, QVariant.String))
        fields.append(QgsField(zwellhead_fieldname, QVariant.String))
        fields.append(QgsField("Above", QVariant.String))
        fields.append(QgsField("Below", QVariant.String))
        contact_fieldname = f"Contacts in {pile.name}"
        fields.append(QgsField(contact_fieldname, QVariant.String))
        output_layer, output_id = self.parameterAsSink(
            parameters,
            self.OUTPUT,
            context,
            fields,
            QgsWkbTypes.PointZ,
            points_layer.sourceCrs(),
        )
        for marker in markers:
            new_feature = QgsFeature()
            new_feature.setFields(fields)
            new_feature.setGeometry(QgsGeometry.fromPoint(QgsPoint(*marker[:3])))
            new_feature.setAttributes(marker)
            output_layer.addFeature(new_feature)

        return {self.OUTPUT: output_id}


def get_contact(pile, formation_below, formation_above):
    """Returns the name of the contact between these formations according to the pile,
    whether it is an erosion or a normal contact
    """
    f_below = pile[formation_below]
    f_above = pile[formation_above]
    idx_below = pile.description.index(f_below)
    idx_above = pile.description.index(f_above)
    if idx_below > idx_above:
        return "Not found"
    erosion = pile.get_erosion_between(formation_below, formation_above)
    if erosion is not None:
        return erosion.name
    # Return the oldest contact above formation_old
    idx = idx_below + 1
    while not isinstance(pile.description[idx], ModellingUnit):
        idx += 1
    return f"Contact {f_below.name} - {pile.description[idx].name}"
