from __future__ import annotations

from functools import cached_property

from qgis.core import QgsApplication, QgsProject
from qgis.PyQt import sip
from qgis.PyQt.QtCore import QObject
from qgis.PyQt.QtWidgets import QAction, QFileDialog, QMessageBox, QWidget

from .layers import (
    FaultNetworkLayerType,
    GmLayer,
    GmLayerType,
    ModelLayerType,
    PileLayerType,
)
from .settings import PluginSettings
from .toolbox import PluginToolbox
from .utils import qicon
from .widgets.faults_widget import FaultNetworkEditionDialog
from .widgets.model_widget import ModelEditionDialog
from .widgets.pile_widget import PileEditionDialog


class PluginInstance(QObject):
    """QGIS plugin instance."""

    @cached_property
    def settings(self):
        return PluginSettings()

    # Required : triggered by qgis.utils.loadPlugin()
    def __init__(self, iface):
        """Constructor (called at plugin load).

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """

        # Call the Qt.QObject constructor
        super().__init__(QgsApplication.instance())
        # Save reference to the QGIS interface
        self.iface = iface
        # Give the QObject the slug of the project
        self.objectName = self.pluginName = self.settings.get("name")
        # Register the custom layer types
        self.layer_types = [
            GmLayerType(),
            PileLayerType(),
            FaultNetworkLayerType(),
            ModelLayerType(),
        ]
        for t in self.layer_types:
            QgsApplication.pluginLayerRegistry().addPluginLayerType(t)

    # Required : triggered by qgis.utils.startPlugin()
    def initGui(self):
        """Called at plugin start (not load)"""

        self.initProcessings()

        # MENUS:
        #   Extensions > ForGEO > ...
        self.menu = self.iface.pluginMenu().addMenu(
            qicon(),
            self.pluginName,
        )
        self.menu.addAction(self.about)
        self.menu.addSeparator()
        self.menu.addAction(self.add_geomodel)
        self.menu.addSeparator()
        self.menu.addAction(self.create_pile)
        self.menu.addAction(self.create_model)
        self.menu.addAction(self.create_faultnet)
        #   Layer > Create Layer > ...
        self.iface.newLayerMenu().addAction(self.create_pile)
        #   Layer > Add Layer > ...
        self.iface.addLayerMenu().addAction(self.add_geomodel)

        # TOOLBARS
        #   Plugin toolbar > ...
        self.iface.addToolBarIcon(self.create_pile)
        self.iface.addToolBarIcon(self.create_faultnet)
        self.iface.addToolBarIcon(self.create_model)

    # Required : triggered by qgis.utils.unloadPlugin()
    def unload(self):
        """Removes the plugin menus/toolbars/toolboxes/events"""

        # MENUS:
        #   Extensions > ForGEO > ...
        self.iface.pluginMenu().removeAction(self.menu.menuAction())
        #   Layer > Create Layer > ...
        self.iface.newLayerMenu().addAction(self.create_pile)
        self.iface.newLayerMenu().addAction(self.create_faultnet)
        self.iface.newLayerMenu().addAction(self.create_model)
        #   Layer > Add Layer > ...
        self.iface.addLayerMenu().addAction(self.add_geomodel)

        # TOOLBARS
        #   Plugin toolbar > ...
        self.iface.pluginToolBar().removeAction(self.create_pile)
        self.iface.pluginToolBar().removeAction(self.create_faultnet)
        self.iface.pluginToolBar().removeAction(self.create_model)

        # remove the processing toolbox
        if self.toolbox and not sip.isdeleted(self.toolbox):
            QgsApplication.instance().processingRegistry().removeProvider(self.toolbox)

        # unregistered custom layers
        for t in self.layer_types:
            QgsApplication.pluginLayerRegistry().removePluginLayerType(t.name())

        # delete
        self.deleteLater()

    def initProcessings(self):
        """Register the plugin toolbox (processings provider)"""
        self.toolbox = PluginToolbox()
        if self.toolbox.algorithms:
            QgsApplication.instance().processingRegistry().addProvider(self.toolbox)

    @cached_property
    def about(self) -> QAction:
        """Display plugin details"""

        def callback():
            dlg = QWidget(self.iface.mainWindow())
            dlg.setWindowIcon(qicon())

            content = f"<b>{self.pluginName}</b><br>"
            content += f"<i>{self.settings.get('description')}</i><br><br>"
            content += f"{self.settings.get('about')}<br><br>"

            content += f"<b>Author</b> : <a href=mailto:{self.settings.get('email')}>{self.settings.get('author')}</a><br>"
            if homepage := self.settings.get("homepage", None):
                content += f"<b>Homepage</b> : <a href={homepage}>GitLab</a><br>"
            content += f"<b>Source code</b> : <a href={self.settings.get('repository')}>GitLab</a><br>"
            if tracker := self.settings.get("tracker", None):
                content += f"<b>Repport issue</b> : <a href={tracker}>GitLab</a><br>"
            if doc := self.settings.get("documentation", None):
                content += f"<b>Documentation</b> : <a href={doc}>GitLab</a><br>"

            content += f"<br><b>Version</b> : {self.settings.get('version')}<br>"
            content += f"<i><b>Requires</b> QGIS &#x02265; {self.settings.get('qgisMinimumVersion')}</i><br>"
            if plugins := self.settings.get("plugin_dependencies"):
                content += f"<i><b>Depends on</b> : {plugins.strip(',')}</i><br>"
            if deps := self.settings.get("pip_dependencies"):
                content += f"<i><b>Requirements</b> : {deps.strip(',')}</i><br>"

            QMessageBox.about(dlg, "About", content)
            dlg.deleteLater()

        action = QAction(
            qicon("SP_FileDialogInfoView"), f"About {self.pluginName}", parent=self
        )
        action.triggered.connect(callback)
        return action

    @cached_property
    def create_pile(self) -> QAction:
        """Open widget to create a pile"""

        def callback():
            PileEditionDialog.new()

        action = QAction(qicon("add_pile"), "Create new stratigraphy", parent=self)
        action.triggered.connect(callback)
        return action

    @cached_property
    def create_model(self) -> QAction:
        """Open widget to create a model : associate the pile with data and interpolation parameters"""

        def callback():
            ModelEditionDialog.new()

        action = QAction(qicon("add_model"), "Create new model", parent=self)
        action.triggered.connect(callback)
        return action

    @cached_property
    def create_faultnet(self) -> QAction:
        """Open widget to create a fault network"""

        def callback():
            FaultNetworkEditionDialog.new()

        action = QAction(
            qicon("add_faultnetwork"), "Create new structural schema", parent=self
        )
        action.triggered.connect(callback)
        return action

    @cached_property
    def add_geomodel(self) -> QAction:
        """Open a GeoModeller project"""

        def callback():
            for file in QFileDialog.getOpenFileNames(
                parent=None,
                caption="Select a geological model",
                filter="GeoModeller files (*.xml)",
            )[0]:
                layer = GmLayer(source=file)
                if layer.isValid():
                    QgsProject.instance().addMapLayer(layer)

        action = QAction(qicon("add_geomodel"), "Open GeoModeller project", parent=self)
        action.triggered.connect(callback)
        return action
